/**
 * @version $Id: map.js 31235 2014-09-06 15:49:40Z astarostin $
 * ------------------------------------------------------------------------------
 * Map camera calibration tool
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2013
 * ------------------------------------------------------------------------------
 */

define("geozoneconfigurator", ["module", "api", "log", "jquery", "jquery.mask", "jquery-ajaxfileupload", "mapzoneeditor"], function(module) {
	"use strict";

	/**
	 * @type {MapZoneEditor}
	 */
	var MapZoneEditor = require("mapzoneeditor");
	var cameraObj = module.config().obj;

	var buildingList = [];

	var api = new API();
	/**
	 * @type {MapZoneEditor}
	 */
	var mapZoneEditor = null;

	$(document).ready(function () {
		$("#target_zone_manager").mask();

		// only for identity, for now
		if (cameraObj != 53)
		{
			$("#object_list").hide();
		}

		mapZoneEditor = new MapZoneEditor(cameraObj, ".map");

		mapZoneEditor.init(null, false)
			.fail(function () {
				Log.error("Cannot init map");
			})
			.done(function (responseGetBlock) {
				// only for identity for now
				if (cameraObj == 53)
				{
					loadList(cameraObj);
				}

				$("#target_zone_manager").unmask();

				$("#remove_floor").click(function () {
					var buildingNumber = $("#building_list select option:selected").val();
					var floorNumber = $("#floor_list select option:selected").val();
					if (buildingNumber && floorNumber)
					{
						removeFloor(buildingNumber, floorNumber);
					}
				});

				$("#floor_list").click(function(){
					var buildingNumber = $("#building_list select option:selected").val();
					var floorNumber = $("#floor_list select option:selected").val();
					if (buildingNumber && floorNumber)
					{
						mapZoneEditor.hideAllExceptFloor(buildingNumber, floorNumber);
						mapZoneEditor.panToFloor(buildingNumber, floorNumber);
					}
				});

				$("#floor_list button.add").click(function () {
					$("#object_list").mask();
					var obj = cameraObj;
					var name = $("#floor_list input.name").val();
					if (name.length == 0)
					{
						Log.error("Please specify floor name");
						$("#object_list").unmask();
						return;
					}

					var buildingNumber = $("#building_list select option:selected").val();

					$.ajaxFileUpload({
						url:           '/api/call/setBlock',
						secureuri:     false,
						fileElementId: 'block',
						dataType:      'json',
						data: {
							"obj":          obj,
							"description":  name,
							"fromPDFToPNG": true,
							"token":        getToken()
						},
						success: function (data, status) {
							if (data.error != '')
							{
								alert(__("Error on file upload: ") + data.error);
							} else {
								var blockId = data.blockId;
								var url = data.url;

								Log.info(__("Floor added"));

								addFloorToList(buildingNumber, blockId, name, url);
							}

							$("#object_list").unmask();
						},
						error: function (data, status, e) {
							Log.error(e);

							$("#object_list").unmask();
						}
					});
				});

				$(".zoneList input[type=radio][name=zoneType]").change(function(){
					var type = $(this).val();

					var types = {
						include: mapZoneEditor.ZONE_INCLUDE,
						exclude: mapZoneEditor.ZONE_EXCLUDE
					};
					var zoneType = types[type];
					mapZoneEditor.changeZoneType(zoneType);
				});

				mapZoneEditor.getMap().on('draw:drawstart draw:editstart', function (e) {
					$(".panel .zoneList .header").trigger("click");
				});

				mapZoneEditor.getMap().on('draw:edited draw:deleted', function (e) {
					var layers = e.layers;
					layers.eachLayer(function (layer) {
						console.log(layer);
					});
				});

				$(".zoneList select").on("change", function(){
					var index = $(this).find("option:selected").val();
					var layer = mapZoneEditor.getEditableLayers().getLayers()[index];

					// async call to prevent freeze of map, issue of leafletjs
					setTimeout(function(){
						var bounds = layer.getBounds();
						if (bounds.isValid())
						{
							mapZoneEditor.getMap().fitBounds(bounds);
						}
					}.bind(this), 0);
				});

				var layers = mapZoneEditor.getEditableLayers().getLayers();
				layers.forEach(function(zone, index){
					$(".zoneList select").append('<option value="' + index + '">' + index + '</option>');
				});
			});

		// accordion
		$(".panel").on("click", ".element .header", function(){
			var parent = $(this).parent();
			if (!parent.hasClass("open"))
			{
				$(".panel .element.open").removeClass("open");
				parent.addClass("open");

				$(".panel .element:not(.open) .content").slideUp();

				$(".panel .element.open .content").slideDown();
			}
		});

		$(".buttons button.back").click(function () {
			window.history.go(-1);
		});

		$(".buttons button.cancel").click(function () {
			window.location.reload();
		});

		$(".buttons button.apply").click(function () {
			mapZoneEditor.savePrimitives();

			// only for identity for now
			if (cameraObj == 53)
			{
				mapZoneEditor.saveLayers();
				mapZoneEditor.saveDefaultPosition();
			}
		});

		$("#target_list select").change(function () {
			var obj = $(this).find("option:selected").data("obj");
			var id = $(this).find("option:selected").data("targetid");

			mapZoneEditor.panToTarget(obj, id);
		});

		$("#target_zone_manager button.pantocamera").click(function () {
			var position = mapZoneEditor.getCameraPosition();
			mapZoneEditor.panTo(position);
		});

		$("#building_list button.add").click(function(){
			var name = $("#building_list input.name").val();
			if (!name)
			{
				Log.warning(__("Fill building name"));
				return;
			}

			addBuildingToList(name);
		});

		$("#building_list select").on("change", function(){
			var buildingNumber = $(this).find("option:selected").val();

			$("#floor_list").removeClass("hidden");
			$("#floor_list select option").remove();

			var floorList = buildingList[buildingNumber].floorList;

			for (var i = 0; i < floorList.length; i++)
			{
				var floor = floorList[i];
				$("#floor_list select").append('<option value="' + i + '">' + floor.name + '</option>');
			}
		});

		$("#building_list button.remove").click(function(){
			var buildingNumber = $("#building_list select").find("option:selected").val();

			if (buildingNumber)
			{
				removeBuilding(buildingNumber);
			}
		});
	});

	/**
	 * load list of layers associated with object
	 *
	 * @param {number} obj
	 */
	function loadList(obj)
	{
		$.when(
			api.getAttributes({obj: obj}),
			api.getBlocksInfo({obj: obj})
		)
			.done(function(responsegetAttribute, responseBlockInfo){
				var attributes = responsegetAttribute[0].list;

				var LAYER_LIST = JSON.parse(attributes.LAYER_LIST);

				/*
				Layer list example:

				var list = {
					objects: [{
						name: "Building 102",
						layers: [{
							name: "Floor 1",
							blockId: 5,
							polygon: {},
							objects: {"101": {
								position: [lat, lng]
							}}
						}]
					}]
				};
				*/

				var blockList = responseBlockInfo[0].list;
				var objectList = LAYER_LIST.objects ? LAYER_LIST.objects : [];
				for (var i = 0; i < objectList.length; i++)
				{
					var building = objectList[i];
					var name = building.name;
					var floors = building.layers;

					addBuildingToList(name);

					for (var j = 0; j < floors.length; j++)
					{
						var floor = floors[j];
						var layerName = floor.name;
						var layerBlockId = floor.blockId;
						var layerPolygon = floor.polygon;
						var layerObjects = floor.objects;

						if (!blockList[layerBlockId])
						{
							continue;
						}

						var block = blockList[layerBlockId];
						var mime = block.mime;
						var url = block.url;
						var description = block.description;

						// only image/png image of floor for now
						addFloorToList(i, layerBlockId, layerName, url, layerPolygon);
					}
				}
			});
	}

	/**
	 *
	 * @param {string} name
	 */
	function addBuildingToList(name)
	{
		var building = {
			name: name,
			floorList: []
		};

		buildingList.push(building);

		var buildingNumber = mapZoneEditor.addBuilding(building.name);

		$("#building_list select").append('<option value="' + buildingNumber + '">' + building.name + '</option>');
	}

	/**
	 *
	 * @param {number} buildingNumber
	 * @param {number} blockId
	 * @param {string} name
	 * @param {string} url
	 * @param {L.Polygon} [polygon]
	 */
	function addFloorToList(buildingNumber, blockId, name, url, polygon)
	{
		var floor = {
			blockId: blockId,
			name: name,
			url: url
		};

		buildingList[buildingNumber].floorList.push(floor);

		var floorNumber = mapZoneEditor.addFloor(buildingNumber, floor.blockId, floor.name, floor.url, polygon);

		$("#floor_list select").append('<option value="' + floorNumber + '">' + floor.name + '</option>');
	}

	/**
	 *
	 * @param {number} buildingNumber
	 */
	function removeBuilding(buildingNumber)
	{
		var blockIdList = [];
		var floorList = buildingList[buildingNumber].floorList;
		for (var i = 0; i < floorList.length; i++)
		{
			var floor = floorList[i];
			blockIdList.push(floor.blockId);
		}

		$("#object_list").mask();
		api.deleteBlocks({
			obj: cameraObj,
			blockIdList: JSON.stringify(blockIdList)
		})
			.fail(function (code, message) {
				Log.error("[" + code + "] " + message);
			})
			.done(function (response) {
				Log.info(__("Building is removed"));

				mapZoneEditor.removeBuilding(buildingNumber);

				$('#building_list select option[value="' + buildingNumber + '"]').remove();

				buildingList.splice(buildingNumber, 1);
			})
			.always(function () {
				$("#object_list").unmask();
			});
	}

	/**
	 *
	 * @param {number} buildingNumber
	 * @param {number} floorNumber
	 */
	function removeFloor(buildingNumber, floorNumber)
	{
		var blockId = buildingList[buildingNumber].floorList[floorNumber].blockId;
		$("#object_list").mask();
		api.deleteBlocks({
			obj: cameraObj,
			blockIdList: JSON.stringify([blockId])
		})
			.fail(function (code, message) {
				Log.error("[" + code + "] " + message);
			})
			.done(function (response) {
				Log.info(__("Floor is removed"));

				mapZoneEditor.removeFloor(buildingNumber, floorNumber);

				$('#floor_list select option[value="' + floorNumber + '"]').remove();

				buildingList[buildingNumber].floorList.splice(floorNumber, 1);
			})
			.always(function () {
				$("#object_list").unmask();
			});
	}
});
