/**
 * @version $id:$
 * ------------------------------------------------------------------------------
 * GEO calibrator
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2013
 * ------------------------------------------------------------------------------
 */

define("geocalibratorptz", ["jquery", "api", "mediaplayer", "flatmap", "utils", "/sdi/ge/js/geo_lib.js"], function(){
	"use strict";

	/**
	 * @type {FlatMap}
	 */
	var FlatMap = require("flatmap");

	/**
	 * @constructor
	 */
	function GEOCalibratorPTZ()
	{
		/**
		 * @type {API}
		 * @private
		 */
		this._api = new API();

		/**
		 * @type {{obj: null|number, player: MediaPlayer, attributes: {}}}
		 * @private
		 */
		this._player = {
			obj: null,
			player: new MediaPlayer(),
			attributes: {}
		};

		/**
		 * @type {FlatMap}
		 * @private
		 */
		this._map = new FlatMap(".player .map");

		/**
		 * @type {L.Marker}
		 * @private
		 */
		this._marker = null;

		/**
		 * @type {L.Polyline}
		 * @private
		 */
		this._polyline = null;

		/**
		 * @type {L.FeatureGroup}
		 * @private
		 */
		this._testLayer = null;

		/**
		 * @type {number}
		 * @private
		 */
		this._pointColor = 0xFFFF0000;

		/**
		 * @type {number}
		 * @private
		 */
		this._selectedPointColor = 0xFF00FF00;

		/**
		 * @type {Object}
		 * @private
		 */
		this._config = {};

		/**
		 * @type {number}
		 * @private
		 */
		this._currentConfigObj = null;

		/**
		 * @type {MediaPlayer}
		 * @private
		 */
		this._associatedPlayer = this._player.player;

		this.DEG2RAD = Math.PI / 180;
		this.RAD2DEG = 180 / Math.PI;
		this.M = [];

		this.internal = [];
		this.external = [];

		this.scale = 0;
		this.math_config = {number: 'number'};
		//this.math = mathjs(this.math_config);

		this._isTest = false;
		this._isCameraPlace = false;

		this._attributes = {};

		this._offsetAngleFromNorth = 0;
		this._aPan = 0;
		this._offsetTilt = 0;
		this._geMapAngle = 0;

		this._showAttributes = {
			"CAM_GEO_LAT": "Latitude [deg]",
			"CAM_GEO_LONG": "Longitude [deg]",
			"CAM_GEO_ALT": "Altitude [deg]",
			"CAM_GEO_NORTH_OFFSET": "North offset angle"
		};
	}

	/**
	 * @param {number} obj
	 */
	GEOCalibratorPTZ.prototype.init = function(obj)
	{
		var self = this;

		this._player.obj = obj;
		this._currentConfigObj = obj;

		$.when(
			this._api.getAttributes({obj: obj}),
			this._api.getIdentityAttributes(),
			this._player.player.init(".player .video")
		)
			.fail(function(code, message){
				Log.error(message);
			})
			.done(function(getAttributesResponse, responseIdentityAttributes, playerInitResponse){
				var identity = responseIdentityAttributes[0].list;
				if (!playerInitResponse)
				{
					self._player.player.subscribe("frame", function(timestamp, width, height){
						this.drawCrosshair(width, height, self._selectedPointColor);

						var date = new Date();
						date.setTime(timestamp);
						$(".player .timestamp").val(date.getHours() + ":" + date.getMinutes() + ":" + date.getSeconds() + "." + date.getMilliseconds());
					}, "player");
				}

				self._player.attributes = getAttributesResponse[0].list;

				self._attributes = self._player.attributes;
				var CAM_GEO_LAT = parseFloat(self._player.attributes.CAM_GEO_LAT);
				var CAM_GEO_LONG = parseFloat(self._player.attributes.CAM_GEO_LONG);
				var zoomLevel = self._map.DEFAULT_ZOOM_LEVEL;

				if (isNaN(CAM_GEO_LAT) || isNaN(CAM_GEO_LONG))
				{
					var DEFAULT_POSITION = JSON.parse(identity.DEFAULT_POSITION);

					CAM_GEO_LAT = DEFAULT_POSITION.lat;
					CAM_GEO_LONG = DEFAULT_POSITION.lng;
					zoomLevel = DEFAULT_POSITION.zoomLevel;
				}

				self._map.initMap(CAM_GEO_LAT, CAM_GEO_LONG, zoomLevel)
					.done(function(){
						var position = [CAM_GEO_LAT, CAM_GEO_LONG];
						var caption = "[" + obj + "] " + self._player.attributes.NAME;

						self._map.addCamera(obj, position, caption);

						self._marker = new L.Marker(position)
							.addTo(self._map.getMap());

						self._polyline = new L.Polyline(
							[self._map.getCameraMarker().getLatLng(), self._marker.getLatLng()],
							{color: 'green'})
							.addTo(self._map.getMap());

						self._map.getMap().on("click", function(e){
							if (self._isCameraPlace)
							{
								self._map.getCameraMarker().setLatLng(e.latlng);

								self.setCameraEditorParameterValue("CAM_GEO_LAT", e.latlng.lat);
								self.setCameraEditorParameterValue("CAM_GEO_LONG", e.latlng.lng);
							} else {
								self._marker.setLatLng(e.latlng);
							}

							var cameraPos = self._map.getCameraMarker().getLatLng();
							var markerPos = self._marker.getLatLng();

							self._geMapAngle = angleBy2Points(
								cameraPos.lat * Math.PI / 180,
								cameraPos.lng * Math.PI / 180,
								markerPos.lat * Math.PI / 180,
								markerPos.lng * Math.PI / 180
							) * 180 / Math.PI;

							var tmpA = self.correctAngle(-self._geMapAngle);

							var S = cameraPos.distanceTo(markerPos);
							self._offsetTilt = -Math.atan2(self._attributes["CAM_GEO_ALT"], S) * 180 / Math.PI;

							self.ptzAbsoluteChange(obj, tmpA, Math.round(self._offsetTilt), "abs");

							self._polyline.setLatLngs([cameraPos, markerPos]);
						});
					});

				self._player.player.play(obj);
			});

		$(".buttons button.back").click(function (){
			window.history.go(-1);
		});

		$(".buttons button.cancel").click(function (){
			window.location.reload();
		});

		$(".buttons button.apply").click(function(){

			for (var attribute in self._showAttributes)
			{
				if (isNaN(self._attributes[attribute]))
				{
					Log.error(__("Please specify: ") + self._showAttributes[attribute]);
					return;
				}
			}
			self._api.setAttributes({
				obj: self._player.obj,
				attributes: JSON.stringify({
					CAM_GEO_LAT: self._attributes.CAM_GEO_LAT,
					CAM_GEO_LONG: self._attributes.CAM_GEO_LONG,
					CAM_GEO_ALT: self._attributes.CAM_GEO_ALT,
					CAM_GEO_NORTH_OFFSET: self._attributes.CAM_GEO_NORTH_OFFSET
				})
			})
				.fail(function (code, message) {
					Log.error("[" + code + "]" + __(message));
				})
				.done(function () {
					Log.info(__("Configuration saved"));
				});
		});

		$(".pointList button.camera").click(function(){
			self._isCameraPlace = true;
			self._map.getCameraMarker().dragging.enable();

			Log.info("Click on map to place camera");
		});

		$(".pointList button.marker").click(function(){
			self._isCameraPlace = false;
			self._map.getCameraMarker().dragging.disable();
		});

		$(".north_offset_minus_20").click(function(){
			self.cameraPtzCalibrate(-20);
		});
		$(".north_offset_minus_5").click(function(){
			self.cameraPtzCalibrate(-5);
		});
		$(".north_offset_minus_05").click(function(){
			self.cameraPtzCalibrate(-0.5);
		});

		$(".north_offset_plus_05").click(function(){
			self.cameraPtzCalibrate(0.5);
		});
		$(".north_offset_plus_5").click(function(){
			self.cameraPtzCalibrate(5);
		});
		$(".north_offset_plus_20").click(function(){
			self.cameraPtzCalibrate(20);
		});

		$(".tilt_minus_5").click(function(){
			self.offsetTiltChange(-5);
		});
		$(".tilt_minus_05").click(function(){
			self.offsetTiltChange(-0.5);
		});

		$(".tilt_plus_05").click(function(){
			self.offsetTiltChange(0.5);
		});
		$(".tilt_plus_5").click(function(){
			self.offsetTiltChange(5);
		});
	};

	GEOCalibratorPTZ.prototype.cameraPtzCalibrate = function(angle)
	{
		this._aPan += angle;
		//if(this._aPan > this._aPan) this._aPan = 180;
		//if(this._aPan < -180) this._aPan = -180;
		if (this._aPan > 180)
		{
			this._aPan = -360 + this._aPan;
		}
		if (this._aPan < -180)
		{
			this._aPan = 360 - this._aPan;
		}
		var aGE = -this._geMapAngle;
		if (this._geMapAngle > 180)
		{
			aGE = 360 - this._geMapAngle;
		}

		this._offsetAngleFromNorth = -Math.round(aGE - this._aPan);
		var mod = Math.abs(this._offsetAngleFromNorth);
		var sig = this._offsetAngleFromNorth / mod;
		if (mod > 180)
		{
			this._offsetAngleFromNorth = (-sig) * (360 - mod);
		}

		this.setCameraEditorParameterValue("CAM_GEO_NORTH_OFFSET", this._offsetAngleFromNorth);

		this.ptzAbsoluteChange(this._player.obj, Math.round(this._aPan), Math.round(this._offsetTilt), "abs");
	};

	GEOCalibratorPTZ.prototype.offsetTiltChange = function(inc)
	{
		var cameraPos = this._map.getCameraMarker().getLatLng();
		var markerPos = this._marker.getLatLng();

		this._offsetTilt += inc;
		if (this._offsetTilt > 90)
		{
			this._offsetTilt = 90;
		}
		if (this._offsetTilt < -90)
		{
			this._offsetTilt = -90;
		}

		var cameraLat = cameraPos.lat * Math.PI / 180;
		var cameraLng = cameraPos.lng * Math.PI / 180;
		var lat = markerPos.lat * Math.PI / 180;
		var lng = markerPos.lng * Math.PI / 180;

		var S = cameraPos.distanceTo(markerPos);

		var alt = Math.round(Math.abs(Math.tan(this._offsetTilt * Math.PI / 180) * S) * 1000) / 1000; // round to mm
		this.setCameraEditorParameterValue("CAM_GEO_ALT", alt);

		this.ptzAbsoluteChange(this._player.obj, Math.round(this._aPan), Math.round(this._offsetTilt), "abs");
	};

	GEOCalibratorPTZ.prototype.correctAngle = function(geMapAngle)
	{
		var angle = geMapAngle;
		if (geMapAngle > 180)
		{
			angle = 360 - geMapAngle;
		}
		angle = Math.round(angle + this._offsetAngleFromNorth);
		if (Math.abs(angle) > 180)
		{
			if (angle > 0)
			{
				angle = -(360 - angle);
			} else {
				angle = 360 + angle;
			}
		}

		return angle;
	};

	GEOCalibratorPTZ.prototype.ptzAbsoluteChange = function(obj, pan, tilt, mode)
	{
		var pt = "<PTZ_Command>do.ptz?dev=" + obj + "&mode=" + mode + "&pt=" + pan + "," + tilt + "</PTZ_Command>";

		$.ajax({
			url: "/ptz/cgi-bin/send_message.pl",
			data: {data: pt},
			success: function () {}
		});
	};

	GEOCalibratorPTZ.prototype.setCameraEditorParameterValue = function(param, value)
	{
		this._attributes[param] = value;

		this.showAttribute(param, value);
	};

	/**
	 * @param {string} attribute
	 * @param {string} value
	 */
	GEOCalibratorPTZ.prototype.showAttribute = function(attribute, value)
	{
		if (!this._showAttributes[attribute])
		{
			return;
		}

		$("#attribute_wrapper div.attribute[data-attr='" + attribute + "']").html(value);
	};

	return GEOCalibratorPTZ;
});
