/**
 * @version $id:$
 * ------------------------------------------------------------------------------
 * GEO calibrator
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2013
 * ------------------------------------------------------------------------------
 */

define("geocalibrator", ["jquery", "api", "mediaplayer", "flatmap", "pointlist", "utils"], function(){
	"use strict";

	/**
	 * @type {FlatMap}
	 */
	var FlatMap = require("flatmap");
	/**
	 * @type {PointList}
	 */
	var PointList = require("pointlist");
	//var mathjs = require("mathjs");

	/**
	 * @constructor
	 * @augments PointList
	 */
	function GEOCalibrator()
	{
		/**
		 * @type {API}
		 * @private
		 */
		this._api = new API();

		/**
		 * @type {{obj: null|number, player: MediaPlayer, attributes: {}}}
		 * @private
		 */
		this._player = {
			obj: null,
			player: new MediaPlayer(),
			attributes: {}
		};

		/**
		 * @type {FlatMap}
		 * @private
		 */
		this._map = new FlatMap(".player .map");

		/**
		 * @type {L.Marker}
		 * @private
		 */
		// this._marker = new L.Marker([0,0]);

		/**
		 * @type {L.FeatureGroup}
		 * @private
		 */
		this._testLayer = null;

		/**
		 * @type {number}
		 * @private
		 */
		this._pointColor = 0xFFFF0000;

		/**
		 * @type {number}
		 * @private
		 */
		this._selectedPointColor = 0xFF00FF00;

		/**
		 * @type {Object}
		 * @private
		 */
		this._config = {};

		/**
		 * @type {number}
		 * @private
		 */
		this._currentConfigObj = null;

		/**
		 * @type {MediaPlayer}
		 * @private
		 */
		this._associatedPlayer = this._player.player;

		this.DEG2RAD = Math.PI / 180;
		this.RAD2DEG = 180 / Math.PI;
		this.M = [];

		this.internal = [];
		this.external = [];

		this.scale = 0;
		this.math_config = {number: 'number'};
		//this.math = mathjs(this.math_config);

		this._isTest = false;
		this._isCameraPlace = false;
	}

	Utils.inherit(GEOCalibrator, PointList);

	/**
	 * @param {number} obj
	 */
	GEOCalibrator.prototype.init = function(obj)
	{
		var self = this;

		this._player.obj = obj;
		this._currentConfigObj = obj;

		$.when(
			this._api.getAttributes({obj: obj}),
			this._api.getIdentityAttributes(),
			this._player.player.init(".player .video")
		)
			.fail(function(code, message){
				Log.error(message);
			})
			.done(function(getAttributesResponse, responseIdentityAttributes, playerInitResponse){
				var identity = responseIdentityAttributes[0].list;
				if (!playerInitResponse)
				{
					self._player.player.subscribe("frame", function(timestamp, width, height){
						this.drawCrosshair(width, height, self._selectedPointColor);

						var date = new Date();
						date.setTime(timestamp);
						$(".player .timestamp").val(date.getHours() + ":" + date.getMinutes() + ":" + date.getSeconds() + "." + date.getMilliseconds());
					}, "player");

					self._player.player.subscribe("imageclick", function(width, height, x, y){
						x = Math.round(x);
						y = Math.round(y);

						var pointColor = self._pointColor;
						var pointWidth = 20;
						var pointHeight = 20;
						var pointX = x;
						var pointY = y;

						if (self._isTest)
						{
							var global = self.screenToGlobal(pointX, pointY, width, height);
							if (global)
							{
								self._marker.setLatLng(global);
							}
						} else
						if (!self._isCameraPlace)
						{
							var pointId = self._addPoint(pointX, pointY, pointWidth, pointHeight, pointColor);
							var text = pointId + ". (" + pointX + ", " + pointY + ")";

							$(".pointList select.list").append('<option value="' + pointId + '">' + text + '</option>');
							$(".pointList select.list option").prop("selected", false);
							$(".pointList select.list option[value=" + pointId + "]")
								.prop("selected", true)
								.change();
						}
					}, "zoom");
				}

				self._player.attributes = getAttributesResponse[0].list;

				var GEO_CALIBRATOR_CONFIG = JSON.parse(self._player.attributes.GEO_CALIBRATOR_CONFIG);
				if (!$.isPlainObject(GEO_CALIBRATOR_CONFIG))
				{
					GEO_CALIBRATOR_CONFIG = {};
				}
				self._config = GEO_CALIBRATOR_CONFIG;

				var pointList = self._getCurrentPointList();
				if (!$.isEmptyObject(pointList))
				{
					for (var pointId in pointList)
					{
						if (!pointList.hasOwnProperty(pointId))
							continue;

						var point = pointList[pointId];

						point.primitiveList[0].id = self._addEllipse(null, point.x - point.width / 2, point.y - point.height / 2, point.width, point.height, point.color);
						point.primitiveList[1].id = self._addText(null, String(pointId), point.x + point.width, point.y + point.height, point.height, point.color);

						var text = pointId;
						if (point.position)
						{
							text = pointId + ". (" + point.x + ", " + point.y + ")" + " (" + point.position.lat + ", " + point.position.lng + ")";
						}
						$(".pointList select.list").append('<option value="' + pointId + '">' + text + '</option>');
					}

					$(".player .pointList").show();
				}

				var CAM_GEO_LAT = parseFloat(self._player.attributes.CAM_GEO_LAT);
				var CAM_GEO_LONG = parseFloat(self._player.attributes.CAM_GEO_LONG);
				var zoomLevel = self._map.DEFAULT_ZOOM_LEVEL;

				if (isNaN(CAM_GEO_LAT) || isNaN(CAM_GEO_LONG))
				{
					var DEFAULT_POSITION = JSON.parse(identity.DEFAULT_POSITION);

					CAM_GEO_LAT = DEFAULT_POSITION.lat;
					CAM_GEO_LONG = DEFAULT_POSITION.lng;
					zoomLevel = DEFAULT_POSITION.zoomLevel;
				}

				self._map.initMap(CAM_GEO_LAT, CAM_GEO_LONG, zoomLevel)
					.done(function(){
						var position = [CAM_GEO_LAT, CAM_GEO_LONG];
						var caption = "[" + obj + "] " + self._player.attributes.NAME;

						self._map.addCamera(obj, position, caption);

						self._marker = new L.Marker([0,0]).addTo(self._map.getMap());
						self._map.getMap().on("click", function(e){
							var selectedOption = $(".pointList select.list option:selected");
							var selectedPointId = selectedOption.val();
							var pointList = self._getCurrentPointList();

							if (self._isCameraPlace)
							{
								self._map.getCameraMarker().setLatLng(e.latlng);
							} else
							if (!self._isTest && selectedPointId)
							{
								self._marker.setLatLng(e.latlng);

								var pos = self._marker.getLatLng();
								var point = pointList[selectedPointId];
								var position = point.position = {lat: pos.lat, lng: pos.lng, alt: 0};
								var text = selectedPointId + ". (" + point.x + ", " + point.y + ")" + " (" + position.lat + ", " + position.lng + ")";
								selectedOption.text(text);
							}
						});
					});

				self._player.player.play(obj);
			});

		$(".buttons button.back").click(function (){
			window.history.go(-1);
		});

		$(".buttons button.cancel").click(function (){
			window.location.reload();
		});

		$(".buttons button.apply").click(function(){
			var GEO_CALIBRATOR_CONFIG = self._config;

			var cameraMarker = self._map.getCameraMarker().getLatLng();
			var lat = cameraMarker.lat;
			var lng = cameraMarker.lng;

			self.calibrate()
				.done(function(){
					self._api.setAttributes({
						obj: self._player.obj,
						attributes: JSON.stringify({
							GEO_CALIBRATOR_CONFIG: JSON.stringify(GEO_CALIBRATOR_CONFIG),
							CAM_GEO_LAT: lat,
							CAM_GEO_LONG: lng
						})
					})
						.fail(function (code, message) {
							Log.error("[" + code + "]" + __(message));
						})
						.done(function () {
							Log.info(__("Configuration saved"));
						});
				});
		});

		$(".pointList button.remove").click(function(){
			var selectedPointId = $(".pointList select.list option:selected").val();
			if (selectedPointId)
			{
				self._removePoint(selectedPointId);
				$(".pointList select.list option[value='" + selectedPointId + "']").remove();
				$(".pointList select.list option").first()
					.prop("selected", true)
					.trigger("change");
			}
		});

		$(".pointList button.edit").click(function(){
			self._isTest = false;
			self._isCameraPlace = false;

			$(".pointList select.list").prop("disabled", false);
			$(".pointList button.remove").prop("disabled", false);
			self._map.getCameraMarker().dragging.disable();

			Log.info("Click on camera and then select projection point on map");

			if (self._testLayer)
			{
				self._map.getMap().removeLayer(self._testLayer);
			}
		});

		$(".pointList button.test").click(function(){
			self._isTest = true;
			self._isCameraPlace = false;

			$(".pointList select.list").prop("disabled", true);
			$(".pointList button.remove").prop("disabled", true);
			self._map.getCameraMarker().dragging.disable();

			Log.info("Click on video to see projection point on the map");

			self.calibrate()
				.done(function(){
					if (self._testLayer)
					{
						self._map.getMap().removeLayer(self._testLayer);
					}

					self._testLayer = new L.FeatureGroup();
					self._map.getMap().addLayer(self._testLayer);

					self.testFullCover(self._player.player._oldImageWidth, self._player.player._oldImageHeight);
				});
		});

		$(".pointList button.camera").click(function(){
			self._isTest = false;
			self._isCameraPlace = true;

			$(".pointList select.list").prop("disabled", true);
			$(".pointList button.remove").prop("disabled", true);
			self._map.getCameraMarker().dragging.enable();

			Log.info("Click on map to place camera");

			if (self._testLayer)
			{
				self._map.getMap().removeLayer(self._testLayer);
			}
		});

		$(".pointList select.list").change(function(){
			var selectedPointId = $(".pointList select.list option:selected").val();
			if (selectedPointId)
			{
				var pointList = self._getCurrentPointList();
				var point = pointList[selectedPointId];

				if (point.position)
				{
					self._marker.setLatLng([point.position.lat, point.position.lng]);
					self._map.getMap().panTo([point.position.lat, point.position.lng]);
				} else {
					self._marker.setLatLng([0, 0]);
				}

				self._selectPoint(selectedPointId);
			}
		});
	};

	GEOCalibrator.prototype.calibrate = function()
	{
		var deferred = $.Deferred();

		var self = this;

		function print(value)
		{
			var precision = 20;
			//console.log(self.math.format(value, precision));
			console.log(value);
		}

		this.internal = [];
		this.external = [];

		var width = this._player.player._oldImageWidth;
		var height = this._player.player._oldImageHeight;

		var pointList = this._config[this._currentConfigObj].pointList;
		var correctPointCount = 0;
		for (var pointId in pointList)
		{
			var point = pointList[pointId];

			if (!point.position)
				continue;

			correctPointCount++;

			var x = point.x;
			var y = point.y;
			this.internal.push(Utils.normalize([x, y, 1], width, height));

			var X = point.position.lat;
			var Y = point.position.lng;
			var Z = 0;

			this.external.push([X, Y, 1]);
		}

		if (correctPointCount < 4)
		{
			Log.warning("Please specify minimum 4 points");
			deferred.reject();

			return deferred.promise();
		}

		// this.optimize();

		var internal = [];
		var external = [];
		for (var i = 0; i < this.internal.length; i++)
		{
			internal.push([this.internal[i][0], this.internal[i][1]]);
			external.push([this.external[i][0], this.external[i][1]]);
		}

		var internal_json = JSON.stringify(internal);
		var external_json = JSON.stringify(external);

		print(internal_json);
		print(external_json);

		/**
		 * find matrix M with least square method
		 *
		 * M [X Y Z 1]^T = [u v 1]^T
		 * [X Y Z 1] M^T = [u v 1]
		 *
		 * In = [u v 1]
		 * Ex = [X Y Z 1]
		 *
		 * Ex M^T = In
		 * (Ex^T Ex) M^T = Ex^T In
		 * (Ex^T Ex)^-1 (Ex^T Ex) M^T = (Ex^T Ex)^-1 Ex^T In
		 *
		 * M^T = (Ex^T Ex)^-1 Ex^T In
		 */

		/*
		var In = this.internal;
		print("In = "); print(In);
		var InT = this.math.transpose(In);
		var Ex = this.external;
		print("Ex = "); print(Ex);
		var ExT = this.math.transpose(Ex);
		*/

		/**
		 * AP = A^T (A A^T)^-1
		 */
		// var ExTEx = this.math.multiply(ExT, Ex);
		// var ExTExT = this.math.transpose(ExTEx);
		// var inv = this.math.multiply(ExTExT, this.math.inv(this.math.multiply(ExTEx, ExTExT)));

		/**
		var ExTEx = this.math.multiply(ExT, Ex);
		var ExTEx_inv = this.math.inv(ExTEx);
		this.M = this.math.transpose(
			this.math.multiply(
				this.math.multiply(
					ExTEx_inv, ExT
				), In
			)
		);
		 */

		/**
		 * find matrix H with least square method
		 *
		 * H [u v 1]^T = [X Y 1]^T
		 * [u v 1] H^T = [X Y 1]
		 *
		 * In = [u v 1]
		 * Ex = [X Y Z 1]
		 *
		 * In H^T = Ex
		 * (In^T In) H^T = In^T Ex
		 * (In^T In)^-1 (In^T In) H^T = (In^T In)^-1 In^T Ex
		 *
		 * H^T = (In^T In)^-1 In^T Ex
		 */

		/*var InTIn = this.math.multiply(InT, In);
		var InTIn_inv = this.math.inv(InTIn);
		this.M = this.math.transpose(
			this.math.multiply(
				this.math.multiply(
					InTIn_inv, InT
				), Ex
			)
		);*/

		this._api.findHomography({
			source: JSON.stringify(internal),
			destination: JSON.stringify(external)
		})
			.fail(function (code, message){
				Log.error("[" + code + "] " + message);

				deferred.reject();
			})
			.done(function (response) {
				self.M = response.homography;

				self._config[self._currentConfigObj].H = response.homography;

				print('M = ');
				print(self.M);

				deferred.resolve();
			});

		return deferred.promise();
	};

	GEOCalibrator.prototype.optimize = function()
	{
		this.lat = {
			min: {value: 0, n: 0},
			max: {value: 0, n: 0},
			diff: 0
		};
		this.lat.min.value = this.external[this.lat.min.n][0];
		this.lat.max.value = this.external[this.lat.max.n][0];

		this.lng = {
			min: {value: 0, n: 0},
			max: {value: 0, n: 0},
			diff: 0
		};
		this.lng.min.value = this.external[this.lng.min.n][1];
		this.lng.max.value = this.external[this.lng.max.n][1];

		for (var i = 0; i < this.external.length; i++)
		{
			var point = this.external[i];
			if (point[0] < this.lat.min.value)
			{
				this.lat.min.value = point[0];
				this.lat.min.n = i;
			}
			if (point[1] < this.lng.min.value)
			{
				this.lng.min.value = point[1];
				this.lng.min.n = i;
			}

			if (point[0] > this.lat.max.value)
			{
				this.lat.max.value = point[0];
				this.lat.max.n = i;
			}
			if (point[1] > this.lng.max.value)
			{
				this.lng.max.value = point[1];
				this.lng.max.n = i;
			}
		}

		var mult = 1;
		for (var i = 0; i < this.external.length; i++)
		{
			var point = this.external[i];

			point[0] = (point[0] - this.lat.min.value);
			point[1] = (point[1] - this.lng.min.value);

			point[0] *= mult;
			point[1] *= mult;
			point[2] *= mult;
		}
	};

	GEOCalibrator.prototype.screenToGlobal = function(x, y, width, height)
	{
		try {
			/**
			 * M [X Y Z 1]^T = [u v 1]^T
			 * [X Y Z 1]^T = M^-1 [u v 1]^T
			 *
			 * In = [u v 1]
			 * Ex = [X Y Z 1]
			 *
			 * M Ex^T = In^T
			 * (M^T M) Ex^T = M^T In^T
			 * (M^T M)^-1 (M^T M) Ex^T = (M^T M)^-1 M^T In^T
			 * Ex^T = (M^T M)^-1 M^T In^T
			 */

			var internal = Utils.normalize([x, y, 1], width, height);
			var M = this.M;
			//var MT = this.math.transpose(M);
			var In = internal;
			//var InT = this.math.transpose(In);

			/**
			 * Pseudo-Inverse of a Matrix
			 *
			 * if rank(a) = n <= m then
			 * AP = (A^T A)^-1 A^T

			 * if rank(a) = m <= n then
			 * AP = A^T (A A^T)^-1
			 */

			/*
			var Ex = this.math.multiply(
				this.math.multiply(
					this.math.inv(this.math.multiply(MT, M)), MT
				), InT
			);
			*/

			/**
			 * H [u v 1]^T = [X Y 1]^T
			 *
			 * In = [u v 1]
			 * Ex = [X Y Z 1]
			 *
			 * In H^T = Ex
			 */

			var Ex = Utils.multiply(
				M, In
			);

			var mult = Ex[Ex.length - 1];

			var X = (Ex[0] / mult);// + this.lat.min.value;
			var Y = (Ex[1] / mult);// + this.lng.min.value;

			var global = {lat: X, lng: Y};
		}
		catch (e)
		{
			console.error(e);
		}

		return global;
	};

	/**
	 * equirectangular projection
	 *
	 * @param {number} lat rad
	 * @param {number} lng rad
	 * @returns {{x: number, y: number}}
	 */
	GEOCalibrator.prototype.proj = function(lat, lng)
	{
		return {
			x: lng * Math.cos(lat),
			y: lat
		};
	};

	/**
	 * equirectangular projection
	 *
	 * @param {number} x
	 * @param {number} y
	 * @returns {{lat: number, lng: number}}
	 */
	GEOCalibrator.prototype.projrev = function(x, y)
	{
		return {
			lng: x / Math.cos(y),
			lat: y
		};
	};

	GEOCalibrator.prototype.testFullCover = function(width, height)
	{
		// full cover
		var pointList = [];

		/*var stepX = width / 5;
		var stepY = height / 10;
		for (var i = 0; i <= width; i += stepX)
		{
			for (var j = 0; j <= height; j += stepY)
			{
				var pointX = Math.round(i);
				var pointY = Math.round(j);

				pointList.push([pointX, pointY]);
			}
		}*/

		// corners
		pointList.push([0, 0]);
		pointList.push([width, 0]);
		pointList.push([width, height]);
		pointList.push([0, height]);

		console.log("test points");
		console.log(JSON.stringify(pointList));

		var polygonPointList = [];
		for (var i = 0; i < pointList.length; i++)
		{
			var point = pointList[i];
			var pointX = point[0];
			var pointY = point[1];

			var global = this.screenToGlobal(pointX, pointY, width, height);
			if (global)
			{
				var marker = new L.Marker(global);
				this._testLayer.addLayer(marker);

				polygonPointList.push(global);
			}
		}

		var polygon = new L.Polygon(polygonPointList, {opacity: 0.5, fill: false});
		this._testLayer.addLayer(polygon);
	};

	return GEOCalibrator;
});
