/**
 * @version $Id: session_manager.js 28385 2013-04-17 19:40:06Z astarostin $
 * ------------------------------------------------------------------------------
 * GUI for session management
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2012
 * ------------------------------------------------------------------------------
 */

(function(window){
	"use strict";

	window.SessionManager = SessionManager;

	/**
	 * create interface for password changing
	 *
	 * @param {boolean} isAll
	 * @constructor
	 */
	function SessionManager(isAll)
	{
		var _self = this;
		this._isAll = !!isAll;
		/**
		 * @type {boolean} selector where interface should be created
		 * @private
		 */
		this._selector = null;
		this._api = new API();

		/**
		 * initialise session manager
		 *
		 * @param {string} selector selector of DOM element to which session manager should be appended
		 */
		this.init = function(selector)
		{
			this._selector = selector;

			$(this._selector).off("click", 'input[type="button"]');

			// add events for Stop button
			$(this._selector).on("click", 'input[type="button"]', function(){
				var sessionId = $(this).data("sessionid");
				var userId = $(this).data("userid");
				_self.stopSession({
					userid: userId,
					sessionid: sessionId,
					callback: function(){
						if (sessionId)
						{
							$(_self._selector + ' li[data-sessionid="' + sessionId + '"]').remove();
						} else {
							_self.fillSessionList();
						}
					}
				});
			});
		};

		/**
		 * get session list from server
		 *
		 * @param {function} callback
		 * @private
		 */
		this._getSessionList = function(callback)
		{
			this._api.getSessionList({
				isAll: _self._isAll
			})
			.fail(function(code, message){
				Log.error(message);
			})
			.done(function(response){
				var sessionListByUser = {};

				for (var sessionId in response.list)
				{
					var userid = response.list[sessionId].content.userid;
					if (!sessionListByUser[userid])
						sessionListByUser[userid] = {};

					var element = {};
					element[sessionId] = response.list[sessionId];
					$.extend(sessionListByUser[userid], element);
				}

				callback(sessionListByUser);
			});
		};

		/**
		 * get list of all users
		 *
		 * @param {function} callback
		 * @private
		 */
		this._getUserList = function(callback)
		{
			this._api.getObjectList({
				type: "user",
				isAll: true
			})
			.fail(function(code, message){
				Log.error(message);
			})
			.done(function(response){
				var userListById = {};

				for (var i = 0; i < response.list.length; i++)
				{
					userListById[response.list[i].obj] = response.list[i];
				}

				callback(userListById);
			});
		};

		/**
		 * fill session list
		 *
		 * @param {function} callback
		 */
		this.fillSessionList = function(callback)
		{
			this._getUserList(function(userList){
				_self._getSessionList(function(sessionList){
					var sessionList_html = ['<ul class="object_list treeview-famfamfam treeview">'];

					for (var userid in sessionList)
					{
						sessionList_html.push(
							'<li>' +
								'<div class="clickable_area"><div class="caption_list_element_small"><input type="button" value="Stop" data-userid="' + userid+ '"/> [' + userid + '] ' + userList[userid].name + '</div></div>' +
								'<ul>'
						);

						for (var sessionId in sessionList[userid])
						{
							var description = "";
							if (sessionId == readCookie("PHPSESSID"))
							{
								description = "Your session. ";
							}
							description += sessionList[userid][sessionId].content.user_ip + " " + sessionList[userid][sessionId].content.user_agent;
							sessionList_html.push(
								'<li data-sessionid="' + sessionId + '">' +
									'<div class="folder list_element_small">' +
										'<div class="content">' +
											'<div class="text">' +
												'<div class="label"><input type="button" value="Stop" data-sessionid="' + sessionId + '"/> [' + sessionId + '] ' + sessionList[userid][sessionId].age +' sec </div>' +
												'<div class="description">' + description + '</div>' +
											'</div>' +
										'</div>' +
									'</div>' +
								'</li>'
							);
						}

						sessionList_html.push('</ul></li>');
					}

					sessionList_html.push('</ul>');
					$(_self._selector).html(sessionList_html.join(''));

					if ($.isFunction(callback))
						callback();
				});
			});
		};

		/**
		 * stop session
		 *
		 * @param {Object} params sessionId
		 */
		this.stopSession = function(params)
		{
			var data = {};

			if (params.userid)
			{
				data.userid = params.userid;
			} else {
				data.sessionid = params.sessionid;
			}
			this._api.stopSession(data)
			.fail(function(code, message){
				Log.error(message);
			})
			.done(function(response){
				Log.info(__("Session successfully stopped"));

				if ($.isFunction(params.callback))
				{
					params.callback();
				}
			});
		};
	}

})(window);
