/**
 * @version $Id:$
 * ------------------------------------------------------------------------------
 * GUI for changing password
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2012
 * ------------------------------------------------------------------------------
 */

(function (root, factory) {
    if (typeof define === 'function' && define.amd) {
        // AMD. Register as an anonymous module.
	    define("password_editor", ["jquery", "log", "api"], factory);
    } else {
        // Browser globals
        root.PasswordEditor = factory(root.$, root.Log, root.API);
    }
}(this, function($, Log, API){
	"use strict";

	/**
	 * create interface for password changing
	 *
	 * @constructor
	 */
	function PasswordEditor()
	{
		var _self = this;
		/**
		 * @type {string} selector where interface should be created
		 * @private
		 */
		this._selector = null;
		/**
		 * @type {API}
		 * @private
		 */
		this._api = new API();
		/**
		 * @type {int}
		 * @private
		 */
		this._obj = null;
		/**
		 * @type {boolean}
		 * @private
		 */
		this._onlyChange = null;

		/**
		 * initialise password changer
		 *
		 * @param {Object.<string,*>} params
		 * {
		 *     selector: {string} selector of DOM element to which password changer should be appended
		 *     obj: {number}
		 *     onlyChange: {boolean}
		 *     success: {function}
		 * }
		 */
		this.init = function(params)
		{
			this._selector = params.selector;
			this._obj = params.obj;
			this._onlyChange = typeof params.onlyChange == "undefined" ? true : !!params.onlyChange;

			this._create();

			$(this._selector + " .change_password").click(function(){
				$(_self._selector + ' form[name="change_password"]').slideDown();
				$(_self._selector + ' form[name="set_password"]').slideUp();
			});

			$(this._selector + " .set_password").click(function(){
				$(_self._selector + ' form[name="change_password"]').slideUp();
				$(_self._selector + ' form[name="set_password"]').slideDown();
			});

			$(this._selector + ' form[name="change_password"] input[type="button"]').click(function(){

				var currentPassword = $(_self._selector + ' form[name="change_password"] input[name="current_password"]').val();
				var newPassword = $(_self._selector + ' form[name="change_password"] input[name="new_password"]').val();
				var confirmNewPassword = $(_self._selector + ' form[name="change_password"] input[name="confirm_new_password"]').val();
				if (newPassword != confirmNewPassword)
				{
					Log.error(__("Passwords in 'New password' and 'Confirm new password' fields not equals"));
					return;
				}

				var parameters = {
					currentPassword: currentPassword,
					newPassword: newPassword
				};
				if (_self._obj) parameters.userid = _self._obj;
				_self._api.changePassword(parameters)
					.fail(function(code, message){
						Log.error(message);
					})
					.done(function(response){
						Log.info(__("Password successfully changed"));
						$(_self._selector + ' form[name="change_password"]').slideUp();
						if ($.isFunction(params.success))
						{
							params.success();
						}
					});
			});

			$(this._selector + ' form[name="set_password"] input[type="button"]').click(function(){
				var newPassword = $(_self._selector + ' form[name="set_password"] input[name="new_password"]').val();
				var confirmNewPassword = $(_self._selector + ' form[name="set_password"] input[name="confirm_new_password"]').val();
				if (newPassword != confirmNewPassword)
				{
					Log.error(__("Passwords in 'New password' and 'Confirm new password' fields not equals"));
					return;
				}

				_self._api.setPassword({
					userid: _self._obj,
					password: newPassword
				})
					.fail(function(code, message){
						Log.error(message);
					})
					.done(function(response){
						Log.info(__("Password successfully set"));
						$(_self._selector + ' form[name="set_password"]').slideUp();
						if ($.isFunction(params.success))
						{
							params.success();
						}
					});
			});

			return !!$(this._selector).length;
		};

		/**
		 * create session_manager editor
		 *
		 * @private
		 */
		this._create = function()
		{
			var passwordEditorHTML = [
				'<div class="password_editor">',
					'<span class="link change_password">Change password</span><br>',
					'<form name="change_password" method="post">',
						'<p><label>Current password</label><input type="password" class="password" name="current_password" autocomplete="off"></p>',
						'<p><label>New password</label><input type="password" class="password" name="new_password" autocomplete="off"></p>',
						'<p><label>Confirm new password</label><input type="password" class="password" name="confirm_new_password" autocomplete="off"></p>',
						'<p><input type="button" value="Save"></p>',
					'</form>'
			];
			if (!this._onlyChange)
			{
				passwordEditorHTML.push(
					'<span class="link set_password">Set password (will be marked as expired)</span><br>',
					'<form name="set_password" method="post">',
						'<p><label>New password</label><input type="password" class="password" name="new_password" autocomplete="off"></p>',
						'<p><label>Confirm new password</label><input type="password" class="password" name="confirm_new_password" autocomplete="off"></p>',
						'<p><input type="button" value="Save"></p>',
					'</form>'
				);
			}
			passwordEditorHTML.push('</div>');

			$(this._selector + ' .password_editor').remove();
			$(this._selector).append(passwordEditorHTML.join(''));
		};

		/**
		 * clear input fields in form
		 */
		this.clear = function()
		{
			// clear form
			$(this._selector + ' form[name="change_password"]').hide();
			$(this._selector + ' form[name="set_password"]').hide();
			$(this._selector + ' form input[type="text"]').val("");
		};
	}

	return PasswordEditor;
}));
