/**
 * @version $Id: mediaplayer.js 27088 2012-10-25 12:30:45Z astarostin $
 * ------------------------------------------------------------------------------
 * media player
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2012
 * ------------------------------------------------------------------------------
 */

(function(window){
	"use strict";

	/**
	 * @type {MediaPlayer}
	 */
	window.MediaPlayer = MediaPlayer;

	/**
	 * @constructor
	 */
	function MediaPlayer()
	{
		/**
		 * @type {Boolean}
		 * @private
		 */
		this._isDebug = false;

		/**
		 * @type {API}
		 * @private
		 */
		this._api = new API();

		/**
		 * @type {String}
		 * @private
		 */
		this._selector = "";

		/**
		 * @type {Number|null}
		 * @private
		 */
		this._obj = null;

		/**
		 * @type {String}
		 * @private
		 */
		this._mime = "";

		/**
		 * @type {HTMLElement|EventTarget}
		 * @private
		 */
		this._player = null;

		/**
		 * @type {boolean}
		 * @private
		 */
		this._isArchive = false;

		/**
		 * @type {boolean}
		 * @private
		 */
		this._isPlay = false;

		/**
		 * @type {boolean}
		 * @private
		 */
		this._isStop = true;

		/**
		 * @type {String}
		 * @private
		 */
		this._currentURL = "";

		/**
		 * @type {Date}
		 * @private
		 */
		this._startDate = new Date();
		/**
		 * @type {Date}
		 * @private
		 */
		this._endDate = new Date();

		/**
		 * list of topics
		 *
		 * @type {{newimage: {}, imageclick: {}, zoom: {imageclick: {}}, player: {play: {}, pause: {}, stop: {}, connecting: {}, buffering: {}, frame: {}}}}
		 * @private
		 */
		this._topics = {
			// old events
			playerready: {},
			bufferchange: {},
			snapshotfinish: {},
			newimage: {},
			imageclick: {},
			statuschange: {},
			// new events
			positionchange: {},
			zoom: {
				imageclick: {}
			},
			player: {
				play: {},
				pause: {},
				stop: {},
				connecting: {},
				buffering: {},
				frame: {}
			},
			// not player object events
			archive: {},
			live: {}
		};

		/**
		 * @type {number}
		 */
		this._lastPrimitiveId = 0;

		/**
		 * @type {number}
		 */
		this._retryInterval = 3000;

		/**
		 * @type {{0: string, 10000: string, 10001: string, 10002: string, 10003: string, 10100: string}}
		 * @private
		 */
		this._stopCodeMessage = {
			"0":     "Stop",
			"default": "Something went wrong"
		};
		this._stopCodeMessage[MediaPlayer.RTCPTIMEOUT] = "TCP Timeout";
		this._stopCodeMessage[MediaPlayer.ENDOFSTREAM] = "End of stream";
		this._stopCodeMessage[MediaPlayer.MEDIAFORMATCHANGED] = "Media format was changed";
		this._stopCodeMessage[MediaPlayer.UNKNOWNSTREAMTYPE] = "Unknown stream type";
		this._stopCodeMessage[MediaPlayer.INTERNALERROR] = "Internal error";

		this._isPrintError = true;
	}

	/**
	 * @const
	 * @type {number}
	 */
	MediaPlayer.RTCPTIMEOUT = 10000;
	/**
	 * @const
	 * @type {number}
	 */
	MediaPlayer.ENDOFSTREAM = 10001;
	/**
	 * @const
	 * @type {number}
	 */
	MediaPlayer.MEDIAFORMATCHANGED = 10002;
	/**
	 * @const
	 * @type {number}
	 */
	MediaPlayer.UNKNOWNSTREAMTYPE = 10003;
	/**
	 * @const
	 * @type {number}
	 */
	MediaPlayer.INTERNALERROR = 10100;

	/**
	 * initialize media player
	 *
	 * @param {String} selector
	 * @param {Object} [parameters]
	 * @param {boolean} [isPrintError]
	 * @param {number} [obj]
	 * @returns {Deferred}
	 */
	MediaPlayer.prototype.init = function(selector, parameters, isPrintError, obj)
	{
		var deferred = $.Deferred();

		if (this._player && document.querySelector(this._selector))
		{
			deferred.resolveWith(this, [true]);
			return deferred.promise();
		}

		this._isPrintError = typeof isPrintError == "undefined" ? this._isPrintError : !!isPrintError;

		var settings = $.extend({
			cacheSize: 4, // size of cache in MB
			backgroundColor: 0x2E2E2E, // background color, number in next format 0xRRGGBB
			jitterBufferLength: 0
		}, parameters);

		// check if selector already belongs to object
		var check = function(callback)
		{
			var player = document.querySelector(selector);
			var isAlreadyExists = !!(player && typeof player.init !== "undefined");
			if (isAlreadyExists)
			{
				this._selector = selector;
				this._player = player;
				this._obj = obj;
				callback.apply(this, [isAlreadyExists]);
			} else {
				this._getManifest(function(manifest){
					var platform = "";
					if (navigator.appVersion.indexOf("Win") != -1)
					{
						platform = "win32";
					} else
					if (navigator.appVersion.indexOf("Mac") != -1)
					{
						platform = "macos";
					}

					this._mime = manifest.platform[platform].mime;

					var id = "MediaPlayer_" + Math.round((Math.random() * 10000));
					this._selector = "#" + id;
					var html = '<object id="'+ id + '" width="100%" height="100%" type="' + this._mime + '"></object>';
					$(selector).append(html);
					this._player = document.querySelector(this._selector);
					callback.apply(this, [isAlreadyExists]);
				});
			}
		};

		check.call(this, function(isAlreadyExists){
			var success = false;
			if (typeof this._player.init !== "undefined")
			{
				if (!isAlreadyExists)
				{
					this._player.init(settings);
					this._addEvents();
				}

				success = true;
			}
			if (success)
			{
				deferred.resolveWith(this, [false]);
			} else {
				deferred.rejectWith(this);
			}
		});

		return deferred.promise();
	};

	MediaPlayer.prototype.destroy = function()
	{

	};

	/**
	 * @private
	 */
	MediaPlayer.prototype._addEvents = function()
	{
		this.subscribe("play", function(){
			this._isPlay = true;
		}, "player");

		this.subscribe("stop", function(code, message){
			this._isPlay = false;
			var errorMessage;
			if (message)
			{
				errorMessage = message;
			} else {
				errorMessage = this._stopCodeMessage[code] ? __(this._stopCodeMessage[code]) : __(this._stopCodeMessage["default"]);
			}

			this.printErrorMessage("[" + code + "] " + errorMessage);

			if (code == 0)
			{
				this._isStop = true;
			} else
			if (this._stopCodeMessage[code])
			{
				setTimeout(this._play.bind(this, [true]), this._retryInterval);
			}
		}, "player");

		this.subscribe("pause", function(){
			this._isPlay = false;
			this.printErrorMessage(__("Pause"));
		}, "player");

		this.subscribe("connecting", function(){
			this._isPlay = false;
			this.printErrorMessage(__("Connecting"));
		}, "player");

		this.subscribe("buffering", function(){
			this._isPlay = false;
			this.printErrorMessage(__("Buffering"));
		}, "player");
	};

	/**
	 * @param {String} message
	 */
	MediaPlayer.prototype.printErrorMessage = function(message)
	{
		if (!this._isPrintError) return false;
		if (!this._player) return false;

		if (document.querySelector(this._selector))
			this._player.printError(message);

		return true;
	};

	/**
	 * @param {Number} [obj]
	 * @param {Number} [time]
	 * @returns {Boolean}
	 */
	MediaPlayer.prototype.play = function(obj, time)
	{
		if (!this._player) return false;

		var isGetNewURL = true;

		if (!obj && !time && this._obj)
		{
			isGetNewURL = false;
			this._player.player.play();
		} else  {
			if (!!obj) this._obj = obj;
			if (!this._obj) return false;

			if (this._obj && !time)
			{
				this._isArchive = false;
			} else
			if (this._obj && time)
			{
				this._isArchive = true;

				if (this._startDate.getTime() < time && time < this._endDate.getTime())
				{
					isGetNewURL = false;
					this.moveAbsolute(Math.round(time / 1000));
					this._player.player.play();
				} else {
					var date = new Date();
					date.setTime(time);
					// date.setTime(date.getTime() - date.getUTCMinutes() * 60 * 1000 - date.getUTCSeconds() * 1000 - date.getUTCMilliseconds());
					this._startDate.setTime(date.getTime());
					date.setUTCHours(date.getUTCHours() + 1);
					this._endDate.setTime(date.getTime());
				}
			}
		}

		this._isStop = false;

		this._play(isGetNewURL/*, function(){
			if (time)
			{
				this.moveAbsolute(Math.round(time / 1000));
			}
		}*/);

		return true;
	};

	/**
	 *
	 * @param {boolean} isGetNewURL
	 * @param {function} [callback]
	 * @private
	 */
	MediaPlayer.prototype._play = function(isGetNewURL, callback)
	{
		isGetNewURL && this._getURL(this._isArchive, this._startDate.getTime(), this._endDate.getTime(), function(url, params){
			if (!this._player) return;

			this._currentURL = url;

			var playParams = {
				url: this._currentURL,
				streamOverTcp: params.MEDIA_TRANSPORT == "tcp",
				pixelAspectRatio: isNaN(params.PIXEL_ASPECT_RATIO) ? params.PIXEL_ASPECT_RATIO : parseInt(params.PIXEL_ASPECT_RATIO)
			};

			this._player.player.play(playParams);

			if ($.isFunction(callback))
			{
				callback.apply(this);
			}

			this._player.player.play();
		});
	};

	MediaPlayer.prototype.pause = function()
	{
		if (!this._player) return false;

		this._player.player.pause();

		return true;
	};

	MediaPlayer.prototype.stop = function()
	{
		if (!this._player) return false;

		this._player.player.stop();

		return true;
	};

	MediaPlayer.prototype.frameBackward = function()
	{
		if (!this._player) return false;

		this._player.player.frameBackward();

		return true;
	};

	/**
	 * @param {Number} time
	 * @returns {Boolean}
	 */
	MediaPlayer.prototype.moveAbsolute = function(time)
	{
		if (!this._player) return false;
		if (time <= 0) return false;

		this._player.player.moveAbsolute(time);

		return true;
	};

	/**
	 * @param {Number} time
	 * @returns {Boolean}
	 */
	MediaPlayer.prototype.moveRelative = function(time)
	{
		if (!this._player) return false;
		if (time == 0) return false;

		this._player.player.moveRelative(time);

		return true;
	};

	/**
	 * subscribe for event
	 *
	 * @param {String} eventName event name
	 * @param {Function} callback function
	 * @param {String} [property]
	 */
	MediaPlayer.prototype.subscribe = function(eventName, callback, property)
	{
		if (!this._player) return false;
		if (!eventName) return false;

		var topic = !!property ? this._topics[property][eventName] : this._topics[eventName];
		if (!topic)
		{
			Log.error("Unknown topic '" + eventName + "'");
			return false;
		}

		if (!topic.callbacks)
		{
			topic = {
				callbacks: $.Callbacks()
			};

			if (!!property)
			{
				this._topics[property][eventName] = topic;
				Utils.addEvent(this._player[property], eventName, topic.callbacks.fire.bind(this));
			} else {
				this._topics[eventName] = topic;
				Utils.addEvent(this._player, eventName, topic.callbacks.fire.bind(this));
			}
		}

		topic.callbacks.add(callback);

		return true;
	};

	/**
	 * unsubscribe from event
	 *
	 * @param {String} eventName event name
	 * @param {Function} callback function
	 * @param {String} [property]
	 */
	MediaPlayer.prototype.unsubscribe = function(eventName, callback, property)
	{
		if (!this._player) return false;
		if (!eventName) return false;
		var topic = !!property ? this._topics[property][eventName] : this._topics[eventName];

		if (topic && topic.callbacks)
		{
			if (!!property)
			{
				Utils.removeEvent(this._player[property], eventName, callback);
			} else {
				Utils.removeEvent(this._player, eventName, callback);
			}

			topic.callbacks.remove(callback);
		}

		return true;
	};

	MediaPlayer.prototype._unsubscribeAll = function()
	{

	};

	/**
	 * simlulate publish event from media player object
	 *
	 * @param {String} eventName
	 * @param {Object} data
	 * @param {String} [property]
	 */
	MediaPlayer.prototype.publish = function(eventName, data, property)
	{
		if (!this._player) return false;
		if (!eventName) return false;
		var topic = !!property ? this._topics[property][eventName] : this._topics[eventName];

		if (topic)
		{
			topic.callbacks.fire.apply(this, [data]);
		}

		return true;
	};

	/**
	 * @param {Function} callback
	 * @private
	 */
	MediaPlayer.prototype._getManifest = function(callback)
	{
		var self = this;
		$.ajax({
			url: "/sdi/player/manifest.json",
			cache: false,
			dataType: "json"
		})
			.done(function(manifest){
				callback.apply(self, [manifest]);
			});
	};

	/**
	 * @param {boolean} isArchive
	 * @param {Number} startTime
	 * @param {Number} endTime
	 * @param {Function} callback
	 * @private
	 */
	MediaPlayer.prototype._getURL = function(isArchive, startTime, endTime, callback)
	{
		if (!document.querySelector(this._selector)) return;

		if (!this.isGetUrl)
		{
			this.isGetUrl = true;
		} else {
			return;
		}

		var self = this;

		var streamType = isArchive ? "archive" : "live";

		var parameters = {
			objid: this._obj,
			"return": "mediastreamauth",
			player: "MPLAYER",
			streamtype: streamType,
			streamnum: 1,
			skip_gaps: 1
		};

		if (isArchive)
		{
			parameters.time_start = Math.round(startTime / 1000);
			parameters.time_end = Math.round(endTime / 1000);
		}

		this.checkLocalStream(this._obj, function(additionalParameters){
			parameters = $.extend(parameters, additionalParameters);

			self._api.authorizationManager(parameters)
				.fail(function(code, message){
					if (self._isStop)
					{
						return;
					}

					switch (code)
					{
						case 401:
							self.printErrorMessage(__("Your session has expired"));
							break;
						default:
							self.printErrorMessage("[" + code + "] " + message);
							setTimeout(self._getURL.bind(self, isArchive, startTime, endTime, callback), self._retryInterval);
					}
				})
				.done(function(response){
					if (self._isStop)
					{
						return;
					}

					callback.apply(self, [response.result.url + "&authorizationid=" + response.result.authid, response.result.param]);

					self.textLocalStream(parameters);
				})
				.always(function(){
					self.isGetUrl = false;
				});
		});
	};

	/**
	 * check local content on Avatar
	 *
	 * @param {number} obj
	 * @param {Function} callback
	 */
	MediaPlayer.prototype.checkLocalStream = function(obj, callback)
	{
		var self = this;

		var getAvatarId = function(callback)
		{
			if (self._avatarId)
			{
				callback(self._avatarId);
			} else {
				self._api.getAttribute({obj: obj, attribute: "AVATARID"})
					.done(function(response){
						var avatarId = parseInt(response.value);
						self._avatarId = isNaN(avatarId) ? 0 : avatarId;
						callback(self._avatarId);
					});
			}
		};

		getAvatarId(function(avatarId){
			if (!avatarId)
			{
				callback({});
				return;
			}

			self._api.getIdentityAttribute({attribute: "AVATAR_LOCAL_STREAMING"})
				.done(function(response){
					// if local streaming allowed
					var allowLocalStreaming = response.value == "yes";
					if (!allowLocalStreaming)
					{
						callback({});
						return;
					}

					var streamlocal = readCookie("streamlocal." + self._obj);
					if (streamlocal && streamlocal == "yes")
					{
						callback({streamlocal: true});
						return;
					}

					// get avatar attributes
					self._api.getAttributes({obj: avatarId})
						.done(function(response){
							$.ajax({
								url: "http://" + response.list['IP'] + ":8554/checkuni",
								data: {
									uni: response.list["UNI"]
								},
								dataType: 'json',
								success: function(response) {
									var parameters = {};

									if (response.error != "")
									{
										setCookie("streamlocal." + self._obj, "", -1, "/");
									} else
									if (response.result == "yes")
									{
										parameters.streamlocal = true;

										setCookie("streamlocal." + self._obj, "yes", 60 * 60, "/");
									} else
									if (response.result == "no")
									{
										setCookie("streamlocal." + self._obj, "no", 60 * 60, "/");
									}

									callback(parameters);
								}
							})
						});
				});
		});
	};

	MediaPlayer.prototype.textLocalStream = function(parameters)
	{
		if (parameters.hasOwnProperty('streamlocal'))
		{
			this.addText("STREAM_LOCAL_INFO", "Local streaming", 5, 35, 14, 0xFFFFDD00);

			var self = this;

			setTimeout(function(){
				self.removePrimitive("STREAM_LOCAL_INFO");
			}, 7000);
		}
	};

	/**
	 * @returns {Boolean}
	 */
	MediaPlayer.prototype.isArchive = function()
	{
		return this._isArchive;
	};

	/**
	 * @returns {Boolean}
	 */
	MediaPlayer.prototype.isPlay = function()
	{
		return this._isPlay;
	};

	/**
	 * @returns {Number|null}
	 */
	MediaPlayer.prototype.getObj = function()
	{
		return this._obj;
	};

	/**
	 * @returns {Boolean}
	 */
	MediaPlayer.prototype.removePrimitives = function()
	{
		if (!this._player) return false;

		var primitives = this.getPrimitives();
		for (var id in primitives)
		{
			this.removePrimitive(id);
		}

		return true;
	};

	/**
	 * @returns {Object}
	 */
	MediaPlayer.prototype.getPrimitives = function()
	{
		if (!this._player) return null;

		return $.extend({}, this._player.layer.primitive);
	};

	/**
	 * @param {Object} scheme
	 * @returns {Boolean}
	 */
	MediaPlayer.prototype.setPrimitives = function(scheme)
	{
		if (!this._player) return false;

		this._player.layer.setPrimitives(scheme);

		return true;
	};

	/**
	 * @param {String} id
	 * @param {Object} primitive
	 * @returns {Boolean}
	 * @private
	 */
	MediaPlayer.prototype._updatePrimitive = function(id, primitive)
	{
		if (!this._player) return false;

		this._player.layer.updatePrimitive(id, primitive);

		return true;
	};

	/**
	 * @param {String} id
	 * @returns {Boolean}
	 */
	MediaPlayer.prototype.removePrimitive = function(id)
	{
		if (!this._player) return false;

		this._player.layer.removePrimitive(id);

		return true;
	};

	/**
	 * @param {String|null} id
	 * @param {Number} x
	 * @param {Number} y
	 * @param {Number} width
	 * @param {Number} height
	 * @param {Number} color
	 * @returns {String|null} id
	 */
	MediaPlayer.prototype.addEllipse = function(id, x, y, width, height, color)
	{
		if (!this._player) return null;

		if (!id) id = String(this._lastPrimitiveId++);

		this._updatePrimitive(
			id,
			{
				type: "ellipse",
				x: x,
				y: y,
				width: width,
				height: height,
				stroke: {
					color: color,
					width: 1
				}
			}
		);

		return id;
	};

	/**
	 * @param {String|null} id
	 * @param {Array} points
	 * @param {Number} color
	 * @returns {String|null} id
	 */
	MediaPlayer.prototype.addPolyline = function(id, points, color)
	{
		if (!this._player) return null;

		if (!id) id = String(this._lastPrimitiveId++);

		this._updatePrimitive(
			id,
			{
				type: "polyline",
				points: points,
				stroke: {
					color: color,
					width: 1
				}
			}
		);

		return id;
	};

	/**
	 * @param {String|null} id
	 * @param {String} text
	 * @param {Number} x
	 * @param {Number} y
	 * @param {Number} size
	 * @param {Number} color
	 * @returns {String|null} id
	 */
	MediaPlayer.prototype.addText = function(id, text, x, y, size, color)
	{
		if (!this._player) return null;

		if (!id) id = String(this._lastPrimitiveId++);

		this._updatePrimitive(
			id,
			{
				type: "text",
				text: text,
				x: x,
				y: y,
				size: size,
				color: color
			}
		);

		return id;
	};

	/**
	 * @param {Number} imageWidth
	 * @param {Number} imageHeight
	 * @param {Number} color
	 */
	MediaPlayer.prototype.drawCrosshair = function(imageWidth, imageHeight, color)
	{
		if (this._oldImageWidth != imageWidth || this._oldImageHeight != imageHeight)
		{
			this.isCrosshairDrawn = false;
		}
		if (!this.isCrosshairDrawn) this.isCrosshairDrawn = true; else return;

		var crosshairSize = 100;

		this._crosshair = this._crosshair || [];

		if (this._crosshair.length > 0)
		{
			this.removePrimitive(this._crosshair[0]);
			this.removePrimitive(this._crosshair[1]);
		}

		this._crosshair[0] = this.addPolyline(
			null,
			[
				{x: imageWidth / 2 - crosshairSize / 2, y: imageHeight / 2},
				{x: imageWidth / 2 + crosshairSize / 2, y: imageHeight / 2}
			],
			color
		);
		this._crosshair[1] = this.addPolyline(
			null,
			[
				{x: imageWidth / 2, y: imageHeight / 2 - crosshairSize / 2},
				{x: imageWidth / 2, y: imageHeight / 2 + crosshairSize / 2}
			],
			color
		);

		this._oldImageWidth = imageWidth;
		this._oldImageHeight = imageHeight;
	};

})(window);
