/**
 * @version $id:$
 * ------------------------------------------------------------------------------
 * MediaExporter
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2012
 * ------------------------------------------------------------------------------
 */

(function(window){
	"use strict";


	window.MediaExporter = MediaExporter;

	/**
	 *
	 * @constructor
	 */
	function MediaExporter()
	{
		/**
		 * matrix core plugin
		 * @type {HTMLElement|EventTarget}
		 * @private
		 */
		this._plugin = null;

		/**
		 * mediaexporter plugin
		 * @type {HTMLElement|EventTarget}
		 * @private
		 */
		this._mediaExporter = null;

		/**
		 * list of topics
		 *
		 * @private
		 */
		this._topics = {
			mediaExporter: {
				progress: {},
				error: {},
				log: {}
			}
		};

		/**
		 *
		 * @type {boolean}
		 * @private
		 */
		this._debug = false;
	}

	/**
	 * initialize media player
	 *
	 * @param {String} selector
     * @param {Object} [parameters]
	 * @returns {Deferred}
	 */
	MediaExporter.prototype.init = function(selector, parameters)
	{
		var deferred = $.Deferred();

		if (this._plugin)
		{
			deferred.resolveWith(this, [true]);
			return deferred.promise();
		}

		var settings = $.extend({
		}, parameters);

		this._id = "MatrixCore_" + Math.round((Math.random() * 10000));

		this._getManifest(function(manifest){
			var platform = "";
			if (navigator.appVersion.indexOf("Win") != -1)
			{
				platform = "win32";
			} else
			if (navigator.appVersion.indexOf("Mac") != -1)
			{
				platform = "macos";
			}

			this._mime = manifest.platform[platform].mime;

			var html = '<object id="'+ this._id + '" width="1" height="1" type="' + this._mime + '"></object>';

			$(selector).append(html);

			this._plugin = document.getElementById(this._id);

			var success = false;
			if (typeof this._plugin.mediaExporter !== "undefined")
			{
				this._addEvents();

				this._mediaExporter = this._plugin.mediaExporter;
				this._mediaExporter.init(settings);

				success = true;
			}
			if (success)
			{
				deferred.resolveWith(this, [false]);
			} else {
				deferred.rejectWith(this);
			}
		});

		return deferred.promise();
	};

	/**
	 * subscribe for event
	 *
	 * @param {String} eventName event name
	 * @param {Function} callback function
 	 * @param {String} [property]
	 */
	MediaExporter.prototype.subscribe = function(eventName, callback, property)
	{
		if (!this._plugin) return false;
		if (!eventName) return false;

		var topic = !!property ? this._topics[property][eventName] : this._topics[eventName];
		if (!topic) return false;

		if (!topic.callbacks)
		{
			topic = {
				callbacks: $.Callbacks()
			};

			if (!!property)
			{
				this._topics[property][eventName] = topic;
				Utils.addEvent(this._plugin[property], eventName, topic.callbacks.fire.bind(this));
			} else {
				this._topics[eventName] = topic;
				Utils.addEvent(this._plugin, eventName, topic.callbacks.fire.bind(this));
			}
		}

		topic.callbacks.add(callback);

		return true;
	};

	/**
	 * unsubscribe from event
	 *
	 * @param {String} eventName event name
	 * @param {Function} callback function
	 * @param {String} [property]
	 */
	MediaExporter.prototype.unsubscribe = function(eventName, callback, property)
	{
		if (!this._plugin) return false;
		if (!eventName) return false;
		var topic = !!property ? this._topics[property][eventName] : this._topics[eventName];

		if (topic)
		{
			if (!!property)
			{
				Utils.removeEvent(this._plugin[property], eventName, callback);
			} else {
				Utils.removeEvent(this._plugin, eventName, callback);
			}

			topic.callbacks.remove(callback);
		}

		return true;
	};

	/**
	 * simlulate publish event from media player object
	 *
	 * @param {String} eventName
	 * @param {Object} data
	 * @param {String} [property]
	 */
	MediaExporter.prototype.publish = function(eventName, data, property)
	{
		if (!this._plugin) return false;
		if (!eventName) return false;
		var topic = !!property ? this._topics[property][eventName] : this._topics[eventName];

		if (topic)
		{
			topic.callbacks.fire.apply(this, [data]);
		}

		return true;
	};

	/**
	 * @param {Function} callback
	 * @private
	 */
	MediaExporter.prototype._getManifest = function(callback)
	{
		var self = this;
		$.ajax({
			url: "/sdi/matrix/manifest.json",
			cache: false,
			dataType: "json"
		})
		.done(function(manifest){
			callback.apply(self, [manifest]);
		});
	};


	/**
	 * @private
	 */
	MediaExporter.prototype._addEvents = function()
	{
		if (this._debug)
		{
			this.subscribe("log", function(type, message){
				window.console && window.console.log("[" + type + "] " + message);
			}, "mediaExporter");
		}
	};

	// applet methods

	/**
	 *
	 * @param {string }defaultFilename
	 * @returns {string}
	 */
	MediaExporter.prototype.selectOutputFile = function(defaultFilename)
	{
		if (!this._mediaExporter) return null;

		return this._mediaExporter.selectOutputFile(defaultFilename);
	};

	/**
	 *
	 * @param {string} url
	 * @param {string} outFilename
	 * @returns {boolean}
	 */
	MediaExporter.prototype.startExport = function(url, outFilename)
	{
		if (!this._mediaExporter) return false;

		return this._mediaExporter.startExport(url, outFilename);
	};

	/**
	 *
	 */
	MediaExporter.prototype.cancelExport = function()
	{
		if (!this._mediaExporter) return;

		this._mediaExporter.cancelExport();
	};

})(window);
