/**
 * @version $Id: mapzoneeditor.js 33904 2016-03-15 13:55:50Z astarostin $
 * ------------------------------------------------------------------------------
 * Map camera calibration tool
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2013
 * ------------------------------------------------------------------------------
 */


define("mapzoneeditor", ["jquery", "api", "amq.api", "mediaplayer", "leaflet", "leaflet.imagetransform", "utils", "map2d", "hwj"], function(){
	"use strict";

	/**
	 * @type {Map2D}
	 */
	var Map2D = require("map2d");

	/**
	 * map zone editor
	 *
	 * @param {number} obj camera objid
	 * @param {string} selector map html container id
	 * @constructor
	 * @augments Map2D
	 */
	function MapZoneEditor(obj, selector)
	{
		MapZoneEditor.superproto.constructor.call(this, selector);

		/**
		 * @type {number}
		 * @private
		 */
		this._obj = obj;

		/**
		 *
		 * @type {string}
		 * @private
		 */
		this._type = null;

		this._attributes = null;
	}

	Utils.inherit(MapZoneEditor, Map2D);

	/**
	 * @param {{lat: number, lng: number, zoomLevel: number}} [position]
	 * @param {boolean} [isDrawBuilding]
	 * @returns {Deferred}
	 */
	MapZoneEditor.prototype.init = function(position, isDrawBuilding)
	{
		isDrawBuilding = typeof isDrawBuilding == "undefined" ? true : !!isDrawBuilding;
		
		var deferred = $.Deferred();

		var self = this;

		MapZoneEditor.superproto.init.call(this, position, isDrawBuilding)
			.fail(function(){
				deferred.reject();
			})
			.done(function(){
				// if identity
				if (self._obj == 53)
				{
					self._attributes = self._identity;
				} else
				// search device with current obj
				for (var i = 0; i < self._deviceList.length; i++)
				{
					var device = self._deviceList[i];
					if (device.obj == self._obj)
					{
						self._attributes = device.attributes;
						break;
					}
				}

				self.addDrawControl(JSON.parse(self._attributes.TARGET_ZONE));

				deferred.resolve();
			});

		return deferred.promise();
	};

	MapZoneEditor.prototype.initControls = function()
	{
		MapZoneEditor.superproto.initControls.apply(this, arguments);
	};

	/**
	 *
	 * @param {string} name
	 * @param {[]} [layers]
	 * @returns {number} building number
	 */
	MapZoneEditor.prototype.addBuilding = function(name, layers)
	{
		layers = typeof layers != "undefined" ? layers : [];

		var building = {
			name: name,
			layers: layers
		};

		return this._objectList.push(building) - 1;
	};

	/**
	 *
	 * @param {number} buildingNumber
	 */
	MapZoneEditor.prototype.removeBuilding = function(buildingNumber)
	{
		var building = this._objectList[buildingNumber];
		if (!building)
		{
			return;
		}

		var floorList = building.layers;
		for (var i = 0; i < floorList.length; i++)
		{
			var floor = floorList[i];

			this._mapLayers.removeLayer(floor.layer);
			this._map.removeLayer(floor.layer);
		}

		this._objectList.splice(buildingNumber, 1);
	};

	/**
	 *
	 * @param {number} buildingNumber
	 * @param {number} blockId
	 * @param {string} name
	 * @param {string} url
	 * @param {L.Polygon} polygon
	 * @returns {number} floor number
	 */
	MapZoneEditor.prototype.addFloor = function(buildingNumber, blockId, name, url, polygon)
	{
		var latLngs = polygon ? L.geoJson(polygon).getLayers()[0].getLatLngs() : this.boundsToLatLngs(this._map.getBounds());

		var floor = {
			name: name,
			blockId: blockId,
			url: url,
			bounds: latLngs,
			layer: new L.LayerGroup(),
			polygon: null,
			objects: []
		};

		var boundsArray = floor.bounds.map(function(bound){
			return [bound.lat, bound.lng];
		});

		var imageOverlay = new L.ImageTransform(floor.url, boundsArray, {opacity: 0.5, attribution: floor.name})
			.addTo(floor.layer);

		floor.polygon = new L.Polygon(boundsArray, {fill: false})
			.addTo(floor.layer);

		var updateBounds = function () {
			var anchors = anchorMarkers.map(function (marker) {
				return marker.getLatLng();
			});

			imageOverlay.setAnchors(anchors);
			floor.polygon.setLatLngs(anchors);
			floor.bounds = floor.polygon.getLatLngs();
		};

		var anchorMarkers = boundsArray.map(function (anchor) {
			return new L.Marker(anchor, {draggable: true})
				.addTo(floor.layer)
				.on('drag', updateBounds);
		});

		this._mapLayers.addOverlay(floor.layer, this._objectList[buildingNumber].name + ": " + floor.name);

		floor.layer.addTo(this._map);

		return this._objectList[buildingNumber].layers.push(floor) - 1;
	};

	/**
	 *
	 * @param {number} buildingNumber
	 * @param {number} floorNumber
	 */
	MapZoneEditor.prototype.removeFloor = function(buildingNumber, floorNumber)
	{
		var floor = this._objectList[buildingNumber].layers[floorNumber];
		if (!floor)
		{
			return;
		}

		this._mapLayers.removeLayer(floor.layer);
		this._map.removeLayer(floor.layer);

		this._objectList[buildingNumber].layers.splice(floorNumber, 1);
	};

	/**
	 *
	 * @param {number} buildingNumber
	 * @param {number} floorNumber
	 */
	MapZoneEditor.prototype.panToFloor = function(buildingNumber, floorNumber)
	{
		var floor = this._objectList[buildingNumber].layers[floorNumber];
		if (!floor)
		{
			return;
		}

		var polygon = floor.polygon;
		if (!polygon)
		{
			return;
		}

		var bounds = polygon.getBounds();
		if (bounds.isValid())
		{
			this._map.fitBounds(bounds)
		}
	};

	/**
	 *
	 * @param {number} buildingNumber
	 * @param {number} floorNumber
	 */
	MapZoneEditor.prototype.hideAllExceptFloor = function(buildingNumber, floorNumber)
	{
		var floor = this._objectList[buildingNumber].layers[floorNumber];
		if (!floor)
		{
			return;
		}

		var floorList= this._objectList[buildingNumber].layers;
		for (var i = 0; i < floorList.length; i++)
		{
			if (i != floorNumber)
			{
				this._map.removeLayer(floorList[i].layer);
			}
		}

		if (!this._map.hasLayer(floor.layer))
		{
			this._map.addLayer(floor.layer);
		}
	};

	MapZoneEditor.prototype.savePrimitives = function()
	{
		this._api.setAttributes({
			obj: this._obj,
			attributes: JSON.stringify({TARGET_ZONE: this.getZoneGeoJSON()})
		})
			.fail(function(code, message){
				Log.error(message);
			})
			.done(function(/*response*/){
				Log.info(__("Zone is saved"));
			});
	};

	MapZoneEditor.prototype.saveLayers = function()
	{
		/*
		var layerList = {};

		for (var blockId in this._floorList)
		{
			var floor = this._floorList[blockId];
			layerList[blockId] = {
				polygon: floor.polygon.toGeoJSON()
			};
		}
		*/

		/*
		Layer list example:

		var list = {
			objects: [{
				name: "Building 102",
				layers: [{
					name: "Floor 1",
					blockId: 5,
					polygon: {},
					objects: {"101": {
						position: [lat, lng]
					}}
				}]
			}]
		};
		*/

		var objectList = [];
		for (var i = 0; i < this._objectList.length; i++)
		{
			var building = this._objectList[i];
			var name = building.name;
			var floors = building.layers;

			var object = {
				name: name,
				layers: []
			};

			for (var j = 0; j < floors.length; j++)
			{
				var floor = floors[j];
				var layerName = floor.name;
				var layerBlockId = floor.blockId;
				var layerPolygon = floor.polygon;
				var layerObjects = floor.objects;

				var layer = {
					name: layerName,
					blockId: layerBlockId,
					polygon: layerPolygon.toGeoJSON(),
					objects: layerObjects
				};

				object.layers.push(layer);
			}

			objectList.push(object);
		}

		var layerList = {objects: objectList};

		this._api.setAttributes({
			obj: this._obj,
			attributes: JSON.stringify({LAYER_LIST: layerList})
		})
			.fail(function(code, message){
				Log.error(message);
			})
			.done(function(/*response*/){
				Log.info(__("Layers is saved"));
			});
	};

	MapZoneEditor.prototype.saveDefaultPosition = function()
	{
		var position = this._map.getCenter();
		var lat = position.lat;
		var lng = position.lng;
		var zoomLevel = this._map.getZoom();
		var DEFAULT_POSITION = {lat: lat, lng: lng, zoomLevel: zoomLevel};

		this._api.setAttributes({
			obj: this._obj,
			attributes: JSON.stringify({DEFAULT_POSITION: DEFAULT_POSITION})
		})
			.fail(function(code, message){
				Log.error(message);
			})
			.done(function(/*response*/){
				Log.info(__("Default position is saved"));
			});
	};

	/**
	 * @returns {[number, number]} [lat, lng]
	 */
	MapZoneEditor.prototype.getCameraPosition = function()
	{
		return [this._attributes.CAM_GEO_LAT, this._attributes.CAM_GEO_LONG];
	};

	return MapZoneEditor;
});
