/**
 * @version $Id: Map2D.js 30659 2014-06-01 22:55:46Z astarostin $
 * ------------------------------------------------------------------------------
 * Map camera calibration tool
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2013
 * ------------------------------------------------------------------------------
 */


define("map2d",
	[
		"jquery",
		"api",
		"amq.api",
		"mediaplayer",
		"leaflet",
		"leaflet.imagetransform",
		"leaflet.marker.bouncemarker",
		/*"leaflet.marker.edgemarker",*/
		"leaflet.marker.markercluster",
		"utils",
		"flatmap",
		"hwj",
		"jquery-ui"
	], function(){
	"use strict";

	/**
	 * @type {FlatMap}
	 */
	var FlatMap = require("flatmap");

	/**
	 * map zone editor
	 *
	 * @param {string} selector map html container id
	 * @constructor
	 * @augments FlatMap
	 */
	function Map2D(selector)
	{
		Map2D.superproto.constructor.call(this, selector);

		/**
		 * camera obj
		 * @type {number}
		 * @protected
		 */
		this._obj = null;

		/**
		 * @type {API}
		 * @protected
		 */
		this._api = new API();

		/**
		 * @type {number}
		 * @private
		 */
		this._colorInListIndex = 0;
		/**
		 * @type {string[]}
		 * @private
		 */
		this._colorList = [
			"#ED1B35", // red
			"#44B85C", // green
			"#0078BF", // blue
			"#19C1F3", // light blue
			"#894E35", // brown
			"#F58631", // orange
			"#8E479C", // purple
			"#FFCB31", // yellow
			"#B3D445", // lime
			"#79CDCD", // teal
			"#ACBFE1"  // blue grey
		];
		/*
		this._colorList = [
			"#b58900", // yellow
			"#d30102", // red
			"#268bd2", // blue
			"#859900", // green
			"#2aa198", // cyan
			"#6c71c4", // violet
			"#d33682", // magenta
			"#cb4b16"  // orange
		];
		*/
		/**
		 * @type {string}
		 * @private
		 */
		this._grayedOutColor = "#E4E4E4";

		/**
		 * @type {MediaPlayer}
		 * @private
		 */
		this._player = new MediaPlayer();

		this._targetList = {};

		/**
		 * @type {number} target lifetime in ms
		 * @private
		 */
		this._targetLifetime = 15000;

		/**
		 * @type {number} max trail path after marker
		 * @private
		 */
		this._maxPathLength = 3;

		this._currentLockTarget = {obj: null, targetId: null};

		this._objectList = [];

		this._deviceList = [];

		this._identity = null;
	}

	Utils.inherit(Map2D, FlatMap);

	/**
	 * @param {{lat: number, lng: number, zoomLevel: number}} [position]
	 * @param {boolean} [isDrawBuilding]
	 * @returns {Deferred}
	 */
	Map2D.prototype.init = function(position, isDrawBuilding)
	{
		isDrawBuilding = typeof isDrawBuilding == "undefined" ? true : !!isDrawBuilding;
		
		var deferred = $.Deferred();
		var self = this;			

		$.when(
			this._api.getObjectList({withAttributes: true}),
			this._api.getLastTargets(),
			this._api.getIdentityAttributes()
		)
			.fail(function(code, message){
				Log.error(message);

				deferred.reject();
			})
			.done(function(responseObjectList, responseLastTargets, responseIdentityAttributes){
				self._identity = responseIdentityAttributes[0].list;

				var DEFAULT_POSITION = position ? position : JSON.parse(self._identity.DEFAULT_POSITION);

				self.initMap(DEFAULT_POSITION.lat, DEFAULT_POSITION.lng, DEFAULT_POSITION.zoomLevel)
					.done(function(){
						self._editableLayers = new L.geoJson(JSON.parse(self._identity.TARGET_ZONE), {
							style: function (feature) {
								var zoneType = feature.properties.zoneType ? feature.properties.zoneType : self._polygonIncludeOptions;
								return self.getPolygonOptions(zoneType);
							}
						});
						self._map.addLayer(self._editableLayers);

						self._deviceList = responseObjectList[0].list;
						var i;
						for (i = 0; i < self._deviceList.length; i++)
						{
							var device = self._deviceList[i];
							if (device.attributes.ARCHSTATE == "on"
								&& device.attributes.CAM_GEO_CALIBRATION != "no")
							{
								var CAM_GEO_LAT = parseFloat(device.attributes.CAM_GEO_LAT);
								var CAM_GEO_LONG = parseFloat(device.attributes.CAM_GEO_LONG);

								if (isNaN(CAM_GEO_LAT) || isNaN(CAM_GEO_LONG))
								{
									continue;
								}

								var position = [CAM_GEO_LAT, CAM_GEO_LONG];
								var caption = "[" + device.obj + "] " + device.name;

								if (device.otype == "D" && device.subtype == "C")
								{
									self.addCamera(device.obj, position, caption);
								} else
								if (device.otype == "D" && device.subtype == "S")
								{
									self.addSensor(device.obj, position, caption);
								}
							}
						}

						var lastTargetList = responseLastTargets[0].list;
						for (i = 0; i < lastTargetList.length; i++)
						{
							var target = lastTargetList[i];

							var obj = parseInt(target.objid, 10);
							var targetId = target.targetid;
							var time = parseInt(target.time);
							var lat = parseFloat(target.targetlatitude);
							var lng = parseFloat(target.targetlongitude);
							var alt = parseFloat(target.targetaltitude);

							self.addTarget(obj, targetId, time, lat, lng, alt);
						}

						self.initControls();

						// plugin for adding edge markers
						// L.edgeMarker().addTo(self._map);

						// cluster markers of devices
						var markerClusterGroup = new L.MarkerClusterGroup({
							disableClusteringAtZoom: 17
						});
						for (i = 0; i < self._deviceMarkerList.length; i++)
						{
							var marker = self._deviceMarkerList[i];
							markerClusterGroup.addLayer(marker);
						}
						self._map.addLayer(markerClusterGroup);

						// for test purposes
						/*
						self._map.on("click", function(event){
							if (!self.list) self.list = [];
							self.list.push(event.latlng);
						});
						*/

						if (isDrawBuilding)
						{
							self.loadImageLayers();
						}

						deferred.resolve();
					});
			});

		return deferred.promise();
	};

	/**
	 * add target to list and map
	 *
	 * @param {number} obj
	 * @param {string} targetId
	 * @param {number} time
	 * @param {number} lat
	 * @param {number} lng
	 * @param {number} alt
	 */
	Map2D.prototype.addTarget = function(obj, targetId, time, lat, lng, alt)
	{
		if (!this._targetList[obj])
		{
			this._targetList[obj] = {};
		}

		if (!this._targetList[obj][targetId])
		{
			var title = "[" + obj + "] " + targetId;

			var self = this;
			var onClick = function()
			{
				self.lockTarget(obj, targetId);
			};

			var marker = new L.Marker([lat, lng], {
				title: title,
				riseOnHover: true,
				bounceOnAdd: true,
				bounceOnAddOptions: {
					bounceOnAddDuration: 500
				}
			})
				.bindPopup(title)
				.addTo(this._map)
				.on("click", onClick)
				.on("popupclose", function(){
					self.unlockTarget(obj, targetId);
				});

			var color = this._colorList[this._colorInListIndex];
			var path = new L.Polyline([[lat, lng]], {color: color})
				.addTo(this._map)
				.on("click", onClick);

			this._targetList[obj][targetId] = {
				time: time,
				lat: lat,
				lng: lng,
				alt: alt,
				marker: marker,
				path: path,
				color: color,
				isGrayedOut: false,
				isHidden: false
			};

			this._colorInListIndex++; if (this._colorInListIndex >= this._colorList.length) this._colorInListIndex = 0;

			$("#target_list select").append('<option data-targetid="' + targetId + '" data-obj="' + obj + '" value="' + obj + '_' + targetId + '">[' + obj + '] ' + targetId + '</option>');
		} else {
			var target = this._targetList[obj][targetId];
			target.time = time;
			target.lat = lat;
			target.lng = lng;
			target.alt = alt;

			// TODO: add animated target move
			// var fx = new L.PosAnimation();
			// fx.run(target.marker, [lat, lng], 0.5);

			target.marker.setLatLng([lat, lng]);

			target.path.addLatLng([lat, lng]);

			if (this._currentLockTarget.obj == obj && this._currentLockTarget.targetId == targetId)
			{
				this.panTo([lat, lng]);
			}
		}
	};

	/**
	 * remove old targets from map
	 */
	Map2D.prototype.cleanTargets = function()
	{
		try
		{

		var self = this;
		for (var targetObj in this._targetList)
		{
			var targetList = this._targetList[targetObj];
			for (var targetId in targetList)
			{
				var target = targetList[targetId];

				// remove not updated targets
				if (Date.now() - target.time > this._targetLifetime)
				{
					$("#target_list select option[value='" + targetObj + '_' + targetId + "']").remove();

					target.marker.on("remove", function () {
						try {
							if (target.removed)
							{
								delete self._targetList[targetObj][targetId];
							} else {
								target.removed = true;
							}
						}
						catch(e)
						{
							console.log("on remove marker:");
							console.log(e);
						}
					});

					target.path.on("remove", function () {
						try {
							if (target.removed)
							{
								delete self._targetList[targetObj][targetId];
							} else {
								target.removed = true;
							}
						}
						catch(e)
						{
							console.log("on remove path:");
							console.log(e);
						}
					});

					this._map.removeLayer(target.marker);
					this._map.removeLayer(target.path);
				} else {
					// simplify path
					var path = target.path.getLatLngs();
					if (path.length > this._maxPathLength)
					{
						path.shift();
					}
				}
			}
		}

		}
		catch (e)
		{
			console.log("clean:");
			console.log(e);
		}
	};

	Map2D.prototype.initControls = function()
	{
		var self = this;

		AMQ.subscribe("topic://amq.gis.topic", function(messageList){
			messageList.forEach(function(message){
				try
				{
					var messageXMLDoc = $.parseXML(message);
				}
				catch (e)
				{
					Log.error("error parsing xml: " + e);
					return;
				}

				var messageXML = $(messageXMLDoc).find("message");
				var type = messageXML.attr("type");

				var targetId, obj;
				var lat, lng, alt;

				if (type == "gisTrgtRadar"
					|| type == "gisTrgtMobile"
					|| type == "gisTrgtUgs"
					|| type == "gisTrgtAvatar")
				{
					obj = parseInt(messageXML.attr("objid"), 10);

					switch (type)
					{
						case "gisTrgtRadar":
						case "gisTrgtUgs":
							targetId = messageXML.attr("id");
							break;
						case "gisTrgtMobile":
							targetId = "iStream";
							break;
						case "gisTrgtAvatar":
							targetId = "AVTR_" + obj;
							break;
					}

					lat = parseFloat(messageXML.attr("lat"));
					lng = parseFloat(messageXML.attr("lon"));
					alt = parseFloat(messageXML.attr("alt"));
				} else
				if (type == "targetRegister")
				{
					targetId = messageXML.attr("targetId");
					obj = parseInt(messageXML.attr("objId"), 10);

					lat = parseFloat(messageXML.attr("targetLatitude"));
					lng = parseFloat(messageXML.attr("targetLongitude"));
					alt = parseFloat(messageXML.attr("targetAltitude"));
				} else {
					return;
				}

				if (!isNaN(lat) && !isNaN(lng) && !isNaN(alt))
				{
					var time = Date.now();
					self.addTarget(obj, targetId, time, lat, lng, alt);
				}
			});

			self.cleanTargets();
		});

		// clean targets every this._targetLifetime ms
		setInterval(function(){
			self.cleanTargets();
		}, this._targetLifetime);
	};

	/**
	 * @param {number} obj
	 * @param {string} id
	 */
	Map2D.prototype.panToTarget = function(obj, id)
	{
		var target = this._targetList[obj][id];
		var lat = target.lat;
		var lng = target.lng;
		var alt = target.alt;
		this._map.panTo([lat, lng], {animate: false});

		// TODO: correct logic of panning to PTZ camera to target
		// window.HWJoystick.sendCommand("target=" + id + "&mode=abs&lat=" + lat + "&long=" + lng + "&elev=" + alt + "&slew=" + this._obj, false);
	};

	/**
	 * @param {number} obj
	 * @param {string} id
	 */
	Map2D.prototype.lockTarget = function(obj, id)
	{
		if (this._targetList[obj][id].isHidden)
		{
			return;
		}

		if (this._currentLockTarget.obj && this._currentLockTarget.targetId)
		{
			this._targetList[this._currentLockTarget.obj][this._currentLockTarget.targetId].marker.closePopup();
		}

		this._currentLockTarget.obj = obj;
		this._currentLockTarget.targetId = id;

		this._targetList[obj][id].marker.openPopup();

		this.panToTarget(obj, id);

		$("#target_list select option:selected").prop("selected", false);
		$("#target_list select option[data-obj='" + obj + "'][data-targetid='" + id + "']").prop("selected", true);

		// "hide" other markers and paths
		for (var targetObj in this._targetList)
		{
			if (targetObj == obj)
			{
				continue;
			}

			var targets = this._targetList[targetObj];
			for (var targetId in targets)
			{
				var target = targets[targetId];

				if (!target.isGrayedOut)
				{
					target.isGrayedOut = true;
					var greyMarkerIcon = new L.Icon({
						iconUrl: '/sdi/map/images/marker-icon-grey.png',
						iconSize: [25, 41],
						iconAnchor: [12, 41],
						popupAnchor: [1, -34],
						shadowSize: [41, 41]
					});
					target.marker.setIcon(greyMarkerIcon);
					target.path.setStyle({color: this._grayedOutColor});
				}
			}
		}
	};

	/**
	 * @param {number} obj
	 * @param {string} id
	 */
	Map2D.prototype.unlockTarget = function(obj, id)
	{
		if (this._currentLockTarget.obj == obj && this._currentLockTarget.targetId == id)
		{
			this._currentLockTarget.obj = null;
			this._currentLockTarget.targetId = null;
			this._targetList[obj][id].marker.closePopup();

			$("#target_list select option:selected").prop("selected", false);

			// "show" other markers
			for (var targetObj in this._targetList)
			{
				var targets = this._targetList[targetObj];
				for (var targetId in targets)
				{
					var target = targets[targetId];

					target.isGrayedOut = false;
					var normalIcon = new L.Icon.Default();
					target.marker.setIcon(normalIcon);
					target.path.setStyle({color: target.color});
				}
			}
		}
	};

	/**
	 * @param {number} obj
	 * @param {string} id
	 */
	Map2D.prototype.showTarget = function(obj, id)
	{
		var target = this._targetList[obj][id];
		if (target.isHidden)
		{
			target.isHidden = false;

			target.marker.bindPopup(target.marker.options.title);
			target.marker.setOpacity(1);

			target.path.setStyle({opacity: 0.5});

			var targetOption = $("#target_list select option[data-obj='" + obj + "'][data-targetid='" + id + "']");
			targetOption.html("[" + obj + "] " + id);
		}
	};

	/**
	 * @param {number} obj
	 * @param {string} id
	 */
	Map2D.prototype.hideTarget = function(obj, id)
	{
		var target = this._targetList[obj][id];
		if (!target.isHidden)
		{
			target.isHidden = true;

			this.unlockTarget(obj, id);

			target.marker.unbindPopup();
			target.marker.setOpacity(0.1);

			target.path.setStyle({opacity: 0.1});

			var targetOption = $("#target_list select option[data-obj='" + obj + "'][data-targetid='" + id + "']");
			targetOption.html("hidden " + targetOption.html());
		}
	};

	/**
	 * @param {Array} position [lat, lng]
	 */
	Map2D.prototype.panTo = function(position)
	{
		this._map.panTo(position, {animate: false});
	};

	/**
	 *
	 * @param {number} obj
	 * @param {string} caption
	 */
	Map2D.prototype.showVideoPlayer = function(obj, caption)
	{
		if ($("#camera").is(":visible"))
		{
			return;
		}

		var self = this;

		var buttons = [];

		// show buttons in Matrix only
		if (parent.mx)
		{
			var event_id = null; //window['device_' + obj_id].event_id;

			var resourceTree = parent.resourceTree;
			if (resourceTree.getObject(obj).attr("POSITIONCTL") != "none"
				&& resourceTree.cred(obj, 'p'))
			{
				buttons.push({
					text: __("PTZ"),
					click: function(){
						var parameters = {
							ve_ptz: "yes",
							obj_id: obj,
							cell_id: "_ve"
						};
						parent.mx.PTZ.load(parameters);
					}
				});
			}

			if (false && event_id != null)
			{
				buttons.push({
					text: __("Ackn"),
					click: function(){
						if (window['device_' + obj])
						{
							window['device_' + obj].event_id = null;

							$("button.ackn")
								.prop("disabled", true)
								.attr("title", __("No new events present"));

							// removeAlertFromDevice(window['device_' + obj_id], -1)
						}

						parent.mx.ELog2.submitAction("Acknowledge", event_id, obj);
					}
				});
			}

			buttons.push({
				text: __("Pop-Up"),
				click: function(){
					parent.mx.ELog2.elM2DEventAckn(obj.toString());
				}
			});

			buttons.push({
				text: __("Pop-Up in Tab"),
				click: function(){
					var cameraName = caption;
					var tab = parent.mx.CT.createNewTab();
					if (tab != null)
					{
						var objDest = tab.getFirstChild();
						tab.doShow();
						var newCell = parent.mx.MATRIX2.createCell(obj, cameraName, objDest, false);
						newCell.setAttribute("close_previous", "false");
						newCell.setAttribute("dst_from", "0");
						newCell.setAttribute("dst_to", "0");
						window.setTimeout(function() {
							parent.mx.MATRIX2.setSelectedGroup(newCell.getDescendantOfName("cell_select"), null, false);
						}, 0);
					}
				}
			});
		}

		$("#camera").dialog({
			buttons: buttons,
			title: caption,
			autoOpen: true,
			appendTo: "#map_wrapper",
			position: {my: "center", at: "center", of: "#map_wrapper"},
			maxWidth: 640,
			maxHeight: 480,
			minWidth: 160,
			minHeight: 160,
			create: function(event, ui) {
			},
			open: function(event, ui) {
				self._player.init("#camera div.player", {backgroundColor: 0xCACACA})
					.fail(function(){
						Log.error(__("Player not initialized"));
					})
					.done(function(){
						this.play(obj);
					});
			},
			close: function(event, ui) {
				self._player.stop();
				$(this).dialog("destroy");
			}
		});
	};

	Map2D.prototype.boundsToLatLngs = function(latLngBounds)
	{
		latLngBounds = L.latLngBounds(latLngBounds);
		return [
			latLngBounds.getNorthWest(),
			latLngBounds.getNorthEast(),
			latLngBounds.getSouthEast(),
			latLngBounds.getSouthWest()
		];
	};

	/**
	 *
	 * @param {string} name
	 * @param {[]} [layers]
	 * @returns {number} building number
	 */
	Map2D.prototype.addBuilding = function(name, layers)
	{
		layers = typeof layers != "undefined" ? layers : [];

		var building = {
			name: name,
			layers: layers
		};

		return this._objectList.push(building) - 1;
	};

	/**
	 *
	 * @param {number} buildingNumber
	 * @param {number} blockId
	 * @param {string} name
	 * @param {string} url
	 * @param {L.Polygon} polygon
	 * @returns {number} floor number
	 */
	Map2D.prototype.addFloor = function(buildingNumber, blockId, name, url, polygon)
	{
		var latLngs = polygon ? L.geoJson(polygon).getLayers()[0].getLatLngs() : this.boundsToLatLngs(this._map.getBounds());

		var floor = {
			name: name,
			url: url,
			bounds: latLngs,
			layer: new L.LayerGroup(),
			polygon: null
		};

		var boundsArray = floor.bounds.map(function(bound){
			return [bound.lat, bound.lng];
		});

		var imageOverlay = new L.ImageTransform(floor.url, boundsArray, {opacity: 0.5, attribution: floor.name})
			.addTo(floor.layer);

		floor.polygon = new L.Polygon(boundsArray, {fill: false, stroke: false})
			.addTo(floor.layer);

		this._mapLayers.addOverlay(floor.layer, this._objectList[buildingNumber].name + ": " + floor.name);

		floor.layer.addTo(this._map);

		return this._objectList[buildingNumber].layers.push(floor) - 1;
	};

	Map2D.prototype.loadImageLayers = function()
	{
		var self = this;
		$.when(
			this._api.getIdentityAttributes(),
			this._api.getBlocksInfo({obj: 53})
		)
			.fail(function(code, message){
				Log.error("[" + code + "] " + message);
			})
			.done(function(responseIdentityAttributes, responseBlockInfo){
				var identity = responseIdentityAttributes[0].list;

				var LAYER_LIST = JSON.parse(identity.LAYER_LIST);

				var blockList = responseBlockInfo[0].list;
				var objectList = LAYER_LIST.objects ? LAYER_LIST.objects : [];
				for (var i = 0; i < objectList.length; i++)
				{
					var building = objectList[i];
					var name = building.name;
					var floors = building.layers;

					self.addBuilding(name);

					for (var j = 0; j < floors.length; j++)
					{
						var floor = floors[j];
						var layerName = floor.name;
						var layerBlockId = floor.blockId;
						var layerPolygon = floor.polygon;
						var layerObjects = floor.objects;

						var block = blockList[layerBlockId];
						var mime = block.mime;
						var url = block.url;
						var description = block.description;

						// only image/png image of floor for now
						self.addFloor(i, layerBlockId, layerName, url, layerPolygon);
					}
				}
			});
	};

	return Map2D;
});
