/**
 * @version $Id: log.js 33061 2015-09-12 20:42:04Z astarostin $
 * ------------------------------------------------------------------------------
 * Logging and alert message on screen
 * include <script scr="/sdi/lib/log.js"></script>
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions LLC 2015
 * ------------------------------------------------------------------------------
 */

(function (root, factory) {
	if (typeof define === 'function' && define.amd) {
		// AMD. Register as an anonymous module.
		define([], function () {
			return (root.Log = factory());
		});
	} else if (typeof module === 'object' && module.exports) {
		// Node. Does not work with strict CommonJS, but
		// only CommonJS-like environments that support module.exports,
		// like Node.
		module.exports = factory();
	} else {
		// Browser globals (root is window)
		root.Log = factory();
	}
}(this, function(){
	"use strict";

	var initLog = function()
	{
		var styles =
			"#messages {position:fixed;top:0px;right:0px;width:350px;margin:0px;padding:2px;background:transparent;z-index:2}" +
			"#messages div{color:#fff;padding:7px;margin-bottom:7px;opacity:0.85;}" +
			"#messages .error{background:#98001b}" +
			"#messages .info{background:#0d8529}" +
			"#messages .warning{background:#dd6; color:#333}" +
			"#messages .closeButton{cursor: pointer;float: right;height: 12px;line-height: 12px;margin: 0;padding: 0;text-align: center;vertical-align: middle;width: 12px;}";

		var addLoadEvent = function(func)
		{
			var oldonload = window.onload;
			if (typeof window.onload != 'function')
			{
				window.onload = func;
			}
			else
			{
				window.onload = function()
				{
					if (oldonload)
					{
						oldonload();
					}
					func();
				}
			}
		};

		var import_style = function(src)
		{
			if ((src === null || src === undefined)) return;
			var imprt = document.createElement('style');
			imprt.setAttribute("type", "text/css");
			if (imprt.styleSheet) imprt.styleSheet.cssText = src;
			else imprt.appendChild(document.createTextNode(src));
			document.getElementsByTagName('head')[0].appendChild(imprt);
		};

		var addAll = function()
		{
			var messageBox = document.createElement('div');
			messageBox.id = "messages";
			if (document.body.firstChild) document.body.insertBefore(messageBox, document.body.firstChild);
			else document.body.appendChild(messageBox);
			import_style(styles);
		};


		if (document.body === null) return addLoadEvent(function()
		{
			addAll();
		});
		addAll();
	};

	initLog();

	var history = []; // store logs to an array for reference

	var isWriteToConsole = false;

	var Log =
	{
		writeToConsole: function(status)
		{
			isWriteToConsole = status;
		},

		info: function (mtext, mtype, howlong)
		{
			if (isWriteToConsole)
			{
				switch (mtype)
				{
					case "info":
						console.info(mtext);
					break;
					case "warning":
						console.warn(mtext);
					break;
					case "error":
						console.error(mtext);
					break;
				}

				return;
			}

			mtype = mtype || 'info';
			howlong = howlong || 15000;

			Log.debug(mtype, mtext);

			if (document.getElementById('messages') === null)
			{
				setTimeout(function()
				{
					Log.info(mtext, mtype, howlong)
				}, 100);
				return;
			}

			var alarm = document.createElement('div');
			alarm.className = mtype;
			alarm.innerHTML = mtext;

			alarm.del = function ()
			{
				document.getElementById('messages').removeChild(alarm);
			};

			var closeButton = document.createElement('div');
			closeButton.className = "closeButton";
			closeButton.innerHTML = "X";

			closeButton.onclick = function ()
			{
				alarm.style.display = "none";
			};

			alarm.appendChild(closeButton);

			document.getElementById('messages').appendChild(alarm);
			setTimeout(alarm.del, howlong);
		},

		warning: function (mtext, howlong)
		{
			howlong = howlong || 15000;
			Log.info(mtext, "warning", howlong);
		},

		error: function (mtext, howlong)
		{
			howlong = howlong || 15000;
			Log.info(mtext, "error", howlong);
		},

		isDebug: false,

		getHistory: function ()
		{
			return history;
		},

		debug: function ()
		{
			if (Log.isDebug)
			{
				history.push(arguments);
				if (console)
				{
					console.log(Array.prototype.slice.call(arguments));
				}
			}
		}
	};

	window.oldAlert = window.alert;
	// replace alert function with Log error message
	window.alert = function()
	{
		if (isWriteToConsole)
		{
			window.oldAlert(arguments[0]);
			return;
		}

		if (arguments && arguments[0])
		{
			Log.error(String(arguments[0]).replace(/\n/g,"<br/>"));
		}
	};

	var oldOnError = window.onerror;
	// override previous handler.
	window.onerror = function (errorMsg, url, lineNumber)
	{
		// if exist, call previous handler
		if (oldOnError)
		{
			return oldOnError(errorMsg, url, lineNumber);
		}

		Log.error(errorMsg + "<br/>line:" + lineNumber + "<br/>url:"+ url);

		return false;
	};

	return Log;
}));
