/**
 * @version $Id: flatmap.js 33098 2015-09-18 16:50:56Z astarostin $
 * ------------------------------------------------------------------------------
 * Flat Map
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2013
 * ------------------------------------------------------------------------------
 */

define("flatmap",
	[
		"jquery",
		"api",
		"leaflet",
		"leaflet.draw",
		"leaflet.scale",
		"leaflet.layer.google",
		"leaflet.geosearch",
		"leaflet.geosearch.google",
		"leaflet.styledlayercontrol"
	],
	function(){
	"use strict";

	/**
	 * @param {string} selector
	 * @constructor
	 */
	function FlatMap(selector)
	{
		/**
		 * @type {string}
		 * @private
		 */
		this._selector = selector;

		/**
		 * @type {L.Map}
		 * @private
		 */
		this._map = null;

		/**
		 * @type {L.Control.Layers}
		 * @private
		 */
		this._mapLayers = null;

		/**
		 * @type {L.Marker[]}
		 * @protected
		 */
		this._deviceMarkerList = [];

		/**
		 * @type {L.geoJson}
		 * @private
		 */
		this._editableLayers = null;

		this._polygonIncludeOptions = {
			color: '#bada55'
		};

		this._polygonExcludeOptions = {
			color: '#ff5555'
		};

		/**
		 * @type {L.Control.GeoSearch}
		 * @private
		 */
		this._searchControl = null;

		/**
		 * @type {L.Control.Draw}
		 * @private
		 */
		this._drawControl = null;

		/**
		 * @type {API}
		 * @private
		 */
		this._api = new API();

		/**
		 * @type {number}
		 * @private
		 */
		this._maxZoom = 20;

		/**
		 * @type {string}
		 * @constant
		 */
		this.ZONE_INCLUDE = "include";
		/**
		 * @type {string}
		 * @constant
		 */
		this.ZONE_EXCLUDE = "exclude";

		this._currentZoneType = this.ZONE_INCLUDE;

		/**
		 * @type {number}
		 * @constant
		 */
		this.DEFAULT_ZOOM_LEVEL = 17;
	}

	/**
	 * @param {number} lat
	 * @param {number} lng
	 * @param {number} [zoomLevel] from 1 to _maxZoom
	 * @returns {Deferred}
	 */
	FlatMap.prototype.initMap = function(lat, lng, zoomLevel)
	{
		var deferred = $.Deferred();

		zoomLevel = typeof zoomLevel == "undefined" ? this.DEFAULT_ZOOM_LEVEL : zoomLevel;

		// some times leaflet incorrect search this path
		// L.Icon.Default.imagePath = "/sdi/lib/js/3rdparty/leaflet/images";

		var initPosition = new L.LatLng(lat, lng);
		this._map = new L.Map(document.querySelector(this._selector), {
			attributionControl: false,
			maxZoom: this._maxZoom
		});

		this._map.on('load', deferred.resolve);

		this._map.setView(initPosition, zoomLevel);

		var osmLayer = new L.TileLayer('http://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
			maxZoom: this._maxZoom,
			maxNativeZoom: 19
		});

		this._mapLayers = new L.Control.Layers({
			'OSM external': osmLayer
		}, {}, {collapsed: false})
			.addTo(this._map);

		var self = this;

		this._api.getIdentityAttributes()
			.done(function(response){
				var attributes = response.list;
				var isWMSLayer = attributes.GEO_SERVER != "no" && attributes.GEO_SERVER_WMS_URL;
				if (isWMSLayer)
				{
					var wmsLayer = new L.TileLayer.WMS(attributes.GEO_SERVER_WMS_URL, {
					    layers: attributes.GEO_SERVER_LAYER_LIST,
					    format: 'image/png',
					    transparent: true,
						maxZoom: self._maxZoom
					});

					self._map.addLayer(wmsLayer);
					self._mapLayers.addBaseLayer(wmsLayer, 'OSM');
				}

				require(["async!http://maps.googleapis.com/maps/api/js?v=3&sensor=false"], function(){
					var googleRoadmapLayer = new L.Google('ROADMAP', {
						maxZoom: self._maxZoom,
						maxNativeZoom: 20
					});
					var googleHybridLayer = new L.Google('HYBRID', {
						maxZoom: self._maxZoom,
						maxNativeZoom: 20
					});

					if (!isWMSLayer)
					{
						self._map.addLayer(googleHybridLayer);
					}

					self._mapLayers.addBaseLayer(googleHybridLayer, 'Google hybrid');
					self._mapLayers.addBaseLayer(googleRoadmapLayer, 'Google roadmap');

					self._searchControl = new L.Control.GeoSearch({
					    provider: new L.GeoSearch.Provider.Google(),
						showMarker: false
					})
						.addTo(self._map);
				});
			});

		this._map.addControl(new L.Control.Scale());

		return deferred.promise();
	};

	/**
	 *
	 * @param {string} TARGET_ZONE GeoJSON
	 */
	FlatMap.prototype.addDrawControl = function(TARGET_ZONE)
	{
		var self = this;

		if (this._editableLayers)
		{
			this._map.removeLayer(this._editableLayers);
		}

		this._editableLayers = new L.geoJson(TARGET_ZONE, {
			style: function (feature) {
				var zoneType = feature.properties.zoneType ? feature.properties.zoneType : self._polygonIncludeOptions;
				return self.getPolygonOptions(zoneType);
			}
		});

		this._map.addLayer(this._editableLayers);

		this._drawControl = new L.Control.Draw({
			draw: {
				polyline: false,
				polygon: {
					showArea: true,
					metric: true,
					allowIntersection: false,
					drawError: {
						color: '#CCe100',
						message: "<strong>Oh!<strong> you can't draw that!"
					},
					shapeOptions: this._polygonIncludeOptions
				},
				circle: false,
				rectangle: false,
				marker: false
			},
			edit: {
				featureGroup: this._editableLayers
			}
		});
		this._map.addControl(this._drawControl);

		this._map.on('draw:created', function (e){
			var type = e.layerType;
			var layer = e.layer;

			var geoJsonLayer = layer.toGeoJSON();
			geoJsonLayer.properties["zoneType"] = self._currentZoneType;

			self._editableLayers.addData(geoJsonLayer);
		});
	};

	FlatMap.prototype.fitZoneBounds = function()
	{
		// async call to prevent freeze of map, issue of leafletjs
		setTimeout(function(){
			var bounds = this._editableLayers.getBounds();
			if (bounds.isValid())
			{
				this._map.fitBounds(bounds);
			}
		}.bind(this), 0);
	};

	FlatMap.prototype.fitToAllBounds = function()
	{
		// async call to prevent freeze of map, issue of leafletjs
		setTimeout(function(){
			var bounds = this._map.getBounds();
			if (bounds.isValid())
			{
				this._map.fitBounds(bounds);
			}
		}.bind(this), 0);
	};

	/**
	 * @returns {string}
	 */
	FlatMap.prototype.getZoneGeoJSON = function()
	{
		return JSON.stringify(this._editableLayers.toGeoJSON());
	};

	/**
	 * @returns {L.Map}
	 */
	FlatMap.prototype.getMap = function()
	{
		return this._map;
	};

	/**
	 * @returns {L.geoJson}
	 */
	FlatMap.prototype.getEditableLayers = function()
	{
		return this._editableLayers;
	};

	/**
	 *
	 * @param {number} obj
	 * @param {Array} position
	 * @param {string} caption
	 */
	FlatMap.prototype.addCamera = function(obj, position, caption)
	{
		var cameraIcon = new L.Icon({
			iconUrl: "/sdi/map/images/marker-icon-camera-small.png",
			iconSize: [24, 24],
			iconAnchor: [2, 24],
			popupAnchor: [0, -24]
		});
		var self = this;
		var cameraMarker = new L.Marker(position, {
			icon: cameraIcon,
			title: caption,
			riseOnHover: true
		})
			.addTo(this._map)
			.on("click", function(){
				self._map.panTo(position);
				self.showVideoPlayer(obj, caption);
			});

		this._deviceMarkerList.push(cameraMarker);
	};

	/**
	 *
	 * @param {number} obj
	 * @param {Array} position
	 * @param {string} caption
	 */
	FlatMap.prototype.addSensor = function(obj, position, caption)
	{
		var sensorIcon = new L.Icon({
			iconUrl: "/sdi/map/images/marker-icon-sensor-small.png",
			iconSize: [27, 25],
			iconAnchor: [13, 25],
			popupAnchor: [0, -25]
		});
		var self = this;
		var sensorMarker = new L.Marker(position, {
			icon: sensorIcon,
			title: caption,
			riseOnHover: true
		})
			.addTo(this._map)
			.on("click", function(){
				self._map.panTo(position);
			});

		sensorMarker.bindPopup(caption);

		this._deviceMarkerList.push(sensorMarker);
	};

	/**
	 * @returns {L.Marker}
	 */
	FlatMap.prototype.getCameraMarker = function()
	{
		return this._deviceMarkerList[0];
	};

	/**
	 *
	 * @param {number} obj
	 * @param {string} caption
	 */
	FlatMap.prototype.showVideoPlayer = function(obj, caption)
	{

	};

	/**
	 *
	 * @param {ZONE_INCLUDE|ZONE_EXCLUDE} zoneType
	 */
	FlatMap.prototype.changeZoneType = function(zoneType)
	{
		this._currentZoneType = zoneType;

		this._drawControl.setDrawingOptions({
			polygon: {
				shapeOptions: this.getPolygonOptions(this._currentZoneType)
			}
		});
	};

	/**
	 *
	 * @param {ZONE_INCLUDE|ZONE_EXCLUDE} zoneType
	 * @returns {*}
	 */
	FlatMap.prototype.getPolygonOptions = function(zoneType)
	{
		var polygonOptions = {};
		polygonOptions[this.ZONE_INCLUDE] = this._polygonIncludeOptions;
		polygonOptions[this.ZONE_EXCLUDE] = this._polygonExcludeOptions;

		return polygonOptions[zoneType];
	};

	/**
	 *
	 * @returns {{lat: number, lng: number, zoomLevel: number}}
	 */
	FlatMap.prototype.getCurrentPosition = function()
	{
		var position = this._map.getCenter();
		var lat = position.lat;
		var lng = position.lng;
		var zoomLevel = this._map.getZoom();

		return {lat: lat, lng: lng, zoomLevel: zoomLevel};
	};

	/**
	 *
	 * @param {{lat: number, lng: number, zoomLevel: number}} position
	 */
	FlatMap.prototype.setCurrentPosition = function(position)
	{
		this._map.setView([position.lat, position.lng], position.zoomLevel);
	};

	return FlatMap;
});
