/**
 * @version $Id: eventlogclient.js 31894 2015-03-09 14:49:38Z atsybulnik $
 * ------------------------------------------------------------------------------
 * EventLog client
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2013
 * ------------------------------------------------------------------------------
 */

(function(window){
	"use strict";

	/**
	 * @type {EventLogClient}
	 */
	window.EventLogClient = EventLogClient;

	/**
	 *
	 * @constructor
	 */
	function EventLogClient()
	{
		/**
		 * matrix core plugin
		 * @type {HTMLElement|EventTarget}
		 * @private
		 */
		this._plugin = null;

		/**
		 * eventlogclient plugin
		 * @type {HTMLElement|EventTarget}
		 * @private
		 */
		this._eventLog = null;

		/**
		 * list of topics
		 *
		 * @private
		 */
		this._topics = {
			eventLog: {
				objectchange: {},
				objectaction: {},
				elogaction: {},
				log: {}
			}
		};

		/**
		 *
		 * @type {boolean}
		 * @private
		 */
		this._debug = false;
	}

	/**
	 * initialize media player
	 *
	 * @param {String} selector
     * @param {Object} [parameters]
	 * @returns {Deferred}
	 */
	EventLogClient.prototype.init = function(selector, parameters)
	{
		var deferred = $.Deferred();

		if (this._plugin)
		{
			deferred.resolveWith(this, [true]);
			return deferred.promise();
		}

		var settings = $.extend({
		}, parameters);

		this._id = "MatrixCore_" + Math.round((Math.random() * 10000));

		this._getManifest(function(manifest){
			var platform = "";
			if (navigator.appVersion.indexOf("Win") != -1)
			{
				platform = "win32";
			} else
			if (navigator.appVersion.indexOf("Mac") != -1)
			{
				platform = "macos";
			}

			this._mime = manifest.platform[platform].mime;

			var html = '<object id="'+ this._id + '" width="1" height="1" type="' + this._mime + '"></object>';

			$(selector).append(html);

			this._plugin = document.getElementById(this._id);

			var success = false;
			if (typeof this._plugin.eventLog !== "undefined")
			{
				this._addEvents();

				this._eventLog = this._plugin.eventLog;
				this._eventLog.init(settings);

				success = true;
			}
			if (success)
			{
				deferred.resolveWith(this, [false]);
			} else {
				deferred.rejectWith(this);
			}
		});

		return deferred.promise();
	};

	/**
	 * subscribe for event
	 *
	 * @param {String} eventName event name
	 * @param {Function} callback function
 	 * @param {String} [property]
	 */
	EventLogClient.prototype.subscribe = function(eventName, callback, property)
	{
		if (!this._plugin) return false;
		if (!eventName) return false;

		var topic = !!property ? this._topics[property][eventName] : this._topics[eventName];
		if (!topic) return false;

		if (!topic.callbacks)
		{
			topic = {
				callbacks: $.Callbacks()
			};

			if (!!property)
			{
				this._topics[property][eventName] = topic;
				Utils.addEvent(this._plugin[property], eventName, topic.callbacks.fire.bind(this));
			} else {
				this._topics[eventName] = topic;
				Utils.addEvent(this._plugin, eventName, topic.callbacks.fire.bind(this));
			}
		}

		topic.callbacks.add(callback);

		return true;
	};

	/**
	 * unsubscribe from event
	 *
	 * @param {String} eventName event name
	 * @param {Function} callback function
	 * @param {String} [property]
	 */
	EventLogClient.prototype.unsubscribe = function(eventName, callback, property)
	{
		if (!this._plugin) return false;
		if (!eventName) return false;
		var topic = !!property ? this._topics[property][eventName] : this._topics[eventName];

		if (topic)
		{
			if (!!property)
			{
				Utils.removeEvent(this._plugin[property], eventName, callback);
			} else {
				Utils.removeEvent(this._plugin, eventName, callback);
			}

			topic.callbacks.remove(callback);
		}

		return true;
	};

	/**
	 * simlulate publish event from media player object
	 *
	 * @param {String} eventName
	 * @param {Object} data
	 * @param {String} [property]
	 */
	EventLogClient.prototype.publish = function(eventName, data, property)
	{
		if (!this._plugin) return false;
		if (!eventName) return false;
		var topic = !!property ? this._topics[property][eventName] : this._topics[eventName];

		if (topic)
		{
			topic.callbacks.fire.apply(this, [data]);
		}

		return true;
	};

	/**
	 * @param {Function} callback
	 * @private
	 */
	EventLogClient.prototype._getManifest = function(callback)
	{
		var self = this;
		$.ajax({
			url: "/sdi/matrix/manifest.json",
			cache: false,
			dataType: "json"
		})
		.done(function(manifest){
			callback.apply(self, [manifest]);
		});
	};

	/**
	 * @private
	 */
	EventLogClient.prototype._addEvents = function()
	{
		if (this._debug)
		{
			this.subscribe("log", function(type, message){
				window.console && window.console.log("[" + type + "] " + message);
			}, "eventLog");
		}

		var self = this;

		AMQ.subscribe("topic://skm.eventlog.notifications", function(messageList){
			var eventList = [];

			messageList.forEach(function(message){
				try
				{
					var event = JSON.parse(message);

					// TODO: remove this hack, he is needed because FireBreath incorrectly parse JS object in FireFox
					var witnesses = event.eventNotification.event.witnesses;
					var objList = Object.keys(witnesses);
					objList.forEach(function(obj){
						witnesses["_" + obj] = $.extend({}, witnesses[obj]);
						delete witnesses[obj];
					});

					eventList.push(event);
				}
				catch (e)
				{
					window.console && window.console.error(e.name + " " + e.message);
				}
			});

			self.addEvents(eventList);
		});

		AMQ.subscribe("topic://skm.om.notifications", function(messageList){
			var objectList = [];

			messageList.forEach(function(message){
				try
				{
					var object = JSON.parse(message);

					// TODO: remove this hack, he is needed because FireBreath incorrectly parse JS object in FireFox
					var ObjectNotification = object.ObjectNotification;
					var ObjectNotificationList = Object.keys(ObjectNotification);
					ObjectNotificationList.forEach(function(obj){
						ObjectNotification["_" + obj] = $.extend({}, ObjectNotification[obj]);
						delete ObjectNotification[obj];
					});

					objectList.push(object);
				}
				catch (e)
				{
					window.console && window.console.error(e.name + " " + e.message);
				}
			});

			self.addObjects(objectList);
		});

		this.subscribe("log", function(type, message){
			// console.log(type, message);
		})
	};


	// AppletIface implementation

	/**
	 *
	 * @deprecated
	 * @returns Object
	 */
	EventLogClient.prototype.getServerURL = function()
	{
		if (!this._eventLog) return null;

		return this._eventLog.getServerURL();
	};

	/**
	 *
	 * @deprecated
	 * @param {number} channel
	 * @param {string} func
	 */
	EventLogClient.prototype.invokeCallback = function(channel, func)
	{
		if (!this._eventLog) return;

		this._eventLog.invokeCallback(channel, func);
	};

	/**
	 * @deprecated
	 * @returns {string}
	 */
	EventLogClient.prototype.getSession = function()
	{
		if (!this._eventLog) return null;

		return this._eventLog.getSession();
	};

	// Event Log API

	/**
	 *
	 * @param {boolean} all
	 * @returns {string}
	 */
	EventLogClient.prototype.getEvents = function(all)
	{
		if (!this._eventLog) return null;

		return this._eventLog.getEvents(all);
	};

	/**
	 *
	 * @param {Array} eventList
	 */
	EventLogClient.prototype.addEvents = function(eventList)
	{
		if (!this._eventLog) return;

		this._eventLog.addEvents(eventList);
	};

	/**
	 *
	 * @param {number} tag
	 * @param {boolean} tagOnFlag
	 * @param {number} eventType
	 * @param {number} priority
	 * @param {string} objids
	 * @param {string} message
	 * @param {number} startTs
	 * @param {number} endTs
	 * @param {number} pageNum
	 * @param {number} pageSize
	 * @returns {boolean}
	 */
	EventLogClient.prototype.setEventFilter = function(tag, tagOnFlag, eventType, priority, objids, message, startTs, endTs, pageNum, pageSize)
	{
		if (!this._eventLog) return false;

		return this._eventLog.setEventFilter(tag, tagOnFlag, eventType, priority, objids, message, startTs, endTs, pageNum, pageSize);
	};

	EventLogClient.prototype.reload = function()
	{
		if (!this._eventLog) return;

		this._eventLog.reload();
	};

	/**
	 *
	 * @returns {number}
	 */
	EventLogClient.prototype.getStatus = function()
	{
		if (!this._eventLog) return null;

		/*$output = array();
		exec(escapeshellcmd($command), $output, $return_var);
		if ($return_var != 0)
		{
			$error = sprintf(__("Return value: %s"), $return_var);
			return array(
				'error' => $error
			);
		}*/


		return this._eventLog.getStatus();
	};

	/**
	 *
	 * @param {string} name
	 * @param {Object} [parameters]
	 * @param {Object} [properties]
	 * @param {Array} [actions]
	 * @returns {Deferred}
	 */
	EventLogClient.prototype.submitAction = function(name, parameters, properties, actions)
	{
		var deferred = $.Deferred();
		if (!this._eventLog) return deferred.reject();

		parameters = parameters || {};
		properties = properties || {};
		actions = actions || [];

		var api = new API();
		api.submitAction({
			name: name,
			parameters: JSON.stringify(parameters),
			properties: JSON.stringify(properties),
			actions: JSON.stringify(actions)
		})
			.fail(function(code, message){
				var errorMessage = "[" + code + "] " + message;
				window.console && window.console.error(errorMessage);
				deferred.reject();
			})
			.done(function(response){
				deferred.resolve(response);
			});

		return deferred.promise();
	};

	// Object Management API

	/**
	 * request object list from server
	 *
	 * @param {string} jsonStr JSON string in the following format:
	 * {
	 *  "objids":[91, 92, 93]					- list of obj ids
	 *	, "properties":["NAME","TYPE",...], 	- (optional) list of properties
	 *	, "properties":"true" 					  or indicator that all obj properties shall be included
	 *  , "methods":"true"						- (optional) - indicate that method description shall be included
	 * }
	 * @returns {string} JSON string
	 * {"status":"OK",
	 *  "objects":[
	 *     {"objid":"<OBJID>",
	 *      "properties":[{"name":"NAME", "value":"VALUE"},{...}],
	 *      "methods":[{"name":"NAME", "displayname":"DISPLAYNAME", "desc": "DESC"},{...}]}]}
	 * or
	 * {"status":"ERROR", "MSG":"<TEXT>"}
	 */
	EventLogClient.prototype.getObjects = function(jsonStr)
	{
		if (!this._eventLog) return null;

		return this._eventLog.getObjects(jsonStr);
	};

	/**
	 * Method adds object into list of monitored objects and
	 * requests current object properties and methods from server
	 *
	 * @param {number} objid
	 * @returns {string} JSON string in the following format:
	 * {"status":"OK",
	 *  "object":{
	 *      "objid":"<OBJID>",
	 *      "properties":[{"name":"NAME", "value":"VALUE"},{...}],
	 *      "methods":[{"name":"NAME", "displayname":"DISPLAYNAME", "desc": "DESC"},{...}]}}
	 * or
	 * {"status":"ERROR", "MSG":"<TEXT>"}
	 */
	EventLogClient.prototype.addMonitoredObject = function(objid)
	{
		if (!this._eventLog) return null;

		return this._eventLog.addMonitoredObject(objid);
	};

	/**
	 * Remove object from monitored object list
	 *
	 * @param {number} objid
	 */
	EventLogClient.prototype.removeMonitoredObject = function(objid)
	{
		if (!this._eventLog) return;

		this._eventLog.removeMonitoredObject(objid);
	};

	/**
	 * removeAllMonitoredObjects
	 */
	EventLogClient.prototype.removeAllMonitoredObjects = function()
	{
		if (!this._eventLog) return;

		this._eventLog.removeAllMonitoredObjects();
	};

	/**
	 * Get list of object which were updated
	 *
	 * @returns {string}
	 * {"status":"OK",
	 *  "changes":[{"objid":"<OBJID>",
	 *   "properties":[{"name":"NAME", "value":"VALUE"},{...}]
	 *  }, {...}]}
	 *  or {"status":"ERROR", "MSG":"<TEXT>"}
	 */
	EventLogClient.prototype.getChanges = function()
	{
		if (!this._eventLog) return null;

		return this._eventLog.getChanges();
	};

	/**
	 * getObjectReceiverStatus
	 *
	 * @returns {number} 0 - notification receiver is off, 1 - connection/cache recovery,
	 *  2 - standby mode, cache is empty, 3 - cache contains new data
	 */
	EventLogClient.prototype.getObjectReceiverStatus = function()
	{
		if (!this._eventLog) return null;

		return this._eventLog.getObjectReceiverStatus();
	};

	/**
	 * invokeObjectMethod
	 *
	 * @param {number} objid
	 * @param {string} methodName
	 * @param {number} userId
	 * @param {string} userName
	 * @returns {string}
	 * {"status":"OK"}
	 * or {"status":"ERROR", "MSG":"<TEXT>"}
	 */
	EventLogClient.prototype.invokeObjectMethod = function(objid, methodName, userId, userName)
	{
		if (!this._eventLog) return null;

		return this._eventLog.invokeObjectMethod(objid, methodName, userId, userName);
	};

	/**
	 *
	 * @param {Array} objectList
	 */
	EventLogClient.prototype.addObjects = function(objectList)
	{
		if (!this._eventLog) return;

		this._eventLog.addObjects(objectList);
	};

})(window);
