/**
 * @version $Id:$
 * ------------------------------------------------------------------------------
 * workink local and global coordinates
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions LLC 2011
 * ------------------------------------------------------------------------------
 */
(function(window, undefined){
	"use strict";

	/**
	 * @type {GMap}
	 */
	window.GMap = GMap;

	/**
	 * @constructor
	 */
	function GMap()
	{
		var self = this;

		/**
		 * @type {google.maps.Map}
		 * @private
		 */
		this.map = null;

		var _params = {
			requestToGE: false,
			horizon: 8000,
			precision: {
				firstStep: 100, // meters
				secondStep: 2 // meters
			},
			camera: {
				fovX: 20.0,
				fovY: 15.0,
				tiltAngel: 0.0,
				azimuthAngle: 0.0,
				latitude: 0.0,
				longitude: 0.0,
				height: 1.0, // height relative to ground
				resolution: {
					width: 640,
					height: 480
				},
				homography: null
			},
			image: {
				x: 0,
				y: 0
			},
			target: {
				latitude: 0,
				longitude: 0,
				altitude: 0,
				distance: 0
			}
		};

		/**
		 * @type {google.maps.LatLng}
		 * @private
		 */
		this._cameraPoint = null;

		/**
		 * initialize google map on block
		 *
		 * @param {string} id
		 */
		this.init = function(id)
		{
			// init map
			var mapOptions = {
				zoom: 16,
				center: this._cameraPoint,
				scaleControl: true,
				// mapTypeId: google.maps.MapTypeId.TERRAIN
				mapTypeId: google.maps.MapTypeId.HYBRID
			};
			this.map = new google.maps.Map(document.getElementById(id), mapOptions);

			// draw start marker
			self.addMarker(this._cameraPoint);

			google.maps.event.addListener(this.map, 'click', function(event){
				self.addMarker(event.latLng);
			});
		};

		/**
		 * add market on map
		 *
		 * @param location
		 * @param {string} [label]
		 * @returns {google.maps.Marker}
		 */
		this.addMarker = function(location, label)
		{
			var pos = "(" + location.lat() + ", " + location.lng() + ")";

			var title = label ? pos + " " + label : pos;

			return new google.maps.Marker({
				position: location,
				map: this.map,
				title: title,
				zIndex: 999
			});
		};

		/**
		 * mark target
		 *
		 * @param {number} x
		 * @param {number} y
		 * @returns {Deferred}
		 */
		this.markTarget = function(x, y)
		{
			var deferred = $.Deferred();

			this.getTargetOnSurface(x, y)
				.done(function(latitude, longitude, H0, distance){
					// self.addMarker(new google.maps.LatLng(latitude, longitude), H0 + " " + distance);

					_params.target.latitude = latitude;
					_params.target.longitude = longitude;
					_params.target.altitude = H0 - _params.camera.height;
					_params.target.distance = distance;

					deferred.resolveWith(self);
				})
				.fail(function(){
					deferred.rejectWith(self);
				});

			return deferred.promise();
		};

		/**
		 * get plane coordinates of visible earth surface on image
		 *
		 * @returns {Deferred}
		 */
		this.getVisiblePlane = function()
		{
			var deferred = $.Deferred();

			$.when(
				this.getTargetOnSurface(0, 0),
				this.getTargetOnSurface(_params.camera.resolution.width - 1, 0),
				this.getTargetOnSurface(_params.camera.resolution.width - 1, _params.camera.resolution.height - 1),
				this.getTargetOnSurface(0, _params.camera.resolution.height - 1)
			)
				.done(function(target1, target2, target3, target4){
					// clock wise, from left upper corner
					deferred.resolve(target1, target2, target3, target4);
				})
				.fail(function(){
					deferred.reject();
				});

			return deferred.promise();
		};

		/**
		 * convert from degree to radian
		 * @param {number} value
		 */
		this.degToRad = function(value)
		{
			return value * Math.PI / 180;
		};

		/**
		 * convert from radian to degree
		 *
		 * @param {number} value
		 */
		this.radToDeg = function(value)
		{
			return value * 180 / Math.PI;
		};

		/**
		 * get camera parameters
		 *
		 * @returns {Object}
		 */
		this.getCameraParameters = function()
		{
			return _params.camera;
		};

		/**
		 * set camera parameters
		 *
		 * @param {Object} parameters
		 */
		this.setCameraParameters = function(parameters)
		{
			for (var param in parameters)
			{
				if (_params.camera[param] !== "undefined")
				{
					_params.camera[param] = parameters[param];
				}
			}

			/*
			if (typeof parameters["latitude"] !== "undefined" || typeof parameters["longitude"] !== "undefined")
			{
				this._cameraPoint = new google.maps.LatLng(_params.camera.latitude, _params.camera.longitude);
			}
			*/
		};

		/**
		 * get parameters
		 *
		 * @returns {Object} object
		 */
		this.getParameters = function()
		{
			return _params;
		};

		/**
		 * set parameters
		 *
		 * @param {Object} parameters
		 */
		this.setParameters = function(parameters)
		{
			for (var param in parameters)
			{
				if (_params[param] !== "undefined")
				{
					_params[param] = parameters[param];
				}
			}
		};

		/**
		 * set target id
		 *
		 * @param {string} id
		 */
		this.setTargetId = function(id)
		{
			_params.target.id = id;
		};

		/**
		 *
		 * @param {number} x
		 * @param {number} y
		 * @param {boolean} [requestToGE]
		 * @returns {{point: google.maps.LatLng, distance: number, tilt: number}}
		 */
		this.localToGlobal = function(x, y, requestToGE)
		{
			var internal = Utils.normalize([x, y, 1], _params.camera.resolution.width, _params.camera.resolution.height);
			var M = _params.camera.homography;
			var In = internal;

			var Ex = Utils.multiply(
				M, In
			);

			var mult = Ex[Ex.length - 1];

			var pointLat = Ex[0] / mult;
			var pointLong = Ex[1] / mult;

			var distance = this.distanceBetween2Points(this.degToRad(_params.camera.latitude), this.degToRad(_params.camera.longitude), this.degToRad(pointLat), this.degToRad(pointLong));
			// TODO: correct this
			var pixTilt = this.degToRad(30);

			return {
				point: {lat: pointLat, lng: pointLong},
				distance: distance,
				tilt: pixTilt
			};
		};

		/**
		 * get target on earth surface from point on screen
		 *
		 * @param {number} x
		 * @param {number} y
		 * @returns {Deferred}
		 */
		this.getTargetOnSurface = function(x, y)
		{
			var deferred = $.Deferred();

			_params.image.x = x;
			_params.image.y = y;

			var global = this.localToGlobal(x, y, _params.requestToGE);
			var pointLat = global.point.lat;
			var pointLong = global.point.lng;
			var distance = global.distance;

			var args = [pointLat, pointLong, 0, distance];

			deferred.resolveWith(this, args);

			return deferred.promise();
		};

		/**
		 * get target on earth surface by calling google earth API
		 *
		 * @param pathElevationRequest
		 * @param {number} length
		 * @param {number} addLength
		 * @param {number} pixTilt
		 * @param {number} H0
		 * @returns {Deferred}
		 */
		this.getTarget = function(pathElevationRequest, length, addLength, pixTilt, H0)
		{
			var deferred = $.Deferred();
			var self = this;

			var elevationService = new google.maps.ElevationService();
			elevationService.getElevationAlongPath(
				pathElevationRequest,
				function(results, status)
				{
					var distance = 0;
					var path = [];
					var args = [];
					if (status !== google.maps.ElevationStatus.OK)
					{
						var global = self.localToGlobal(_params.image.x, _params.image.y, false);
						var pointLat = global.point.lat;
						var pointLong = global.point.lng;
						var point = new google.maps.LatLng(self.radToDeg(pointLat), self.radToDeg(pointLong));
						distance = global.distance;

						path = [point, point];
						args = [path, H0, 0, distance];

						deferred.resolveWith(self, args);
						return;
						/*
						Log.error("Error: " + status);
						deferred.reject();
						/*
						google.maps.ElevationStatus.INVALID_REQUEST indicating the service request was malformed
						google.maps.ElevationStatus.OVER_QUERY_LIMIT indicating that the requestor has exceeded quota
						google.maps.ElevationStatus.REQUEST_DENIED indicating the service did not complete the request, likely because on an invalid parameter
						google.maps.ElevationStatus.UNKNOWN_ERROR indicating an unknown error
						*/
					}

					if (H0 == null)  H0 = results[0].elevation + _params.camera.height;
					var prevDistance = 0;
					var i = 1;
					for (i = 1; i < results.length; i++)
					{
						distance = addLength + i * length / results.length;
						var H = distance * Math.sin(pixTilt);
						var Hn = results[i].elevation + H;
						if (Hn >= H0)
						{
							path = [results[i - 1].location, results[i].location];
							var diffDistance = distance - prevDistance;
							args = [path, H0, diffDistance, prevDistance];

							deferred.resolveWith(self, args);

							break;
						}
						prevDistance = distance;
					}
					if (i == results.length)
					{
						path = [results[results.length - 1].location, results[results.length - 1].location];
						args = [path, H0, diffDistance, prevDistance];

						deferred.resolveWith(self, args);
					}
				}
			);

			return deferred.promise();
		};

		/**
		 *
		 * @param {number} lat1
		 * @param {number} lon1
		 * @param {number} lat2
		 * @param {number} lon2
		 * @returns {number}
		 */
		this.distanceBetween2Points = function(lat1, lon1, lat2, lon2)
		{
			var distance = 2 * Math.asin(Math.sqrt(
				Math.sin((lat1 - lat2) / 2) * Math.sin((lat1 - lat2) / 2) +
				Math.cos(lat1) * Math.cos(lat2) * ( Math.sin((lon1 - lon2) / 2) * Math.sin((lon1 - lon2) / 2))
			));
		    return 6371000 * distance;
		}
	}

})(window);
