/**
 * @version $Id:$
 * ------------------------------------------------------------------------------
 * camera GEO calibration
 * ------------------------------------------------------------------------------
 * @author Sergey Abaev
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2013
 * ------------------------------------------------------------------------------
 */

(function(){
	"use strict";

var minMode = window.minMode;

var api = new API();
var oldAttributes = {};
var attributes = {};

var ge_3d_mapping_mode = window.ge_3d_mapping_mode;
var ge_3d_mapping_hosted_key = window.ge_3d_mapping_hosted_key;
var ge_3d_mapping_hosted_datasource = window.ge_3d_mapping_hosted_datasource;

var objId = window.objId;
var jsServerName = window.jsServerName;
var jsServerPort = window.jsServerPort;
var jsServerProto = window.jsServerProto;
var ge = null;

var dev_type = window.dev_type;

var showAttributes = {
	"CAM_GEO_LAT": "Latitude [deg]",
	"CAM_GEO_LONG": "Longitude [deg]",
	"CAM_GEO_ALT": "Altitude [deg]",
	"CAM_GEO_NORTH_OFFSET": "North offset angle",
	"CAM_GEO_AZIMUTH": "Heading [deg]"
};

/**
 * @param {string} attribute
 * @param {string} value
 */
function showAttribute(attribute, value)
{
	if (!showAttributes[attribute])
	{
		return;
	}

	$("#attribute_wrapper div.attribute[data-attr='" + attribute + "']").html(value);
}

$(document).ready(function() {
	if (minMode)
	{
		$("#attribute_wrapper").hide();
		$("#camera").hide();
		$(".tabs").css("margin", "0 0");
		$(".tabs .tabs_menu").hide();
		$(".tabs .tabs_footer").hide();
		$(".tabs_content .buttons").hide();

		$("#calibration_block_wrapper").hide();

		$("#map3d_wrapper").css("width", "100%");
		$("#map3d_container").css("height", "450px");
		$("#map3d_search_device").hide();
	}

	$("#device_icon").attr("src", checkDeviceIconURL());

	$(".buttons button.back").click(function(){
		window.history.go(-1);
	});

	$(".buttons button.cancel").click(function(){
		window.location.reload();
	});

	$(".buttons button.apply").click(function(){
		saveAttributes();
	});

	if (dev_type == "camera" && !minMode)
	{
		var player = new MediaPlayer();

		player.init("#camera", {backgroundColor: 0xCACACA})
			.fail(function(){
				Log.error(__("Player not initialized"));
			})
			.done(function(isAgain){
				if (!isAgain)
				{
					this.subscribe("frame", function(timestamp, width, height){
						this.drawCrosshair(width, height, 0xFF00FF00);
					}, "player");
				}

				this.play(objId);
			});
	}

	$("#calibration_ruler").click(function(e){
		if (attributes["POSITIONCTL"] != "none")
		{
			cameraPtzCalibrate(e.originalEvent);
		}
	});

	$("#calibrator4 .up").click(function(){
		offsetTiltChange(1);
	});

	$("#calibrator4 .down").click(function(){
		offsetTiltChange(-1);
	});

	$("#map3d_search_device").click(function(){
		showDevice();
	});

	$("#map3d_search_model").click(function(){
		doSearchModel();
	});

	$("#map3d_search_address").click(function(){
		doSearch();
	});

	$("#map3d_search_data").keydown(function(event){
		if(event.keyCode == 13)
		{
			doSearch();
		}
	});

	ge_initialize();
});

if(ge_3d_mapping_mode == 'hosted') {
	if (!("google" in window)) { window.google = {}; }
	if (!("loader" in window.google)) { window.google.loader = {}; }
	if (!("earth" in window.google)) { window.google.earth = {}; }
	window.google.loader.ApiKey = ge_3d_mapping_hosted_key;
	window.google.loader.KeyVerified = true;
	window.google.earth.allowUsageLogging = false;
	window.google.earth.setErrorUrl("/sdi/ge/ge_error.php");
} else {
	google.load("earth", "1");
	google.load("maps", "3.x", {other_params: "sensor=false"});
}

var aPAN = null;
var offsetTilt = 0;
var offsetAngleFromNorth = null;

var gePlayerW = 400;
var gePlayerH = 300;
var geMapAngle = null;

//angle between NORTH and target
var targetAngle = 0;

var obj_3d_models_list = [];
var obj_3d_models_placemarks = {};

/*
 initialize google earth
 */
function ge_initialize() {
	api.getAttributes({obj: objId})
		.fail(function(code,message){
			Log.error(message);
		})
		.done(function(response){
			oldAttributes = $.extend({}, response.list);
			attributes = response.list;

			for (var attribute in attributes)
			{
				var value = attributes[attribute];
				showAttribute(attribute, value);
			}

			if(google){
				if(google.loader.LoadFailure) {
					$("#helpMessage").html(__("Install Google Earth plugin first!"));
				}
				else if (document.getElementById('map3d')) {
					document.getElementById("map3d_wrapper").style.display = "inline-block";

					if(ge_3d_mapping_mode == 'hosted') {
						google.earth.createInstance('map3d', ge_success, ge_failure, {database: ge_3d_mapping_hosted_datasource});
					} else {
						google.earth.createInstance("map3d", ge_success, ge_failure);
					}
				}
			}
			else {
				window.setTimeout(function() {
					ge_initialize();
				}, 250);
			}
		});
}

function setCameraEditorParameterValue(param, value, objid)
{
	attributes[param] = value;

	setTimeout(showAttribute.bind(this, param, value), 0);
}

function saveAttributes()
{
	var attributesForSave = {};
	for (var key in attributes)
	{
		if (typeof oldAttributes[key] != "undefined" && attributes[key] != oldAttributes[key])
		{
			attributesForSave[key] = attributes[key];
		}
	}

	api.setAttributes({
		obj: objId,
		attributes: JSON.stringify(attributesForSave)
	})
		.fail(function(code, message){
			Log.error(message);
		})
		.done(function(){
			Log.info("Save completed");
		})
}

/**
 *
 * @param {number} geMapAngle
 * @returns {number}
 */
function correctAngle(geMapAngle)
{
	var angle = geMapAngle;
	if (geMapAngle > 180) angle = 360 - geMapAngle;
	angle = Math.round(angle + offsetAngleFromNorth);
	if (Math.abs(angle) > 180) {
		if (angle > 0) {
			angle = -(360 - angle);
		} else {
			angle = 360 + angle;
		}
	}

	return angle;
}

/**
 *
 * @returns {boolean}
 */
function isDeviceIconSelected()
{
	return $("#device_icon").get(0).style.borderWidth == "1px";
}

/**
 *
 * @returns {boolean}
 */
function isTargetIconSelected()
{
	return $("#target_icon").get(0).style.borderWidth == "1px";
}

/**
 * on ge initialization success callback
 *
 * @param {GEPlugin} object
 */
function ge_success(object) {
	get_3d_models_data(obj_3d_models_list, null);

	ge = object;
	window.ge = ge;

	if (minMode)
	{
		parent.geoCalibrator.setEarthObject(ge);
		var isMove = false;
		google.earth.addEventListener(ge.getGlobe(), "click", function(event){
			setTimeout(parent.geoCalibrator.onEarthClick.bind(parent.geoCalibrator, event), 0);
		});
	}

	//ge.getLayerRoot().enableLayerById(ge.LAYER_BORDERS, true);
	////ge.getLayerRoot().enableLayerById(ge.LAYER_BUILDINGS, true);
	//ge.getLayerRoot().enableLayerById(ge.LAYER_ROADS, true);
	// Display all features by default. Potentially - introduce feature switcher.
	var geFeatures = ge.getLayerRoot().getFeatures().getChildNodes();
	for(var i=0; i<geFeatures.getLength(); i++) {
		ge.getLayerRoot().enableLayerById(geFeatures.item(i).getId(), true);
	}

	ge.getOptions().setAtmosphereVisibility(true);
	ge.getOptions().setStatusBarVisibility(true);
	ge.getWindow().setVisibility(true);
	ge.getNavigationControl().setVisibility(ge.VISIBILITY_AUTO);

	document.getElementById("device_icon").onclick = changeActivePlacemark;
	document.getElementById("target_icon").onclick = changeActivePlacemark;

	// parent.document.getElementById("CAM_GEO_LAT_" + objId).onchange = updateCameraParams;
	// parent.document.getElementById("CAM_GEO_LONG_" + objId).onchange = updateCameraParams;
	// parent.document.getElementById("CAM_GEO_ALT_" + objId).onchange = updateCameraParams;
	// parent.document.getElementById("CAM_GEO_RANGE_" + objId).onchange = updateCameraParams;
	// parent.document.getElementById("CAM_GEO_TILT_" + objId).onchange = updateCameraParams;
	// parent.document.getElementById("CAM_GEO_AZIMUTH_" + objId).onchange = updateCameraParams;

	// parent.document.getElementById("CAM_GEO_MODEL_" + objId).onchange = on_model_select;
	// parent.document.getElementById("CAM_GEO_LAYER_" + objId).onchange = on_layer_select;
	// parent.document.getElementById("CAM_GEO_ALT_MODE_" + objId).onchange = on_altitude_mode_change;

	offsetAngleFromNorth = parseInt(attributes["CAM_GEO_NORTH_OFFSET"], 10);

	if (dev_type != "camera") {
		$("#calibrator1").css("visibility", "hidden");
		$("#calibrator2").css("visibility", "hidden");
		$("#calibrator3").css("visibility", "hidden");
		$("#calibrator4").css("visibility", "hidden");
	}

	if(attributes["CAM_GEO_MODEL"] != ""){
		$("#map3d_search_device").prop("disabled", false);
		$("#map3d_search_model").prop("disabled", false);
	}

	updateCameraParams();

	if (!window.globeWindowEventListener) {
		window.globeWindowEventListener = function(event) {
			// run async to prevent block browser
			setTimeout(function(){
				if (isDeviceIconSelected()) {

					setCameraEditorParameterValue("CAM_GEO_LAT", event.getLatitude(), objId);
					setCameraEditorParameterValue("CAM_GEO_LONG", event.getLongitude(), objId);

					updateCameraParams();

					window.pointDevice.setLatitude(event.getLatitude());
					window.pointDevice.setLongitude(event.getLongitude());

					if (window.pointTarget != null) {
						geMapAngle = angleBy2Points(window.pointDevice.getLatitude() * Math.PI / 180,
								window.pointDevice.getLongitude() * Math.PI / 180,
								window.pointTarget.getLatitude() * Math.PI / 180,
								window.pointTarget.getLongitude() * Math.PI / 180) * 180 / Math.PI;
						var tmpA = correctAngle(-geMapAngle);

						var cameraLat = window.pointDevice.getLatitude() * Math.PI / 180;
						var cameraLng = window.pointDevice.getLongitude() * Math.PI / 180;
						var lat = window.pointTarget.getLatitude() * Math.PI / 180;
						var lng = window.pointTarget.getLongitude() * Math.PI / 180;

						var S = 6371000 * distanceBetween2Points(cameraLat, cameraLng, lat, lng);
						offsetTilt = -Math.atan2(attributes["CAM_GEO_ALT"], S) * 180 / Math.PI;

						ptzAbsoluteChange(objId, tmpA, Math.round(offsetTilt), "abs");
					}
				} else
				if (isTargetIconSelected()) {
					updateTargetParams(event.getLatitude(), event.getLongitude());
					geMapAngle = angleBy2Points(window.pointDevice.getLatitude() * Math.PI / 180,
							window.pointDevice.getLongitude() * Math.PI / 180,
							window.pointTarget.getLatitude() * Math.PI / 180,
							window.pointTarget.getLongitude() * Math.PI / 180) * 180 / Math.PI;

					var tmpA = correctAngle(-geMapAngle);

					var cameraLat = window.pointDevice.getLatitude() * Math.PI / 180;
					var cameraLng = window.pointDevice.getLongitude() * Math.PI / 180;
					var lat = window.pointTarget.getLatitude() * Math.PI / 180;
					var lng = window.pointTarget.getLongitude() * Math.PI / 180;

					var S = 6371000 * distanceBetween2Points(cameraLat, cameraLng, lat, lng);
					offsetTilt = -Math.atan2(attributes["CAM_GEO_ALT"], S) * 180 / Math.PI;

					var heading = calcAzimuth(cameraLat, cameraLng, lat, lng);
					if (attributes["POSITIONCTL"] == "none")
					{
						setCameraEditorParameterValue("CAM_GEO_AZIMUTH", heading);
					}

					ptzAbsoluteChange(objId, tmpA, Math.round(offsetTilt), "abs");
				}
			}, 0);
		}
	}
	google.earth.addEventListener(ge.getGlobe(), "click", window.globeWindowEventListener);
	google.earth.addEventListener(ge.getWindow(), "click", window.globeWindowEventListener);

	load_model_on_init();

	ge_monitor();
}

/*
 on ge initialization failure callback
 */
function ge_failure(object) {
	//alert("Google Earth not initialized!");
	//TODO user notification
}

function ge_monitor(){
	if(google.loader.LoadFailure || !google.loader.KeyVerified || google.earth.e){
		alert(__("Google Earth plugin crashed. Component reloaded."));
		window.location.href = window.location.href;
	}
	else{
		window.setTimeout(function() { ge_monitor(); }, 20000);
	}
}

/*
 load device model on init
 */
function load_model_on_init(){
	var model_id = attributes["CAM_GEO_MODEL"];
	var layer_id = attributes["CAM_GEO_LAYER"];
	if(model_id != '' && model_id != ''){
		var model = get_model_data_by_id(obj_3d_models_list, model_id);
		var layer = get_model_layer_data_by_id(obj_3d_models_list, model_id, layer_id);
		if(model && layer){
			load_layer_file(layer, obj_3d_models_placemarks);
		}
		else{
			window.setTimeout(function(){ load_model_on_init(); }, 500);
		}
	}
}

function cameraPtzCalibrate(e) {
	if ('target' in window && window.target != null) {
		if (!e) e = window.event;

		var offX = 0;
		var obj = $("#calibration_ruler");
		var strCaption = __("Pan Change [deg]");
		if (obj.attr("title") != strCaption){
			obj.attr("title", strCaption);
		}
		if (!e.offsetX) {
			offX = e.layerX - (findElementPos(obj.get(0))).left;
		}
		else {
			offX = e.offsetX;
		}
		if (isNaN(offX)) offX = 0;
		aPAN += (offX - Math.round(obj.width() / 2)) / 10;
		//if(aPAN > 180) aPAN = 180;
		//if(aPAN < -180) aPAN = -180;
		if (aPAN > 180) aPAN = -360 + aPAN;
		if (aPAN < -180) aPAN = 360 - aPAN;
		var aGE = -geMapAngle;
		if (geMapAngle > 180) aGE = 360 - geMapAngle;

		offsetAngleFromNorth = -Math.round(aGE - aPAN);
		var mod = Math.abs(offsetAngleFromNorth);
		var sig = offsetAngleFromNorth / mod;
		if (mod > 180) offsetAngleFromNorth = (-sig) * (360 - mod);

		setCameraEditorParameterValue("CAM_GEO_NORTH_OFFSET", offsetAngleFromNorth, objId);
		ptzAbsoluteChange(objId, Math.round(aPAN), Math.round(offsetTilt), "abs");
	}
	else {
		//document.getElementById("calibration_ruler").title = "Attn: Please, place the target on the map first.";
		alert(__("You have to put target on the 3D map before you use controls to center camera on the target."));
		return;
	}
}

function offsetTiltChange(inc) {
	if (!('target' in window) || window.target == null) {
		alert( __("You have to put target on the 3D map before you use controls to center camera on the target."));
		return;
	}
	offsetTilt += inc;
	if (offsetTilt > 90) offsetTilt = 90;
	if (offsetTilt < -90) offsetTilt = -90;

	var cameraLat = window.pointDevice.getLatitude() * Math.PI / 180;
	var cameraLng = window.pointDevice.getLongitude() * Math.PI / 180;
	var lat = window.pointTarget.getLatitude() * Math.PI / 180;
	var lng = window.pointTarget.getLongitude() * Math.PI / 180;

	var S = 6371000 * distanceBetween2Points(cameraLat, cameraLng, lat, lng);

	var alt = Math.round(Math.abs(Math.tan(offsetTilt * Math.PI / 180) * S) * 1000) / 1000; // round to mm
	setCameraEditorParameterValue("CAM_GEO_ALT", alt);

	ptzAbsoluteChange(objId, Math.round(aPAN), Math.round(offsetTilt), "abs");
}

function updateCameraParams() {
	if (minMode) return;

	var lat = attributes["CAM_GEO_LAT"];
	var lon = attributes["CAM_GEO_LONG"];
	var alt = attributes["CAM_GEO_ALT"];
	var altMode = attributes["CAM_GEO_ALT_MODE"];
	var range = attributes["CAM_GEO_RANGE"];
	var tilt = attributes["CAM_GEO_TILT"];
	var heading = attributes["CAM_GEO_AZIMUTH"];

	if(lat == "" && lon == ""){
		$("#map3d_search_device").prop("disabled", true);
		return;
	}
	else{
		$("#map3d_search_device").prop("disabled", false);
	}

	var isNum = true;
	if (IsNumeric(lat)) {
		lat = lat * 1.0;
	} else {
		isNum = false;
	}
	if (IsNumeric(lon)) {
		lon = lon * 1.0;
	} else {
		isNum = false;
	}
	if (IsNumeric(alt)) {
		alt = alt * 1.0;
	} else {
		alt = 0.0;
	}
	if (IsNumeric(range)) {
		range = range * 1.0;
	} else {
		range = 500.0;
	}
	if (IsNumeric(tilt)) {
		tilt = tilt * 1.0;
	} else {
		tilt = 0.0;
	}
	if (IsNumeric(heading)) {
		heading = heading * 1.0;
	} else {
		heading = 0.0;
	}

	var lookAt = ge.createLookAt('');

	if (isNum) {
		lookAt.setLatitude(lat);
		lookAt.setLongitude(lon);
		lookAt.setAltitude(alt);
		if (altMode == "absolute") {
			var altM = ge.ALTITUDE_ABSOLUTE;
		} else {
			var altM = ge.ALTITUDE_RELATIVE_TO_GROUND;
		}
		lookAt.setAltitudeMode(altM);
		lookAt.setRange(range);
		lookAt.setTilt(tilt);
		lookAt.setHeading(heading);
		ge.getView().setAbstractView(lookAt);

		if (!('device' in window)) {
			// create icon style for the placemark
			var icon = ge.createIcon('');
			icon.setHref(checkDeviceIconURL()/*'http://maps.google.com/mapfiles/kml/shapes/webcam.png'*/);
			var style = ge.createStyle('');
			style.getIconStyle().setIcon(icon);

			// create a point geometry
			var point = ge.createPoint('');
			point.set(lat, lon, alt, altM, true, false);

			// create the point placemark and add it to Earth
			var pointPlacemark = ge.createPlacemark('');
			pointPlacemark.setName('');
			pointPlacemark.setGeometry(point);
			pointPlacemark.setStyleSelector(style);

			ge.getFeatures().appendChild(pointPlacemark);

			window.device = pointPlacemark;
			window.pointDevice = point;

			if (dev_type == "camera" && !$.browser.safari && !$.browser.mozilla) {
				google.earth.addEventListener(window.device, 'click', function(event) {
					if (isDeviceIconSelected()) {
						return;
					}
					var div = document.createElement('DIV');
					var close_left = gePlayerW - 30;
					var content = '<div onclick="closeBalloon()" title="Close" style="width:12px;height:12px;background-image:url(\'/TibcoGI/JSXAPPS/mx/images/design1/m3d/ge_close_baloon.png\');cursor:pointer;margin-left:'+close_left+'px">&nbsp;</div>';

					 content+= '<div><iframe id = "id_ballon_iframe"' +
							'src = "/TibcoGI/JSXAPPS/mx/cell_player.php?name=geCalibrationPlayer&camera=' + objId + '&width=' + gePlayerW + '&height=' + gePlayerH + '"' +
							'width = "' + gePlayerW + '"' +
							'height = "' + gePlayerH + '"' +
							'scrolling = "no"' +
							'border = "0" / ></div>';
					div.innerHTML = content;

					// create the balloon and show it in Earth
					var balloon = ge.createHtmlDivBalloon('');
					balloon.setMaxWidth(800);
					balloon.setFeature(window.device);
					balloon.setContentDiv(div);
					balloon.setCloseButtonEnabled(false);
					ge.setBalloon(balloon);

					/*window.setTimeout(function() {
						var iframe = document.getElementById("id_ballon_iframe");
						iframe.contentWindow._mplayer_geCalibrationPlayer.showCenterSign(true);
						// Kept for further development

						 iframe.contentWindow._mplayer_geCalibrationPlayer._onImageClick = function (x,y)
						 {
						 clickOnScreenPTZ(obj_id, livePlayerWidth, livePlayerHeight, x, y);
						 };

					}, 1000);*/
				});
			}
			var layer_id = attributes["CAM_GEO_LAYER"];
			if(layer_id != ''){
				var model_id = attributes["CAM_GEO_MODEL"];
				var layer = get_model_layer_data_by_id(obj_3d_models_list, model_id, layer_id);
				if(layer){
					on_altitude_mode_change();
				}
				else{
					window.setTimeout(function(){ on_altitude_mode_change(true); }, 100);
				}
			}
		}
		else {
			window.pointDevice.set(lat, lon, alt, altM, true, false)
		}
	}
}

function closeBalloon(){
	document.getElementById("id_ballon_iframe").src = "";
	ge.setBalloon(null);
}

function updateTargetParams(lat, lon) {
	var alt = attributes["CAM_GEO_ALT"];
	var altMode = attributes["CAM_GEO_ALT_MODE"];
	if (IsNumeric(alt)) {
		alt = alt * 1.0;
	} else {
		alt = 0.0;
	}
	if (altMode == "absolute") {
		var altM = ge.ALTITUDE_ABSOLUTE;
	} else {
		var altM = ge.ALTITUDE_RELATIVE_TO_GROUND;
	}

	if (!('target' in window) || window.target == null) {
		// create icon style for the placemark
		var icon = ge.createIcon('');
		icon.setHref('http://maps.google.com/mapfiles/kml/paddle/red-circle.png');
		var style = ge.createStyle('');
		style.getIconStyle().setIcon(icon);

		// create a point geometry
		var point = ge.createPoint('');
		point.set(lat, lon, alt, altM, true, false)

		// create the point placemark and add it to Earth
		var pointPlacemark = ge.createPlacemark('');
		pointPlacemark.setName('');
		pointPlacemark.setGeometry(point);
		pointPlacemark.setStyleSelector(style);

		ge.getFeatures().appendChild(pointPlacemark);
		window.target = pointPlacemark;
		window.pointTarget = point;

		$("#calibration_ruler").attr("title", __("Pan Change [deg]"));
	}
	else {
		window.pointTarget.set(lat, lon, alt, altM, true, false)
	}

	geMapAngle = angleBy2Points(window.pointDevice.getLatitude() * Math.PI / 180,
			window.pointDevice.getLongitude() * Math.PI / 180,
			window.pointTarget.getLatitude() * Math.PI / 180,
			window.pointTarget.getLongitude() * Math.PI / 180) * 180 / Math.PI;


	aPAN = correctAngle(-geMapAngle);
}

function changeActivePlacemark(e) {
	var srcE = (e ? e.target : window.event.srcElement);
	var deviceIcon = $("#device_icon");
	var targetIcon = $("#target_icon");
	var c_border = deviceIcon.get(0).style.borderWidth;
	var t_border = targetIcon.get(0).style.borderWidth;

	if (srcE.id == "device_icon") {
		if(isTargetIconSelected()){
			if(window.target != null){
				ge.getFeatures().removeChild(window.target);
				window.target = null;
				window.pointTarget = null;
			}
			targetIcon.css("border-width", "0px");
		}

		if (!isDeviceIconSelected()) {
			deviceIcon.css("border-width", "1px");
		} else {
			deviceIcon.css("border-width", "0px");
		}
	} else
	if (srcE.id == "target_icon") {
		deviceIcon.css("border-width", "0px");
		if (!isTargetIconSelected()) {
			targetIcon.css("border-width", "1px");
		} else {
			targetIcon.css("border-width", "0px");
			if(window.target != null){
				ge.getFeatures().removeChild(window.target);
				window.target = null;
				window.pointTarget = null;
			}
		}
	}
}

function checkDeviceIconURL() {
	var deviceIcon = "";
	if (dev_type.search(/camera/) != -1) {
		deviceIcon = '/m3d/camera.gif';
	} else if (dev_type.search(/door/) != -1) {
		deviceIcon = "/resources/door.png";
	} else if (dev_type.search(/sensor/) != -1) {
		deviceIcon = "/resources/sensor.png";
	} else if (dev_type.search(/relay/) != -1) {
		deviceIcon = "/resources/relay.png";
	}

	var url_prefix = jsServerProto + "://" + jsServerName;
	if (jsServerPort != 80) {
		url_prefix += ":" + jsServerPort;
	}

	//return correct icon
	return url_prefix + "/TibcoGI/JSXAPPS/mx/images/design1" + deviceIcon;
}

/*
 on 3d model select
 */
function on_model_select(){
	var model_id = attributes["CAM_GEO_MODEL"];
	var find_model_btn = $("#map3d_search_model");

	if(model_id == ""){
		attributes["CAM_GEO_ALT_MODE"] = "relativeToGround";
		// alt_mode.prop("disabled", false);
		find_model_btn.prop("disabled", true);
	}
	else{
		attributes["CAM_GEO_ALT_MODE"] = "relativeToModel";
		// alt_mode.prop("disabled", true);
		find_model_btn.prop("disabled", false);
	}

	hide_all_models_placemarks(obj_3d_models_placemarks);
}

/*
 on 3d model layer select
 */
function on_layer_select(){
	var model_id = attributes["CAM_GEO_MODEL"];
	var layer_id = attributes["CAM_GEO_LAYER"];

	if(model_id == ""){
		return;
	}

	hide_all_models_placemarks(obj_3d_models_placemarks);

	var model = get_model_data_by_id(obj_3d_models_list, model_id);
	var layer = get_model_layer_data_by_id(obj_3d_models_list, model_id, layer_id);

	if(model.layers === undefined){
		window.setTimeout(function(){ on_layer_select(); }, 500);
		return;
	}

	for (var i = 0; i < model.layers.length; i++) {
		if(model.layers[i].attributes.NUMBER <= layer.attributes.NUMBER){
			var placemark = get_layer_placemark(obj_3d_models_placemarks, model.layers[i].obj);
			if (placemark == null) {
				if(!$.isEmptyObject(model.layers[i].blocks)){
					load_layer_file(model.layers[i], obj_3d_models_placemarks);
				}
				else if(model.layers[i].attributes.NUMBER == layer.attributes.NUMBER){
					alert(__("No 3D file in this layer!"));
				}
			}
			else{
				placemark.setVisibility(true);
			}
		}
	}
	on_altitude_mode_change();
}

/*
 move device due to altitude mode value
 */
function on_altitude_mode_change(is_wait){
	var alt = parseFloat(attributes["CAM_GEO_ALT"]);
	var alt_mode = attributes["CAM_GEO_ALT_MODE"];
	var device_placemark = window.pointDevice;
	if(alt_mode == "absolute"){
		device_placemark.setAltitude(alt);
	}
	else if(alt_mode == "relativeToGround"){
		device_placemark.setAltitude(alt);
	}
	else if(alt_mode == "relativeToModel"){
		var model_id = attributes["CAM_GEO_MODEL"];
		var layer_id = attributes["CAM_GEO_LAYER"];
		var layer_data = get_model_layer_data_by_id(obj_3d_models_list, model_id, layer_id);
		var model_data = get_model_data_by_id(obj_3d_models_list, model_id);

		if((layer_data == null && is_wait) || device_placemark === undefined){
			window.setTimeout(function(){ on_altitude_mode_change(true); }, 500);
			return;
		}

		var layer_height = calculate_layer_height(model_data, layer_data);
		var new_alt = alt + layer_height;
		device_placemark.setAltitude(new_alt);
	}
}

//-------------------------------->
// click-on-screen PTZ Functions
function clickOnScreenPTZ(objId, playerW, playerH, x, y) {
	var Direction_C = "<PTZ_Command>do.ptz?dev=" + objId + "&mode=rel&size=" + playerW + 'x' + playerH + "&xy=" + x + "," + y + "</PTZ_Command>";

	$.ajax({
		url: "/ptz/cgi-bin/send_message.pl",
		data: {data: Direction_C},
		success: function() {}
	});
}

function ptzAbsoluteChange(objId, pan, tilt, mode) {
	var pt = "<PTZ_Command>do.ptz?dev=" + objId + "&mode=" + mode + "&pt=" + pan + "," + tilt + "</PTZ_Command>";

	$.ajax({
		url: "/ptz/cgi-bin/send_message.pl",
		data: {data: pt},
		success: function() {}
	});
}

function findElementPos(obj) {
	var curleft = 0;
	var curtop = 0;
	if (obj.offsetParent) {
		while (obj.offsetParent) {
			curleft += obj.offsetLeft - obj.scrollLeft;
			curtop += obj.offsetTop - obj.scrollTop;
			var position = '';
			if (obj.style && obj.style.position) position = obj.style.position.toLowerCase();
			if ((position == 'absolute') || (position == 'relative')) break;
			while (obj.parentNode != obj.offsetParent) {
				obj = obj.parentNode;
				curleft -= obj.scrollLeft;
				curtop -= obj.scrollTop;
			}
			obj = obj.offsetParent;
		}
	}
	else {
		if (obj.x) curleft += obj.x;
		if (obj.y) curtop += obj.y;
	}
	return {left:curleft,top:curtop};
}

/*
 search location
 */
function searchLocation(location) {
	var geocoder = new google.maps.Geocoder();

	geocoder.geocode({
		address: location.toString()
		},
		function(GeocoderResults) {
			if (GeocoderResults) {
				var lat = GeocoderResults[0].geometry.location.lat();
				var lng = GeocoderResults[0].geometry.location.lng();
				var lookAt = ge.createLookAt('');
				lookAt.set(lat, lng, 10, ge.ALTITUDE_RELATIVE_TO_GROUND, 0, 0, 3000);
				ge.getView().setAbstractView(lookAt);
			}
		}
	);
}

/*
 search location wrapper
 */
function doSearch() {
	searchLocation($('#map3d_search_data').val());
}

/*
 find model location
 */
function doSearchModel() {
	var model_id = attributes["CAM_GEO_MODEL"];
	var model = get_model_data_by_id(obj_3d_models_list, model_id);
	var empty_layers = 0;
	var load_file = null;

	if (model && model.layers)
	{
		for (var i = 0; i < model.layers.length; i++) {
			var placemark = get_layer_placemark(obj_3d_models_placemarks, model.layers[i].obj);
			if (placemark == null) {
				if(!$.isEmptyObject(model.layers[i].blocks)){
					load_file = model.layers[i];
				}
				else{
					empty_layers++;
				}
			}
			else{
				var lat = placemark.getGeometry().getLocation().getLatitude();
				var lon = placemark.getGeometry().getLocation().getLongitude();
				change_view(lat, lon, 30, 100, null, 30);
				load_file = null;
			}
		}

		//if all layers are empty
		if(empty_layers == model.layers.length){
			alert(__("No files uploaded for this 3D model!"));
		}
	}

	//if need to load file
	if(load_file != null){
		load_layer_file(load_file, obj_3d_models_placemarks, true);
		window.setTimeout(function(){ doSearchModel(); }, 500);
	}

}

/*
 show camera location
 */
function showDevice() {
	var lat = parseFloat(attributes["CAM_GEO_LAT"]);
	var lon = parseFloat(attributes["CAM_GEO_LONG"]);
	var alt = parseFloat(attributes["CAM_GEO_ALT"]);
	var altMode = attributes["CAM_GEO_ALT_MODE"];
	var range = parseFloat(attributes["CAM_GEO_RANGE"]);
	var tilt = parseFloat(attributes["CAM_GEO_TILT"]);
	var heading = parseFloat(attributes["CAM_GEO_AZIMUTH"]);

	var lookAt = ge.createLookAt('');

	lookAt.setLatitude(lat);
	lookAt.setLongitude(lon);
	lookAt.setAltitude(alt);
	var altM = ge.ALTITUDE_RELATIVE_TO_GROUND;
	if (altMode == "absolute") {
		altM = ge.ALTITUDE_ABSOLUTE;
	}
	lookAt.setAltitudeMode(altM);
	lookAt.setRange(range);
	lookAt.setTilt(tilt);
	lookAt.setHeading(heading);
	ge.getView().setAbstractView(lookAt);
}

})();
