/**
 * @version $id:$
 * ------------------------------------------------------------------------------
 * description
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2013
 * ------------------------------------------------------------------------------
 */

(function(){
	"use strict";

	var analytics = window.analytics;

	/**
	 * formBuilder is responsible for building html input fields and forms, their serializing and deserializing
	 */
	analytics.formBuilder = {

		/**
		 * Build HTML list of input fields accordingly to given schema
		 * @param {Object} data schema of input fields
		 * @param id
		 * @returns {String}
		 */
		getFormHtml: function(data, id) {
			var key, field, html = '';

			for (key in data) {
				field = data[key];

				var category = '';
				if (field.category) {
					category = "data-category = '" + JSON.stringify(field.category) + "'";
				}

				var visible = typeof field.visible == "undefined" ? true : field.visible;
				var style = visible ? "" : 'style="display: none;"';
				html += '<li ' + category + ' ' + style + '><div class="clear">' + this.getFieldHtml(key, field) + '</div></li>';
			}

			return '<ul ' + (id ? 'id="' + id + '"' : '') + '>' + html + '</ul>';
		},

		buildOVForm: function(formId, schema, categories) {

			// Get all uncategorized parameters and add them first
			var uncategorizedParamsSchema = {};
			var categorizedParamsSchema = {};
			var key, field, html;

			for (key in schema) {
				field = schema[key];
				if (!field.category || !$.isArray(field.category)) {
					uncategorizedParamsSchema[key] = field;
				} else {
					categorizedParamsSchema[key] = field;
				}
			}

			html = this.getFormHtml(uncategorizedParamsSchema);
			html += this.getCategoriesFilterHtml(categories);
			html += this.getFormHtml(categorizedParamsSchema, 'ov_additional_params');

			$(formId).html(html);

			var catFormInputs = $('#ov_additional_params li');

			$(formId + ' select[name=category_filter]').change(function() {
				var category = this.value;

				catFormInputs
					.hide()
					.filter(function() {
						var $this = $(this);

						// View only elements without category (needed to be visible all the time) and current category
						return !$this.data('category') || $.inArray(category, $this.data('category')) != -1;
					}).show();

			}).change();

			$(formId + ' input.switch_parameter').change(function() {
				if (this.checked) {
					$(this).parent().next().find("input, select").prop('disabled', false);
					$(this).parents('li').removeClass('disabled');
				} else {
					$(this).parent().next().find("input, select").prop('disabled', true);
					$(this).parents('li').addClass('disabled');
				}
			}).parents('li').addClass('disabled').end() // all checkable inputs must be disabled initially
			  .parent().next().find("input, select").prop('disabled', true);
		},

		getCategoriesFilterHtml: function(categories) {
			var html = '';
			for (var key in categories) {
				html += '<option value="' + key + '">' + categories[key] + '</option>';
			}

			return '<div style="margin-top: 20px; white-space: nowrap;">Additional parameters: <select name="category_filter">' + html + '</select></div>';
		},

		getFieldHeader: function(key, field) {

			var header = field.description || i;

			if (field.properties && field.properties.meaunits) {
				header += ' (' + field.properties.meaunits + ')';
			}

			if (field.longdescr) {
				header = '<span title="' + field.longdescr + '">' + header + '</span>';
			}
			if (field.checkable) {
				header = '<label><input type="checkbox" name="' + key + '_enabled" class="switch_parameter" /> ' + header + '</label>';
			}

			return header;
		},

		getFieldHtml: function(key, field) {
			var result = null;
			if (field.type === 'boolean' || field.type === 'bool') {
				result = this.getCheckFieldHtml(key, field);
			}
			if (field.type === 'string' && $.isArray(field['enum'])) {
				result = this.getSelectFieldHtml(key, field);
			}else if (field.type === 'string') {
				result = this.getStringFieldHtml(key, field);
			}
			if (field.type === 'integer' || field.type === 'number') {
				result = this.getNumberFieldHtml(key, field);
			}
			return result;
		},

		/**
		 * Get HTML for checkbox field
		 * @param {String} key name of field
		 * @param {Object} field schema for select field
		 * @returns {String} HTML
		 */
		getCheckFieldHtml: function(key, field) {
			return '<label><input type="checkbox" name="' + key + '" /> ' + this.getFieldHeader(key, field) + '</label>';
		},

		/**
		 * Get HTML for range field, including min and max values and step (0.01 for floats, 1 for integers)
		 * @param {String} key name of field
		 * @param {Object} field schema of field
		 * @returns {String} HTML
		 */
		getNumberFieldHtml: function(key, field) {

			var bounds = (field.minimum != undefined && field.maximum != undefined) ? 'min="' + field.minimum + '" max="' + field.maximum + '"' : '';
			var step = (field.type === 'integer') ? ' step="1" ': ' step="0.01" '; // Range input step
			var floatrange = (field.floatrange === 1) ? 'floatrange ="1"' : "";
			var value = field['default'] ? field['default'] : '';
			var html = '';

			if (bounds) {
				html = '<input type="range" name="' + key + '" ' + bounds + step + floatrange + ' value="' + value + '" />';
			} else {
				html = '<input type="text" class="text" name="' + key + '" value="' + value + '" />';
			}

			return this.getFieldHeader(key, field) + '<div class="input_wrapper">' + html + '</div>';
		},

		/**
		 * Get HTML for select combobox
		 * @param {String} key name of field
		 * @param {Object} field schema for select field
		 * @returns {String} HTML
		 */
		getSelectFieldHtml: function(key, field) {
			var html = '<select name="' + key + '" ' + (field.multiple ? 'multiple' : '') + '>';

			for (var i=0; i<field['enum'].length; i++) {
				var value, label;

				if ($.isPlainObject(field['enum'][i])) {
					for (value in field['enum'][i]) {}
					label = field['enum'][i][value];
				} else {
					value = label = field['enum'][i];
				}
				html += '<option value="' + value + '">' + label + '</option>';
			}

			html += '</select>';

			return this.getFieldHeader(key, field) + '<div class="input_wrapper">' + html + '</div>';
		},

		/**
		 * Get HTML for range field, including min and max values and step (0.01 for floats, 1 for integers)
		 * @param {String} key name of field
		 * @param {Object} field schema of field
		 * @returns {String} HTML
		 */
		getStringFieldHtml: function(key, field) {
			var maxLength = field['maxLength'] ? 'size="'+field['maxLength']+'"' : '';
			var value = field['default'] ? field['default'] : '';
			var html = '';
			html = '<input type="text" class="text" ' + maxLength + ' name="' + key + '" value="' + value + '" />';

			return this.getFieldHeader(key, field) + '<div class="input_wrapper">' + html + '</div>';
		},
		/**
		 * Get HTML for event rule
		 * @param {String} name Name of event rule
		 * @param {Object} data Schema of event rule
		 * @returns {String} HTML
		 */
		getRuleHtml: function(name, data) {
			var html = '<label><input type="checkbox" name="' + name + '" /> ' + data.description + '</label>';
			if (data.properties) {
				html += this.getFormHtml(data.properties);
			}
			return '<li id="rule_' + name + '">' + html + '</li>';
		},

		/**
		 * Check correctness of given number
		 * @param {String} value field value, supposed to be numeric
		 * @param {String} type number type, integer|number
		 * @returns {Number|Boolean} Numerical value or false if value is not a number
		 */
		checkNumber: function(value, type) {
			value = $.trim(value);

			if (value === '')
				return null;

			if (type === 'integer') {
				if (/^[+-]?[0-9]+$/.test(value)) {
					return parseInt(value, 10);
				} else {
					return false;
				}
			}
			if (type === 'number') {
				if (/^[-+]?[0-9]*\.?[0-9]+$/.test(value)) {
					// return fixed point float with 2 digits after point
					return parseFloat(parseFloat(value).toFixed(2));
				} else {
					return false;
				}
			}
			return false;
		},

		/**
		 * Get "checkable" parameters
		 * @param id dom id of form to get parameters
		 */
		getSwitchFields: function(id) {
			var fields = {};

			$(id).find('input.switch_parameter:checkbox').each(function() {
				fields[this.name] = this;
			});

			return fields;
		},

		/**
		 * get DOM fields array by name
		 * @param id dom id of form
		 */
		getFields: function(id) {
			var fields = {};

			$(id).find('input:not(.switch_parameter), select').each(function() {
				fields[this.name] = this;
			});

			return fields;
		},

		/**
		 * Store form field from given block to JS hash
		 * @param {String} id DOM id of block which contains input field
		 * @param {Object} schema schema of input fields; if given, we check correctness of field values in compliance with schema
		 * @throws {Object} throws inputError when schema is given and some field is not correct
		 * @returns {Object} JS hash with values of fields
		 */
		serializeForm: function(id, schema) {
			var result = {}, field, fieldSchema, value, key;

			var switchFields = this.getSwitchFields(id);
			var fields = this.getFields(id);

			for (key in schema) {

				field = fields[key];
				if (!field)
					continue;

				fieldSchema = schema[key];

				if (fieldSchema.checkable === true) {
					result[key + '_enabled'] = switchFields[key + '_enabled'].checked;
				}

				switch (fieldSchema.type) {
					case 'boolean':
					case 'checkbox':
						value = field.checked ? true : false;
					break;
					case 'integer':
					case 'number':
						value = this.checkNumber(field.value, fieldSchema.type);
						if (value === false)
							throw {
								name: 'inputError',
								message: 'Incorrect number value in field "' + fieldSchema.description + '"',
								field: field
							};
					break;
					default:
						value = $(field).val();
				}

				result[key] = value;

			}
			return result;
		},

		/**
		 * Fill with data all input fields in given block
		 * @param {String} id DOM id of block
		 * @param {Object} data Data to be filled, name=>value
		 * @returns {boolean}
		 */
		unserializeForm: function(id, data) {

			if (!data)
				return false;

			var fields = $(id).find('input, select');

			for (var name in data) {
				var value = data[name] == null ? 0 : data[name];

				var field = fields.filter('[name="' + name + '"]');
				if (!field.size())
					continue;

				field = field[0];

				if (field.type === 'checkbox') {
					field.checked = value;
				} else {
					$(field).val(value);
				}

				// Trigger events to get consistent state of system and correct view of rangeinputs
				// i.e. some checkboxes turn on/off some rules, so we must trigger change() event for them
				field = $(field);
				field.triggerHandler( "changeField!", [name, value] );
				field.change();
				if (field.data("rangeinput")) {
					field.data("rangeinput").setValue(value);
				}
			}

			return true;
		}
	};

})();
