/**
 * @version $Id: ptzCalibrator.js 33597 2016-02-16 16:00:29Z astarostin $
 * ------------------------------------------------------------------------------
 * description
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2012
 * ------------------------------------------------------------------------------
 */

define("ptzcalibrator", ["jquery", "api", "utils", "i18n", "mediaplayer", "pointlist", "hwj"], function(){
	"use strict";

	var PointList = require("pointlist");

	/**
	 * @constructor
	 * @augments PointList
	 */
	function PTZCalibrator()
	{
		/**
		 * @type {Object}
		 * @private
		 */
		this._config = {};

		/**
		 * @type {API}
		 * @private
		 */
		this._api = new API();

		/**
		 * @type {Array}
		 * @private
		 */
		this._player = [
			{
				obj: null,
				player: new MediaPlayer(),
				attributes: {}
			},
			{
				obj: null,
				player: new MediaPlayer(),
				attributes: {}
			}
		];

		/**
		 * @type {number}
		 * @private
		 */
		this._currentConfigObj = null;

		/**
		 * @type {MediaPlayer}
		 * @private
		 */
		this._associatedPlayer = this._player[0].player;

		/**
		 * @type {number}
		 * @private
		 */
		this._pointColor = 0xFFFF0000;

		/**
		 * @type {number}
		 * @private
		 */
		this._selectedPointColor = 0xFF00FF00;

		/**
		 * @type {{pan: null|number, tilt: null|number, hfov: null|number, step: number}}
		 * @private
		 */
		this._position = {
			pan: null,
			tilt: null,
			hfov: null,
			step: 10
		};
	}

	Utils.inherit(PTZCalibrator, PointList);

	PTZCalibrator.prototype.init = function(obj)
	{
		var self = this;

		// skm350
		this._player[0].obj = obj;

		var player1 = $("#player1");
		var player2 = $("#player2");

		this._load();

		$(".buttons button.back").click(function(){
			window.history.go(-1);
		});

		$(".buttons button.cancel").click(function(){
			self._load();
		});

		$(".buttons button.apply").click(function(){
			var PTZ_CALIBRATOR_CONFIG = self._config;

			self._api.setAttributes({
				obj: self._player[0].obj,
				attributes: JSON.stringify({PTZ_CALIBRATOR_CONFIG: JSON.stringify(PTZ_CALIBRATOR_CONFIG)})
			})
				.fail(function(code, message){
					Log.error("[" + code + "]" + __(message));
				})
				.done(function(){
					Log.info(__("Configuration saved"));
				});
		});

		player2.find(".playerPTZControls select.ptz_step").change(function(){
			self._position.step = parseFloat(this.value);
		});

		player2.find(".playerPTZControls .ptz_top").click(function(){
			var new_pos = self._position.tilt - self._position.step;
			self._position.tilt = new_pos > -90 ? new_pos : -90;

			player2.find(".move_pan_tilt input.tilt").val(self._position.tilt);
			self._goToPosition();
		});
		player2.find(".playerPTZControls .ptz_bottom").click(function(){
			var new_pos = self._position.tilt + self._position.step;
			self._position.tilt = new_pos < 90 ? new_pos : 90;

			player2.find(".move_pan_tilt input.tilt").val(self._position.tilt);
			self._goToPosition();
		});
		player2.find(".playerPTZControls .ptz_right").click(function(){
			var new_pos = self._position.pan + self._position.step;
			self._position.pan = new_pos < 180 ? new_pos : new_pos - 360;

			player2.find(".move_pan_tilt input.pan").val(self._position.pan);
			self._goToPosition();
		});
		player2.find(".playerPTZControls .ptz_left").click(function(){
			var new_pos = self._position.pan - self._position.step;
			self._position.pan = new_pos > -180 ? new_pos : new_pos + 360;

			player2.find(".move_pan_tilt input.pan").val(self._position.pan);
			self._goToPosition();
		});
		player2.find(".playerPTZControls .ptz_zoom_in").click(function(){
			var attributes = self._player[1].attributes;
			if (self._position.hfov > parseFloat(attributes.PTZ_HFOV_MIN))
			{
				var new_pos = self._position.hfov - self._position.step;
				self._position.hfov = new_pos > parseFloat(attributes.PTZ_HFOV_MIN) ? new_pos : parseFloat(attributes.PTZ_HFOV_MIN);

				player2.find(".move_fov input.hfov").val(self._position.hfov);
				self._zoomToPosition();
			}
		});
		player2.find(".playerPTZControls .ptz_zoom_out").click(function(){
			var attributes = self._player[1].attributes;
			if (self._position.hfov < parseFloat(attributes.PTZ_HFOV_MAX))
			{
				var new_pos = self._position.hfov + self._position.step;
				self._position.hfov = new_pos < parseFloat(attributes.PTZ_HFOV_MAX) ? new_pos : parseFloat(attributes.PTZ_HFOV_MAX);

				player2.find(".move_fov input.hfov").val(self._position.hfov);
				self._zoomToPosition();
			}
		});

		$(".pointList button.remove").click(function(){
			var selectedPointId = $(".pointList select.list option:selected").val();
			if (selectedPointId)
			{
				self._removePoint(selectedPointId);
				$(".pointList select.list option[value='" + selectedPointId + "']").remove();
				$(".pointList select.list option").first().prop("selected", true);

				self._calculatePositionLimit();
				self._displayPositionLimit();
			}
		});

		$(".pointList button.apply").click(function(){
			var selectedOption = $(".pointList select.list option:selected");
			var selectedPointId = selectedOption.val();
			var pointList = self._getCurrentPointList();
			if (selectedPointId)
			{
				var point = pointList[selectedPointId].position = $.extend({}, self._position);
				var text = selectedPointId + ". pan = " + point.pan + ", tilt = " + point.tilt + ", hfov = " + point.hfov;
				selectedOption.text(text);

				self._calculatePositionLimit();
				self._displayPositionLimit();
			}
		});

		$(".pointList select.list").change(function(){
			var selectedPointId = $(".pointList select.list option:selected").val();
			if (selectedPointId)
			{
				var pointList = self._getCurrentPointList();
				var point = pointList[selectedPointId];
				if (point.position)
				{
					self._position = $.extend({}, point.position);
					$(".move_pan_tilt input.pan").val(point.position.pan);
					$(".move_pan_tilt input.tilt").val(point.position.tilt);
					$(".move_fov input.hfov").val(point.position.hfov);

					self._goToPosition()
						.then(setTimeout(self._zoomToPosition.bind(self), 500));
				}
				self._selectPoint(selectedPointId);
			}
		});
	};

	/**
	 * @private
	 */
	PTZCalibrator.prototype._calculatePositionLimit = function()
	{
		var point;
		var pointList = this._getCurrentPointList();
		var positionLimit = this._getCurrentPositionLimit();
		if (Object.keys(pointList).length > 0)
		{
			point = pointList[Object.keys(pointList)[0]];
			positionLimit.pan.min = positionLimit.pan.max = point.position.pan;
			positionLimit.tilt.min = positionLimit.tilt.max = point.position.tilt;
			positionLimit.hfov.min = positionLimit.hfov.max = point.position.hfov;
		}
		for (var id in pointList)
		{
			if (!pointList.hasOwnProperty(id))
				continue;

			point = pointList[id];

			if (!point.position)
				continue;

			var pan = point.position.pan;
			var tilt = point.position.tilt;
			var hfov = point.position.hfov;

			if (pan < positionLimit.pan.min)
			{
				positionLimit.pan.min = pan;
			}
			if (pan > positionLimit.pan.max)
			{
				positionLimit.pan.max = pan;
			}
			if (tilt < positionLimit.tilt.min)
			{
				positionLimit.tilt.min = tilt;
			}
			if (tilt > positionLimit.tilt.max)
			{
				positionLimit.tilt.max = tilt;
			}
			if (hfov < positionLimit.hfov.min)
			{
				positionLimit.hfov.min = hfov;
			}
			if (tilt > positionLimit.hfov.max)
			{
				positionLimit.hfov.max = hfov;
			}
		}
	};

	PTZCalibrator.prototype._displayPositionLimit = function()
	{
		var positionLimit = this._getCurrentPositionLimit();
		$(".pointList .limitList input.pan_limit").val(positionLimit.pan.min + " < pan < " + positionLimit.pan.max);
		$(".pointList .limitList input.tilt_limit").val(positionLimit.tilt.min + " < tilt < " + positionLimit.tilt.max);
		$(".pointList .limitList input.hfov_limit").val(positionLimit.hfov.min + " < hfov < " + positionLimit.hfov.max);
	};

	PTZCalibrator.prototype._load = function()
	{
		$(".pointList select.list option").remove();

		this._player[0].player.removePrimitives();

		var self = this;

		var player1 = $("#player1");
		var player2 = $("#player2");

		this._api.getAttributes({
			obj: this._player[0].obj
		})
			.fail(function(code, message){
				Log.error("[" + code + "]" + __(message));
			})
			.done(function(response){
				self._player[0].attributes = $.extend({}, response.list);

				// attribuets of first camera
				var attributes = self._player[0].attributes;
				var PTZ_CALIBRATOR_CONFIG = $.parseJSON(attributes.PTZ_CALIBRATOR_CONFIG);
				if (!$.isPlainObject(PTZ_CALIBRATOR_CONFIG))
				{
					PTZ_CALIBRATOR_CONFIG = {};
				}
				self._config = PTZ_CALIBRATOR_CONFIG;

				var PTZ_ASSOCIATED_CAMERAS = $.parseJSON(attributes.PTZ_ASSOCIATED_CAMERAS);
				if (PTZ_ASSOCIATED_CAMERAS.length == 0)
				{
					Log.error(__("PTZ camera is not associated with wide camera"));
					return;
				}

				// load config
				self._player[1].obj = PTZ_ASSOCIATED_CAMERAS[0];
				self._currentConfigObj = self._player[1].obj;

				// load position limit
				var positionLimit = self._getCurrentPositionLimit();
				$(".pointList .limitList input.pan_limit").val(positionLimit.pan.min + " < pan < " + positionLimit.pan.max);
				$(".pointList .limitList input.tilt_limit").val(positionLimit.tilt.min + " < tilt < " + positionLimit.tilt.max);
				$(".pointList .limitList input.hfov_limit").val(positionLimit.hfov.min + " < hfov < " + positionLimit.hfov.max);

				// init first camera
				self._player[0].player.init("#player1 .video", {backgroundColor: 0xCACACA})
					.fail(function(){
						Log.error(__("Player not initialized"));
					})
					.done(function(isAgain){
						// load point list
						var pointList = self._getCurrentPointList();
						if (!$.isEmptyObject(pointList))
						{
							for (var pointId in pointList)
							{
								if (!pointList.hasOwnProperty(pointId))
									continue;

								var point = pointList[pointId];

								point.primitiveList[0].id = self._addEllipse(null, point.x, point.y, point.width, point.height, point.color);
								point.primitiveList[1].id = self._addText(null, String(pointId), point.x + point.width, point.y + point.height, point.height, point.color);

								var text = pointId;
								if (point.position)
								{
									text = pointId + ". pan = " + point.position.pan + ", tilt = " + point.position.tilt + ", hfov = " + point.position.hfov;
								}
								$(".pointList select.list").append('<option value="' + pointId + '">' + text + '</option>');
							}

							player1.find(".pointList").show();
							player2.find(".playerPTZControls").show();
						}

						if (!isAgain)
						{
							this.subscribe("frame", function(timestamp/*, width, height*/){
								var date = new Date();
								date.setTime(timestamp);
								player1.find(".timestamp").val(date.getHours() + ":" + date.getMinutes() + ":" + date.getSeconds() + "." + date.getMilliseconds());
							}, "player");

							this.subscribe("imageclick", function(width, height, x, y){
								x = Math.round(x);
								y = Math.round(y);

								var pointColor = self._pointColor;
								var pointWidth = 30;
								var pointHeight = 30;
								var pointX = x;
								var pointY = y;

								var pointId = self._addPoint(pointX, pointY, pointWidth, pointHeight, pointColor);

								$(".pointList select.list").append('<option value="' + pointId + '">' + pointId + '</option>');
								$(".pointList select.list option").prop("selected", false);
								$(".pointList select.list option[value=" + pointId + "]")
									.prop("selected", true)
									.change();

								player1.find(".pointList").show();
								player2.find(".playerPTZControls").show();
							}, "zoom");
						}

						this.play(self._player[0].obj);
					});

				// attributes of second camera
				self._api.getAttributes({
					obj: self._player[1].obj
				})
					.fail(function(code, message){
						Log.error("[" + code + "]" + __(message));
					})
					.done(function(response){
						self._player[1].attributes = $.extend({}, response.list);

						var attributes = self._player[1].attributes;

						window.HWJoystick.init()
							.done(function(){
								window.HWJoystick.setObj(self._player[1].obj);
							});

						$(".pointList select.list option:first")
							.prop("selected", true)
							.change();

						// init second camera
						self._player[1].player.init("#player2 .video", {backgroundColor: 0xCACACA})
							.fail(function(){
								Log.error(__("Player not initialized"));
							})
							.done(function(isAgain){
								if (!isAgain)
								{
									this.subscribe("frame", function(timestamp, width, height){
										this.drawCrosshair(width, height, self._selectedPointColor);

										var date = new Date();
										date.setTime(timestamp);
										player2.find(".timestamp").val(date.getHours() + ":" + date.getMinutes() + ":" + date.getSeconds() + "." + date.getMilliseconds());
									}, "player");
								}

								this.play(self._player[1].obj);
							});

						player2.find(".move_pan_tilt button.move").click(function(){
							var pan = parseFloat($(".move_pan_tilt input.pan").val());
							var tilt = parseFloat($(".move_pan_tilt input.tilt").val());

							self._position.pan = pan;
							self._position.tilt = tilt;

							self._goToPosition();
						});

						player2.find(".move_fov button.move").click(function(){
							self._position.hfov = parseFloat($(".move_fov input.hfov").val());

							window.HWJoystick.sendCommand("mode=abs&hfov=" + self._position.hfov);
						});

						self._position.pan = 0; // parseFloat(attributes.PTZ_PAN_MIN);
						self._position.tilt = 0; // parseFloat(attributes.PTZ_TILT_MIN);
						self._position.hfov = parseFloat(attributes.PTZ_HFOV_MAX);
						self._position.step = parseFloat(player2.find(".playerPTZControls select.ptz_step").val());

						player2.find(".move_pan_tilt input.pan").val(self._position.pan);
						player2.find(".move_pan_tilt input.tilt").val(self._position.tilt);

						player2.find(".move_fov input.hfov").val(self._position.hfov);
					});
			});
	};

	/**
	 * @returns {Deferred}
	 * @private
	 */
	PTZCalibrator.prototype._goToPosition = function()
	{
		return window.HWJoystick.sendCommand("mode=abs&pt=" + this._position.pan + "," + this._position.tilt);
	};

	/**
	 * @returns {Deferred}
	 * @private
	 */
	PTZCalibrator.prototype._zoomToPosition = function()
	{
		return window.HWJoystick.sendCommand("mode=abs&hfov=" + this._position.hfov);
	};

	/**
	 * @returns {Object}
	 * @private
	 */
	PTZCalibrator.prototype._getCurrentPositionLimit = function()
	{
		return this._getCurrentConfig().positionLimit;
	};

	return PTZCalibrator;
});
