/**
 * @version $Id: hwj.js 33597 2016-02-16 16:00:29Z astarostin $
 * ------------------------------------------------------------------------------
 * Logic for communicate with hardware joystick
 * HWJoystick - object with methods for creating applet and communication with him
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @author Sergey Koloney
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2013
 *
 */
(function (window){
	"use strict";

window.HWJoystick = {
	id: "HWJoystick",
	hwj: null,
	hwjCalibratorId: null,
	obj: null,
	priority: null,
	lock_incheck: false,
	userid: null,
	commandPreset: null,
	speed: 1, // current move speed, can be changed by the sensitivity slider
	axisState: {"0": 0, "1": 0},
	zoomAxis: 3,  // Zoom axis; needed to quickly find value of zoom axis when moving x or y to provide simultaneous zooming and moving

	/* Speed commands support */
	speedCommandDelay: 500,   // Delay between two consecutive calls of sendModeSpeedCommand
	minDelayBetween2Command: 100,   // minimum delay between two consecutive calls of sendModeSpeedCommand
	isCommandPending: false,  // pending speed command flag
	lastCommandSent: 0,       // time when last ajax command was sent to ptz server
	timerSendSpeedCommand: null,

	isDebug: false,

	defaultPreset: {
		button: {
			"0": "mode=step&zoom=in",
			"1": "mode=step&zoom=out",
			"2": "mode=preset&goto=1",
			"3": "mode=preset&goto=2",
			"4": "mode=preset&goto=3",
			"5": "mode=preset&goto=4",
			"6": "mode=preset&goto=5",
			"7": "mode=preset&goto=6",
			"8": "mode=preset&goto=7",
			"9": "mode=preset&goto=8",
			"10": "mode=preset&goto=9",
			"11": "",
			"12": "",
			"99": "" // do not remove
		},
		axis: {
			"1":  "-tilt",
			"3":  "mode=speed&zoom="
		},
		deadzone: {
			"default": 5
		}
	},

	/**
	 * list of all command presets avaliable
	 * @var {Object}
	 */
	commandPresets: {},

	/**
	 * HWJ parameters
	 * @var {Object}
	 */
	parameters: {},

	/**
	 * initialization of HWJ
	 *
	 * @returns {Deferred}
	 */
	init: function()
	{
		var deferred = $.Deferred();

		var self = this;
		HWJoystick._getManifest(function(manifest){
			var platform = "";
			if (navigator.appVersion.indexOf("Win") != -1)
			{
				platform = "win32";
			} else
			if (navigator.appVersion.indexOf("Mac") != -1)
			{
				platform = "macos";
			}

			var mime = manifest.platform[platform].mime;

			if ($("#" + HWJoystick.id).length)
			{
				HWJoystick.destroy();
			}
			var html = '<object id="' + HWJoystick.id + '" type="' + mime + '" width="1" height="1"></object>';

			$("body").append(html);

			var hwj = document.getElementById(HWJoystick.id);

			var result = false;

			if (hwj.version)
			{
				addEvent(hwj, "hwjload", self.onHWJLoad);
				addEvent(hwj, "axis", self.onAxis);
				addEvent(hwj, "hat", self.onHat);
				addEvent(hwj, "button", self.onButton);
				addEvent(hwj, "enddeadzonecalibrate", self.onEndDeadZoneCalibrate);
				addEvent(hwj, "log", self.onLog);

				if (typeof hwj.init !== "undefined")
				{
					hwj.init(HWJoystick.isDebug);

					result = true;
				} else {
					$("#" + HWJoystick.id).remove();
				}
			}

			if (result)
			{
				deferred.resolve();
			} else {
				deferred.reject();
			}
		});

		return deferred.promise();
	},

	/**
	 * @param {Function} callback
	 * @private
	 */
	_getManifest: function(callback)
	{
		var self = this;
		$.ajax({
			url: "/ptz/manifest.json",
			cache: false,
			dataType: "json"
		})
			.done(function(manifest){
				callback.apply(self, [manifest]);
			});
	},

	/**
	 * load command presets from db, check current joystick name and set appropriate command preset
	 * `
	 * @returns {Deferred}
	 */
	loadCommandPresets: function()
	{
		var deferred = $.Deferred();

		var api = new API();
		api.getIdentityAttribute({
			attribute: 'JOYSTICK_PRESETS'
		})
			.done(function(response){
				if (response.value)
				{
					HWJoystick.commandPresets = JSON.parse(response.value);
				}
			})
			.always(function(){
				// get joystick name and search it in command presets
				var info = HWJoystick.getInfo();
				var preset = HWJoystick.defaultPreset;
				// check for old format, it does not consist axis property
				if (HWJoystick.commandPresets[info.name] && HWJoystick.commandPresets[info.name].axis)
				{
					preset = HWJoystick.commandPresets[info.name];
				}
				HWJoystick.setCommandPreset(preset);

				deferred.resolve();
			});

		return deferred.promise();
	},

	/**
	 * Set current preset for joystick commands
	 *
	 * @param {Object} commandPreset
	 */
	setCommandPreset: function(commandPreset)
	{
		// Clone object
		this.commandPreset = $.extend(true, {}, commandPreset);

		var info = HWJoystick.getInfo();
		for (var name in info.axis)
		{
			if (!info.axis.hasOwnProperty(name))
				continue;

			var value = 0; // info.axis[i].value

			// set speed value accordingly to current axis position if sensitivity assigned for some axis
			if (HWJoystick.commandPreset.axis[name] && HWJoystick.commandPreset.axis[name].indexOf('sensitivity') >= 0)
			{
				HWJoystick.setSpeed(HWJoystick.commandPreset.axis[name], HWJoystick.axisState[name]);
			}

			// Find zooming axis
			if (HWJoystick.commandPreset.axis[name] && HWJoystick.commandPreset.axis[name].indexOf('mode=speed&zoom=') >= 0)
			{
				HWJoystick.zoomAxis = name;
				HWJoystick.axisState[HWJoystick.zoomAxis] = 0;
			}
		}

		// if preset contains DeadZone settings, send them to applet
		if (HWJoystick.commandPreset.deadzone)
		{
			HWJoystick.setDeadZone(HWJoystick.commandPreset.deadzone);
		}
	},

	/**
	 * remove HWJ from page
	 */
	destroy: function()
	{
		$("#" + HWJoystick.id).remove();
		HWJoystick.hwj = null;
	},

	/**
	 * get joystick info
	 *
	 * @returns {Object} joystick info
	 */
	getInfo: function()
	{
		return HWJoystick.parameters;
	},

	startWaitEvents: function()
	{
		if (HWJoystick.hwj)
		{
			HWJoystick.hwj.startWaitEvents();
		}
	},

	/**
	 * set obj of PTZ device in ssytem
	 *
	 * @param {Number} obj
	 */
	setObj: function(obj)
	{
		HWJoystick.obj = obj;
		if (obj)
		{
			HWJoystick.priority = HWJoystick.getPTZPriority();
		}
	},

	/**
	 *
	 * @returns {Number}
	 */
	getPTZPriority: function(){
		var priority = 1;

		var api = new API(true);
		api.getObjectList({
			type: 'role',
			withAttributes: true
		})
			.done(function(response){
				for (var i = 0; i < response.list.length; i++) {
					api.getAttribute({
						obj: response.list[i].obj,
						attribute: 'PTZ_PRIORITY'
					})
						.done(function(response) {
							if(response.value > priority){
								priority = response.value;
							}
						});
				}
			});

		return priority;
	},

	/**
	 * Send info about deadzone to applet
	 *
	 * @param {Object} axis data for deadzone, ex. {0: 0.1, 1: 0.05, 2: 0}
	 */
	setDeadZone: function(axis)
	{
		if (HWJoystick.hwj)
		{
			HWJoystick.hwj.setDeadZone(axis);
		}
	},

	/**
	 * Set speed of joystick accordingly to value of sensitivity axis
	 *
	 * @param {String} sens sensitivity|-sensitivity, how to compute speed
	 * @param {Number} value value of axis
	 */
	setSpeed: function(sens, value)
	{
		// Reversed sensitivity
		if (sens == "-sensitivity") {
			value = -value;
		}

		// axis returns values -100..100, we need values 0.1 .. 1
		value = ((value + 100) / 200);
		if (value < 0.1) value = 0.1;
		HWJoystick.speed = value;
	},

	/**
	 * send PTZ command to server, and call callback function after server response or error
	 *
	 * @param {string} command for send
	 * @param {boolean} [addDev] add dev or not
	 * @param {number} [priority]
	 * @returns {Deferred}
	 */
	sendCommand: function(command, addDev, priority)
	{
		addDev = typeof addDev == "undefined" ? true : !!addDev;
		priority = priority || HWJoystick.priority;

		if (addDev && !(HWJoystick.obj && command))
		{
			return $.Deferred().reject();
		}

		var deferred = $.Deferred();

		// mode=speed&pt=0,0
		HWJoystick.isCommandPending = true;
		HWJoystick.lastCommandSent  = new Date().getTime();

		var devString = addDev ? "dev=" + HWJoystick.obj + "&" : "";
		command = "<PTZ_Command>do.ptz?" + devString + command + "&priority=" + priority + "</PTZ_Command>";
		command = command.replace(/--/g, ''); // delete '--' to provide "reverse" behaviour

		if (HWJoystick.isDebug)
		{
			window.console && window.console.log("Send: " + command);
		}

		/*
		window.console.log(command);
		HWJoystick.isCommandPending = false;
		*/

		this._jqXHR && this._jqXHR.abort();
		this._jqXHR = $.ajax({
			url: '/ptz/cgi-bin/send_message.pl',
			dataType: "json",
			data: {
				data: command
			}
		})
			.done(function(data){
				// check priority lock
				if (data.lockinfo && data.lockinfo.allow_override != undefined)
				{
					HWJoystick.checkLock(data.lockinfo.allow_override);
				}
			})
			.always(function(){
				HWJoystick.isCommandPending = false;

				deferred.resolve();
			});

		return deferred.promise();
	},

	/**
	 *
	 * @param {Number} x
	 * @param {Number} y
	 * @param {Number} width
	 * @param {Number} height
	 */
	gotoXY: function(x, y, width, height)
	{
		HWJoystick.sendCommand("mode=rel&size=" + width + 'x' + height + "&xy=" + x + ',' + y);
	},

	/**
	 * check lock
	 *
	 * @param {String} lockOverride lock override state
	 */
	checkLock: function(lockOverride)
	{
		if (HWJoystick.lock_incheck)
		{
			return;
		}

		HWJoystick.lock_incheck = true;

		switch (lockOverride)
		{
			case "sameuser":
				if (window.mx) window.mx.PTZ.lockDevice();
				break;
			case "no":
				Log.info(__("Another lock is higher priority. You have to wait until controls are released. Please try again later."));
				break;
			case "yes":
				var getLock = confirm(__("Another lock is in place. Do you want to override?"));
				if (getLock)
				{
					HWJoystick.overrideLock();
					if (window.mx) window.mx.PTZ.lockDevice();
				}
				break;
		}

		HWJoystick.lock_incheck = false;
	},

	/**
	 * override lock
	 */
	overrideLock: function()
	{
		HWJoystick.sendCommand("mode=lock&cmd=override");
	},

	/**
	 * Send speed command; when user use axis, this function calls every speedCommandDelay milliseconds
	 * until user releases joystick.
	 * Every 5 iterations it checks joystick position to prevent endless movement if we skipped "stop" command for some reason.
	 *
	 * Next speed command can be sent only if:
	 * 1) speedCommandDelay milliseconds passed since last command was sent;
	 * 2) last ajax command was completed
	 *
	 * @param {String} command    speed command
	 * @param {Number} iteration  number of function iteration
	 */
	sendModeSpeedCommand: function(command, iteration)
	{
		if (!HWJoystick.obj)
			return;

		/*
		// Once in a 10 iterations check current position of axis to prevent endless movement
		// (if last event from joystick with axis=0 was lost)
		// TODO: check code and remove it
		if (iteration > 5)
		{
			var info = HWJoystick.getInfo();
			for (var i in info.axis)
			{
				if (!info.axis.hasOwnProperty(i))
					continue;

				// If axis state changed send event
				if (HWJoystick.axisState[info.axis[i].name] && HWJoystick.axisState[info.axis[i].name] != info.axis[i].value)
				{
					HWJoystick.onAxis(info.axis[i].name, info.axis[i].value);
				}
			}
			return;
		}
		*/

		// if previous command completed,
		// and
		// delay time passed or this new command
		// we can send another command
		if (!HWJoystick.isCommandPending
			//&&
			//(new Date().getTime() - HWJoystick.lastCommandSent > HWJoystick.minDelayBetween2Command)
		)
		{
			HWJoystick.sendCommand(command);
		}

		HWJoystick.stopSendSpeedCommand();
		HWJoystick.timerSendSpeedCommand = setTimeout(
			function()
			{
				HWJoystick.sendModeSpeedCommand(command, iteration + 1);
			},
			HWJoystick.speedCommandDelay
		);
	},

	/**
	 * Clear sendModeSpeedCommand timeout
	 */
	stopSendSpeedCommand: function()
	{
		if (HWJoystick.timerSendSpeedCommand !== null)
		{
			clearTimeout(HWJoystick.timerSendSpeedCommand);
		}
	},

	/**
	 * Function is called when user use joystick axis
	 *
	 * @param {String} axis   name of axis
	 * @param {Number} value  value of axis
	 */
	onAxis: function(axis, value)
	{
		if (HWJoystick.isDebug)
		{
			console.log(axis, value);
		}
		HWJoystick.axisState[axis] = value;

		// Check for reversed Y behaviour
		if (axis == 1 && HWJoystick.commandPreset.axis[1] && HWJoystick.commandPreset.axis[1] == '-tilt')
		{
			HWJoystick.axisState[axis] = -value;
		}

		var command = "";
		var isStop = false;

		if (axis == 0 || axis == 1) // for ptz command: || axis === HWJoystick.zoomAxis
		{
			command = "mode=speed&pt=" +
				Math.round(HWJoystick.axisState["0"] * HWJoystick.speed) + "," +
				Math.round(HWJoystick.axisState["1"] * HWJoystick.speed);
				// for ptz command: Math.round(HWJoystick.axisState[HWJoystick.zoomAxis] * HWJoystick.speed);

			if (HWJoystick.axisState["0"] == 0 && HWJoystick.axisState["1"] == 0) // for ptz command: && HWJoystick.axisState[HWJoystick.zoomAxis] === 0
			{
				isStop = true;
			}
		} else
		if (HWJoystick.commandPreset.axis[axis])
		{
			// Sensitivity command
			if (HWJoystick.commandPreset.axis[axis].indexOf("sensitivity") != -1)
			{
				HWJoystick.setSpeed(HWJoystick.commandPreset.axis[axis], value);
				return;
			}

			command = HWJoystick.commandPreset.axis[axis] + (Math.round(HWJoystick.axisState[axis] * HWJoystick.speed));
			if (HWJoystick.axisState[axis] == 0)
			{
				isStop = true;
			}
		}

		if (isStop)
		{
			HWJoystick.stopSendSpeedCommand();
			HWJoystick.sendCommand(command);
		} else {
			HWJoystick.sendModeSpeedCommand(command, 1);
		}
	},

	/**
	 *
	 * @param {String} hat
	 * @param {Number} value
	 */
	onHat: function(hat, value)
	{
	},

	/**
	 * Function is called when user click on joystick button
	 *
	 * @param {String} button joystick button name
	 * @param {Number} value
	 */
	onButton: function(button, value)
	{
		if (value == 1 && HWJoystick.commandPreset.button[button])
		{
			HWJoystick.sendCommand(HWJoystick.commandPreset.button[button]);
		}
	},

	/**
	 *
	 * @param {Object} parameters
	 */
	onEndDeadZoneCalibrate: function(parameters)
	{
	},

	/**
	 *
	 * @param {Number} code
	 * @param {String} message
	 */
	onLog: function(code, message)
	{
		switch (code)
		{
			case 0: Log.info(message); break;
			case 1: Log.warning(message); break;
		}
	},

	/**
 	 * @param {Boolean} success
	 */
	onLoad: function(success)
	{

	},

	/**
	 *
	 * @param {Boolean} success
	 * @param {Object} parameters
	 */
	onHWJLoad: function(success, parameters)
	{
		if (success)
		{
			HWJoystick.hwj = document.getElementById(HWJoystick.id);
			// Set default command preset
			HWJoystick.setCommandPreset(HWJoystick.defaultPreset);

			HWJoystick.parameters = parameters;

			// Load presets
			HWJoystick
				.loadCommandPresets()
				.done(function(){
					HWJoystick.startWaitEvents();
					HWJoystick.onLoad(success);
				});
		}
	}
};

/**
 * add event listener
 *
 * @param {HTMLElement|EventTarget} obj
 * @param {String} eventName
 * @param {Function} callback
 */
function addEvent(obj, eventName, callback)
{
	// IE11
	if (((navigator.appName == 'Microsoft Internet Explorer') || ((navigator.appName == 'Netscape') && (new RegExp("Trident/.*rv:([0-9]{1,}[\.0-9]{0,})").exec(navigator.userAgent) != null))))
	{
		obj["on" + eventName] = callback;
	} else
	// IE
	if ($.browser.msie)
	{
		obj.attachEvent("on" + eventName, callback);
	} else {
		obj.addEventListener(eventName, callback, false);
	}
}

})(window);
