#!/usr/bin/perl
#  $Id: ptz_xviewpelco.pl 25370 2012-03-09 21:19:34Z teetov $
# -----------------------------------------------------------------------------
#  PTZ driver for Pelco D protocol for cameras connected to XView serial port
# -----------------------------------------------------------------------------
#  Author: Andrey Fomenko
#  Edited by: 
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT LLC, 2004-2005
# -----------------------------------------------------------------------------

use strict;
use Socket;
use MIME::Base64();
use IO::File;
use IO::Select;
use IO::Socket;
use Fcntl;

use NextCAM::Init;
use NextCAM::PTZ;
use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

my $log=get_logger('NEXTCAM::PTZ::PTZ_XVIEWPELCO');

httpHeaders( {
    "Accept" =>"*/*",
    "Connection" => "Keep-Alive"
} );

my $query_str = shift || "'POSITIONCTL'=>'XviewPelco'";
$log->info("Starting XviewPelco PTZ DRIVER, query string: [$query_str]");

my ($APL,$APL_CONF)=($ENV{APL},$ENV{APL_CONF});
my (%conf,$cam,$cmd,$par,$last_cmd,$last_mode);
my $WAIT  = 5;		# default wait
my $FLASH = "yes";	# default flash

$SIG{HUP}=\&load_dev_conf;
load_dev_conf();

# -----------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# -----------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
                                   PeerPort => $TCP_PORT,
                                   Proto     => "tcp",
                                   Type      => SOCK_STREAM)
    or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';

my %commands;
my $ready;
my $trycount = 30;

while(1) {
	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/i;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/
				|| $commands{$cam} =~ /speed iris=0/
				|| $commands{$cam} =~ /speed gain=0/
				)
			{
				; # do not override stop command!
			}else{
				$commands{$cam} = $cmd;
			}
		} else {
			$commands{$cam} = $cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)
	
	$cmd = '';
	
	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		delete $commands{$camera};
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
		
		httpAuth($conf{$cam}{USRNAME}, $conf{$cam}{PASSWD}) if $conf{$cam}{USRNAME} && $conf{$cam}{PASSWD};
		
		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam,$mode,"pt", $params[0].','.$params[1] ,\%options);
			camCmd($cam,$mode,"zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam,$mode,$cmd,$par,\%options);
		}
	}

	select(undef,undef,undef,.1) if not $cmd;

	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
	}
	
	if(!$cmd) {
		select(undef,undef,undef,.4);
		if(!$trycount--) {
			$trycount = 30;
			print $socket "test\0" or $log->logdie("Couldn't communicate to socket $TCP_PORT: $@");
		}
	}

} # while(1)

# -------------------------------------------------------------- camFlush -----
sub camFlush {
  my ($dev)=@_;
  httpAuth($conf{$dev}{USRNAME}, $conf{$dev}{PASSWD});
  checkURL("http://$conf{$dev}{DEVIP}/admin/serial_comm.cgi?flush=$FLASH");
} # sub camFlush

# ----------------------------------------------------------- cmdTransmit -----
sub cmdTransmit {
  my ($dev,@cmd)=@_;
  my $bb;
  my $str='';
  foreach my $b (@cmd) {
    $bb = sprintf("%02X ",$b);
    $log->debug($bb,':');
    $str .= $bb;
  }
  $str=~s/\s//g;
  httpAuth($conf{$dev}{USRNAME}, $conf{$dev}{PASSWD});
  checkURL("http://$conf{$dev}{DEVIP}/admin/serial_comm.cgi?flush=$FLASH&write=$str&wait=$WAIT&read=12");
} # sub cmdTransmit

# ------------------------------------------------------- cmdStepTransmit -----
sub cmdStepTransmit {
  my ($dev,$speed,$mode,$cmd) = @_;
  
  $log->debug("cmdStepTransmit($dev,$speed,$mode,$cmd)");
  
  my $speed_txt = "very_slow";
  $speed_txt = "slow" if $speed > 25;
  $speed_txt = "normal" if $speed > 50;
  $speed_txt = "fast" if $speed > 75;

  httpAuth($conf{$dev}{USRNAME}, $conf{$dev}{PASSWD});
  my $camera = $conf{$dev}->{CAMERA};  
  if(length($cmd) <= 5 ) {
    checkURL("http://$conf{$dev}{DEVIP}/cgi-bin/ptz.cgi?camera=$camera&speed=$speed_txt&$mode=$cmd");
  }
  else {
    $cmd =~ /(down|up)(left|right)/;
    my ($ud,$lr) = ($1,$2);
    checkURL("http://$conf{$dev}{DEVIP}/cgi-bin/ptz.cgi?camera=$camera&speed=$speed_txt&$mode=$ud");
    checkURL("http://$conf{$dev}{DEVIP}/cgi-bin/ptz.cgi?camera=$camera&speed=$speed_txt&$mode=$lr");
  }
}

# -------------------------------------------------------------- CheckSum -----
sub checkSum {
  my @arr = @_;
  return ($arr[1]+$arr[2]+$arr[3]+$arr[4]+$arr[5]) % 256;
} # sub checkSum

# --------------------------------------------------------------- camStop -----
sub camStop {
  my ($dev) = @_;
  my @cmd = ( 0xFF, 0, 0, 0, 0, 0, 0 );
  $cmd[1]=$conf{$dev}{PTZID}; # camera hardware ID
  $cmd[6]=checkSum(@cmd);
  cmdTransmit($dev,@cmd);
}

# -------------------------------------------------------------- camReset -----
sub camReset {
  my ($dev) = @_;
  my @cmd = ( 0xFF, 0, 0, 0, 0, 0, 0 );
  $cmd[1]=$conf{$dev}{PTZID};
  #Load defaults
  $cmd[3]=0x29; $cmd[6]=checkSum(@cmd);
  cmdTransmit($dev,@cmd);
  # Auto-focus: byte 5: 0-auto/1-on/2-off
  $cmd[3]=0x2B; $cmd[5]=0; $cmd[6]=checkSum(@cmd);
  cmdTransmit($dev,@cmd);
  # Auto-iris: byte 5: 0-auto/1-on/2-off
  $cmd[3]=0x2D; $cmd[5]=0; $cmd[6]=checkSum(@cmd);
  cmdTransmit($dev,@cmd);
  # Auto white balance: byte 5: 1-on/2-off
  $cmd[3]=0x33; $cmd[5]=2; $cmd[6]=checkSum(@cmd);
  cmdTransmit($dev,@cmd);
  # Set focus speed: byte 5: 0..3
  $cmd[3]=0x27; $cmd[5]=1; $cmd[6]=checkSum(@cmd);
  cmdTransmit($dev,@cmd);
} # sub camReset

# ------------------------------------------------------------- camFZspeed -----
sub camFZspeed {
  my ($dev,$FZ,$speed) = @_;
  $log->debug("camFZspeed($dev,$FZ,$speed)");
  my @cmd = ( 0xFF, 0, 0, 0, 0, 0, 0 );
  $cmd[1]=$conf{$dev}{PTZID};
  $cmd[3]=$FZ=='F'?0x27:0x25;
  $cmd[5]=sprintf("%d",abs($speed)/25);
  $cmd[6]=checkSum(@cmd);
  cmdTransmit($dev,@cmd);
} # sub camReset

# ---------------------------------------------------------------- PTspeed ----
sub PTspeed {
  my ($speed)=@_;
  #return 255 if(abs($speed)>99); # turbo speed - does not work with spectra !!!
  $speed=100 if abs($speed)>100;
  return sprintf("%2d",abs($speed)/100*63);
} # sub PTspeed

# ---------------------------------------------------------------- camCmd -----
sub camCmd {
  my ($dev,$mode,$cmd,$param,$options)=@_;
  my %options = %$options;
  my @cmd = ( 0xFF, 0, 0, 0, 0, 0, 0 );
  $log->debug("camCmd: DEVID=$dev mode: $mode command: $cmd parameter: $param");
  $cmd[1]=$conf{$dev}{PTZID}; # camera hardware ID
  
  # here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
  if($conf{$dev}{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
     if($mode=~/speed/i || $mode=~/step/i || $mode=~/abs/i || $mode=~/rel/i || $mode=~/smooth/i) {
        $log->debug("Setting timeout +$conf{$dev}{PTZ_PRESET1TIMEOUT}");
        $conf{$dev}{TIMEOUT} = time + $conf{$dev}{PTZ_PRESET1TIMEOUT};
     }
     elsif($mode=~/preset/i && $cmd=~/goto/i && $param!=1) {
        $log->debug("Setting timeout (preset) +$conf{$dev}{PTZ_PRESET1TIMEOUT}");
        $conf{$dev}{TIMEOUT} = time + $conf{$dev}{PTZ_PRESET1TIMEOUT};
    }
  }

  if(not $cmd=~/stop/) { # non-Stop
    # set camera speed
    my $spd= $conf{$dev}{PTZSPEED}?$conf{$dev}{PTZSPEED}:30;
    $cmd[4]=$cmd[5]=$conf{$dev}{PTZSPEED}?$conf{$dev}{PTZSPEED}:0x20;
    $cmd[4]=0x3F if $cmd[4]>0x3F;
    $cmd[5]=0x3F if $cmd[5]>0x3F;
   if ($mode=~/speed/i){	# mode=speed
    if($cmd=~/PT/i) { # RPT
      my ($p,$t) = split(/,/,$param);
      $cmd[4]=PTspeed($p);
      $cmd[5]=PTspeed($t);
      $log->debug("RPT $p $t");
      $cmd[3] |= 0x02 if $p>0;
      $cmd[3] |= 0x04 if $p<0;
      $cmd[3] |= 0x08 if $t<0;
      $cmd[3] |= 0x10 if $t>0;
    }
    elsif($cmd=~/Z/i) { 
      camFZspeed($dev,'Z',$param);
      $log->debug("ZOOM $param");
      $cmd[3] |= 0x20 if $param>0;
      $cmd[3] |= 0x40 if $param<0;
    }
    elsif($cmd=~/focus/i) {
      camFZspeed($dev,'F',$param);
      $log->debug("FOCUS $param");
      $cmd[2] |= 0x01 if $param>0;
      $cmd[3] |= 0x80 if $param<0;
    }
    elsif($cmd=~/iris/i) {
      camFZspeed($dev,'F',$param);
      $log->debug("IRIS $param");
      $cmd[2] |= 0x02 if $param>0;
      $cmd[2] |= 0x04 if $param<0;
    }
   }elsif($mode=~/rel/i) { # mode=rel: "<devid> rel size=640x480 xy=225,152"
      $param =~ /(\d+)x(\d+)/;
      my ($rel_size_x,$rel_size_y) = ($1,$2);
      $options{xy} =~ /(\d+),(\d+)/;
      my ($rel_click_x,$rel_click_y) = ($1,$2);
      my ($rel_p, $rel_t) = ( 200*($rel_click_x-$rel_size_x/2)/$rel_size_x, 200*($rel_click_y-$rel_size_y/2)/$rel_size_y );
      $log->debug("REL [$rel_size_x] [$rel_size_y] [$rel_click_x] [$rel_click_y] [$rel_p] [$rel_t]");
      $cmd[4]=PTspeed($rel_p);
      $cmd[5]=PTspeed($rel_t);
      $cmd[3] |= 0x02 if $rel_p>0;
      $cmd[3] |= 0x04 if $rel_p<0;
      $cmd[3] |= 0x08 if $rel_t<0;
      $cmd[3] |= 0x10 if $rel_t>0;
      $cmd[6]=checkSum(@cmd);
      cmdTransmit($dev,@cmd);
      select(undef,undef,undef,.05);
      camStop($dev);
      camFlush($dev);
      return;
   }elsif ($mode=~/settings/i){ # settings
    if($cmd=~/autoiris/i) { 
      $cmd[3] = 0x2D;
      $cmd[4] = 0x00;
      $cmd[5] = 0x00 if $param=~/auto/i;
      $cmd[5] = 0x01 if $param=~/on/i;
      $cmd[5] = 0x02 if $param=~/off/i;
    }
    elsif($cmd=~/autofocus/i) { 
      $cmd[3] = 0x2B;
      $cmd[4] = 0x00;
      $cmd[5] = 0x00 if $param=~/auto/i;
      $cmd[5] = 0x01 if $param=~/on/i;
      $cmd[5] = 0x02 if $param=~/off/i;
    }
    elsif($cmd=~/whitebalance/i) { #'awb_on','g', 'awb_off','b'
      $cmd[3] = 0x33;
      $cmd[4] = 0x00;
      $cmd[5] = 0x01 if $param=~/[auto|on]/i;
      $cmd[5] = 0x02 if $param=~/off/i;
    }
   }elsif($mode=~/step/i){       # mode=step  /Step by step positioning/
      cmdStepTransmit($dev,$spd,$cmd,$param);
      return;   
# Commented by ASTAL
#       $cmd[4]=$cmd[5]=($conf{$dev}{PTZSPEED})?PTspeed($conf{$dev}{PTZSPEED}):7;
#       $cmd[4]=$cmd[5]=30;
#      my $timeout = 0.02;
#    if($cmd=~/move/i) { # step pan/tilt
#      $cmd[3] |= 0x02 if $param=~/right/i;
#      $cmd[3] |= 0x04 if $param=~/left/i;
#      $cmd[3] |= 0x08 if $param=~/up/i;
#      $cmd[3] |= 0x10 if $param=~/down/i;
#    } elsif ($cmd=~/zoom/i){
#      $cmd[3] |= 0x20 if $param=~/in/i;
#      $cmd[3] |= 0x40 if $param=~/out/i;
#    } elsif ($cmd=~/focus/i){    
#      $cmd[2] |= 0x01 if $param=~/near/i;
#      $cmd[3] |= 0x80 if $param=~/far/i;
#      $timeout = 0.02;		# reduce timeout for focusing
#    } elsif ($cmd=~/iris/i){
#      $cmd[2] |= 0x02 if $param=~/open/i;
#      $cmd[2] |= 0x04 if $param=~/close/i;
#    }
#      $cmd[6]=checkSum(@cmd);
#      cmdTransmit($dev,@cmd);
#      select(undef,undef,undef,$timeout);
#      camStop($dev);
   }elsif($mode=~/hardware/i){   # mode=hardware  /Hardware reset,defaults/
    if($cmd=~/do/i && $param=~/defaults/i) { # init
      camReset($dev);
      return;
    }
   } elsif($mode=~/abs/i){                # mode=abs  /Absloute positioning/
    if($cmd=~/center/i) { # Center
      camReset($dev);
      $cmd[3]=0x07;
      $cmd[5]=0x22;
    }elsif($cmd=~/PT/i){            # Pan, Tilt
         $log->debug("ABS:  PT $param");
         $cmd[2]=0x00;
         $param=~/^(\-?.+)?\,(\-?.+)$/;
         my $p = int($1*100);
         my $t = int($2*100);
         $log->debug("ABS! pt=$1,$2 ($p,$t)");
	 $p += 36000 if $p<0;
	 $t += 36000 if $t<0;
	 my $p_hex = sprintf("%04X",$p);
	 my $t_hex = sprintf("%04X",$t);
         $log->debug("==== P_HEX= $p_hex");
         $p_hex =~/^(\w\w)(\w\w)$/;
         ($cmd[4],$cmd[5]) = (hex($1),hex($2));
         $cmd[3]=0x4B;	# For Pan Position
         $cmd[6]=checkSum(@cmd);
         cmdTransmit($dev,@cmd);	#sending Pan first
         $cmd[3]=0x4D;	# For Tilt Position
         $t_hex =~/^(\w\w)(\w\w)$/;
         ($cmd[4],$cmd[5]) = (hex($1),hex($2));
         # command will be send later 
      } elsif ($cmd=~/Z/i){	# Absolute zoom
         $log->debug("ABS:  Z $param");
	 $param = int( ($param/184)*65535 );
	 $cmd[2]=0x00;
	 $cmd[3]=0x4F;
	 my $z_hex = sprintf("%04X",$param);
         $log->debug("ABS: ZOOM $z_hex");
         $z_hex =~/^(\w\w)(\w\w)$/;
         ($cmd[4],$cmd[5]) = (hex($1),hex($2));
      }
   } elsif($mode=~/preset/i){ # presets
      $cmd[3]= 0x03 if $cmd =~/save/i; 
      $cmd[3]= 0x05 if $cmd =~/clear/i; 
      $cmd[3]= 0x07 if $cmd =~/goto/i; 
      $cmd[5]= $param;
   } elsif($mode=~/smooth/i) { # Right/Left/Up/Down/Tele/Wide/focus_Near/focus_Far/iris_Open/irIs_close
      # set movement bits
      if ($cmd=~/move/i) { # pan/tilt
	$cmd[3] |= 0x02 if $param=~/right/i;
	$cmd[3] |= 0x04 if $param=~/left/i;
	$cmd[3] |= 0x08 if $param=~/up/i;
	$cmd[3] |= 0x10 if $param=~/down/i;
      } elsif($cmd=~/zoom/i) { # zoom
	$cmd[3] |= 0x20 if $param=~/in/i;
	$cmd[3] |= 0x40 if $param=~/out/i;
      } elsif($cmd=~/focus/i) { # focus
	$cmd[2] |= 0x01 if $param=~/near/i; # reduce speed of focusing
	$cmd[3] |= 0x80 if $param=~/far/i;
      } elsif($cmd=~/iris/i) { # iris
	$cmd[2] |= 0x02,$cmd[4]=$cmd[5]=0 if $param=~/open/i;
	$cmd[2] |= 0x04,$cmd[4]=$cmd[5]=0 if $param=~/close/i;  
      }
    }
  }
  # calculate command check-sum
  $cmd[6]=checkSum(@cmd);
  cmdTransmit($dev,@cmd);
  camFlush($dev);
}

# --------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {
 %conf = GetCfgs( eval("($query_str)") );    # Load configurations
 $log->debug("Config read as:");
 $log->debug("-------------------------------------");
 foreach my $dev (keys %conf) {
   $log->debug("DEVID=$conf{$dev}{DEVID}");
   $log->debug("CAMERAMODEL=$conf{$dev}{CAMERAMODEL}");
   $log->debug("POSITIONCTL=$conf{$dev}{POSITIONCTL}");
   $log->debug("PTZID=$conf{$dev}{PTZID}");
   $log->debug("PTZSPEED=$conf{$dev}{PTZSPEED}");
   $log->debug("PTZHWPORT_SPEED=$conf{$dev}{PTZHWPORT_SPEED}");
   $log->debug("PTZHWPORT_BITS=$conf{$dev}{PTZHWPORT_BITS}");
   $log->debug("PTZHWPORT_STOPBITS=$conf{$dev}{PTZHWPORT_STOPBITS}");
   $log->debug("PTZHWPORT_PARITY=$conf{$dev}{PTZHWPORT_PARITY}");
   $log->debug("-------------------------------------");
   httpAuth($conf{$dev}{USRNAME}, $conf{$dev}{PASSWD});
   checkURL("http://$conf{$dev}{DEVIP}/admin/control.cgi?P_bps=$conf{$dev}{PTZHWPORT_SPEED}&P_data=$conf{$dev}{PTZHWPORT_BITS}&P_stop=$conf{$dev}{PTZHWPORT_STOPBITS}&P_parity=$conf{$dev}{PTZHWPORT_PARITY}");
 } # foreach $dev
}


# -------------------------------------------------------------- nonblock -----
#   nonblock($socket) puts socket into nonblocking mode
sub nonblock {
    my $socket = shift;
    my $flags;
    
    $flags = fcntl($socket, F_GETFL, 0)
            or die "Can't get flags for socket: $!\n";
    fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
            or die "Can't make socket nonblocking: $!\n";
}

