#!/usr/bin/perl
#  $Id: ptz_ultrak_kd6.pl 25370 2012-03-09 21:19:34Z teetov $
# -----------------------------------------------------------------------------
#  PTZ engine for Ultrak KD6 protocol, TCP and SERIAL transports
# -----------------------------------------------------------------------------
#  Unlike other engines this one handles different transport types in the same 
#  time (TCP sockets, URL and Serial )
# -----------------------------------------------------------------------------
#  Author: Andriy Fomenko
#  Edited by: 
#  QA by:  Christopher C Gettings
#  Copyright: videoNEXT LLC 2004
# -----------------------------------------------------------------------------

use strict;

# -----------------------------------------------------------------------------
# Constants section
# -----------------------------------------------------------------------------

# -----------------------------------------------------------------------------
# Parameters analysed from configuration file:
# POSITIONCTL=Ultrak_KD6
# PTZID=1
# PTZMONITOR=1
# PTZ_TRANSPORT=TCP/serial
# PTZ_TCP_PORT=5555
# PTZHWPORT=/dev/ttyUSB0
# PTZHWPORT_BITS=8
# PTZHWPORT_PARITY=none
# PTZHWPORT_SPEED=2400
# PTZHWPORT_STOPBITS=1
#

use POSIX;
use IO::Socket;
use IO::Select;
use Tie::RefHash;
use IO::File;
use Socket;
use Fcntl;
use Tie::RefHash;
use NextCAM::Init;
use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

my $log=get_logger('NEXTCAM::PTZ::PTZ_ULTRASK_KD6');

$SIG{HUP}=\&load_dev_conf;

my (%conf, %sock, %port, $cam, $cmd, $par, $usrpsw, $last_cmd, $last_mode, $cmd_executed);
my %ready     = ();
tie %ready, 'Tie::RefHash';

my $APL=$ENV{APL} || $log->logdie('Variable APL is not set');
my $APL_CONF=$ENV{APL_CONF} || $log->logdie('Variable APL_CONF is not set');

load_dev_conf();

open(PID,"> $APL/var/ptz/ptz_ultrak_kd6.pid"); print PID $$; close PID;
my $handles=new IO::Select();
my $pipe = new IO::File "+<$APL/var/ptz/ultrak_kd6";
$handles->add($pipe);

my %commands;
my $ready;
my $lastcmdtime = time;

while(2){
	while((($ready) = IO::Select->select($handles,undef,undef,.1))){
		foreach my $in (@$ready) {
			next if not $cmd=<$in>;
			chomp $cmd;
			load_dev_conf(),next if $cmd=~/HUP/i;
			next if not $cmd=~/^(\d+)/;
			$cam=$1;
			if(defined($commands{$cam})){
				if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/
				|| $commands{$cam} =~ /speed iris=0/
				|| $commands{$cam} =~ /speed gain=0/
				)
				{
					; # do not override stop command!
				}else{
					$commands{$cam} = $cmd;
				}
			} else {
				$commands{$cam} = $cmd;
			}
		} # foreach my $in
	} # while((($ready)
	
	$cmd_executed = 0;
	
	foreach my $camera (keys %commands){
		if($cmd=~/^===/) {
			delete $commands{$camera};
			next
		}
		$cmd=$commands{$camera};
		$log->debug("FILTERED: $cmd");
		if(not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/) {
			delete $commands{$camera};
			next
		}
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		$log->debug("SPLIT: $cam :: $mode :: $cmd :: $par :: $options");
		delete $commands{$camera};
		next if not defined $conf{$cam};
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
		$usrpsw = "$conf{$cam}{USRNAME}:$conf{$cam}{PASSWD}" if $conf{$cam}{USRNAME} && $conf{$cam}{PASSWD};
		
		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam,$mode,"pt", $params[0].','.$params[1] ,\%options);
			camCmd($cam,$mode,"zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam,$mode,$cmd,$par,\%options);
		}
		
		$cmd_executed=1;
		$lastcmdtime = time;
	}
	# check that cameras health only if no user activity was detected in 5 seconds
	# checkCamerasHealth() if (!$cmd_executed) and (time-$lastcmdtime > 5); ;
	# `echo === > $APL/var/ptz/icd001`;
	#purgeBuffers();
}
exit 0;


# ---------------------------------------------------------------- camCmd -----
sub camCmd {
    my ($dev,$mode,$cmd,$param,$options)=@_;
    my %options = %$options;    
    my $resp;
    my @cmd;                                # Byte | Description
    $cmd[0] = $conf{$dev}{PTZMONITOR};      #   0  | Monitor number
    $cmd[1] = $conf{$dev}{PTZID}-1;         #   1  | Camera number

    if($mode=~/step/i) { # ==================================== step =====
        if($cmd=~/move/i) { # --------------------------------- move -----
        }
        elsif($cmd=~/zoom/i) { # ------------------------------ zoom -----
        }
        elsif($cmd=~/focus/i) { # ---------------------------- focus -----
        }
        elsif($cmd=~/iris/i or $cmd=~/gain/i) { # -------- iris/gain -----
        }
    }
    elsif($mode=~/preset/i) { # ============================= preset =====
        return if $param <0 or $param > 9;
        if($cmd=~/goto/i) {
        }
        elsif($cmd=~/save/i) {
        }
        elsif($cmd=~/clear/i) {
        }
    }
    elsif($mode=~/speed/i) { # =============================== speed =====
        if($cmd=~/pt/i) { # ------------------------------------- pt -----
            my ($p,$t) = split(/,/,$param); 
            $log->debug("PT: [$p,$t]");
            $cmd[4] = $cmd[5] = 0;
            $cmd[2] = sprintf("%2d",abs($p*1.26)); $cmd[2] += 0x80 if $p <= 0;
            $cmd[3] = sprintf("%2d",abs($p*1.26)); $cmd[3] += 0x80 if $p <= 0;
        }
        elsif($cmd=~/z/i) { # ------------------------------------ z -----
            $cmd[2] = $cmd[3] = $cmd[5] = 0;
            $cmd[4] = sprintf("%2d",abs($param*1.26));
            $cmd[4] += 0x80 if $param <= 0;
        }
        elsif($cmd=~/focus/i) { # ---------------------------- focus -----
            $cmd[2] = $cmd[3] = $cmd[4] = 0;
            $cmd[5] = $param?
                             $param>0? 0x08 : 0x10
                           : 0x00;
        }
        elsif($cmd=~/[iris|gain]/i) { # ------------------- iris/gain -----
            $cmd[2] = $cmd[3] = $cmd[4] = 0;
            $cmd[5] = $param?
                             $param>0? 0x02 : 0x04
                           : 0x00;
        }
    }
    elsif($mode=~/hardware/i) { # ========================= hardware =====
        if($cmd=~/do/i) {
        }
    }
    elsif($mode=~/rel/i) { # =================================== rel =====
        $param =~ /(\d+)x(\d+)/;
        my ($rel_size_x,$rel_size_y) = ($1,$2);
        $options{xy} =~ /(\d+),(\d+)/;
        my ($rel_click_x,$rel_click_y) = ($1,$2);
        my ($rel_p, $rel_t) = ( 200*($rel_click_x-$rel_size_x/2)/$rel_size_x, 200*($rel_click_y-$rel_size_y/2)/$rel_size_y );
        $log->debug('Relative movement: [',$rel_p,'] [',$rel_t,']');
    }
    elsif($mode=~/settings/i) { # ========================= settings =====
        if($cmd=~/whitebalance/i) {
        }
        elsif($cmd=~/gain/i) {
        }
        elsif($cmd=~/autofocus/i) {
        }
        elsif($cmd=~/autoiris/i) {
        }
    }
    # default behavior - calculate checksum and send command to device
    @cmd = checkSum(@cmd);
    cmdTransmit($dev,@cmd);
} # sub camCmd

# -------------------------------------------------------------- checkSum -----
sub checkSum
{
  my (@arr) = @_;
  # p0 = XOR of bits 00 03 06 11 14 17 22 25 30 33 36 41 44 47 52
  my $p0 = ($arr[0] & 0x01) ^ (($arr[0] & 0x08)>>3) ^ (($arr[0] & 0x40)>>6)
         ^ (($arr[1] & 0x02)>>1) ^ (($arr[1] & 0x10)>>4) ^ (($arr[1] & 0x80)>>7)
         ^ (($arr[2] & 0x04)>>2) ^ (($arr[2] & 0x20)>>5)
         ^ ($arr[3] & 0x01) ^ (($arr[3] & 0x08)>>3) ^ (($arr[3] & 0x40)>>6)
         ^ (($arr[4] & 0x02)>>1) ^ (($arr[4] & 0x10)>>4) ^ (($arr[4] & 0x80)>>7)
         ^ (($arr[5] & 0x04)>>2);
  # p1 = XOR of bits 01 04 07 12 15 20 23 26 31 34 37 42 45 50 53
  my $p1 = (($arr[0] & 0x02)>>1) ^ (($arr[0] & 0x10)>>4) ^ (($arr[0] & 0x80)>>7)
         ^ (($arr[1] & 0x04)>>2) ^ (($arr[1] & 0x20)>>5)
         ^ ($arr[2] & 0x01) ^ (($arr[2] & 0x04)>>3) ^ (($arr[2] & 0x40)>>6)
         ^ (($arr[3] & 0x02)>>1) ^ (($arr[3] & 0x10)>>4) ^ (($arr[3] & 0x80)>>7)
         ^ (($arr[4] & 0x04)>>2) ^ (($arr[4] & 0x20)>>5)
         ^ ($arr[5] & 0x01) ^ (($arr[5] & 0x08)>>3);
  # p2 = XOR of bits 02 05 10 13 16 21 24 27 32 35 40 43 46 51 54
  my $p2 = (($arr[0] & 0x04)>>2) ^ (($arr[0] & 0x20)>>5)
         ^ ($arr[1] & 0x01) ^ (($arr[1] & 0x08)>>3) ^ (($arr[1] & 0x40)>>6)
         ^ (($arr[2] & 0x02)>>1) ^ (($arr[2] & 0x10)>>4) ^ (($arr[2] & 0x80)>>7)
         ^ (($arr[3] & 0x04)>>2) ^ (($arr[3] & 0x20)>>5)
         ^ ($arr[4] & 0x01) ^ (($arr[4] & 0x08)>>3) ^ (($arr[4] & 0x40)>>6)
         ^ (($arr[5] & 0x02)>>1) ^ (($arr[5] & 0x10)>>4);
         
  $log->debug("p0=$p0 p1=$p1 p2=$p2 arr[5]=$arr[5]");       
  
  $log->debug($p2 << 7);
  
  $arr[5] &=  0x1F;
  $arr[5] += (($p2 << 7) | ($p2 << 6) | ($p2 << 5));
  
  $log->debug("arr[5]=$arr[5]");
  
  return @arr;
} # sub checkSum


# ----------------------------------------------------------- cmdTransmit -----
sub cmdTransmit {
    my ($dev,@cmd)=@_;
    
    my $command='';
    my $b;
    foreach $b (@cmd) { $command .= sprintf("%02X ",$b); }
    $log->debug('cmdTransmit( ',$dev, '=> ',$command,')');    
    if($conf{$dev}{PTZ_TRANSPORT}=~/TCP/) {
        my $bts = '';
        foreach my $b (@cmd) { $bts .= chr($b); }
        my $skt = $sock{ $conf{$dev}{PTZ_TCP_PORT} }{SOCK};
        print $skt $bts;
        select(undef,undef,undef,.1);
        foreach my $sss ( $sock{$conf{$dev}{PTZ_TCP_PORT}}{SEL}->can_read(1) ) {
            my $data = '';
            eval {
                alarm 1;
                $sss->recv($data, POSIX::BUFSIZ, 0);
                $command='';
                for($b=0;$b<length($data);$b++) { $command .= sprintf("%02X ",ord(substr($data,$b,1))); }
                $log->debug('Received:',$command);
            };
            alarm 0;
            return $data;
        }  # foreach      
    }
    else { # serial connection
        foreach $b (@cmd) { $port{$conf{$dev}{PTZHWPORT}}->write(chr($b)); }
        $port{$conf{$dev}{PTZHWPORT}}->write_drain;
        my ($count_in, $string_in) = $port{$conf{$dev}{PTZHWPORT}}->read(50);
        return substr($string_in,0,$count_in);
    } # if($conf{$dev}{PTZ_TRANSPORT}=~/TCP/)
} # sub cmdTransmit


# ---------------------------------------------------------- purgeBuffers -----
# go other devices and purge buffers to prevent chaotic movements due to
# messages stuck in buffers
sub purgeBuffers {
    foreach my $dev ( keys %conf ) {
        my $skt = $sock{ $conf{$dev}{PTZ_TCP_PORT} }{SOCK};
        foreach my $sss ( $sock{$conf{$dev}{PTZ_TCP_PORT}}{SEL}->can_read(1) ) {
            my $data = '';
            eval {
                alarm 1;
                $sss->recv($data, POSIX::BUFSIZ, 0);
                if( length($data)) {
                    my $command='';
                    for($b=0;$b<length($data);$b++) { $command .= sprintf("%02X ",ord(substr($data,$b,1))); }
                    $log->debug('PURGED:',$command);
                }
            };
            alarm 0;
        }
    }
} # sub purgeBuffers

# --------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {

    $log->debug('load_dev_conf');
    
    # fisrst, close everything
    foreach my $skt (keys %sock) {
        close($sock{$skt}{SOCK});
        delete $sock{$skt};
    }
    foreach my $p ( keys %port ) {
        $port{$p}->close;
        delete $port{$p};
    }

    %conf = GetCfgs( ('POSITIONCTL'=>'Ultrak_KD6') );     # Load configurations
    foreach my $dev (keys %conf) {
        $log->debug("[$dev]");
        $log->debug("DEVID=$conf{$dev}{DEVID}");
        $log->debug("POSITIONCTL=$conf{$dev}{POSITIONCTL}");
        $log->debug("PTZID=$conf{$dev}{PTZID}");
        $log->debug("PTZ_TRANSPORT=$conf{$dev}{PTZ_TRANSPORT}");
        if($conf{$dev}{PTZ_TRANSPORT}=~/TCP/) {
            $log->debug("PTZ_TCP_PORT=$conf{$dev}{PTZ_TCP_PORT}");
            if(not defined($sock{$conf{$dev}{PTZ_TCP_PORT}})) {
                $sock{$conf{$dev}{PTZ_TCP_PORT}}{SOCK} = IO::Socket::INET->new(PeerAddr => $conf{$dev}{DEVIP},
                                                        PeerPort => $conf{$dev}{PTZ_TCP_PORT},
                                                        Proto    => "tcp",
                                                        Type     => SOCK_STREAM)
                    or $log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n"),next;
                $sock{$conf{$dev}{PTZ_TCP_PORT}}{SOCK}->autoflush(1);
                nonblock($sock{$conf{$dev}{PTZ_TCP_PORT}}{SOCK});
                $sock{$conf{$dev}{PTZ_TCP_PORT}}{SEL} = IO::Select->new($sock{$conf{$dev}{PTZ_TCP_PORT}}{SOCK});
            }
        }
        else {
            $log->debug("PTZHWPORT=$conf{$dev}{PTZHWPORT}");
            $log->debug("PTZSPEED=$conf{$dev}{PTZSPEED}");
            $log->debug("PTZHWPORT_SPEED=$conf{$dev}{PTZHWPORT_SPEED}");
            $log->debug("PTZHWPORT_BITS=$conf{$dev}{PTZHWPORT_BITS}");
            $log->debug("PTZHWPORT_STOPBITS=$conf{$dev}{PTZHWPORT_STOPBITS}");
            $log->debug("PTZHWPORT_PARITY=$conf{$dev}{PTZHWPORT_PARITY}");
            if(not defined($port{$conf{$dev}{PTZHWPORT}})) {
                $log->debug("Initialise serial port: $conf{$dev}{PTZHWPORT}");
                $port{$conf{$dev}{PTZHWPORT}} = new Device::SerialPort($conf{$dev}{PTZHWPORT}) || $log->error("Can't open port $conf{$dev}{PTZHWPORT}: $!\n"),next;
                $port{$conf{$dev}{PTZHWPORT}}->handshake("none"); # none / rts / xoff
                $port{$conf{$dev}{PTZHWPORT}}->baudrate($conf{$dev}{PTZHWPORT_SPEED});
                $port{$conf{$dev}{PTZHWPORT}}->parity($conf{$dev}{PTZHWPORT_PARITY});
                $port{$conf{$dev}{PTZHWPORT}}->databits($conf{$dev}{PTZHWPORT_BITS});
                $port{$conf{$dev}{PTZHWPORT}}->stopbits($conf{$dev}{PTZHWPORT_STOPBITS});

                $port{$conf{$dev}{PTZHWPORT}}->buffers(4096, 4096);
                $port{$conf{$dev}{PTZHWPORT}}->read_const_time(20);  # milliseconds
                $port{$conf{$dev}{PTZHWPORT}}->read_char_time(5);
            }
        } # if($conf{$dev}{PTZ_TRANSPORT}=~/TCP/)
    } # foreach $dev
} # sub load_dev_conf

# -------------------------------------------------------------- nonblock -----
sub nonblock {
    my ($fd) = @_;
    my $flags = fcntl($fd, F_GETFL,0);
    fcntl($fd, F_SETFL, $flags|O_NONBLOCK);
}


# ----------===== Changes log =====----------
# $Log$
# Revision 1.3  2005/10/17 17:54:38  afomenko
# logger include file changed
#
# Revision 1.2  2005/01/11 15:49:12  afomenko
# fixes on customer site - still does not work
#
# Revision 1.1  2005/01/06 19:54:49  afomenko
# "ultrak_kd6" support added
#
