#!/usr/bin/perl
#  $Id: ptz_toshibaIKWB.pl 25370 2012-03-09 21:19:34Z teetov $
# -----------------------------------------------------------------------------
#  PTZ driver for Toshiba IKWB
# -----------------------------------------------------------------------------
#  Author: Andrey Fomenko
#  Edited by: 
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT LLC, 2004-2005
# -----------------------------------------------------------------------------

use strict;
use Socket;
use MIME::Base64();
use IO::File;
use IO::Select;
use IO::Socket;
use Fcntl;

use NextCAM::Init;
use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

my $log=get_logger('NEXTCAM::PTZ::PTZ_TOSHIBAIKWB');

my $query_str = shift || "'POSITIONCTL'=>'ToshibaIKWB','CAMERAMODEL'=>'Toshiba'";
$log->info("Starting ToshibaIKWB PTZ DRIVER, query string: [$query_str]");

$SIG{HUP}=\&load_dev_conf;

my $APL=$ENV{APL};
my (%conf,$cam,$cmd,$usrpsw,$par,$last_cmd,$last_mode);

load_dev_conf();

# -----------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# -----------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
                                   PeerPort => $TCP_PORT,
                                   Proto     => "tcp",
                                   Type      => SOCK_STREAM)
    or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';

my %commands;
my $ready;
my $trycount = 30;

while(1) {
	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/i;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/
				|| $commands{$cam} =~ /speed iris=0/
				|| $commands{$cam} =~ /speed gain=0/
				)
			{
				; # do not override stop command!
			}else{
				$commands{$cam} = $cmd;
			}
		} else {
			$commands{$cam} = $cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)
	
	$cmd = '';
	
	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		delete $commands{$camera};
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
		$usrpsw = "$conf{$cam}{USRNAME}:$conf{$cam}{PASSWD}" if $conf{$cam}{USRNAME} && $conf{$cam}{PASSWD};
		
		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam,$mode,"pt", $params[0].','.$params[1] ,\%options);
			camCmd($cam,$mode,"zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam,$mode,$cmd,$par,\%options);
		}
	}

	select(undef,undef,undef,.1) if not $cmd;

	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
	}
	
	if(!$cmd) {
		select(undef,undef,undef,.4);
		if(!$trycount--) {
			$trycount = 30;
			print $socket "test\0" or $log->logdie("Couldn't communicate to socket $TCP_PORT: $@");
		}
	}

} # while(1)



# =============================================================================

# ---------------------------------------------------------------- camCmd -----
sub camCmd
{
  my ($dev,$mode,$cmd,$param,$options)=@_;
  my %options = %$options;
  $log->debug("camCmd: DEVID=$dev  command: $cmd  optional parameter: $param");
  my $URL="http://$conf{$dev}{DEVIP}";
  my $moveURL="http://$conf{$dev}{DEVIP}/pantiltapi.cgi?cont_2";
  my $relURL="http://$conf{$dev}{DEVIP}/directmoveapi.cgi?cont_3";
  my $curURL="http://$conf{$dev}{DEVIP}/getcurrentptpointapi.htm";

  my $spd=($options{speed}=~/^\d+/) ? $options{speed}:$conf{$dev}->{PTZSPEED};
  $spd=sprintf("%02x",$spd);
  my $zoomspd = $conf{$dev}->{PTZZOOMSPEED};
  $zoomspd=sprintf("%02x",$zoomspd);

  # here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
  if($conf{$dev}->{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
     if($mode=~/speed/i || $mode=~/step/i || $mode=~/abs/i || $mode=~/rel/i || $mode=~/smooth/i) {
        $log->debug("Setting timeout +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
        $conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
     }
     elsif($mode=~/preset/i && $cmd=~/goto/i && $param!=1) {
        $log->debug("Setting timeout (preset) +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
        $conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
    }
  }

  # we need to "terminate" each command in smooth mode because it can move forever if used with no STOP !
  if ($last_mode=~/smooth/i) {
     checkURL("$URL=8101060100000303FF") if $last_cmd=~/move/i;
     checkURL("$URL=8101060100000303FF") if $last_cmd=~/zoom/i;  # zoom
#     checkURL("$URL=8101040700FF") if $last_cmd=~/zoom/i;  # zoom
     checkURL("$URL=8101040B00FF") if $last_cmd=~/iris/i;  # iris
     checkURL("$URL=8101040800FF") if $last_cmd=~/focus/i;  # focus
     $log->debug("Last mode was <$last_mode>. Last command <$last_cmd>");
  }

 if ($mode=~/speed/i){  
    if($cmd=~/PT/i){ # RPT
	my ($p,$t) = split(/,/,$param);
	
	my ($vv,$ww) = (sprintf("%02x",abs($p/4)), sprintf("%02x",abs($t/4)));
	my $cccc = '';
	for ($p,$t) {
	   $cccc .= '01' if $_>0;	# right or up
	   $cccc .= '02' if $_<0;	# left or down
	   $cccc .= '03' if $_==0;	# no movement in this direction	
	}
        $log->debug("PT: $p $t $cccc");
        checkURL("$URL=81010601${vv}${ww}${cccc}FF");
    }
    elsif($cmd=~/Z/i){ # RZ
       if($param ne 0) {
          $zoomspd = sprintf("%01d",abs($zoomspd/100*7));
          $zoomspd = 7 if $zoomspd > 7;
          my $cmdcode= ($param<0)? "3$zoomspd" : "2$zoomspd";
          checkURL("$URL=81010407${cmdcode}FF")
       } else {
          checkURL("$URL=8101040700FF");
       }
    }
    elsif($cmd=~/focus/i){ # RF
       if($param ne 0) {
          $spd = sprintf("%01d",abs($spd/100*7));
          $spd = 7 if $spd > 7;
          my $cmdcode= ($param<0)? "3$spd" : "2$spd";
          checkURL("$URL=81010408${cmdcode}FF")
       } else {
          checkURL("$URL=8101040800FF")
       }
    }
 } elsif($mode=~/abs/i){		# mode=abs  /Absloute positioning/
    if($cmd=~/PT/i){ 		# Pan, Tilt
      $log->debug("abs $param");
      $param=~/(\-?\d+)\s*,\s*(\-?\d+)\s*$/;
      my $p = -1 * int(int("$1")*14.4);
      my $t=int((int("$2")-32.5)*14.4);
      $log->debug("ABS! pt=$1,$2 ($p,$t)");
      if($p>=0) {
        $p = substr(sprintf("%04X",$p),0,4)
      }
      else {
        $p = substr(sprintf("%04X",$p),4,4)
      }
      if($t>=0) {
        $t = substr(sprintf("%04X",$t),0,4)
      }
      else {
        $t = substr(sprintf("%04X",$t),4,4)
      }
      $log->debug("pt=$1,$2 ($p,$t)");
      $p='0'.substr($p,0,1).'0'.substr($p,1,1).'0'.substr($p,2,1).'0'.substr($p,3,1);
      $t='0'.substr($t,0,1).'0'.substr($t,1,1).'0'.substr($t,2,1).'0'.substr($t,3,1);
      checkURL("$URL=810106021010${p}${t}FF");
    }
    elsif($cmd=~/Z/) {		# Zoom
      $param=~/(\d+)\s*$/;
      $log->debug("z=$1");
    }
    elsif($cmd=~/center/i) {	# center
      checkURL("$URL=8101062403FF"); # Auto Pan-Tilt Speed -> OFF
      checkURL("$URL=8101040603FF"); # Digital-Zoom -> ON
      checkURL("$URL=8101040B00FF"); # Iris reset
      checkURL("$URL=8101043802FF"); # Focus reset
      checkURL("$URL=81010604FF");   # Home
    }

 } elsif($mode=~/step/i){	# mode=step  /Step by step positioning/
   $log->debug("SPEED! $options{speed} $spd");
   if($cmd=~/move/i) { # step pan/tilt
      my $pan = $param=~/[left|right]/i ? $spd : "00";
      my $tlt = $param=~/[up|down]/i ? $spd : "00";
      $log->debug("PARAM=$param SPEED=$spd PAN=$pan TILT=$tlt");
      if   ($param=~/^left$/i)     { checkURL("$moveURL=1&ok=dummy.htm"); }
      elsif($param=~/^right$/i)  { checkURL("$moveURL=2&ok=dummy.htm"); }
      elsif($param=~/^up$/i)  { checkURL("$moveURL=4&ok=dummy.htm"); }
      elsif($param=~/^down$/i)  { checkURL("$moveURL=8&ok=dummy.htm"); }
      elsif($param=~/^upleft$/i) { checkURL("$moveURL=4&ok=dummy.htm"); select (undef,undef,undef, .5);checkURL("$moveURL=1&ok=dummy.htm"); }
      elsif($param=~/^upright$/i) { checkURL("$moveURL=4&ok=dummy.htm"); select (undef,undef,undef, .5); checkURL("$moveURL=2&ok=dummy.htm"); }
      elsif($param=~/^downleft$/i) { checkURL("$moveURL=8&ok=dummy.htm"); select (undef,undef,undef, .5); checkURL("$moveURL=1&ok=dummy.htm"); }
      elsif($param=~/^downright$/i) { checkURL("$moveURL=8&ok=dummy.htm"); select (undef,undef,undef, .5); checkURL("$moveURL=2&ok=dummy.htm"); }
      select(undef,undef,undef,.05);
     # checkURL("$URL=8101060100000303FF");
   }
   elsif ($cmd=~/zoom/i) { # zoom
      $spd = sprintf("%01d",$spd/100*7); $spd = 7 if $spd > 7;
      my $cmdcode= ($param=~/out/i)? "3$spd" : "2$spd";
      $log->debug("ZOOM!!! PARAM=$param SPEED=$spd ZOOM");
      checkURL("$URL=81010407${cmdcode}FF");
      select(undef,undef,undef,.1);
      checkURL("$URL=8101040700FF")
   }
   elsif ($cmd=~/focus/i) { # focus
      $spd = sprintf("%01d",$spd/100*7); $spd = 7 if $spd > 7;
      my $cmdcode= ($param=~/near/i)? "3$spd" : "2$spd";
      checkURL("$URL=81010408${cmdcode}FF");
      select(undef,undef,undef,.1);
      checkURL("$URL=8101040800FF")
   }

 } elsif($mode=~/hardware/i){	# mode=hardware  /Hardware reset,defaults/
  if($cmd=~/do/i && $param=~/defaults/i) { # init
    checkURL("$URL=8101062403FF"); # Auto Pan-Tilt Speed -> OFF
    checkURL("$URL=8101040603FF"); # Digital-Zoom -> ON
    checkURL("$URL=8101040B00FF"); # Iris reset
    checkURL("$URL=8101043802FF"); # Focus reset
  }

 } elsif($mode=~/smooth/i){ 	# mode=smooth  /Old style PTZ. Goes to direction 
			 	# untill stop (or any other command is sent)/
  if($cmd=~/move/i) { # pan/tilt
    my $pan = ( $param=~/[left|right]/) ? $spd : "00";
    my $tlt = ( $param=~/[down|up]/) ? $spd : "00";
    $log->debug("PARAM=$param SPEED=$spd PAN=$pan TILT=$tlt");
    if   ($param=~/^left$/i)	{ checkURL("$URL=81010601${pan}000203FF"); }
    elsif($param=~/^right$/i)	{ checkURL("$URL=81010601${pan}000103FF"); }
    elsif($param=~/^up$/i) 	{ checkURL("$URL=8101060100${tlt}0302FF"); }
    elsif($param=~/^down$/i)	{ checkURL("$URL=8101060100${tlt}0301FF"); }
    elsif($param=~/^upleft$/i)	{ checkURL("$URL=81010601${pan}${tlt}0202FF"); }
    elsif($param=~/^upright$/i)	{ checkURL("$URL=81010601${pan}${tlt}0102FF"); }
    elsif($param=~/^downleft$/i){ checkURL("$URL=81010601${pan}${tlt}0201FF"); }
    elsif($param=~/^downright$/i){ checkURL("$URL=81010601${pan}${tlt}0101FF"); }
  }
  elsif($cmd=~/zoom/i) { # zoom
    $spd = sprintf("%01d",$spd/100*7); 
    $spd = 7 if $spd > 7;
    $param = (defined $param) ? lc($param) : "stop";
    my %code = (
           "in"            => "810104072${spd}FF",
           "out"           => "810104073${spd}FF",
	   "stop"	   => "8101040700FF"
    );
       checkURL("$URL=$code{$param}")
  }
  elsif($cmd=~/iris/i) { # iris
    $param = (defined $param) ? lc($param) : "reset";
    my %code = (
	   "reset"	=> "8101040B00FF",
           "open"       => "8101040B02FF",
           "close"      => "8101040B03FF"
    );
       checkURL("$URL=$code{$param}")
  }
  elsif($cmd=~/focus/i) { # focus
    $spd = sprintf("%01d",$spd/100*7); $spd = 7 if $spd > 7;
    my $cmdcode= ($param=~/near/)? "3$spd" : "2$spd";
    checkURL("$URL=81010408${cmdcode}FF")
  }

 } elsif($mode=~/preset/i){ # presets
    my $pos=sprintf("%02d",$param);
    my $code;
    $code = '00' if $cmd=~/clear/;
    $code = '01' if $cmd=~/save/;
    $code = '02' if $cmd=~/goto/;

    checkURL("$URL=8101043F${code}${pos}FF");
  }
  elsif($mode=~/settings/i){ # settings
     if ($cmd=~/witebalance/i) {
	$param = (defined $param) ? lc($param) : "auto";
	my %code = (
		"auto"		=> "8101043500FF",
		"indoor"	=> "8101043501FF",
		"outdoor"	=> "8101043502FF",
		"onepushwb"	=> "8101043503FF",
		"atw"		=> "8101043504FF",
		"manual"	=> "8101043505FF",
		"onepushtrigger"=> "8101043506FF"
	);
    	checkURL("$URL=$code{$param}")
     } elsif ($cmd=~/gain/i) {
	$param = (defined $param) ? lc($param) : "reset";
	my ($p,$q) = split(/,/,$options{pq}) if $options{pq};
	my %code = (
		"reset"		=> "8101040C00FF",
		"up"		=> "8101040C02FF",
		"down"		=> "8101040C03FF",
		"direct"	=> "8101044C000${p}0${q}FF"
	);
    	checkURL("$URL=$code{$param}")
     } elsif ($cmd=~/iris/i) {
        $param = (defined $param) ? lc($param) : "reset";
        my ($p,$q) = split(/,/,$options{pq}) if $options{pq};
        my %code = (
                "reset"         => "8101040B00FF",
                "up"            => "8101040B02FF",
                "down"          => "8101040B03FF",
                "direct"        => "8101044B000${p}0${q}FF"
        );
        checkURL("$URL=$code{$param}")
     } elsif ($cmd=~/backlight/i) {
        $param = (defined $param) ? lc($param) : "off";
        my %code = (
                "on"		=> "8101043302FF",
                "off"		=> "8101043303FF",
        );
        checkURL("$URL=$code{$param}")
     } elsif ($cmd=~/autofocus/i) {
        $param = (defined $param) ? lc($param) : "on";
        my %code = (
                "on"		=> "8101043802FF",
                "off"		=> "8101043803FF",
        );
        checkURL("$URL=$code{$param}")
     } elsif ($cmd=~/digitalzoom/i) {
        $param = (defined $param) ? lc($param) : "on";
        my %code = (
                "on"		=> "8101040602FF",
                "off"		=> "8101040603FF",
        );
        checkURL("$URL=$code{$param}")
     }
  }
  elsif($mode=~/rel/i){ # relative positioning (recentering)
      $param =~ /(\d+)x(\d+)/;
      my ($rel_size_x,$rel_size_y) = ($1,$2);
      $options{xy} =~ /(\d+),(\d+)/;
      my ($rel_click_x,$rel_click_y) = ($1,$2);
     # $param = int(640/$rel_size_x*$rel_click_x).",".int(480/$rel_size_y*$rel_click_y);
      $param = int($rel_click_x/26)*256+int($rel_click_y/13);
      $log->debug("x=($rel_click_x/26)*256+int($rel_click_y/13))");
      my $current=currentPOS("$curURL"); # check current position
      checkURL("$relURL=${param}&ok=dummy.htm");
      $log->debug("CURRENT POS=$current"); #POSOX debug
      $log->debug("PARAM=$param");
      my $cur_x=$current/0xFF;
      my $cur_y=$current%0xFF;
      $log->debug("CUR_X and CUR_Y= $cur_x, $cur_y");
      return;
     if ($cmd=~/xy/i && $param=~/^\??\d+,\d+$/) {
	my $size = ($options{size}) ? $options{size} : "640x480";
	$param=~s/(\?)//g;
	if ($size ne "640x480") {
	   my ($width,$height) = split (/x/,$size);
	   my ($x,$y) = split(/,/,$param);
	   $param = int(640/$width*$x).",".int(480/$height*$y);
	}
     $log->debug("PARAM=$param SIZE=$size PARAM=$param");
     checkURL("$relURL=${param}&ok=dummy.htm");
     }
  }
  $last_cmd = $cmd;
  $last_mode= $mode;
}


# --------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {
 %conf = GetCfgs( eval("($query_str)") );     # Load configurations
 $log->debug("Config read as:");
 foreach my $dev (keys %conf) {
   camParams($dev);
   $log->debug("[$dev]");
   $log->debug("CAMERAMODEL=$conf{$dev}{CAMERAMODEL}");
   $log->debug("POSITIONCTL=$conf{$dev}{POSITIONCTL}");
   $log->debug("DEVIP=$conf{$dev}{DEVIP}");
 }
}

# ------------------------------------------------------------- camParams -----
sub camParams {
  my ($_dev)=@_;
  if($conf{$_dev}{PTZPARAMS}=~/(.+):(.+):(.+):(.+)/) {
    $conf{$_dev}{pan}=$1;
    $conf{$_dev}{tilt}=$2;
    $conf{$_dev}{zoom}=$3;
    $conf{$_dev}{speed}=$4;
  }
  elsif($conf{$_dev}{PTZPARAMS} eq 'SONY EVI-D30/31') {
    $conf{$_dev}{pan}=1.66;
    $conf{$_dev}{tilt}=0.54;
    $conf{$_dev}{zoom}=625;
    $conf{$_dev}{speed}=20;
  }
  elsif($conf{$_dev}{PTZPARAMS} eq 'SONY EVI-D100/D100P') {
    $conf{$_dev}{pan}=2.78;
    $conf{$_dev}{tilt}=0.69;
    $conf{$_dev}{zoom}=10000;
    $conf{$_dev}{speed}=20;
  }
  else {
    $conf{$_dev}{pan}=1;
    $conf{$_dev}{tilt}=1;
    $conf{$_dev}{zoom}=1000;
    $conf{$_dev}{speed}=100;
  }

  if($conf{$_dev}{PTZSTEPS}=~/(.+):(.+):(.+)/) {
    $conf{$_dev}{span}=$1;
    $conf{$_dev}{stilt}=$2;
    $conf{$_dev}{szoom}=$3;
  }
  else {
    $conf{$_dev}{span}=50;
    $conf{$_dev}{stilt}=50;
    $conf{$_dev}{szoom}=50;
  }
  $conf{$_dev}{p_pos}=0;
  $conf{$_dev}{t_pos}=0;
  $conf{$_dev}{z_pos}=0;
} # sub camParams


# -------------------------------------------------------------- checkURL -----
sub checkURL {
  my @answ;
  my $val=-2;
  my($url)=@_;  #print "URL: $url\n";
  my($server,$port,$path)=($url=~/^http:\/\/([\w\.]+):?(\d+)?\/(.*)/)? ($1,$2,$3):('','');
  $port = 80 if not $port;
  my $iaddr = inet_aton($server) || die "no host: $server";
  my $paddr = sockaddr_in($port, $iaddr);
  my $proto = getprotobyname('tcp');
  my $auth=($usrpsw)?"\nAuthorization: Basic ". MIME::Base64::encode($usrpsw,''):'';
  $log->debug("$usrpsw \n checkURL($url)\t");
  eval {
    local $SIG{ALRM}=sub{die 'ALARM'};
    alarm 2;
    socket(SOCK, PF_INET, SOCK_STREAM, $proto) || die "socket: $!";
    connect(SOCK, $paddr) || die "connect: $!";
    autoflush SOCK;
    my $get=qq(GET /$path HTTP/1.0$auth
Accept: image/gif, image/x-xbitmap, image/jpeg, image/pjpeg, image/png
Accept-Charset: iso-8859-1,*,utf-8
Accept-Encoding: gzip
Accept-Language: en
User-Agent: Mozilla/4.76 [en] (X11; U; Linux 2.2.16 i686)

);
    print SOCK $get;
    $val = -1;
    @answ=<SOCK>;  #print "@answ\n";
    close (SOCK);
    alarm 0;
  } or $val=-2; # eval
  alarm 0;
  foreach (@answ) { next if not /.+=(\d)/; return $1;}
  return $val;
} # sub checkURL

# ----------------------------------------------------------- currentPOS-----
sub currentPOS{
  my @answ;
  my $val=-2;
  my($url)=@_;  #print "URL: $url\n";
  my($server,$port,$path)=($url=~/^http:\/\/([\w\.]+):?(\d+)?\/(.*)/)? ($1,$2,$3):('','');
  $port = 80 if not $port;
  my $iaddr = inet_aton($server) || die "no host: $server";
  my $paddr = sockaddr_in($port, $iaddr);
  my $proto = getprotobyname('tcp');
  my $auth=($usrpsw)?"\nAuthorization: Basic ". MIME::Base64::encode($usrpsw,''):'';
  $log->debug("$usrpsw \n currentURL($url)\t");
  eval {
    local $SIG{ALRM}=sub{die 'ALARM'};
    alarm 2;
    socket(SOCK, PF_INET, SOCK_STREAM, $proto) || die "socket: $!";
    connect(SOCK, $paddr) || die "connect: $!";
    autoflush SOCK;
    my $get=qq(GET /$path HTTP/1.0$auth
Accept: image/gif, image/x-xbitmap, image/jpeg, image/pjpeg, image/png
Accept-Charset: iso-8859-1,*,utf-8
Accept-Encoding: gzip
Accept-Language: en
User-Agent: Mozilla/4.76 [en] (X11; U; Linux 2.2.16 i686)

);
    print SOCK $get;
    $val = -1;
    @answ=<SOCK>;  #print "@answ";
    close (SOCK);
    alarm 0;
  } or $val=-2; # eval
  alarm 0;
  foreach (@answ) { next if not /.+=(\d)/; return $1;}

  $answ[$#answ]=~ s/\D.*//; #bad code revew later
  $val = $answ[7]; #bad code revew later
  $log->debug("CURR POS= $val \n");#bad code revew later
  
  return $val;
} # sub currentPOS

# ----------------------------------------------------------- truncDigits -----
sub truncDigits{
  my($val)=@_;
  return sprintf("%5.5f",$val)
} # sub truncDigits

# -------------------------------------------------------------- nonblock -----
#   nonblock($socket) puts socket into nonblocking mode
sub nonblock {
    my $socket = shift;
    my $flags;
    
    $flags = fcntl($socket, F_GETFL, 0)
            or die "Can't get flags for socket: $!\n";
    fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
            or die "Can't make socket nonblocking: $!\n";
}
