#!/usr/bin/perl
# $Id: ptz_splitter.pl 25370 2012-03-09 21:19:34Z teetov $
# -----------------------------------------------------------------------------
#  This script manages multiple processes handling one type of device, but
#  splitting them based on split criteria passed into command line.
#
#  The main purpose is to create multiple instances of PTZ drivers handling
#  their own hardware, so they do not create a bottleneck
# -----------------------------------------------------------------------------
#  Author: Andriy Fomenko
#  Edited by: 
#  QA by:  Christopher C Gettings
#  Copyright: videoNEXT LLC, (c) 2005
# -----------------------------------------------------------------------------
#  Command line:
#    ptz_splitter.pl "engine_name" "conf_query_string" "split_field"
#      where:
#        engine_name       is like "ptz_axis_v2.pl"
#        conf_query_string is like "'POSITIONCTL'=>'Axis','CAMERAMODEL'=>'axis'"
#        split_field       is like "DEVIP"
#
#  This script will launch/kill engines as needed, each engine will connect
#  to PTZ_server via socket and receive commands in broadcasts, so it has
#  to be able to ignore commands for non-handled devices. 
#  Process will receive only one command line parameter - query string to 
#  query for devices
# -----------------------------------------------------------------------------
use strict;
use Socket;
use IO::Socket;
use Fcntl;
use POSIX;
use NextCAM::Init;

use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

my $log=get_logger('NEXTCAM::PTZ::SPLITTER');

my $engine    = shift || $log->logdie('Engine name is not provided');
my $query_str = shift || $log->logdie('Query string is not provided');
my $split_fld = shift || $log->logdie('Splitting field name is not provided');

# -----------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# -----------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
                                   PeerPort => $TCP_PORT,
                                   Proto     => "tcp",
                                   Type      => SOCK_STREAM)
    or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");
nonblock($socket);
print $socket "PTZ DRIVER\n";

TerminateAll(); # kill driver processes if any are still in memory
#$SIG{CHLD} = 'IGNORE';

$SIG{CHLD} = \&REAPER;
sub REAPER {
	my $pid;
	$pid = waitpid(-1, &WNOHANG);
	if ($pid == -1) { # no child waiting.  Ignore it.
	} elsif (WIFEXITED($?)) {
		$log->debug("Process $pid exited.");
	} else {
		$log->debug("False alarm on $pid.");
	}
	$SIG{CHLD} = \&REAPER;          # in case of unreliable signals
}

my ($cmd, %conf, $dev, $drv, $key, %drvs);
$cmd = 'first-loop-entrance';

while(1) {
    if($cmd eq 'first-loop-entrance') {
        $cmd = '';
    }
    else {
        if(not defined($cmd=<$socket>)) {
            $cmd = '';
        }
        if($cmd =~ /HUP/) {
            $log->debug('HURRAY!! We hove some usefull stuff to do!');
        }
        else {
            sleep 10;
            # check if processes are alive
            my $pids = `ps ax|cut -c 1-6`;
            foreach $drv ( keys %drvs ) {
                next if not defined($drvs{$drv}{PID});
                next if $drvs{$drv}{PID} == -1;
                my $pid = sprintf("%5d",$drvs{$drv}{PID});
                next if $pids=~/$pid/mi;
                $drvs{$drv}{PID} = -1; # schedule to restart
            }    
            goto updown;
        }
    }
    UpdateCfgs( \%conf, eval("($query_str)") );

    # go over registered drivers/devices, mark devices as non-active
    foreach $drv ( keys %drvs ) {
        foreach $key ( keys %{$drvs{$drv}} ) {
            next if $key eq 'PID';
            $drvs{$drv}{$key} = 0;
        }
    }
    
    # go over devices in the system, mark them as active if they are present
    foreach $dev ( keys %conf ) {
        if($conf{$dev}{TRG} == -1) {
            delete $conf{$dev};
            next
        }
        $drv = $conf{$dev}{$split_fld};
        $log->debug("[$dev] POSITIONCTL=$conf{$dev}{POSITIONCTL} CAMERAMODEL=$conf{$dev}{CAMERAMODEL} DRV:$drv");
        if(not defined($drvs{$drv})) {
            # driver for this resource was not created yet
            $drvs{$drv}{PID} = -1; # flag to start process
            $drvs{$drv}{$dev} = 1; # stands for new/active device
        }
        else {
            # driver is defined, lets mark device as active and proceed
            $drvs{$drv}{$dev} = 1; # stands for new/active device
        }
    }
    updown:
    # go over registered drivers/devices again, check for necessity to bring drivers up/down
    foreach $drv ( keys %drvs ) {
        $drvs{$drv}{COUNT} = 0; # usage counter
        foreach $key ( keys %{$drvs{$drv}} ) {
            next if $key eq 'PID';
            $drvs{$drv}{COUNT}++ if($drvs{$drv}{$key});
        }
        
        if(! $drvs{$drv}{COUNT}) {
            # this driver is useless, let's terminate it
            next if $drvs{$drv}{PID} == -1; # should not happen, but who knows
            Terminate($drv);
        }
        else {
            # driver is used by endines, check if we need to start one
            next if $drvs{$drv}{PID} != -1; # ignore running
            StartProcess($drv);
        }
    }
}

# -------------------------------------------------------------- nonblock -----
sub nonblock {
    my ($fd) = @_;
    my $flags = fcntl($fd, F_GETFL,0);
    fcntl($fd, F_SETFL, $flags|O_NONBLOCK);
}

# ---------------------------------------------------------- TerminateAll -----
sub TerminateAll {
    $log->debug('TerminateAll:',$engine);
    system("kill -9 `ps ax|grep $engine|grep -v grep|grep -v splitter|cut -c 1-6`");
}

# ------------------------------------------------------------- Terminate -----
sub Terminate {
    my $drv = shift;
    $log->debug("Terminate($drv)");
    system("kill -9 $drvs{$drv}{PID}");
}

# ---------------------------------------------------------- StartProcess -----
sub StartProcess {
    my $drv = shift;
    # first, create a command line for the process
    my $cmd_line = "$engine \"$query_str,'$split_fld'=>'$drv'\"";
    $drvs{$drv}{PID} = fork();
    if(!$drvs{$drv}{PID}) { # here goes child
        exec($cmd_line);
        exit 0;
    }
    $log->debug("StartProcess($drv)-> $drvs{$drv}{PID} [$cmd_line]");
}

