#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#  PTZ driver for VISCA protocol - Modified from pvpaeo
# ------------------------------------------------------------------------------
#  Author: Jeffrey Scott
#  Edited by: Andriy Fomenko
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT Network Solutions, Inc, 2010
# ------------------------------------------------------------------------------
#  VISCA PTZ driver
#  
# Requires STOMP
#

use strict;
use POSIX;
use Socket;
use IO::File;
use IO::Select;
use IO::Socket;
use Fcntl;
use JSON;
use XML::Simple;
use Net::STOMP::Client;
use Data::Dumper;
use Math::Trig;

use NextCAM::Init;
use Log::Log4perl "get_logger";

require "$ENV{APL}/common/bin/logger.engine";

# Constants
my $CHECK_DANTE_TIMEOUT = 5;

my $log=get_logger('NEXTCAM::PTZ');

my $query_str = shift || "'POSITIONCTL'=>'Pelco'";
$log->info("Starting VISCA PTZ DRIVER, query string: [$query_str]");

$SIG{HUP}=\&load_dev_conf;

my $APL=$ENV{APL};
my (%conf, %sock, $cam, $cmd, $par, $last_cmd, $last_mode, $lastchecktime);

load_dev_conf();

# ------------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# ------------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
			PeerPort => $TCP_PORT,
			Proto     => "tcp",
			Type      => SOCK_STREAM)
		or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

my $stomp = Net::STOMP::Client->new( uri=>"stomp://localhost:61613" );
$stomp->message_callback(sub{return(1) } );
eval{
	$stomp->connect();

	my $sid = $stomp->uuid();
	$stomp->subscribe(
		destination	=> "/topic/amq.gis.topic",
		id		=> $sid,
	);
	sub msg_cb($$) {
		my ($self, $frame) = @_;
		return(0) unless $frame->command() eq "MESSAGE";
		return($frame->body());
	}
};

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';
$lastchecktime = time;

my %commands;
my $trycount = 30;
my $c = 0;
my $lastpostime = time;

my $deg_p=142;
my $deg_t=142;
my $deg_cal = 0;
my $vID=1;


while(1) {
	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam}=~/speed pt=0,0/ || $commands{$cam}=~/speed zoom=0/ || $commands{$cam}=~/speed focus=0/ || $commands{$cam}=~/speed iris=0/ || $commands{$cam}=~/speed gain=0/ || $commands{$cam}=~/speed ptz=0,0,0/)
			{
				; # do not override stop command!
			}else{
				$commands{$cam} = $cmd;
			}
		} else {
			$commands{$cam} = $cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)
	
	$cmd = '';
	
	# STOMP stuff
	eval{
		my $frame = $stomp->wait_for_frames(callback => \&msg_cb, timeout=>0);
		writePos($cam,1,$1) if ($frame=~/rf_g.*usid="(.*)"/); 
		my $ref = XMLin($frame);
		#$log->debug("\n>>>>> goPos called <<<<< \n");
		goPos($cam, $ref) if ($ref->{type} eq 'amqGisPtzSet');
	};
	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		delete $commands{$camera};
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
		
		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam,$mode,"pt", $params[0].','.$params[1] ,\%options);
			camCmd($cam,$mode,"zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam,$mode,$cmd,$par,\%options);
		}
	}
	
	select(undef,undef,undef,.1) if not $cmd;
	
	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
	}
	
	if(!$cmd) {
		select(undef,undef,undef,.4);
		if(!$trycount--) {
			$trycount = 30;
			print $socket "test\0" or $log->logdie("Couldn't communicate to socket $TCP_PORT: $@");
		}
	}
	
} # while(1)


# ------------------------------------------------------------ cmdTransmit -----
sub cmdTransmit {
    my ($dev,$delay, @cmd)=@_;
	
	$vID = sprintf("%1d",$conf{$dev}{PTZID}); # VISCA ID on daisy-chain

    my $command='';
    my $b;
	my $ret;
    foreach $b (@cmd) { 
		$command .= sprintf("%02X ",$b); 
		$command .= sprintf("\r\n") if $b==0xff;
	}
	
    $log->debug('cmdTransmit( ',$dev, '=> ',$command,')');    
	
    my $bts = '';
    foreach my $b (@cmd) { $bts .= chr($b); }
	
    my $retries = 1;
secondaryPass:
    my $skt = $sock{ $conf{$dev}{DEVIP} }{SOCK};
    eval {
        local $SIG{PIPE} = sub { die "Error writing to socket" };
        print $skt $bts;
    };
    if($@) {
        $log->error($@);
        # camera did not answer or socket error occured - try to close socket and reopen it
        eval{ $sock{$conf{$dev}{DEVIP}}{SOCK}->close(); };
        $log->error("Re-open socket connection: $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT}");
        $sock{$conf{$dev}{DEVIP}}{SEL} = undef;
        select(undef,undef,undef,.1);
        $sock{$conf{$dev}{DEVIP}}{SOCK} = IO::Socket::INET->new(PeerAddr => $conf{$dev}{DEVIP},
		PeerPort => $conf{$dev}{PTZ_TCP_PORT},
		Proto    => "tcp",
		Type     => SOCK_STREAM);
        if(! $sock{$conf{$dev}{DEVIP}}{SOCK} ) {
            $log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
            delete $sock{$conf{$dev}{DEVIP}};
            return;
        }
        $sock{$conf{$dev}{DEVIP}}{SOCK}->autoflush(1);
        nonblock($sock{$conf{$dev}{DEVIP}}{SOCK});
        $sock{$conf{$dev}{DEVIP}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}}{SOCK});
        $log->info('Socket was closed, reconnected');
        goto secondaryPass if $retries--;
		
        return;
    }
	
    select(undef,undef,undef,$delay);
    foreach my $sss ( $sock{ $conf{$dev}{DEVIP} }{SEL}->can_read(1) ) {
		my $data = '';
		$ret='';
		my $loop = 0;
		do{
			#$log->debug("Recieve Loop.");
			eval {
				alarm 1;
				$sss->recv($data, POSIX::BUFSIZ, 0);
				
				for($b=0;$b<length($data);$b++) { 
					$ret .= sprintf("%02X ",ord(substr($data,$b,1))); 
				}
			};
			select(undef,undef,undef,0.01);
			$loop++;
		} while (!($ret=~/90\ 51/) && $loop < 100);
        alarm 0;
        $log->debug('Received:',$ret);
        return $ret;
    }  # foreach
    
	
} # sub cmdTransmit

# --------------------------------------------------------------- CheckSum -----
sub checkSum
{
	my @arr = @_;
	return ($arr[1]+$arr[2]+$arr[3]+$arr[4]+$arr[5]) % 256;
}
#checkSum

# ---------------------------------------------------------------- camStop--
sub camStop
{
	my ($dev) = @_;
	my @cmd = ( 0xFF, 0, 0, 0, 0, 0, 0 );
	return @cmd;
}

# --------------------------------------------------------------- camReset -----
sub camReset
{
	my ($dev) = @_;
	#my @cmd = ( 0xFF, 0, 0, 0, 0, 0, 0 );
	my @cmd = (0xFF, 0x00, 0x32, 0x77, 0x01, 0x11);

} # sub camReset

# ------------------------------------------------------------- camFZspeed -----
sub camFZspeed
{
	my ($dev,$FZ,$speed) = @_;
	$log->debug("camFZspeed($dev,$FZ,$speed)");
	my @cmd = ( 0xFF, 0, 0, 0, 0, 0, 0 );
	$cmd[1]=0;
	$cmd[3]=$FZ=='F'?0x27:0x25;
	$cmd[5]=sprintf("%d",abs($speed)/25);
	cmdTransmit($dev,0,@cmd);
} # sub camFZspeed

# ----------------------------------------------------------------- PTspeed ----
sub PTspeed {
	my ($speed)=@_;
	#return 255 if(abs($speed)>99); # turbo speed - does not work with spectra !!!
	$speed=100 if abs($speed)>100;
	my $val = int(abs($speed/100*0xfe));
	$val=254 if ($val > 254);
	return $val;
} # sub PTspeed

# ----------------------------------------------------------------- FZspeed ----
sub FZspeed {
	my ($speed)=@_;
	$speed=100 if abs($speed)>100;
	return 0 if $speed == 0;
	return (abs($speed)>70) ? 7 : 2; # 7 is high speed, 2 is low speed
} # sub FZspeed

# ----------------------------------------------------------------- checkReg ----
sub checkReg {
	my ($data, $mask, @cmd) = @_;
	
	my $reg;
	foreach my $a (@cmd) { $reg .= sprintf("%02X ", $a); }
	$data=~/$reg$mask/i;
	#$log->debug("Matched:: $1 :: with '$reg($mask)'");
	return $1;
}

# ----------------------------------------------------------------- getRange ----


# ----------------------------------------------------------------- getPos ----
sub getPos {
	my ($dev) = @_;
	my ($fov, $zoom, $focus, $az, $el);
	
	# PT, in steps
	my $rsp = cmdTransmit($cam, 0.1, (0x80+$vID, 0x09, 0x06, 0x52, 0xff));
	$rsp=~s/\s*//g;
	$rsp =~ /90510(\w)0(\w)0(\w)0(\w)0(\w)0(\w)0(\w)0(\w)0(\w)0(\w)FF/i;
	$az = hex("$1$2$3$4$5") / $deg_p + $conf{$dev}->{CAM_GEO_NORTH_OFFSET}; # response is in steps, convert to degrees
	$log->debug("NORTH offset: ".$conf{$dev}->{CAM_GEO_NORTH_OFFSET});
	$az %= 360;
	$el = hex("$6$7$8$9$10") / $deg_t - 90;
	
	$log->debug("Degrees (from steps) Az: $az El:$el");
	

	# zoom
	$fov = getFov();
	
	# focus
	$rsp = cmdTransmit($cam, 0.1, (0x80+$vID, 0x09, 0x04, 0x48, 0xff));
	$rsp =~ s/\s*//g;
	$rsp =~ /90500(\w)0(\w)0(\w)0(\w)ff/i;
	$focus = hex("$1$2$3$4");

	$log->debug("Got back - \n FOV: $fov\n AZ:$az\n EL:$el\n Focus:$focus\n");
	return ( FOV => $fov, DEG_AZ => $az, DEG_EL => $el, ZOOM=>$zoom, FOCUS=>$focus);
}

# ----------------------------------------------------------------- goPos ------

sub goPos{
	my ($dev, $ref) = @_;
	my ($range, $az, $el, $obji) = ($ref->{range}, $ref->{az}, $ref->{el}, $ref->{objid});
	my $range_scale = 375.5070; # This should be set by some form of calibration later. Calculated from 30.1752 meters for a zoom level of 0x2c43
	
	#return if not $obji == $dev;
	
	# base command, with fake values
	my @cmd=(0x80+$vID, 0x01, 0x06, 0x52, 0x18, 0x18, 0x01, 0x02, 0x03, 0x04, 0x05, 0x10, 0x20, 0x30, 0x40, 0x50, 0xff);
		
	# camera is accurate for tenth of a degree?
	my $p = $az;
	my $t = $el;
	# negative angles are 360degrees - angle
	$p += 360 if $p<0;
	$t += 90; # 0 is straight down, so add 90 degrees to everything
	$p = int($p*$deg_p);
	$t = int($t*$deg_t);
	$log->debug(sprintf("P T (%02X, %02X)", $p, $t));
	# convert to hex (0A 0B 0C 0D for step-count # ABCD)
	# pan
	$cmd[6] = ($p >> 16) & 0x0f;
	$cmd[7] = ($p >> 12) & 0x0f;
	$cmd[8] = ($p >> 8) & 0x0f;
	$cmd[9] = ($p >> 4) & 0x0f;
	$cmd[10] = $p & 0x0f;
	
	# tilt
	$cmd[11] = ($t >> 16) & 0x0f;
	$cmd[12] = ($t >> 12) & 0x0f;
	$cmd[13] = ($t >> 8) & 0x0f;
	$cmd[14] = ($t >> 4) & 0x0f;
	$cmd[15] = $t & 0x0f;
	
	$log->debug("Going to: az:$az el:$el range:$range");

	#cmdTransmit($dev,0,@cmd);
	
	#scale zoom, will do digital zoom if needed
	my $zoom = int($range * $range_scale);
	$zoom = 0x7000 if $zoom > 0x7000;

	$zoom = 0;
	
	$log->debug("Zoom set = $zoom");

	my @cmd2 = (0x80+$vID, 0x01, 0x04, 0x47, 1, 2, 3, 4, 0xff);
	$cmd2[4] = ($zoom >> 12) & 0x0f;
	$cmd2[5] = ($zoom >> 8 ) & 0x0f;
	$cmd2[6] = ($zoom >> 4 ) & 0x0f;
	$cmd2[7] = $zoom & 0x0f;
	
	# send both zoom and PT at once.
	cmdTransmit($dev,0,(@cmd, @cmd2));

}

sub getSteps{
	my ($dev) = $_;
	my $deg_rsp = cmdTransmit($cam, 0.1, (0x81, 0x09, 0x06, 0x32, 0xff));
	$deg_rsp=~s/\s*//g;
	$deg_rsp =~ 
	/90510(\w)0(\w)0(\w)0(\w)0(\w)0(\w)0(\w)0(\w)0(\w)0(\w)0(\w)0(\w)FF/i;
	$deg_p = hex("$1$2$3$4$5$6") / 256; # shift two places
	$deg_t = $deg_p;
	$log->debug("Recieved steps per degree. Pan:$deg_p Tilt:$deg_t");
	$deg_p = 142 if(not $deg_p);
	$deg_t = 142 if(not $deg_t);
}

# ----------------------------------------------------------------- writePos -----
sub writePos {
	my ($dev, $target, $USID) = @_;	
	my $mTime = time();
	my $dTime = sprintf("%s_%s",$dev,$mTime);
	my %posUpdate = ( getPos($dev) );

	$USID = 0 unless defined $USID;
	eval{
	
		my $range = 42;

		$stomp->send(
			destination	=>"/topic/amq.gis.topic",
			type		=>"amqGisPtzPos",
			time		=>time(), 
			id			=>"$dTime",
			usid		=>"$USID",
			az			=>"$posUpdate{DEG_AZ}",
			fov 		=>"$posUpdate{FOV}",
			el			=>"$posUpdate{DEG_EL}",
			zoom		=>"$posUpdate{ZOOM}",
			focus		=>"$posUpdate{FOCUS}",
			body		=>".",
			range		=>"$range",
		);
		

	};
}

sub getFov {
	my ($dev) = @_;

	my $fov;	

	my $rsp = cmdTransmit($cam, 0.1, (0x80+$vID, 0x09, 0x04, 0x47, 0xff));
	$rsp=~s/\s*//g;
	$rsp =~ /90500(\w)0(\w)0(\w)0(\w)ff/i;
	my $steps = hex("$1$2$3$4");
	$log->debug("Zoom steps is $steps");
	my $zoom = (-59264 / ($steps - 18490)) - 2.68; # more accurate approximation 
	$zoom = 1 if $zoom < 1;
	$log->debug("Zoom factor is $zoom");
	$fov = rad2deg(2 * atan(3.581982/(7*$zoom))); # Fancy Maths!

	$log->debug("FoV is $fov");

	$conf{$dev}->{CAM_GEO_HFOV} = $fov;
	return $fov;
}

# ----------------------------------------------------------------- camCmd -----

sub camCmd {
	my ($dev,$mode,$cmd,$param,$options)=@_;
	my %options = %$options;
	my @cmd;
	$log->debug("camCmd: DEVID=$dev mode: $mode command: $cmd optional parameter: $param");
	
	# here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
	if($conf{$dev}->{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
		if($mode=~/speed/ || $mode=~/step/ || $mode=~/abs/ || $mode=~/rel/ || $mode=~/smooth/) {
			$log->debug("Setting timeout +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
		elsif($mode=~/preset/ && $cmd=~/goto/ && $param!=1) {
			$log->debug("Setting timeout (preset) +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
	}
	
	if (not $deg_cal) {
		getSteps($dev);
		getFov($dev);
		$deg_cal = 1;
	}	
	
	if(not $cmd=~/stop/) { # non-Stop
		# set camera speed
		if ($mode=~/speed/){        # mode speed
			if($cmd=~/pt/i) { # RPT
				my ($p,$t) = split(/,/,$param);
				@cmd = ( 0x80+$vID, 0x01, 0x06, 0x01, 0, 0, 0x03, 0x03, 0xff );
				$cmd[4]=PTspeed($p);
				$cmd[5]=PTspeed($t);
				$log->debug(sprintf("RPT $p(%02X) $t(%02X)",$cmd[4],$cmd[5]));
				$cmd[6] = 0x02 if $p>0;
				$cmd[6] = 0x01 if $p<0;
				$cmd[6] = 0x03 if $p==0;
				$cmd[7] = 0x02 if $t>0;
				$cmd[7] = 0x01 if $t<0;
				$cmd[7] = 0x03 if $t==0;

				cmdTransmit($dev,0,@cmd);
				return;
			}
			elsif($cmd=~/z/i) { # Zoom 
				$log->debug("ZOOM $param");
				@cmd = ( 0x80+$vID, 0x01, 0x04, 0x07, 0, 0xff );
				$cmd[4] = 0x20 if $param>0;
				$cmd[4] = 0x30 if $param<0;
				$cmd[4] |= FZspeed($param);
				cmdTransmit($dev,0,@cmd);
				getFov($dev);
				return;
			}
			elsif($cmd=~/focus/) {
				@cmd = ( 0x80+$vID, 0x01, 0x04, 0x08, 0, 0xff );
				$cmd[4] = 0x20 if $param>0;
				$cmd[4] = 0x30 if $param<0;
				$cmd[4] |= FZspeed($param);
				cmdTransmit($dev,0,@cmd);
				return;
			}
		}
		elsif($mode=~/rel/){ # mode=rel: "<devid> rel size=640x480 xy=225,152"
			$param =~ /(\d+)x(\d+)/;
			my $fov;
			my ($rel_size_x,$rel_size_y) = ($1,$2);
			$options{xy} =~ /(\d+),(\d+)/;
			my ($rel_click_x,$rel_click_y) = ($1,$2);
			eval{
				
				my $fov = $conf{$dev}->{CAM_GEO_HFOV};
				# Calculate % offset from screen center, adjusted for current FoV
				my ($rel_p, $rel_t) = ( $fov*($rel_click_x-$rel_size_x/2)/$rel_size_x, $fov*($rel_click_y-$rel_size_y/2)/$rel_size_y ); 
				$log->debug("REL [$rel_size_x] [$rel_size_y] [$rel_click_x] [$rel_click_y] [$rel_p] [$rel_t]");
				$cmd[4]=PTspeed($rel_p); # % difference from the middle of the screen, +-50%
				$cmd[5]=PTspeed($rel_t);
				@cmd = ( 0x80+$vID, 0x01, 0x06, 0x53, 0x18, 0x18, 01, 02, 03, 04, 05, 10, 20, 30, 40, 50, 0xff);
				
				# Convert to whole (hex) numbers
				$rel_p = int($rel_p * $deg_p);
				$rel_t = int($rel_t * $deg_t * -1); # seems to be inverted
				$log->debug("Relative reposition in steps: $rel_p, $rel_t");
				
				# pan
				$cmd[6]	 = ($rel_p >> 16) & 0x0f;
				$cmd[7]	 = ($rel_p >> 12) & 0x0f;
				$cmd[8]	 = ($rel_p >> 8 ) & 0x0f;
				$cmd[9]  = ($rel_p >> 4 ) & 0x0f;
				$cmd[10] = ($rel_p >> 0 ) & 0x0f;
				
				# tilt
				$cmd[11] = ($rel_t >> 16) & 0x0f;
				$cmd[12] = ($rel_t >> 12) & 0x0f;
				$cmd[13] = ($rel_t >> 8 ) & 0x0f;
				$cmd[14] = ($rel_t >> 4 ) & 0x0f;
				$cmd[15] = ($rel_t >> 0 ) & 0x0f;
			};

			cmdTransmit($dev,0,@cmd);
			return;
		}
		elsif($mode=~/abs/){    # mode=ABS
			if ($cmd=~/pt/) {
				$log->debug("ABS:  PT $param");
				@cmd=(0x80+$vID, 0x01, 0x06, 0x52, 0x18, 0x18, 0x01, 0x02, 0x03, 0x04, 0x05, 0x10, 0x20, 0x30, 0x40, 0x50, 0xff);
				
				#my $deg = 222.22; # each 222.2, can be queried
				
				$param=~/^(\-?.+)?\,(\-?.+)$/;
				# camera is accurate for tenth of a degree
				my $p = $1;
				my $t = $2;
				# negative angles are 360degrees - angle
				$p += 360 if $p<0;
				$t += 90; # 0 is straight down, so add 90 degrees to everything
				$p = int($p*$deg_p);
				$t = int($t*$deg_t);
				$log->debug(sprintf("P T (%02X, %02X)", $p, $t));
				# convert to hex (0A 0B 0C 0D for step-count # ABCD)
				# pan
				$cmd[6] = ($p >> 16) & 0x0f;
				$cmd[7] = ($p >> 12) & 0x0f;
				$cmd[8] = ($p >> 8) & 0x0f;
				$cmd[9] = ($p >> 4) & 0x0f;
				$cmd[10] = $p & 0x0f;
				
				# tilt
				$cmd[11] = ($t >> 16) & 0x0f;
				$cmd[12] = ($t >> 12) & 0x0f;
				$cmd[13] = ($t >> 8) & 0x0f;
				$cmd[14] = ($t >> 4) & 0x0f;
				$cmd[15] = $t & 0x0f;
				cmdTransmit($dev,0,@cmd);	# Sending Pan / Tilt
				return;
			} 
			elsif ($cmd=~/z/){	# Absolute zoom
				$log->debug("ABS:  Z $param");
				my $zoom = int($param * 375.5070);
				cmdTransmit($dev,0,@cmd);	# Sending Pan / Tilt
				@cmd = (0x80+$vID, 0x01, 0x04, 0x47, 1, 2, 3, 4, 0xff);
				$cmd[4] = ($zoom >> 12) & 0x0f;
				$cmd[5] = ($zoom >> 8 ) & 0x0f;
				$cmd[6] = ($zoom >> 4 ) & 0x0f;
				$cmd[7] = $zoom & 0x0f;
				cmdTransmit($dev,0,@cmd);	# Sending Pan / Tilt
				
			}
			elsif($cmd=~/center/) { # center
				my @cmd = (0x80+$vID, 0x01, 0x06, 0x04, 0xff);
				cmdTransmit($dev,0,@cmd);
			}
			
		}
		elsif($mode=~/settings/){ # settings
			if($cmd=~/autofocus/) { 
				@cmd = ( 0x80+$vID, 0x01, 0x04, 0x38, 0x02, 0xff ) if $param=~/on/;
				@cmd = ( 0x80+$vID, 0x01, 0x04, 0x38, 0x03, 0xff ) if $param=~/off/;
				cmdTransmit($dev,0,@cmd);
				return;
			}
			
			elsif($cmd=~/mh/){
				@cmd = (0xff, 0, 0x32, 0x77, 0x01, 0x01);
			}
			elsif($cmd=~/reset/){
				@cmd = (0xff, 0, 0x32, 0x77, 0x01, 0x11);
			}
			elsif($cmd=~/home/){
				@cmd = (0x80+$vID, 0x01, 0x06, 0x04, 0x02, 0xff) if $param=~/set/;
				@cmd = (0x80+$vID,0x01,0x06,0x04,0x03,0xff) if $param=~/goto/;
				cmdTransmit($dev,0, @cmd);
				return;
			}
		}
		elsif($mode=~/step/){       # mode=step  /Step by step positioning/
			my $timeout = 0.2;
			if($cmd=~/move/) { # step pan/tilt
				@cmd=(0x80+$vID, 0x01, 0x06, 0x01, 0x80, 0x80, 0x03, 0x03, 0xff);
				$cmd[6] = 0x01 if $param=~/left/;
				$cmd[6] = 0x02 if $param=~/right/;
				$cmd[7] = 0x01 if $param=~/up/;
				$cmd[7] = 0x02 if $param=~/down/;
				cmdTransmit($dev,0,@cmd);
				select(undef,undef,undef,$timeout);
				$cmd[6]= 0x03;
				$cmd[7]= 0x03;
				cmdTransmit($dev,0,@cmd);
				return;
			} elsif ($cmd=~/zoom/){
				@cmd=(0x80+$vID, 0x01, 0x04, 0x07, 0, 0xff);
				$cmd[4] = 0x02 if $param=~/in/;
				$cmd[4] = 0x03 if $param=~/out/;
				cmdTransmit($dev,0,@cmd);
				select(undef,undef,undef,$timeout);
				$cmd[4]=0;
				cmdTransmit($dev,0,@cmd);
				getFov($dev);
				return;
			} elsif ($cmd=~/focus/){    
				@cmd=(0x80+$vID, 0x01, 0x04, 0x08, 0, 0xff);
				$cmd[4] |= 0x03 if $param=~/near/;
				$cmd[4] |= 0x02 if $param=~/far/;
				cmdTransmit($dev,0,@cmd);
				select(undef,undef,undef,$timeout);
				$cmd[4]=0;
				cmdTransmit($dev,0,@cmd);
				return;
			}
		}
		elsif($mode=~/hardware/){   # mode=hardware  /Hardware reset,defaults/
			if($cmd=~/do/ && $param=~/defaults/) { # init
				cmdTransmit($dev,0,(0x80+$vID, 0x01, 0x06, 0x05, 0xff));
				return;
			}
		} elsif($mode=~/preset/){ # presets
			@cmd = (0x80+$vID, 0x01, 0x04, 0x3f, 0x0f, 0, 0xff);
			$cmd[4]= 0x01 if $cmd =~/save/; 
			$cmd[4]= 0x02 if $cmd =~/goto/;
			$cmd[4]= 0x00 if $cmd =~/clear/;
			$param = 0 if ($param > 100 || $param < 0);
			$cmd[5]= $param;
			cmdTransmit($dev,0,@cmd);
		}
	}	
}

# --------------------------------------------------------------- nonblock -----
#   nonblock($socket) puts socket into nonblocking mode
sub nonblock {
	my $socket = shift;
	my $flags;
	
	$flags = fcntl($socket, F_GETFL, 0)
	or die "Can't get flags for socket: $!\n";
	fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
	or die "Can't make socket nonblocking: $!\n";
}

# ---------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {
	
	$log->info('Configurations refreshed: ',$query_str);
	
	# fisrst, close everything
	foreach my $skt (keys %sock) {
		close($sock{$skt}{SOCK});
		delete $sock{$skt};
	}
	
	%conf = GetCfgs( eval("($query_str)") );     # Load configurations
	foreach my $dev (keys %conf) {
		# in a case when PTZ works on another server (e.g. 3ETI COMTROL Serial Server)
		$conf{$dev}{DEVIP} = $conf{$dev}{PTZ_TCP_IP} if $conf{$dev}{PTZ_TCP_IP};
		
		$log->debug("[$dev]");
		$log->debug("DEVID=$conf{$dev}{DEVID}");
		$log->debug("PTZID=$conf{$dev}{PTZID}");
		$log->debug("PTGROUP=$conf{$dev}{PTZGROUP}");
		$log->debug("PTZSPEED=$conf{$dev}{PTZSPEED}");
		$log->debug("DEVIP=$conf{$dev}{DEVIP}");
		$log->debug("PTZ_TRANSPORT_PROTO=$conf{$dev}{PTZ_TRANSPORT_PROTO}");
		$log->debug("PTZ_TCP_PORT=$conf{$dev}{PTZ_TCP_PORT}");
		
		$log->info("Open socket connection: $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT}");
		if(not defined($sock{$conf{$dev}{DEVIP}})) {
			$sock{$conf{$dev}{DEVIP}}{SOCK} = IO::Socket::INET->new(PeerAddr => $conf{$dev}{DEVIP},
			PeerPort => $conf{$dev}{PTZ_TCP_PORT},
			Proto    => "tcp",
			Type     => SOCK_STREAM);
			if(! $sock{$conf{$dev}{DEVIP}}{SOCK} ) {
				$log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
				delete $sock{$conf{$dev}{DEVIP}};
				next;
			}
			
			$sock{$conf{$dev}{DEVIP}}{SOCK}->autoflush(1);
			nonblock($sock{$conf{$dev}{DEVIP}}{SOCK});
			$sock{$conf{$dev}{DEVIP}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}}{SOCK});
		}
	} # foreach $dev
} # sub load_dev_conf

