#!/usr/bin/perl
#  $Id: ptz_sonyVISCAv2.pl 25370 2012-03-09 21:19:34Z teetov $
# -----------------------------------------------------------------------------
#  SONY PTZ cameras attached to serial port through RS232, VISCA protocol
# -----------------------------------------------------------------------------
#  Author: Andrey Fomenko
#  Edited by: 
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT LLC, 2004-2005
# -----------------------------------------------------------------------------
use strict;
use Device::SerialPort; # qw( :PARAM :STAT 0.07 );
use Socket;
use IO::File;
use IO::Select;
use IO::Socket;
use Fcntl;

use NextCAM::Init;
use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

my $log=get_logger('NEXTCAM::PTZ::PTZ_VISCAV2');

my $query_str = shift || "'POSITIONCTL'=>'SonyVISCA'";
$log->info("Starting SONYVISCA PTZ DRIVER, query string: [$query_str]");

$SIG{HUP}=\&load_dev_conf;

my $APL=$ENV{APL};
my (%conf,%port,$cam,$cmd,$par,$usrpsw,$last_mode,$last_cmd, $cmd_executed);
load_dev_conf();

$last_mode='smooth';
$last_cmd='left';

# -----------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# -----------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
                                   PeerPort => $TCP_PORT,
                                   Proto     => "tcp",
                                   Type      => SOCK_STREAM)
    or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';

my %commands;
my $ready;
my $trycount = 30;

while(1) {
	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/i;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/
				|| $commands{$cam} =~ /speed iris=0/
				|| $commands{$cam} =~ /speed gain=0/
				)
			{
				; # do not override stop command!
			}else{
				$commands{$cam} = $cmd;
			}
		} else {
			$commands{$cam} = $cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)
	
	$cmd = '';
	
	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		delete $commands{$camera};
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
		$usrpsw = "$conf{$cam}{USRNAME}:$conf{$cam}{PASSWD}" if $conf{$cam}{USRNAME} && $conf{$cam}{PASSWD};
		
		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam,$mode,"pt", $params[0].','.$params[1] ,\%options);
			camCmd($cam,$mode,"zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam,$mode,$cmd,$par,\%options);
		}
	}

	select(undef,undef,undef,.1) if not $cmd;

	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
	}
	
	if(!$cmd) {
		select(undef,undef,undef,.4);
		if(!$trycount--) {
			$trycount = 30;
			print $socket "test\0" or $log->logdie("Couldn't communicate to socket $TCP_PORT: $@");
		}
	}

} # while(1)




#} # main while loop

# --------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {
	# close any ports before we go any further
	foreach my $p ( keys %port ) {
		$port{$p}->close;
		delete $port{$p};
	}
	%conf = GetCfgs( eval("($query_str)") );     # Load parameters
	$log->debug("Config read as:");
	$log->debug("-------------------------------------");
	foreach my $dev (keys %conf) {
		$log->debug("[$dev]");
		$log->debug("DEVID=$conf{$dev}{DEVID}");
		$log->debug("CAMERAMODEL=$conf{$dev}{CAMERAMODEL}");
		$log->debug("POSITIONCTL=$conf{$dev}{POSITIONCTL}");
		$log->debug("PTZID=$conf{$dev}{PTZID}");
		$log->debug("PTZHWPORT=$conf{$dev}{PTZHWPORT}");
		$log->debug("PTZSPEED=$conf{$dev}{PTZSPEED}");
		$log->debug("PTZHWPORT_SPEED=$conf{$dev}{PTZHWPORT_SPEED}");
		$log->debug("PTZHWPORT_BITS=$conf{$dev}{PTZHWPORT_BITS}");
		$log->debug("PTZHWPORT_STOPBITS=$conf{$dev}{PTZHWPORT_STOPBITS}");
		$log->debug("PTZHWPORT_PARITY=$conf{$dev}{PTZHWPORT_PARITY}");
		$log->debug("-------------------------------------");
		#
		if(not defined($port{$conf{$dev}{PTZHWPORT}})) {
			$log->debug("Initialise serial port: $conf{$dev}{PTZHWPORT}");
			$port{$conf{$dev}{PTZHWPORT}} = new Device::SerialPort($conf{$dev}{PTZHWPORT}) || die "Can't open port $conf{$dev}{PTZHWPORT}: $!\n";
			$port{$conf{$dev}{PTZHWPORT}}->handshake("none"); # none / rts / xoff

# commented-out old init
#     $port{$conf{$dev}{PTZHWPORT}}->baudrate(9600);
#     $port{$conf{$dev}{PTZHWPORT}}->parity("none"); # none / odd / even   $PortObj->parity_enable("T");
#     $port{$conf{$dev}{PTZHWPORT}}->databits(8);
#     $port{$conf{$dev}{PTZHWPORT}}->stopbits(1);
        #  PTZHWPORT_SPEED=2400|9600
        #  PTZHWPORT_BITS=8|7
        #  PTZHWPORT_STOPBITS=1|2
        #  PTZHWPORT_PARITY=none|odd|even

			$port{$conf{$dev}{PTZHWPORT}}->baudrate($conf{$dev}{PTZHWPORT_SPEED});
			$port{$conf{$dev}{PTZHWPORT}}->parity($conf{$dev}{PTZHWPORT_PARITY});
			$port{$conf{$dev}{PTZHWPORT}}->databits($conf{$dev}{PTZHWPORT_BITS});
			$port{$conf{$dev}{PTZHWPORT}}->stopbits($conf{$dev}{PTZHWPORT_STOPBITS});
			
			$port{$conf{$dev}{PTZHWPORT}}->buffers(4096, 4096);
			$port{$conf{$dev}{PTZHWPORT}}->read_const_time(200);  # milliseconds
			$port{$conf{$dev}{PTZHWPORT}}->read_char_time(15);
			$log->debug("Done" );
			$log->debug("-------------------------------------");
		}
	} # foreach $dev

	foreach my $sp ( keys %port ) {
		$log->debug("Sending address-assign command to serial port");
		$port{$sp}->write(chr(0x88));
		$port{$sp}->write(chr(0x30));
		$port{$sp}->write(chr(0x01));
		$port{$sp}->write(chr(0xFF));
		$port{$sp}->write_drain;
	}

	# close unused ports
	my $found;
	foreach my $p ( keys %port ) {
		$found=0;
		foreach my $dev ( keys %conf ) {
			$found=1 if $conf{$dev}{PTZHWPORT} eq $p;
		}
		if( !$found ) {
			$port{$p}->close;
			delete $port{$p};
		}
	}
}

sub hexdump {
	my ($input) = @_;
	my $s = '';
	for(my $i=0; $i<length($input); $i++) {
		$s .= sprintf("%02X ",ord(substr($input,$i,1)));
	}
	return $s;
}


# ----------------------------------------------------------- cmdTransmit -----
sub cmdTransmit
{
	my ($dev,$cmd)=@_;
	$log->debug('COMMAND: ',hexdump($cmd));
	for (my $i=0; $i < length($cmd); $i+=2) {
		$port{$conf{$dev}{PTZHWPORT}}->write(chr(hex(substr($cmd,$i,2))));
	}
	$port{$conf{$dev}{PTZHWPORT}}->write_drain;
	my ($count_in, $string_in) = $port{$conf{$dev}{PTZHWPORT}}->read(30);
	$log->debug('RESPONSE: ',hexdump($string_in)) if $count_in;
} # sub cmdTransmit

# ---------------------------------------------------------------- camCmd -----
sub camCmd
{
	my ($dev,$mode,$cmd,$param,$options)=@_;
	my %options = %$options;
	$log->debug("camCmd: DEVID=$dev  command: $cmd ($cmd) optional parameter: $param");
	my $vID = sprintf("%1d",$conf{$dev}{PTZID}); # VISCA ID on daisy-chain
	my $spd=($options{speed}=~/^\d+/) ? $options{speed}:$conf{$dev}->{PTZSPEED};
	$spd=sprintf("%02x",$spd);
	my $zoomspd = $conf{$dev}->{PTZZOOMSPEED};
	$zoomspd=sprintf("%02x",$zoomspd);

	# here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
	if($conf{$dev}->{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
		if($mode=~/speed/i || $mode=~/step/i || $mode=~/abs/i || $mode=~/rel/i || $mode=~/smooth/i) {
			$log->debug("Setting timeout +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
		elsif($mode=~/preset/i && $cmd=~/goto/i && $param!=1) {
			$log->debug("Setting timeout (preset) +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
	}


	# we need to "terminate" each command because it can move forever if used with no STOP !
	if ($last_mode=~/smooth/i) {
		cmdTransmit($dev,"8${vID}01060100000303FF") if $last_cmd=~/move/i;
		cmdTransmit($dev,"8${vID}01060100000303FF") if $last_cmd=~/zoom/i;
		#  cmdTransmit($dev,"8${vID}01040700FF") if $last_cmd=~/[wt]/;  # zoom
		cmdTransmit($dev,"8${vID}01040B00FF") if $last_cmd=~/iris/i;  # iris
		cmdTransmit($dev,"8${vID}01040800FF") if $last_cmd=~/focus/i;  # focus
		$log->debug("Last mode was <$last_mode>. Last command <$last_cmd>");
	}

	if ($mode=~/speed/i){
		if($cmd=~/PT/i){ # RPT
			my ($p,$t) = split(/,/,$param);
			my ($vv,$ww) = (sprintf("%02x",abs($p/7)), sprintf("%02x",abs($t/7)));
			my $cccc = '';
			for ($p,$t) {
				$cccc .= '02' if $_>0;       # right or up
				$cccc .= '01' if $_<0;       # left or down
				$cccc .= '03' if $_==0;      # no movement in this direction
			}
			$log->debug("PT: $p $t $cccc");
			cmdTransmit($dev,"8${vID}010601${vv}${ww}${cccc}FF");
		}
		elsif($cmd=~/Z/i){ # RZ
			if($param ne 0) {
				$zoomspd = sprintf("%01d",abs($zoomspd/100*7));
				$zoomspd = 7 if $zoomspd > 7;
				my $cmdcode= ($param<0)? "3$zoomspd" : "2$zoomspd";
				cmdTransmit($dev,"8${vID}010407${cmdcode}FF")
			} else {
				cmdTransmit($dev,"8${vID}01040700FF");
			}
		}
		elsif($cmd=~/focus/i){ # RF
			if($param ne 0) {
				$spd = sprintf("%01d",abs($spd/100*7));
				$spd = 7 if $spd > 7;
				my $cmdcode= ($param<0)? "3$spd" : "2$spd";
				cmdTransmit($dev,"8${vID}010408${cmdcode}FF")
			} else {
				cmdTransmit($dev,"8${vID}01040800FF")
			}
		}
	}elsif($mode=~/rel/i) { # mode=rel: "<devid> rel size=640x480 xy=225,152"
		$param =~ /(\d+)x(\d+)/;
		my ($rel_size_x,$rel_size_y) = ($1,$2);
		$options{xy} =~ /(\d+),(\d+)/;
		my ($rel_click_x,$rel_click_y) = ($1,$2);
		my ($rel_p, $rel_t) = ( 200*($rel_click_x-$rel_size_x/2)/$rel_size_x, 200*($rel_click_y-$rel_size_y/2)/$rel_size_y );
		$log->debug("REL [$rel_size_x] [$rel_size_y] [$rel_click_x] [$rel_click_y] [$rel_p] [$rel_t]");
		my ($vv,$ww) = (sprintf("%02x",abs($rel_p/7)), sprintf("%02x",abs($rel_t/7)));
		my $cccc = '';
		for ($rel_p,$rel_t) {
			$cccc .= '02' if $_>0;       # right or up
			$cccc .= '01' if $_<0;       # left or down
			$cccc .= '03' if $_==0;      # no movement in this direction
		}
		cmdTransmit($dev,"8${vID}010601${vv}${ww}${cccc}FF");
		select(undef,undef,undef,.3);
		cmdTransmit($dev,"8${vID}01060100000303FF");
		return;
	} elsif($mode=~/step/i){       # mode=step  /Step by step positioning/
		if($cmd=~/move/i) { # step pan/tilt
			$param=1 if not $param;
			my $pan = $param=~/[left|right]/i ? $spd : "00";
			my $tlt = $param=~/[up|down]/i ? $spd : "00";

			$log->debug("PARAM=$param SPEED=$spd PAN=$pan TILT=$tlt");
			if   ($param=~/^right$/i)	{ cmdTransmit($dev,"8${vID}010601${pan}000203FF"); }
			elsif($param=~/^left$/i)	{ cmdTransmit($dev,"8${vID}010601${pan}000103FF"); }
			elsif($param=~/^down$/i)	{ cmdTransmit($dev,"8${vID}01060100${tlt}0302FF"); }
			elsif($param=~/^up$/i)	{ cmdTransmit($dev,"8${vID}01060100${tlt}0301FF"); }
			elsif($param=~/^downright$/i){ cmdTransmit($dev,"8${vID}010601${pan}${tlt}0202FF"); }
			elsif($param=~/^downleft$/i) { cmdTransmit($dev,"8${vID}010601${pan}${tlt}0102FF"); }
			elsif($param=~/^upright$/i)	{ cmdTransmit($dev,"8${vID}010601${pan}${tlt}0201FF"); }
			elsif($param=~/^upleft$/i)	{ cmdTransmit($dev,"8${vID}010601${pan}${tlt}0101FF"); }
			select(undef,undef,undef,.05);
			cmdTransmit($dev,"8${vID}01060100000303FF");
		}
		elsif ($cmd=~/zoom/i) { # zoom
			$spd = sprintf("%01d",$spd/100*7); $spd = 7 if $spd > 7;
			my $cmdcode= ($param=~/out/i)? "3$spd" : "2$spd";
			$log->debug("ZOOM!!! PARAM=$param SPEED=$spd ZOOM");
			cmdTransmit($dev,"8${vID}010407${cmdcode}FF");
			select(undef,undef,undef,.1);
			cmdTransmit($dev,"8${vID}01040700FF")
		}
		elsif ($cmd=~/focus/i) { # focus
			$spd = sprintf("%01d",$spd/100*7); $spd = 7 if $spd > 7;
			my $cmdcode= ($param=~/near/i)? "3$spd" : "2$spd";  
			cmdTransmit($dev,"8${vID}010408${cmdcode}FF");
			select(undef,undef,undef,.1);
			cmdTransmit($dev,"8${vID}01040800FF")
		}

	} elsif($mode=~/hardware/i){   # mode=hardware  /Hardware reset,defaults/
		if($cmd=~/do/i && $param=~/defaults/i) { # init
			cmdTransmit($dev,"8${vID}01062403FF"); # Auto Pan-Tilt Speed -> OFF
			cmdTransmit($dev,"8${vID}01040603FF"); # Digital-Zoom -> ON
			cmdTransmit($dev,"8${vID}01040B00FF"); # Iris reset
			cmdTransmit($dev,"8${vID}01043802FF"); # Focus reset
		}

	} elsif($mode=~/abs/i){                # mode=abs  /Absloute positioning/
		if($cmd=~/PT/i){            # Pan, Tilt
			$log->debug("abs $param");
			$param=~/(\-?\d+)\s*,\s*(\-?\d+)\s*$/;
			my $p = -1 * int(int("$1")*14.4);
			my $t=int((int("$2")-32.5)*14.4);
			$log->debug("ABS! pt=$1,$2 ($p,$t)");
			if($p<-880){
				$p=-880;
			}elsif($p>800){
				$p=880;
			}
			if($t<-300) {
				$t=-300;
			}elsif($t>300){
				$t=300;
			}
			if($p>=0) {
				$p = substr(sprintf("%04X",$p),0,4)
			}
			else {
				$p = substr(sprintf("%04X",$p),4,4)
			}
			if($t>=0) {
				$t = substr(sprintf("%04X",$t),0,4)
			}
			else {
				$t = substr(sprintf("%04X",$t),4,4)
			}
			$log->debug("pt=$1,$2 ($p,$t)");
			$p='0'.substr($p,0,1).'0'.substr($p,1,1).'0'.substr($p,2,1).'0'.substr($p,3,1);
			$t='0'.substr($t,0,1).'0'.substr($t,1,1).'0'.substr($t,2,1).'0'.substr($t,3,1);
			cmdTransmit($dev,"8${vID}0106021010${p}${t}FF");
		}
		elsif($cmd=~/center/i) { # center
			cmdTransmit($dev,"8${vID}01062403FF"); # Auto Pan-Tilt Speed -> OFF
			cmdTransmit($dev,"8${vID}01040603FF"); # Digital-Zoom -> ON
			cmdTransmit($dev,"8${vID}01040B00FF"); # Iris reset
			cmdTransmit($dev,"8${vID}01043802FF"); # Focus reset
			cmdTransmit($dev,"8${vID}010604FF");   # Home
		}

	} elsif($mode=~/smooth/i){     # mode=smooth  /Old style PTZ. Goes to direction
									# untill stop (or any other command is sent)/
		if($cmd=~/move/i) { # pan/tilt
			my $pan = ( $param=~/[left|right]/) ? $spd : "00";
			my $tlt = ( $param=~/[down|up]/) ? $spd : "00";
			$log->debug("PARAM=$param SPEED=$spd PAN=$pan TILT=$tlt"); 
			if   ($param=~/^right$/i)	{ cmdTransmit($dev,"8${vID}010601${pan}000203FF"); }
			elsif($param=~/^left$/i)	{ cmdTransmit($dev,"8${vID}010601${pan}000103FF"); }
			elsif($param=~/^down$/i)	{ cmdTransmit($dev,"8${vID}01060100${tlt}0302FF"); }
			elsif($param=~/^up$/i)	{ cmdTransmit($dev,"8${vID}01060100${tlt}0301FF"); }
			elsif($param=~/^downright$/i){ cmdTransmit($dev,"8${vID}010601${pan}${tlt}0202FF"); }
			elsif($param=~/^downleft$/i) { cmdTransmit($dev,"8${vID}010601${pan}${tlt}0102FF"); }
			elsif($param=~/^upright$/i){ cmdTransmit($dev,"8${vID}010601${pan}${tlt}0201FF"); }
			elsif($param=~/^upleft$/i) { cmdTransmit($dev,"8${vID}010601${pan}${tlt}0101FF"); }
		}
		elsif($cmd=~/zoom/i) { # zoom
			$spd = sprintf("%01d",$spd/100*7);
			$spd = 7 if $spd > 7;
			$param = (defined $param) ? lc($param) : "stop";
			my %code = (
				"in"            => "8${vID}0104072${spd}FF",
				"out"           => "8${vID}0104073${spd}FF",
				"stop"          => "8${vID}01040700FF"
			);
			cmdTransmit($dev,"$code{$param}")
		}
		elsif($cmd=~/iris/i) { # iris
			$param = (defined $param) ? lc($param) : "reset";
			my %code = (
				"reset"      => "8${vID}01040B00FF",
				"open"       => "8${vID}01040B02FF",
				"close"      => "8${vID}01040B03FF"
			);
			cmdTransmit($dev,"$code{$param}")
		}
		elsif($cmd=~/focus/i) { # focus
			$spd = sprintf("%01d",$spd/100*7); $spd = 7 if $spd > 7;
			my $cmdcode= ($param=~/near/)? "3$spd" : "2$spd";
			cmdTransmit($dev,"8${vID}010408${cmdcode}FF")
		}
	} elsif($mode=~/preset/i){ # presets
		my $pos=sprintf("%02d",$param);
		my $code;
		$code = '00' if $cmd=~/clear/;
		$code = '01' if $cmd=~/save/;
		$code = '02' if $cmd=~/goto/;
		cmdTransmit($dev,"8${vID}01043F${code}${pos}FF");
	}
	$last_mode= $mode;
	$last_cmd = $cmd;
}

# -------------------------------------------------------------- nonblock -----
#   nonblock($socket) puts socket into nonblocking mode
sub nonblock {
	my $socket = shift;
	my $flags;
	
	$flags = fcntl($socket, F_GETFL, 0)
	         or die "Can't get flags for socket: $!\n";
	fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
	         or die "Can't make socket nonblocking: $!\n";
}

