#!/usr/bin/perl
#  @version $Id: ptz_siqura.pl 25511 2012-04-04 18:59:35Z jscott $
# -----------------------------------------------------------------------------
#  PTZ driver for Siqura cameras, based on Axis HTTP
# -----------------------------------------------------------------------------
#  Author: Jeff Scott
#  Edited by: 
#  QA by:  
#  Copyright: (c) videoNEXT LLC, 2004-2005
# -----------------------------------------------------------------------------

use strict;
use Socket;
use MIME::Base64();
use IO::File;
use IO::Select;
use IO::Socket;
use Fcntl;
use LWP::UserAgent;
use HTTP::Request::Common;

use NextCAM::Init;
use NextCAM::PTZ;
use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

my $log=get_logger('NEXTCAM::PTZ::PTZ_SIQURA');

my $query_str = shift || "'POSITIONCTL'=>'Siqura','CAMERAMODEL'=>'Siqura'";
$log->info("Starting Siqura PTZ DRIVER, query string: [$query_str]");

my $ua = LWP::UserAgent->new();


$SIG{HUP}=\&load_dev_conf;
my $APL=$ENV{APL};
my $APL_CONF=$ENV{APL_CONF};
my (%conf,$cam,$cmd,$usr,$psw,$par,$last_cmd,$last_mode);
load_dev_conf();

# -----------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# -----------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
                                   PeerPort => $TCP_PORT,
                                   Proto    => "tcp",
                                   Type     => SOCK_STREAM)
    or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';

my %commands;
my $ready;
my $trycount = 30;

while(1) {
	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/i;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/
				|| $commands{$cam} =~ /speed iris=0/
				|| $commands{$cam} =~ /speed gain=0/
				)
			{
				; # do not override stop command!
			}else{
				$commands{$cam} = $cmd;
			}
		} else {
			$commands{$cam} = $cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)
	
	$cmd = '';
	
	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		delete $commands{$camera};
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
		$usr = "$conf{$cam}{USRNAME}" if $conf{$cam}{USRNAME};
		$psw = "$conf{$cam}{PASSWD}" if $conf{$cam}{PASSWD};

		httpAuth($conf{$cam}{USRNAME}, $conf{$cam}{PASSWD}) if $conf{$cam}{USRNAME} && $conf{$cam}{PASSWD};
		
		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam,$mode,"pt", $params[0].','.$params[1] ,\%options);
			camCmd($cam,$mode,"zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam,$mode,$cmd,$par,\%options);
		}
	}

	select(undef,undef,undef,.1) if not $cmd;

	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
	}
	
	if(!$cmd) {
		select(undef,undef,undef,.4);
		if(!$trycount--) {
			$trycount = 30;
			print $socket "test\0" or $log->logdie("Couldn't communicate to socket $TCP_PORT: $@");
		}
	}

} # while(1)


# ---------------------------------------------------------------- camCmd -----
sub camCmd
{
	my ($dev,$mode,$cmd,$param,$options)=@_;
	my %options = %$options;
	$log->debug("camCmd: DEVID=[$dev] mode=[$mode] command:[$cmd] param=[$param] ");
	my $URLp = "Data-1.Ptz-1";
	my $URL="http://$conf{$dev}{DEVIP}/params/set.cgi?$URLp.ActualPanPosition";
	my $spd=($options{speed}=~/^\d+/) ? $options{speed}:$conf{$dev}->{PTZSPEED};
	my $zoomspd = $conf{$dev}->{PTZZOOMSPEED};

	# we need to "terminate" each command because it can move forever if used with no STOP !
	if ($last_mode=~/smooth/i) {
		checkURL("$URL&$URLp.Pan=0") 	if $last_cmd=~/move/i;
		checkURL("$URL&$URLp.Tilt=0") 	if $last_cmd=~/move/i;
		checkURL("$URL&$URLp.Zoom=0")	if $last_cmd=~/zoom/i; 
		checkURL("$URL&$URLp.Iris=0")	if $last_cmd=~/iris/i; 
		checkURL("$URL&$URLp.Focus=0")	if $last_cmd=~/focus/i; 
	}

	# here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
	if($conf{$dev}->{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
		if($mode=~/speed/i || $mode=~/step/i || $mode=~/abs/i || $mode=~/rel/i || $mode=~/smooth/i) {
			$log->debug("Setting timeout +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
		elsif($mode=~/preset/i && $cmd=~/goto/i && $param!=1) {
			$log->debug("Setting timeout (preset) +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
	}

	if ($mode=~/speed/i){
		$spd=$conf{$dev}->{PTZSPEED};
		$spd=1 if $spd < 1;
		$spd=100 if $spd > 100;
		if($cmd=~/^pt$/i) { #  pan-tilt
			
			my ($p,$t) = split(/,/,$param);
			$p = int($p*$spd/100);
			$t = -int($t*$spd/100);
			checkURL("$URL&$URLp.Pan=$p&$URLp.Tilt=$t");
		}
		elsif($cmd=~/^zoom$/i) { # zoom
			my $zm=int($param * $zoomspd/100);
			checkURL("$URL&URLp.Zoom=$zm")
		}
		elsif($cmd=~/focus/i) { # focus
			my $fc=int($param*$spd/100);
			checkURL("$URL&URLp.Focus=$fc")
		}
		
		elsif($cmd=~/gain/i) { # iris
			my $fc=int($param*$spd/50);
			checkURL("$URL&Iris=$fc")
		}
		
	} elsif($mode=~/abs/i){    # mode=ABS
	
		if ($cmd=~/pt/i) {
			$param=~/(\-?\d+)\s*,\s*(\-?\d+)\s*$/;
			my $p = $1;
			my $t = $2;
			checkURL("$URL&$URLp.SetPanPosition=$p&$URLp.SetTiltPosition=$t");
		}
		elsif($cmd=~/Z/i) {
			$param=~/(\d+)/;
			my $abs_z = ($1 + 1)*99.98;
			checkURL("$URL&URLp.SetZoomPosition=$abs_z");
		}
		elsif($cmd=~/center/i) { # center
			checkURL("$URL&$URLp.SetPanPosition=0&$URLp.SetTiltPosition=0&$URLp.SetZoomPosition=1");
		}
		
	} elsif($mode=~/rel/i){ # relative positioning (recentering)
		
		#$param =~ /(\d+)x(\d+)/;
		#my ($rel_size_x,$rel_size_y) = ($1,$2);
		#$options{xy} =~ /(\d+),(\d+)/;
		#my ($rel_click_x,$rel_click_y) = ($1,$2);
		#checkURL("$URL&imagewidth=$rel_size_x&imageheight=$rel_size_y&center=$rel_click_x,$rel_click_y");
	} elsif($mode=~/step/i) {       # mode=step  /Step by step positioning/
		if($cmd=~/move/i) { # step pan/tilt
			my ($p,$t) = (0,0);
			$p = 50 if $param=~/right/;
			$p = -50 if $param=~/left/;
			$t = 50 if $param=~/up/;
			$t = -50 if $param=~/down/;
			checkURL("$URL&$URLp.Pan=$p&$URLp.Tilt=$t");
			select(undef,undef,undef,.5);
			checkURL("$URL&URLp.Pan=0&$URLp.Tilt=0");
		}
		elsif($cmd=~/zoom/i) {
			$spd=100;
			my $zm = $param=~/out/i? -$spd : $spd;
			$zm=int($zm);
			checkURL("$URL&$URLp.Zoom=$zm");
			select(undef,undef,undef,.5);
			checkURL("$URL&$URLp.Zoom=0");
		}
		elsif($cmd=~/focus/i) {
			$spd=300;
			my $focus = $param=~/far/i? -$spd : $spd;
			checkURL("$URL&URLp.Focus=$focus");
		}
		elsif($cmd=~/iris/i) { 
			$spd=300;
			my $iris = $param=~/open/i? $spd : -$spd;
			checkURL("$URL&$URLp.Iris=$iris");
		}
	} elsif($mode=~/smooth/i){
		# mode=smooth  /Old style PTZ. Goes to direction
		# until stop (or any other command is sent)/
		if($cmd=~/move/i) { # pan/tilt
			$spd=3 if not $spd;
			my $pan = $param=~/left/i? -$spd : $param=~/right/i? $spd : 0;
			my $tlt = $param=~/down/i? -$spd : $param=~/up/i? $spd : 0;
			checkURL("$URL&$URLp.Pan=$pan&$URLp.Tilt=$tlt");
		}
		elsif($cmd=~/zoom/i) { # zoom
			$zoomspd=1 if not $zoomspd;
			my $zm = $param=~/out/? -$zoomspd : $zoomspd;
			checkURL("$URL&$URLp.Zoom=$zm")
		}
		elsif($cmd=~/iris/i) { # iris
			$spd=1 if not $spd;
			my $iris = $param=~/close/i? -$spd : $spd;
			checkURL("$URL&$URLp.Iris=$iris")
		}
		elsif($cmd=~/focus/i) { # focus
			$spd=1 if not $spd;
			my $focus = $param=~/far/i? -$spd : $spd;
			checkURL("URL&$URLp.Focus=$focus")
		}
	} elsif($mode=~/preset/i){ # presets
		my $code;
		$code = "GoPreset"	if $cmd=~/goto/i;
		$code = "SetPreset"	if $cmd=~/save/i;
		checkURL("$URL&$URLp.$code=$param");
	} elsif($mode=~/settings/i){ # settings
		my $code = '';
		if ($cmd=~/timeout/i) {
			$log->debug("Setting timer: [$param]");
			$conf{$dev}->{TIMEOUT} = time + $param;
		}
		elsif ($cmd=~/autofocus/i) {
			$code = "$URL&$URLp.AutoFocus=";
			$code .= ($param=~/On/i) ? "On" : "Off";
		} elsif ($cmd=~/autoiris/i) {
			$code = "$URL&$URLp.AutoIris=";
			$code .= ($param=~/On/i) ? "On" : "Off";
		}
		checkURL($code) if $code;
	}
	$last_mode= $mode;
	$last_cmd = $cmd;
}

# ------------------------------------------------------------- UpdatePID -----
sub UpdatePID {
	open(PID,"> $APL/var/ptz/ptz_siqura.pid");
	print PID $$;
	close PID;
} # sub UpdatePID

# --------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {
	%conf = GetCfgs( eval("($query_str)") );     # Load configurations
	my $ids='';
	foreach my $dev (keys  %conf) {
		next if not $conf{$dev}->{PTZID} =~/[12]/;
		next if not $conf{$dev}->{DEVIP};
		$ids.=" $dev";
		#camParams($dev);
	}
	$log->info("Loaded configurations for cameras:$ids");
}


# -------------------------------------------------------------- checkURL -----
 sub checkURL { ##
	my $rsp;
	my $val=-2;
	my($url)=@_;  #print "URL: $url\n";
	$log->debug("checkURL($url)");
	eval {
		local $SIG{ALRM}=sub{die 'ALARM'};
		alarm 2;

		my $ua = LWP::UserAgent->new;
		$url =~ tr/ //d;
		my $req = GET $url;
		$req->authorization_basic($usr, $psw);
		$rsp = $ua->request($req);
		
		$val = -1;
		alarm 0;
	} or $val=-2; # eval
	alarm 0;
	return ($rsp->is_success) ? $rsp->content."\n" : "HTTP ERROR [".$rsp->code."]";
 } # sub checkURL

# ----------------------------------------------------------- truncDigits -----
sub truncDigits{
	my($val)=@_;
	return sprintf("%5.5f",$val)
} # sub truncDigits

# -------------------------------------------------------------- nonblock -----
#   nonblock($socket) puts socket into nonblocking mode
sub nonblock {
        my $socket = shift;
        my $flags;
    
        $flags = fcntl($socket, F_GETFL, 0)
                or die "Can't get flags for socket: $!\n";
        fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
                or die "Can't make socket nonblocking: $!\n";
}

