#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#  PTZ driver for Pelco D protocol - Modified for PVPAEO
# ------------------------------------------------------------------------------
#  Author: Jeffrey Scott
#  Edited by: Andriy Fomenko
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT Network Solutions, Inc, 2010
# ------------------------------------------------------------------------------
#  PVPAEO PTZ driver
#  
#
#

use strict;
use POSIX;
use Socket;
use IO::File;
use IO::Select;
use IO::Socket;
use Fcntl;
use JSON;
use XML::Simple;
use Net::STOMP::Client;
use Data::Dumper;

use NextCAM::Init;
use Log::Log4perl "get_logger";

require "$ENV{APL}/common/bin/logger.engine";

# Constants
my $CHECK_DANTE_TIMEOUT = 5;

my $log=get_logger('NEXTCAM::PTZ');

my $query_str = shift || "'POSITIONCTL'=>'pvpaeo'";
$log->info("Starting Pelco PTZ DRIVER, query string: [$query_str]");

$SIG{HUP}=\&load_dev_conf;

my $APL=$ENV{APL};
my (%conf, %sock, $cam, $cmd, $par, $last_cmd, $last_mode, $lastchecktime);

load_dev_conf();

my %dev_pos;

# ------------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# ------------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
PeerPort => $TCP_PORT,
Proto     => "tcp",
Type      => SOCK_STREAM)
or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

my $stomp = Net::STOMP::Client->new( uri=>"stomp://localhost:61613" );
$stomp->message_callback(sub{return(1) } );
eval{
	$stomp->connect();

	my $sid = $stomp->uuid();
	$stomp->subscribe(
		destination	=> "/topic/amq.gis.topic",
		id		=> $sid,
	);
	sub msg_cb($$) {
		my ($self, $frame) = @_;
		return(0) unless $frame->command() eq "MESSAGE";
		return($frame->body());
	}
};

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';
$lastchecktime = time;

my %commands;
my $trycount = 30;
my $c = 0;
my $lastpostime = time;

# ------------------------------------------------------------------------------
my $device = 0; 
# PVPAEO Nighthawk consists of the PT controller 0x00, IR 0x01, Color 0x02
# ------------------------------------------------------------------------------

while(1) {
	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/
				|| $commands{$cam} =~ /speed iris=0/
				|| $commands{$cam} =~ /speed gain=0/
				)
			{
				; # do not override stop command!
			}else{
				$commands{$cam} = $cmd;
			}
		} else {
			$commands{$cam} = $cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)
	
	$cmd = '';

	
	# STOMP stuff
	eval{
		my $frame = $stomp->wait_for_frames(callback => \&msg_cb, timeout=>0);
		#$log->debug("STOMP msg: ", $frame);
		writePos($cam,1,$1) if ($frame=~/rf_g.*usid="(.*)"/); 
		my $ref = XMLin($frame);
		goPos($cam, $ref) if ($ref->{type} eq 'amqGisPtzSet');
	};
	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		delete $commands{$camera};
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
		
		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam,$mode,"pt", $params[0].','.$params[1] ,\%options);
			camCmd($cam,$mode,"zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam,$mode,$cmd,$par,\%options);
		}
	}
	
	#select(undef,undef,undef,.1) if not $cmd;
	
	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
	}
	
	if(!$cmd) {
		select(undef,undef,undef,.4);
		if(!$trycount--) {
			$trycount = 30;
			print $socket "test\0" or $log->logdie("Couldn't communicate to socket $TCP_PORT: $@");
		}
	}
	
} # while(1)


# ------------------------------------------------------------ cmdTransmit -----
sub cmdTransmit {
    my ($dev, $delay, @cmd)=@_;
	
	$cmd[6]=checkSum(@cmd);
	
    my $command='';
    my $b;
	my $ret;
    foreach $b (@cmd) { 
		$command .= sprintf("\r\n") if $b==0xff;
		$command .= sprintf("%02X ",$b); 
	}
    $log->debug('cmdTransmit( ',$dev, '=> ',$command,')');    
	
    my $bts = '';
    foreach my $b (@cmd) { $bts .= chr($b); }
	
    my $retries = 1;
secondaryPass:
    my $skt = $sock{ $conf{$dev}{DEVIP} }{SOCK};
    eval {
        local $SIG{PIPE} = sub { die "Error writing to socket" };
        print $skt $bts;
    };
    if($@) {
        $log->error($@);
        # camera did not answer or socket error occured - try to close socket and reopen it
        eval{ $sock{$conf{$dev}{DEVIP}}{SOCK}->close(); };
        $log->error("Re-open socket connection: $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT}");
        $sock{$conf{$dev}{DEVIP}}{SEL} = undef;
        #select(undef,undef,undef,.1);
        $sock{$conf{$dev}{DEVIP}}{SOCK} = IO::Socket::INET->new(PeerAddr => $conf{$dev}{DEVIP},
		PeerPort => $conf{$dev}{PTZ_TCP_PORT},
		Proto    => "tcp",
		Type     => SOCK_STREAM);
        if(! $sock{$conf{$dev}{DEVIP}}{SOCK} ) {
            $log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
            delete $sock{$conf{$dev}{DEVIP}};
            return;
        }
        $sock{$conf{$dev}{DEVIP}}{SOCK}->autoflush(1);
        nonblock($sock{$conf{$dev}{DEVIP}}{SOCK});
        $sock{$conf{$dev}{DEVIP}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}}{SOCK});
        $log->info('Socket was closed, reconnected');
        goto secondaryPass if $retries--;
		
        return;
    }
	
    select(undef,undef,undef,$delay);
    foreach my $sss ( $sock{ $conf{$dev}{DEVIP} }{SEL}->can_read(.1) ) {
		my $data = '';
        eval {
            alarm 1;
            $sss->recv($data, POSIX::BUFSIZ, 0);
            $ret='';
            for($b=0;$b<length($data);$b++) { 
				$ret .= sprintf("%02X ",ord(substr($data,$b,1))); 
			}
            $log->debug('Received:',$ret);
        };
        alarm 0;
	parsePos($dev,$ret);
        return $ret;
    }  # foreach
    
	
} # sub cmdTransmit

# --------------------------------------------------------------- CheckSum -----
sub checkSum
{
	my @arr = @_;
	return ($arr[1]+$arr[2]+$arr[3]+$arr[4]+$arr[5]) % 256;
} # su
#checkSum

# ---------------------------------------------------------------- camStop--
sub camStop
{
	my ($dev) = @_;
	my @cmd = ( 0xFF, 0, 0, 0, 0, 0, 0 );
	return @cmd;
}

# --------------------------------------------------------------- camReset -----
sub camReset
{
	my ($dev) = @_;
	#my @cmd = ( 0xFF, 0, 0, 0, 0, 0, 0 );
	my @cmd = (0xFF, 0x00, 0x32, 0x77, 0x01, 0x11);
	#Load defaults
	#$cmd[3]=0x29; $cmd[6]=checkSum(@cmd);
	#cmdTransmit($dev,0,@cmd);
	## Auto-focus: byte 5: 0-auto/1-on/2-off
	#$cmd[2]=0x24; $cmd[4]=0x0D; $cmd[3]=0x77;
	#cmdTransmit($dev,0,@cmd);
	## Auto-iris: byte 5: 0-auto/1-on/2-off
	## Auto white balance: byte 5: 1-on/2-off
	#$cmd[3]=0x33; $cmd[5]=2; $cmd[6]=checkSum(@cmd);
	cmdTransmit($dev,0,@cmd);
	# Set focus speed: byte 5: 0..3
	#$cmd[3]=0x27; $cmd[5]=1; $cmd[6]=checkSum(@cmd);
	#cmdTransmit($dev,0,@cmd);
} # sub camReset

# ------------------------------------------------------------- camFZspeed -----
sub camFZspeed
{
	my ($dev,$FZ,$speed) = @_;
	$log->debug("camFZspeed($dev,$FZ,$speed)");
	my @cmd = ( 0xFF, 0, 0, 0, 0, 0, 0 );
	$cmd[1]=0;
	$cmd[3]=$FZ=='F'?0x27:0x25;
	$cmd[5]=sprintf("%d",abs($speed)/25);
	cmdTransmit($dev,0,@cmd);
} # sub camFZspeed

# ----------------------------------------------------------------- PTspeed ----
sub PTspeed {
	my ($speed)=@_;
	#return 255 if(abs($speed)>99); # turbo speed - does not work with spectra !!!
	$speed=100 if abs($speed)>100;
	my $val = int(abs($speed/100*99));
	return $val;
} # sub PTspeed

# ----------------------------------------------------------------- FZspeed ----
sub FZspeed {
	my ($speed)=@_;
	#return 255 if(abs($speed)>99); # turbo speed - does not work with spectra !!!
	$speed=100 if abs($speed)>100;
	return int(abs($speed)/100*127);
} # sub FZspeed

# ----------------------------------------------------------------- checkReg ----
sub checkReg {
	my ($data, $mask, @cmd) = @_;
	
	my $reg;
	foreach my $a (@cmd) { $reg .= sprintf("%02X ", $a); }
	$data=~/$reg$mask/i;
	#$log->debug("Matched:: $1 :: with '$reg($mask)'");
	return $1;
}

# ----------------------------------------------------------------- getRange ----
sub getRange {
	my ($dev) = @_;
	my $i;
	
	# Run Range Finder Status until 'Ready'
	my @cmd = ( 0xff, 0, 0x40, 0x77, 0x02);
	$log->debug("Checking if LRF is ready...");
	for ($i = 0; ! (checkReg(cmdTransmit($dev,0,(@cmd, 0x00)), '(\w{2})', @cmd) == "00") && $i < 30; $i++){ select(undef,undef, undef, 0.15);}
	
	# Run Range Measure Cycle until complete
	$log->debug("Running Range...");
	@cmd = ( 0xff, 0, 0x0f, 0x77, 0x26, 0x00);
	checkReg(cmdTransmit($dev,0,@cmd), '.*$', @cmd);

	# Run Range Finder Status until 'Ready'
	@cmd = ( 0xff, 0, 0x40, 0x77, 0x02);
	$log->debug("Checking if LRF is ready...");
	for ($i=0; !(checkReg(cmdTransmit($dev,0,(@cmd, 0x00)), '(\w{2})', @cmd) == "00") && $i < 30; $i++){ select(undef,undef, undef, 0.15);}
	
	# Get Range value
	$log->debug("Getting Range...");
	@cmd = ( 0xff, 0, 0x4b, 0x77);
	my $range = checkReg(cmdTransmit($dev,0,(@cmd, 0, 0)), '(\w{2} \w{2})', @cmd);
	$range=~s/\s*//g;
	return hex($range);
}

sub parsePos {
	my ($dev,$data) = @_;
	my $deg = 0.0054932;
	
	$data=~s/\s*//g;
	
	if ($data=~/ff000f7716(\w{2})/i){
		$dev_pos{active} = $1;
		$log->debug("Active camera: $1");
	} 
	
	if ($data=~/ff000977(\w{4})/i){
		$dev_pos{deg_az} = hex($1);
		$dev_pos{deg_az} = $dev_pos{deg_az}*$deg;
		$log->debug("Az: $dev_pos{deg_az}($1)");
		$dev_pos{deg_az} -= $conf{$dev}{CAM_GEO_NORTH_OFFSET};
		$log->debug("Az: $dev_pos{deg_az}($1)");
	}
	
	if ($data=~/ff000a77(\w{4})/i){
		($dev_pos{deg_el}) = hex($1);
		($dev_pos{deg_el}) = $dev_pos{deg_el}*$deg;
		$dev_pos{deg_el} = $dev_pos{deg_el} - 360 if ($dev_pos{deg_el} > 260); # 0 to -95 = $deg-360
		$log->debug("El: $dev_pos{deg_el}($1)");
	}
	
	my $fov_vis;
	if ($data=~/ff0050772(\w{3})/i){
		$fov_vis = hex($1)/10;
		$log->debug("Fov vis: $fov_vis($1)");
	}
	my $fov_ir; 
	if ($data=~/ff0051772{(\w{3})/i){	
		$fov_ir = hex($1)/10;
		$log->debug("Fov IR: $fov_vis($1)");
	}
		
	$dev_pos{fov} = ($dev_pos{active}=~/01/) ? $fov_ir : $fov_vis;
	
	if( $data=~/ff00327702(\w{2})/i){
		$log->debug("2x: $1");
		$dev_pos{s2x} = $1 ;
	}
	

	if( $data=~/ff000c77(\w{4})/i){
		$log->debug("focus: $1");
		$dev_pos{focus} = $1; 
	}
	if ($data=~/ff000b77(\w{4})/i){
		$dev_pos{zoom} = $1; 
		$log->debug("zoom: $1");
	}
}

# ----------------------------------------------------------------- getPos ----
sub getPos {
	my ($dev) = @_;
	my $ret;
	my $deg = 0.0054932;

	
	# FOV - Visible
	my @cmd = ( 0xff, 0, 0x50, 0x77, 0x20, 0);
	my @cmd2 = (@cmd, checkSum(@cmd));

	# FOV - IR
	@cmd = ( 0xff, 0, 0x51, 0x77, 0x20, 0);
	@cmd2 = (@cmd2, @cmd, checkSum(@cmd));
	
	# 2x state
	$cmd[2]= 0x32;
	$cmd[4]= 0x02;
	$cmd[5]= 0x02;
	@cmd2 = (@cmd2, @cmd, checkSum(@cmd));

	#az
	$cmd[2]= 0x09;
	$cmd[4]= 0x00;
	$cmd[5]= 0x00;
	@cmd2 = (@cmd2, @cmd, checkSum(@cmd));

	#el
	$cmd[2]= 0x0A;
	@cmd2 = (@cmd2, @cmd, checkSum(@cmd));
	
	#active
	@cmd=(0xff, 0, 0x0f, 0x77, 0x16, 0);
	@cmd2 = (@cmd2, @cmd, checkSum(@cmd));

	@cmd=(0xff, 0, 0x0c, 0x77, 0, 0);
	@cmd2 = (@cmd2, @cmd, checkSum(@cmd));

	@cmd=(0xff, 0, 0x0b, 0x77, 0, 0);
	@cmd2 = (@cmd2, @cmd, checkSum(@cmd));
	
	$ret = cmdTransmit($dev,1,@cmd2);


	#$dev_pos{log->debug("Got back - \n FOV: $dev_pos{fov}\n 2x:$dev_pos{s2x}\n AZ:$dev_pos{deg_az}\n EL:$dev_pos{deg_el}\n CAM:$dev_pos{active}\n Zoom:$dev_pos{zoom}\n Focus:$dev_pos{focus}");
	#return ( FOV => $dev_pos{fov}, DEG_AZ => $dev_pos{deg_az}, DEG_EL => $dev_pos{deg_el}, S2X => $dev_pos{s2x}, DEV => $dev_pos{active}, ZOOM=>$dev_pos{zoom}, FOCUS=>$dev_pos{focus});
}

# ----------------------------------------------------------------- goPos ------

sub goPos{
	my ($dev, $ref) = @_;
	my ($fov, $az, $el, $s2x, $camt) = ($ref->{fov}, $ref->{az}, $ref->{el}, $ref->{s2x}, $ref->{camt});
	my $deg = 0.0054932;

	$log->debug("Going to: az:$az el:$el fov:$fov");
	
	# FOV 
	$fov = int($fov*10);
	my @cmd = ( 0xff, 0, 0x07, 0x77, 0, 0);
	$cmd[5] = ($fov & 0x0f);
	$cmd[4] = ($fov & 0xf0) >> 4;
	$cmd[4] |= 0x40 if $camt==0;
	$cmd[4] |= 0x80 if $camt==1;
	my @cmd2 = (@cmd, checkSum(@cmd));

	$az = int($az/$deg);
	@cmd = ( 0xff, 0, 0x05, 0x77, 0, 0);
	$cmd[5] = ($az & 0x0f);
	$cmd[4] = ($az & 0xf0) >> 4;
	@cmd2 = (@cmd2, @cmd, checkSum(@cmd));

	$el = int($az/$deg);
	@cmd = ( 0xff, 0, 0x06, 0x77, 0, 0);
	$cmd[5] = ($az & 0x0f);
	$cmd[4] = ($az & 0xf0) >> 4;
	@cmd2 = (@cmd2, @cmd, checkSum(@cmd));
	
	# 2x state
	@cmd = ( 0xff, 0, 0x32, 0x77, 0x02, $s2x);
	@cmd2 = (@cmd2, @cmd, checkSum(@cmd)) if $camt == 0;

	my $ret = cmdTransmit($dev,.5,@cmd2);
}

# ----------------------------------------------------------------- writePos -----
sub writePos {
	my ($dev, $target, $USID) = @_;	
	my $mTime = time();
	my $dTime = sprintf("%s_%s",$dev,$mTime);
	my %posUpdate;
	getPos($dev);

	$USID = 0 unless defined $USID;
	eval{
		#$stomp->send(
		#	destination	=>"/topic/amq.gis.topic",
		#	time		=>time(), 
		#	id		=>"$dTime",
		#	usid		=>"$USID",
		#	az		=>"$posUpdate{DEG_AZ}",
		#	fov 		=>"$posUpdate{FOV}",
		#	el		=>"$posUpdate{DEG_EL}",
		#	s2x		=>"$posUpdate{S2X}",
		#	zoom		=>"$posUpdate{ZOOM}",
		#	focus		=>"$posUpdate{FOCUS}",
		#	camt		=>"$posUpdate{DEV}",
		#	body		=>".",
		#);
		
		my $range = -1;
		# $range = getRange($dev);

		$stomp->send(
			destination	=>"/topic/amq.gis.topic",
                        type            =>"amqGisPtzPos",
			time		=>time(),
			usid		=>"$USID",
			range		=>"$range", 
			id		=>"$dTime",
			az		=>"$dev_pos{deg_az}",
			fov 		=>"$dev_pos{fov}",
			el		=>"$dev_pos{deg_el}",
			s2x		=>"$dev_pos{s2x}",
			zoom		=>"$dev_pos{zoom}",
			focus		=>"$dev_pos{focus}",
			camt		=>"$dev_pos{active}",
			body		=>".",
		);
	};
}



# ----------------------------------------------------------------- camCmd -----
sub camCmd {
	my ($dev,$mode,$cmd,$param,$options)=@_;
	my %options = %$options;
	my @cmd = ( 0xff, 0,0,0,0,0);
	$log->debug("camCmd: DEVID=$dev mode: $mode command: $cmd optional parameter: $param");
	
	# here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
	if($conf{$dev}->{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
		if($mode=~/speed/ || $mode=~/step/ || $mode=~/abs/ || $mode=~/rel/ || $mode=~/smooth/) {
			$log->debug("Setting timeout +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
		elsif($mode=~/preset/ && $cmd=~/goto/ && $param!=1) {
			$log->debug("Setting timeout (preset) +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
	}
	
	if(not $cmd=~/stop/) { # non-Stop
		# set camera speed
		if ($mode=~/speed/){        # mode speed
			if($cmd=~/PT/i) { # RPT
				my ($p,$t) = split(/,/,$param);
				$cmd[4]=PTspeed($p);
				$cmd[5]=PTspeed($t);
				$log->debug(sprintf("RPT $p(%02X) $t(%02X)",$cmd[4],$cmd[5]));
				$cmd[3] |= 0x02 if $p>0;
				$cmd[3] |= 0x04 if $p<0;
				$cmd[3] |= 0x10 if $t>0;
				$cmd[3] |= 0x08 if $t<0;

				#cmdTransmit($dev,0,camStop($dev)) if($p==0 && $t==0);
			}
			elsif($cmd=~/z/) { # Zoom 
				@cmd = ( 0xff, 0, 0x0f, 0x77, 0x16);
				my $active = $dev_pos{active};
				if ( $active=~/01/ ) {
					$log->debug("Active = IR") if $active=~/01/;
					@cmd = ( 0xFF, 0, 0, 0, 0, 0 );
					$cmd[3] = 0x20 if $param>0;
					$cmd[3] = 0x40 if $param<0;
					$cmd[5] = FZspeed($param);
				}else{
					$log->debug("Active = TV") if $active=~/00/;
					@cmd=(0xFF, 0, 0x0E, 0x77, 0, 0);
					$cmd[4] = FZspeed($param);
					$cmd[4] ^= 0xFF if $param < 0;
				}
				$log->debug("ZOOM $param");
			}
			elsif($cmd=~/focus/) {
				@cmd = ( 0xff, 0, 0x0f, 0x77, 0x16);
				my $active = $dev_pos{active};
				if ( $active=~/01/ ) {
					$device = 1;
					$log->debug("Active = IR") if $active=~/01/;
					@cmd = ( 0xFF, 0, 0, 0, 0, 0 );
					$cmd[2] = 0x01 if $param>0;
					$cmd[3] = 0x80 if $param<0;
					$cmd[4] = FZspeed($param);
				}else{
					$device = 0;
					$log->debug("Active = TV") if $active=~/00/;
					@cmd = ( 0xFF, 0, 0x0E, 0x77, 0, 0 );
					$cmd[5] = FZspeed($param);
					$cmd[5] ^= 0xFF if $param<0;
				}
				$log->debug("FOCUS $param");
			}
		}
		elsif($mode=~/rel/){ # mode=rel: "<devid> rel size=640x480 xy=225,152"
			$param =~ /(\d+)x(\d+)/;
			my ($rel_size_x,$rel_size_y) = ($1,$2);
			$options{xy} =~ /(\d+),(\d+)/;
			my ($rel_click_x,$rel_click_y) = ($1,$2);
			eval{
				my ($rel_p, $rel_t) = ( 100*($rel_click_x-$rel_size_x/2)/$rel_size_x, 100*($rel_click_y-$rel_size_y/2)/$rel_size_y ); 
				$log->debug("REL [$rel_size_x] [$rel_size_y] [$rel_click_x] [$rel_click_y] [$rel_p] [$rel_t]");
				$cmd[4]=PTspeed($rel_p);
				$cmd[5]=PTspeed($rel_t);
				$cmd[3] |= 0x02 if $rel_p>0;
				$cmd[3] |= 0x04 if $rel_p<0;
				$cmd[3] |= 0x08 if $rel_t<0;
				$cmd[3] |= 0x10 if $rel_t>0;
			};

			cmdTransmit($dev,0,@cmd);
			select(undef,undef,undef,.2);
			cmdTransmit($dev,0,camStop($dev));
			return;
		}
		elsif($mode=~/abs/){    # mode=ABS
			if ($cmd=~/pt/) {
				$log->debug("ABS:  PT $param");
				@cmd=(0xFF, 0, 0, 0x77, 0, 0);
				
				my $deg = 0.0054932; # each binary-digit is .0054932 degrees, or ~ 1/2 * 1/100
				
				$param=~/^(\-?.+)?\,(\-?.+)$/;
				# camera is accurate for tenth of a degree
				my $p = int($1/$deg);
				my $t = int($2/$deg);
				# negative angles are 360degrees - angle
				$p += 360/$deg if $p<0;
				$t += 360/$deg if $t<0;
				$log->debug("ABS! pt=$1,$2 ($p,$t)");
				# convert to 4-digit hex
				($cmd[4],$cmd[5]) = (int($p / 256), $p % 256);
				$log->debug(sprintf("HEX: %04x",$p));
				$log->debug(sprintf("++++ N1 %02x, N2 %02x", $cmd[4], $cmd[5]));
				$cmd[2]=0x05;	# For Pan Position
				cmdTransmit($dev,0,@cmd);	# Sending Pan first
				($cmd[4],$cmd[5]) = (int($t / 0xff),int($t % 0xff));				
				$cmd[2]=0x06;	# For Tilt Position
				cmdTransmit($dev,0,@cmd);	# Sending Tilt
				
			} 
			elsif ($cmd=~/z/){	# Absolute zoom
				$log->debug("ABS:  Z $param");
				my $z_hex = sprintf("%04X",$param);
				# not implemented 
			}
			elsif($cmd=~/center/) { # center
				my @cmd = (0xFF, 0, 0x32, 0x77, 0x01, 0x01);
				cmdTransmit($dev,0,@cmd);
			}
			
		}
		elsif($mode=~/settings/){ # settings
			if($cmd=~/autofocus/) { 
				@cmd = ( 0xFF, 0, 0x0F, 0x77, 0x15, 0 ); #set IR active
				cmdTransmit($dev,0,@cmd);
				@cmd = ( 0xFF, 0, 0x24, 0x77, 0x0D, 0x01 ) if $param=~/on/;
				@cmd = ( 0xFF, 0, 0x24, 0x77, 0x0D, 0 ) if $param=~/auto/;
			}
			elsif($cmd=~/active/){
				@cmd = (0xff, 0, 0x0f, 0x77, 0x24, 0x00);
				$cmd[4] = 0x15 if $param=~/ir/;
				$cmd[4] = 0x14 if $param=~/tv/;
			}
			elsif($cmd=~/edge/){
				@cmd = (0xff, 0, 0x24, 0x77, 0x10, 0x09) if $param=~/on/;
				@cmd = (0xff, 0, 0x24, 0x77, 0x10, 0) if $param=~/off/;
			}
			elsif($cmd=~/tvstab/){
				@cmd = (0xff, 0, 0x32, 0x77, 0x0b, 0x01) if $param=~/on/;
				@cmd = (0xff, 0, 0x32, 0x77, 0x0b, 0x00) if $param=~/off/;
			}
			elsif($cmd=~/irstab/){
				@cmd = (0xff, 0, 0x24, 0x77, 0x0c, 0x25) if $param=~/on/;
				@cmd = (0xff, 0, 0x24, 0x77, 0x0c, 0x24) if $param=~/off/;
			}
			elsif($cmd=~/turbulence/){
				@cmd = (0xff, 0, 0x24, 0x77, 0x13, 0x00) if ($param=~/off/);
				@cmd = (0xff, 0, 0x24, 0x77, 0x13, 0x11) if ($param=~/low/);
				@cmd = (0xff, 0, 0x24, 0x77, 0x13, 0x12) if ($param=~/mild/);
				@cmd = (0xff, 0, 0x24, 0x77, 0x13, 0x13) if ($param=~/medium/);
				@cmd = (0xff, 0, 0x24, 0x77, 0x13, 0x14) if ($param=~/high/);
				@cmd = (0xff, 0, 0x24, 0x77, 0x13, 0x15) if ($param=~/extreme/);
			}
			elsif($cmd=~/ezoom/){
				if($param=~/off/) {
					@cmd = (0xff, 0, 0x24, 0x77, 0x12, 0x00) if $param=~/off/;
				}else{
					@cmd = (0xff, 0, 0x24, 0x77, 0x12, 0x10);
					cmdTransmit($dev,0,@cmd);
					@cmd = (0xff, 0, 0x24, 0x77, 0x12, 0x11) if $param=~/in/;
					@cmd = (0xff, 0, 0x24, 0x77, 0x12, 0x12) if $param=~/out/;
				}	
			}
			elsif($cmd=~/mh/){
				@cmd = (0xff, 0, 0x32, 0x77, 0x01, 0x01);
			}
			elsif($cmd=~/reset/){
				@cmd = (0xff, 0, 0x32, 0x77, 0x01, 0x11);
			}
			elsif($cmd=~/home/){
				if($param=~/set/){
					@cmd = (0xff, 0, 0x09, 0x77);
					my $az = checkReg(cmdTransmit($dev,0,(@cmd, 0, 0)), '(\w{2} \w{2})', @cmd);
					@cmd = (0xff, 0, 0x02, 0x77, 0, 0);
					$az =~/(\w\w)\s*(\w\w)/;
					($cmd[4], $cmd[5]) = (hex($1),hex($2));
					cmdTransmit($dev,0, @cmd);

					@cmd = (0xff, 0, 0x0a, 0x77);
					my $el = checkReg(cmdTransmit($dev,0,(@cmd, 0, 0)), '(\w{2} \w{2})', @cmd);
					@cmd = (0xff, 0, 0x04, 0x77, 0, 0);
					$el =~/(\w\w)\s*(\w\w)/;
					($cmd[4], $cmd[5]) = (hex($1),hex($2));
					cmdTransmit($dev,0, @cmd);
				}

				@cmd = (0xff, 0, 0, 0, 0, 0);
				@cmd = (0xff, 0, 0x32, 0x77, 0x01, 0x03) if $param=~/goto/;
			}
			elsif($cmd=~/nuc/) {
				@cmd = ( 0xFF, 0, 0x0F, 0x77, 0x15, 0 ); #set IR active
				cmdTransmit($dev,0,@cmd);
				@cmd = ( 0xFF, 0, 0x24, 0x77, 0x0e, 0x02 );
				cmdTransmit($dev,0,@cmd);
				@cmd = (0xff, 0, 0x0f, 0x77, 0x17);
				my $ret;
				for(my $i=0; $i < 100; $i++){
					$ret = 	hex(checkReg(cmdTransmit($dev,0, (@cmd, 0x00)), '(\w{2})', @cmd));
					$i = 100 if ($ret != 1);
				}
				$log->debug("NUC Status: $ret") if $ret != 0;
				@cmd = (0xff, 0, 0, 0, 0, 0);
			}
			elsif($cmd=~/lac/) {
				@cmd = ( 0xFF, 0, 0x0F, 0x77, 0x15, 0 ); #set IR active
				cmdTransmit($dev,0,@cmd);
				@cmd = ( 0xFF, 0, 0x24, 0x77, 0x0F, 0x00 ) if $param=~/off/;
				@cmd = ( 0xFF, 0, 0x24, 0x77, 0x0F, 0x0F ) if $param=~/aggressive/;
				@cmd = ( 0xFF, 0, 0x24, 0x77, 0x0F, 0x09 ) if $param=~/normal/;
			}
			elsif($cmd=~/tvfocus/) {
				@cmd = ( 0xFF, 0, 0x0F, 0x77, 0x14, 0 ); #set TV active
				cmdTransmit($dev,0,@cmd);
				@cmd = ( 0xFF, 0, 0x32, 0x77, 0x04, 0 ) if $param=~/on/;
			}
			elsif($cmd=~/pairfunc/) {
				@cmd = ( 0xFF, 0, 0x32, 0x77, 0x07, 0x00 ) if $param=~/bypass/;
				@cmd = ( 0xFF, 0, 0x32, 0x77, 0x07, 0x10 ) if $param=~/modified/;
				@cmd = ( 0xFF, 0, 0x32, 0x77, 0x07, 0x00 ) if $param=~/filteroff/;
				@cmd = ( 0xFF, 0, 0x32, 0x77, 0x07, 0x01 ) if $param=~/filteron/;
			}
			elsif($cmd=~/hot/) {
				@cmd = ( 0xFF, 0, 0x0F, 0x77, 0x15, 0 ); #set IR active
				cmdTransmit($dev,0,@cmd);
				@cmd = ( 0xFF, 0, 0x24, 0x77, 0x03, 0 ) if $param=~/white/;
				@cmd = ( 0xFF, 0, 0x24, 0x77, 0x03, 0x01 ) if $param=~/black/;
			}
			elsif($cmd=~/2x/) {
				@cmd = ( 0xFF, 0, 0x0F, 0x77, 0x14, 0 ); #set TV active
				cmdTransmit($dev,0,@cmd);
				@cmd = ( 0xFF, 0, 0x32, 0x77, 0x02, 0x01 ) if $param=~/in/;
				@cmd = ( 0xFF, 0, 0x32, 0x77, 0x02, 0 ) if $param=~/out/;
			}
			elsif($cmd=~/lrf/){
				writePos($dev, 1);
			}
			elsif($cmd=~/illuminator/){
				@cmd = (0xff, 0, 0x0f, 0x77, 0x25);
				if($param=~/on/){
					$cmd[5] = 0x01;
				}else{
					$cmd[5] = 0x00;
				}
				cmdTransmit($dev,0,@cmd);
			}
			elsif($cmd=~/irpwr/){
				@cmd = (0xff, 0, 0x0f, 0x77, 0x14, 0); # set IR active
				cmdTransmit($dev,0, @cmd);
				$cmd[4] = 0x10 if $param=~/off/;
				$cmd[4] = 0x11 if $param=~/on/;
				cmdTransmit($dev,0, @cmd);
			}
		}
		elsif($mode=~/step/){       # mode=step  /Step by step positioning/
			my $timeout = 0.2;
			if($cmd=~/move/) { # step pan/tilt
				@cmd=(0xff,0,0,0,0,0);
				$cmd[4] = 0x09 if $param=~/left/;
				$cmd[4] = 0x09 if $param=~/right/;
				$cmd[5] = 0x09 if $param=~/up/;
				$cmd[5] = 0x09 if $param=~/down/;
				$cmd[3] |= 0x02 if $param=~/right/;
				$cmd[3] |= 0x04 if $param=~/left/;
				$cmd[3] |= 0x08 if $param=~/up/;
				$cmd[3] |= 0x10 if $param=~/down/;
			} elsif ($cmd=~/zoom/){
				@cmd=(0xff,0,0x32,0x77,0x05,0);
				$cmd[4] = 0x05;
				$cmd[5] |= 0x01 if $param=~/in/;
				$cmd[5] |= 0x02 if $param=~/out/;
			} elsif ($cmd=~/focus/){    
				@cmd=(0xff,0,0x32,0x77,0x05,0);
				$cmd[4] = 0x06;
				$cmd[5] |= 0x01 if $param=~/near/;
				$cmd[5] |= 0x02 if $param=~/far/;
			}
			cmdTransmit($dev,0,@cmd);
			select(undef,undef,undef,$timeout);
			@cmd = camStop($dev);
			cmdTransmit($dev,0,@cmd);
		}
		elsif($mode=~/hardware/){   # mode=hardware  /Hardware reset,defaults/
			if($cmd=~/do/ && $param=~/defaults/) { # init
				camReset($dev);
				return;
			}
		} elsif($mode=~/preset/){ # presets
			$cmd[3]= 0x03 if $cmd =~/save/; 
			$cmd[3]= 0x05 if $cmd =~/clear/; 
			$cmd[3]= 0x07 if $cmd =~/goto/; 
			$cmd[5]= $param;
			writePos($dev, 0);
		}
		# calculate command check-sum
		cmdTransmit($dev,0,@cmd);
	}
	
}
# --------------------------------------------------------------- nonblock -----
#   nonblock($socket) puts socket into nonblocking mode
sub nonblock {
	my $socket = shift;
	my $flags;
	
	$flags = fcntl($socket, F_GETFL, 0)
	or die "Can't get flags for socket: $!\n";
	fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
	or die "Can't make socket nonblocking: $!\n";
}

# ---------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {
	
	$log->info('Configurations refreshed: ',$query_str);
	
	# fisrst, close everything
	foreach my $skt (keys %sock) {
		close($sock{$skt}{SOCK});
		delete $sock{$skt};
	}
	
	%conf = GetCfgs( eval("($query_str)") );     # Load configurations
	foreach my $dev (keys %conf) {
		# in a case when PTZ works on another server (e.g. 3ETI COMTROL Serial Server)
		$conf{$dev}{DEVIP} = $conf{$dev}{PTZ_TCP_IP} if $conf{$dev}{PTZ_TCP_IP};
		
		$log->debug("[$dev]");
		$log->debug("DEVID=$conf{$dev}{DEVID}");
		$log->debug("PTZID=$conf{$dev}{PTZID}");
		$log->debug("PTGROUP=$conf{$dev}{PTZGROUP}");
		$log->debug("PTZSPEED=$conf{$dev}{PTZSPEED}");
		$log->debug("DEVIP=$conf{$dev}{DEVIP}");
		$log->debug("PTZ_TRANSPORT_PROTO=$conf{$dev}{PTZ_TRANSPORT_PROTO}");
		$log->debug("PTZ_TCP_PORT=$conf{$dev}{PTZ_TCP_PORT}");
		
		$log->info("Open socket connection: $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT}");
		if(not defined($sock{$conf{$dev}{DEVIP}})) {
			$sock{$conf{$dev}{DEVIP}}{SOCK} = IO::Socket::INET->new(PeerAddr => $conf{$dev}{DEVIP},
			PeerPort => $conf{$dev}{PTZ_TCP_PORT},
			Proto    => "tcp",
			Type     => SOCK_STREAM);
			if(! $sock{$conf{$dev}{DEVIP}}{SOCK} ) {
				$log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
				delete $sock{$conf{$dev}{DEVIP}};
				next;
			}
			
			$sock{$conf{$dev}{DEVIP}}{SOCK}->autoflush(1);
			nonblock($sock{$conf{$dev}{DEVIP}}{SOCK});
			$sock{$conf{$dev}{DEVIP}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}}{SOCK});
		}
	} # foreach $dev
} # sub load_dev_conf

