#!/usr/bin/perl
#  $$
# -----------------------------------------------------------------------------
#  PTZ driver for QuickSet GeminEye
# -----------------------------------------------------------------------------
#  Author: Serg Pososhenko
#  Edited by: 
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT LLC
# -----------------------------------------------------------------------------

use strict;

use POSIX;
use IO::Socket;
use IO::Select;
use Tie::RefHash;
use IO::File;
use Socket;
use Fcntl;
use Tie::RefHash;
use NextCAM::Init;
use Time::Local;


use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

my $log=get_logger('NEXTCAM::PTZ::PTL10GE');
my $query_str = shift || "'POSITIONCTL'=>'ptl10ge','CAMERAMODEL'=>'axis'";
$log->info("Starting PTL10GE PTZ DRIVER, query string: [$query_str]");


$SIG{HUP}=\&load_dev_conf;

my $APL=$ENV{APL};
my (%conf, %sock, $cam,$cmd,$par,$usrpsw,$last_cmd,$last_mode,$cmd_executed,%respond);
my $ETX = 0x03;
my $STX = 0x02;
my $time_360 = 0;

load_dev_conf();

# -----------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# -----------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
                                   PeerPort => $TCP_PORT,
                                   Proto     => "tcp",
                                   Type      => SOCK_STREAM)
    or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';

my %commands;
my $ready;
my $trycount = 30;
my %lastcmd;
my $lastcmdtime = time;
my $polling = 0;
my %antispoof;
my $BASE_VIEW=47;   # camera view angle on minimum zoom
my $STEP=0.1;       # fraction of base view
my $MAX_ZOOM=255;   # max value of zoom parameter for this camera 

while(1) {
	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/i;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/
				|| $commands{$cam} =~ /speed iris=0/
				|| $commands{$cam} =~ /speed gain=0/
				)
			{
				; # do not override stop command!
			}else{
				$commands{$cam} = $cmd;
			}
		} else {
			$commands{$cam} = $cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)
	
	$cmd = '';
	
	CheckAntispoof();
	
	
	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		if ($cmd=~/\santispoof\s/i) #ANTISPOOF support
		{
			my ($cam,$mode,$comd,$par,$options);
			$log->debug("ANTISPOOF:[$cmd]");
			if ($cmd!~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/)
			{delete $commands{$camera};$cmd=''; next;}
			else
			{
				($cam,$mode,$comd,$par,$options) = ($1,$2,$3,$4,$5);
				$log->debug("ANTISPOOF support: DEVID=$cam mode: $mode command: $comd optional parameter: $par");
			}
			$antispoof{par}='';         
			for(my $b=0;$b<length($par);$b++) { $antispoof{par} .= sprintf(substr($par,$b,1)); }
			$antispoof{cmd}=$comd;
			$antispoof{cam}=$cam;
			$antispoof{options}=$options;
			delete $commands{$camera};
			CheckAntispoof();
			$cmd=$commands{$camera};
		} # end of ANTISPOOF support
		else
		{
			delete $commands{$camera};
		}
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
		$usrpsw = "$conf{$cam}{USRNAME}:$conf{$cam}{PASSWD}" if $conf{$cam}{USRNAME} && $conf{$cam}{PASSWD};
		#$log->debug('Last command was:',$lastcmd{$camera}{CMD});
		if ($lastcmd{$camera}{CMD}=~/gain/i)
		{
			camStop($cam);
			$lastcmd{$camera}{CMD} = 'pt';
		}
		
		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam,$mode,"pt", $params[0].','.$params[1] ,\%options);
			camCmd($cam,$mode,"zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam,$mode,$cmd,$par,\%options);
		}
		
		$lastcmd{$camera}{CMD} = $cmd;
		$lastcmd{$camera}{MODE} = $mode;
		$lastcmd{$camera}{PAR} = $par;
		$lastcmd{$camera}{OPTS} = \%options;
		$polling = 0;
	}
	
	select(undef,undef,undef,.1) if not $cmd;

	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
		$lastcmd{$dev}{CMD} = 'goto';
		$lastcmd{$dev}{MODE} = 'preset';
		$lastcmd{$dev}{PAR} = 1;
		$lastcmd{$dev}{OPTS} = \%options;
		$polling = 0;
	}
	
	if(!$cmd) {
#        $log->debug('===> No CMD procedure!');
		foreach my $dev (keys %conf) {
			my %options = {};
			$polling = 1;
#            $log->debug("===> No CMD procedure!,$dev,$lastcmd{$dev}{MODE},$lastcmd{$dev}{CMD},$lastcmd{$dev}{PAR} time: $time_360");
			if(defined($lastcmd{$dev})) {
				if($lastcmd{$dev}{MODE} eq 'speed') {
					if($lastcmd{$dev}{CMD} eq 'gain' and $lastcmd{$dev}{PAR} == 0) {
						camCmd($dev,'speed','pt','0,0',\%options);
					}
					elsif($lastcmd{$dev}{CMD} eq 'iris' and $lastcmd{$dev}{PAR} == 0) {
						camCmd($dev,'speed','pt','0,0',\%options);
					}
					else {
						camCmd($dev,$lastcmd{$dev}{MODE},$lastcmd{$dev}{CMD},$lastcmd{$dev}{PAR},$lastcmd{$dev}{OPTS});
						if (($time_360)&&((timegm(gmtime) - $time_360) >170))
						{
							$lastcmd{$dev}{CMD} = 'pt';
							$lastcmd{$dev}{MODE} = 'speed';
							$lastcmd{$dev}{PAR} = '0,0';
							$lastcmd{$dev}{OPTS} = \%options;
							camCmd($dev,'step','move','right',\%options);
							$time_360=0;
						}
					}
				}
				else {
					camCmd($dev,'speed','pt','0,0',\%options);
				}
				select(undef,undef,undef,.2);
			}
			else {
				camCmd($dev,'speed','pt','0,0',\%options);
				select(undef,undef,undef,.1);
			}
		}        
	}
} # while(1)


# ----------------------------------------------------------- cmdTransmit -----
sub cmdTransmit {
	my ($dev,@cmd)=@_;
	
	@cmd = checkEscape(@cmd);
	
	my $command='';
	my $b;
	foreach $b (@cmd) { $command .= sprintf("%02X ",$b); }
	$log->debug('cmdTransmit( ',$dev, '=> ',$command,')');    
	
	if(not defined $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK}) {
		$log->debug('Error: socket connection is broken');
		$sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK} = IO::Socket::INET->new(
		                                        PeerAddr => $conf{$dev}{DEVIP},
		                                        PeerPort => $conf{$dev}{PTZ_TCP_PORT},
		                                        Proto    => "tcp",
		                                        Type     => SOCK_STREAM);
		if(! $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK} ) {
			$log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
			delete $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}};
			return;
		}
		$sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK}->autoflush(1);
		nonblock($sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK});
		$sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK});
		return;
	}

	my $bts = '';
	foreach my $b (@cmd) { $bts .= chr($b); }
	my $skt = $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK};
	eval {
		local $SIG{PIPE} = sub { die "Error writing to socket" };
		print $skt $bts;
	};
	$log->error($@) if $@;
	select(undef,undef,undef,.15);
	foreach my $sss ( $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SEL}->can_read(1) ) {
		my $data = '';
		eval {
			alarm 1;
			$sss->recv($data, POSIX::BUFSIZ, 0);
			$command='';
			for($b=0;$b<length($data);$b++) { $command .= sprintf("%02X ",ord(substr($data,$b,1))); }
			$log->debug("Received:$command");
			un_Escape($data);
			respond($data);
		};
		alarm 0;
		return $data;
	}  # foreach      

} # sub cmdTransmit

# -------------------------------------------------------------- CheckSum -----
sub checkSum
{
	my ($len,@arr) = @_;
	my $LRC = 0;
	for ( my $i = 2; $i < $len; $i++) {
		$LRC ^= $arr[$i];
	}
	return $LRC % 256;
	#return ($arr[1]+$arr[2]+$arr[3]+$arr[4]+$arr[5]) % 256;
} # sub checkSum

# ----------------------------------------------------------- checkEscape -----
sub checkEscape
{
	my @arr = @_;
	my @res = ($arr[0]);
	my $r = 1;
	for ( my $i = 1; $i <= $#arr; $i++) {
		if ( $#arr - 1 > $i && ( $arr[$i] == $STX || $arr[$i] == $ETX || $arr[$i] == 0x06 || $arr[$i] == 0x15 || $arr[$i] == 0x1B ) ) {
			$res[$r++] = 0x1B;
			$res[$r++] = $arr[$i] | 0x80;
		} else {
			$res[$r++] = $arr[$i];
		}
	}
	return @res;
} # sub checkEscape

# --------------------------------------------------------------- camStop -----
sub camStop
{
	my ($dev) = @_;
	my %options = {};
	camCmd($dev,'speed','pt','0,0',\%options);
}

# -------------------------------------------------------------- camReset -----
sub camReset
{
	my ($dev) = @_;
	my @cmd = ();
  
	camStop($dev); # Stop motion, if exist

	$cmd[0] = $STX;
	$cmd[1]=0x00;#$conf{$dev}{PTZID};
  
	# Move to Absolute 0/0
	$cmd[2]=0x35; $cmd[3]=checkSum(3,@cmd); $cmd[4]=$ETX;
	cmdTransmit($dev,@cmd);
	# Auto Gain&Level - on
	@cmd = @cmd[0..1];
	$cmd[2]=0x62; $cmd[3]=2; $cmd[4]=0x80; $cmd[5]=0x04; 
	$cmd[6]=checkSum(6,@cmd); $cmd[7]=$ETX;
	cmdTransmit($dev,@cmd);
	# Auto Shutter - on
	@cmd = @cmd[0..1];
	$cmd[2]=0x62; $cmd[3]=2; $cmd[4]=0x80; $cmd[5]=0xF1; 
	$cmd[6]=checkSum(6,@cmd); $cmd[7]=$ETX;
	cmdTransmit($dev,@cmd);
	# Histogram box - off
	@cmd = @cmd[0..1];
	$cmd[2]=0x62; $cmd[3]=2; $cmd[4]=0x80; $cmd[5]=0x95; 
	$cmd[6]=checkSum(6,@cmd); $cmd[7]=$ETX;
	cmdTransmit($dev,@cmd);

} # sub camReset

# ---------------------------------------------------------------- PTspeed ----
sub PTspeed {
	my ($dev,$speed,$ptzspeed)=@_;
	$speed = int( 1.27 * $speed * $ptzspeed / 100.0 );
	if($speed <= 0 ) {
		$speed = -127 if $speed < -127;
		$speed = -$speed * 2;
	} 
	else {
		$speed = 127 if $speed > 127;
		$speed = $speed * 2 + 1;
	}
	return $speed;
} # sub PTspeed
# ---------------------------------------------------------------- camCmd -----
sub camCmd {
	my ($dev,$mode,$cmd,$param,$options)=@_;
	my %options = %$options;
	my @cmd = ( $STX , 0x00); # camera hardware ID
	$log->debug("camCmd: DEVID=$dev mode: $mode command: $cmd optional parameter: $param");

	# here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
	if($conf{$dev}{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
		if(not $polling and ($mode=~/speed/i || $mode=~/step/i || $mode=~/abs/i || $mode=~/rel/i || $mode=~/smooth/i)) {
			$log->debug("Setting timeout +$conf{$dev}{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}{TIMEOUT} = time + $conf{$dev}{PTZ_PRESET1TIMEOUT};
		}
		elsif(not $polling and ($mode=~/preset/i && $cmd=~/goto/i && $param!=1)) {
			$log->debug("Setting timeout (preset) +$conf{$dev}{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}{TIMEOUT} = time + $conf{$dev}{PTZ_PRESET1TIMEOUT};
		}
	}

	if(not $cmd=~/stop/) { # non-Stop
		if ($mode=~/speed/i){        # mode speed
			# $log->debug("MODE->speed, cmd:$cmd  param: $param");
			$cmd[2]=0x31; 
			$cmd[3]=0x00;
			if($cmd=~/PT/i) { # RPT
				my ($p,$t) = split(/,/,$param);
				$cmd[3]=$cmd[3]+0x80 if ($p>0);
				$cmd[3]=$cmd[3]+0x40 if ($t<0);
				$cmd[4]=PTspeed($dev,$p,$conf{$dev}{PTZSPEED});
				$cmd[5]=PTspeed($dev,-$t,$conf{$dev}{PTZSPEED});
				$cmd[6]=0;
				$cmd[7]=0;
				$cmd[8]=0;
				$cmd[9]=0;
				$cmd[10]=checkSum(10,@cmd);
				$cmd[11]=$ETX;
				# $log->debug("RPT $p $t");
			}
			elsif($cmd=~/Z/i) { 
				$cmd[4]=0;
				$cmd[5]=0;
				$cmd[6]=PTspeed($dev,-$param,$conf{$dev}{PTZZOOMSPEED});
				$cmd[7]=0;
				$cmd[8]=PTspeed($dev,-$param,$conf{$dev}{PTZZOOMSPEED});
				$cmd[9]=0;
				$cmd[10]=checkSum(10,@cmd);
				$cmd[11]=$ETX;
				$log->debug("Z $param");
			}
			elsif($cmd=~/focus/i) {
				$log->debug("FOCUS $param");
				$cmd[4]=0;
				$cmd[5]=0;
				$cmd[6]=0;
				$cmd[7]=PTspeed($dev,-$param,$conf{$dev}{PTZSPEED});
				$cmd[8]=0;
				$cmd[9]=PTspeed($dev,-$param,$conf{$dev}{PTZSPEED});
				$cmd[10]=checkSum(10,@cmd);
				$cmd[11]=$ETX;
			}
			elsif($cmd=~/iris/i) {
				$log->debug("IRIS $param");
				my $attr="IRIS".$conf{$dev}{PTZID};
				$cmd[2]=0x64;
				$cmd[3]=0x00;
				$cmd[3]=64 if $conf{$dev}{PTZID} == 2;
				$cmd[3]=$cmd[3]+0x02;
				if ($param > 0)
				{
					if ($respond{$attr}+30 >255) {
						$cmd[4] = 255;
					}
					else {
						$cmd[4] =$respond{$attr}+30;
					}
				} elsif($param < 0) {
					if ($respond{$attr}-30 <0) {
						$cmd[4] = 0;
					}
					else {
						$cmd[4] =$respond{$attr}-30;
					}
				} else {
					camStop($conf{$dev}{DEVID});
					return;
				}
				$cmd[5]=checkSum(5,@cmd);
				$cmd[6]=$ETX;
			}
			elsif($cmd=~/gain/i) {
				$log->debug("GAIN $param");
				my $attr="GAIN".$conf{$dev}{PTZID};
				$cmd[2]=0x64;
				$cmd[3]=0x00;
				$cmd[3]=64 if $conf{$dev}{PTZID} == 2;
				$cmd[3]=$cmd[3]+0x03;
				if ($param > 0) {
					if ($respond{$attr}+20 >255) {
						$cmd[4] = 255;
					}
					else {
						$cmd[4] =$respond{$attr}+30;
					}
				} elsif($param < 0) {
					if ($respond{$attr}-20 <0) {
						$cmd[4] = 0;
					}
					else {
						$cmd[4] =$respond{$attr}-30;
					}
				} else {
					camStop($conf{$dev}{DEVID});
					return;
				}
				$cmd[5]=checkSum(5,@cmd);
				$cmd[6]=$ETX;
			}
		} elsif($mode=~/rel/i) { # mode=rel: "<devid> rel size=640x480 xy=225,152"
			
			my $local_zoom=$respond{ZOOM1};
			$local_zoom=$respond{ZOOM2} if $conf{$dev}{PTZID} == 2; 
			$local_zoom=1 if $local_zoom == 0;
			
			$param =~ /(\d+)x(\d+)/;
			my ($rel_size_x,$rel_size_y) = ($1,$2);
			$options{xy} =~ /(\d+),(\d+)/;
			my ($rel_click_x,$rel_click_y) = ($1,$2);
			my ($rel_p, $rel_t) = ( (3020.01*(256-$local_zoom)/255)*($rel_click_x-$rel_size_x/2.01)/$rel_size_x, (-3120.01*(256-$local_zoom)/255)*($rel_click_y-$rel_size_y/2.01)/$rel_size_y );
			$rel_p = $rel_p*10 if abs($rel_p) < 20; # correction for MAX zoom
			$rel_t = $rel_t*10 if abs($rel_t) < 20; # correction for MAX zoom
			$log->debug("REL [$rel_size_x] [$rel_size_y] [$rel_click_x] [$rel_click_y] [$rel_p] [$rel_t]");
			$cmd[2]=0x34;
			my $p_hex = substr(sprintf("%06X",int($rel_p)),-6);
			my $t_hex = substr(sprintf("%06X",int($rel_t)),-6);
			$log->debug("REL [$rel_p] [$rel_t] {$p_hex} {$t_hex}");
			$p_hex =~/(\w\w)(\w\w)(\w\w)$/;
			($cmd[3],$cmd[4],$cmd[5]) = (hex($3),hex($2),hex($1));
			$t_hex =~/^(\w\w)(\w\w)(\w\w)$/;
			($cmd[6],$cmd[7],$cmd[8]) = (hex($3),hex($2),hex($1));
			$cmd[9]=checkSum(9,@cmd);
			$cmd[10]=$ETX;
		}
		elsif ($mode=~/settings/i){ # settings
			$log->debug('settings',$cmd);
			if($cmd=~/autofocus/i) { 
				# Auto Focus
				$cmd[2]=0x65; 
				$cmd[3]=0;
				$cmd[3]=64 if $conf{$dev}{PTZID} == 2;
				$cmd[3]=$cmd[3]+128 if $param=~/on/i;
				$cmd[4]=checkSum(4,@cmd);
				$cmd[5]=$ETX;
			}
			elsif($cmd=~/autogain/i) { 
				# Auto Gain
				$cmd[2]=0x65; 
				$cmd[3]=0x01;
				$cmd[3]=64 if $conf{$dev}{PTZID} == 2;
				$cmd[3]=$cmd[3]+128 if $param=~/on/i;
				$cmd[4]=checkSum(4,@cmd);
				$cmd[5]=$ETX; 
			}
			elsif($cmd=~/autoiris/i) { 
				# Auto Iris
				$cmd[2]=0x65; 
				$cmd[3]=0x02;
				$cmd[3]=$cmd[3]+64 if $conf{$dev}{PTZID} == 2;
				$cmd[3]=$cmd[3]+128 if $param=~/on/i;
				$cmd[4]=checkSum(4,@cmd);
				$cmd[5]=$ETX;             
			}
			elsif($cmd=~/autobalance/i) { 
				# Auto White Balance
				$cmd[2]=0x65; 
				$cmd[3]=0x03;
				$cmd[3]=$cmd[3]+64 if $conf{$dev}{PTZID} == 2;
				$cmd[3]=$cmd[3]+128 if $param=~/on/i;
				$cmd[4]=checkSum(4,@cmd);
				$cmd[5]=$ETX;             
			}
			elsif($cmd=~/autoir/i) { 
				# Auto IR mode
				$cmd[2]=0x65; 
				$cmd[3]=0x04;
				$cmd[3]=$cmd[3]+64 if $conf{$dev}{PTZID} == 2;
				$cmd[3]=$cmd[3]+128 if $param=~/on/i;
				$cmd[4]=checkSum(4,@cmd);
				$cmd[5]=$ETX;             
			}
			elsif($cmd=~/pictflip/i) { 
				# Picture Flip
				$cmd[2]=0x65; 
				$cmd[3]=0x05;
				$cmd[3]=$cmd[3]+64 if $conf{$dev}{PTZID} == 2;
				$cmd[3]=$cmd[3]+128 if $param=~/on/i;
				$cmd[4]=checkSum(4,@cmd);
				$cmd[5]=$ETX;             
			}
			elsif($cmd=~/autostab/i) { 
				# Auto Stabilization
				$cmd[2]=0x65; 
				$cmd[3]=0x06;
				$cmd[3]=$cmd[3]+64 if $conf{$dev}{PTZID} == 2;
				$cmd[3]=$cmd[3]+128 if $param=~/on/i;
				$cmd[4]=checkSum(4,@cmd);
				$cmd[5]=$ETX;             
			}
			elsif ($cmd=~/polarity/i){
				$cmd[2]=0x65; 
				$cmd[3]=0x07;
				$cmd[3]=$cmd[3]+64 if $conf{$dev}{PTZID} == 2;
				$cmd[3]=$cmd[3]+128 if $param=~/white/i;
				$cmd[4]=checkSum(4,@cmd);
				$cmd[5]=$ETX;      
			}
			elsif($cmd=~/irmode/i) { 
				# IR mode
				$cmd[2]=0x65; 
				$cmd[3]=0x08;
				$cmd[3]=$cmd[3]+64 if $conf{$dev}{PTZID} == 2;
				$cmd[3]=$cmd[3]+128 if $param=~/on/i;
				$cmd[4]=checkSum(4,@cmd);
				$cmd[5]=$ETX;             
			}
			elsif($cmd=~/digizoom/i) { 
				# Digital Zoom
				$cmd[2]=0x65; 
				$cmd[3]=0x09;
				$cmd[3]=$cmd[3]+64 if $conf{$dev}{PTZID} == 2;
				$cmd[3]=$cmd[3]+128 if $param=~/on/i;
				$cmd[4]=checkSum(4,@cmd);
				$cmd[5]=$ETX;             
			}
			# elsif($cmd=~/pictflip/i) { 
			elsif($cmd=~/camerapower/i) { 
				# Camera Power
				$cmd[2]=0x65; 
				$cmd[3]=0x0A;
				$cmd[3]=$cmd[3]+64 if $conf{$dev}{PTZID} == 2;
				$cmd[3]=$cmd[3]+128 if $param=~/on/i;
				$cmd[4]=checkSum(4,@cmd);
				$cmd[5]=$ETX;             
			}
			elsif($cmd=~/reset/i) {
				$cmd[2]=0x31; 
				$cmd[3]=0x00;
				$cmd[4]=0;
				$cmd[5]=0;
				$cmd[6]=0;
				$cmd[7]=0;
				$cmd[8]=checkSum(8,@cmd);
				$cmd[9]=$ETX;
				$log->debug("SETTINGS/RESET");
			}
		} elsif($mode=~/step/i){       # mode=step  /Step by step positioning/ 
			if($cmd=~/move/i) { # step pan/tilt
				$cmd[2]=0x34; $cmd[3]=$cmd[4]=$cmd[5]=$cmd[6]=$cmd[7]=$cmd[8]=0;
				my $local_zoom=$respond{ZOOM1};
				$local_zoom=$respond{ZOOM2} if $conf{$dev}{PTZID} == 2;
				
				if ( $param=~/right/i ) {
					$cmd[3]=int(($BASE_VIEW*(256-$local_zoom)/(20*$MAX_ZOOM))*100); #0xC8 #- 2 degree right;
					$cmd[3]=20 if $cmd[3]< 20;
				}
				elsif ( $param=~/left/i ) {
					$cmd[4]=0xFF;$cmd[5]=0xFF;
					$cmd[3]=255 - (int(($BASE_VIEW*(256-$local_zoom)/(20*$MAX_ZOOM))*100));#$cmd[3]=0x38; $cmd[4]=0xFF;$cmd[5]=0xFF;# 2 degree left
					$cmd[3]=235 if $cmd[3] > 235;
				}
				if ( $param=~/down/i ) {
					$cmd[7]=0xFF;$cmd[8]=0xFF;
					$cmd[6]=255 - (int(($BASE_VIEW*(256-$local_zoom)/(20*$MAX_ZOOM))*100));#$cmd[6]=0x38; $cmd[7]=0xFF;$cmd[8]=0xFF;# 2 degree down
					$cmd[6]=235 if $cmd[6] > 235;
				}
				elsif ( $param=~/up/i ) {
					$cmd[6]=int(($BASE_VIEW*(256-$local_zoom)/(20*$MAX_ZOOM))*100); #0xC8 ;#- 2 degree up;
					$cmd[6]=20 if $cmd[6]< 20;
				}
				$cmd[9]=checkSum(9,@cmd); $cmd[10]=$ETX;
			} elsif ($cmd=~/zoom/i){
				my $attr="ZOOM".$conf{$dev}{PTZID};
				$cmd[2]=0x64;
				$cmd[3]=0x00;
				$cmd[3]=64 if $conf{$dev}{PTZID} == 2;
				$cmd[4] = $respond{$attr} +10 if $param=~/in/i;
				$cmd[4] = $respond{$attr} -10 if $param=~/out/i;
				$cmd[5]=checkSum(5,@cmd);
				$cmd[6]=$ETX;
			} elsif ($cmd=~/focus/i){    
				my $attr="FOCUS".$conf{$dev}{PTZID};
				$cmd[2]=0x64;
				$cmd[3]=0x00;
				$cmd[3]=64 if $conf{$dev}{PTZID} == 2;
				$cmd[3]=$cmd[3]+0x01;
				$cmd[4] = $respond{$attr}+10 if $param=~/near/i;
				$cmd[4] = $respond{$attr}-10 if $param=~/far/i;
				$cmd[4] = 0x00 if $cmd[4] < 0;
				$cmd[5]=checkSum(5,@cmd);
				$cmd[6]=$ETX;
			} elsif ($cmd=~/iris/i){
				my $attr="IRIS".$conf{$dev}{PTZID};
				$cmd[2]=0x64;
				$cmd[3]=0x00;
				$cmd[3]=64 if $conf{$dev}{PTZID} == 2;
				$cmd[3]=$cmd[3]+0x02;
				$cmd[4] = $respond{$attr}+30 if $param=~/up/i;
				$cmd[4] = $respond{$attr}-30 if $param=~/down/i;
				$cmd[4] = 0 if $cmd[4]<0;
				$cmd[4] = 255 if $cmd[4]>255;
				$cmd[5]=checkSum(5,@cmd);
				$cmd[6]=$ETX;
			} elsif ($cmd=~/gain/i){
				my $attr="GAIN".$conf{$dev}{PTZID};
				$cmd[2]=0x64;
				$cmd[3]=0x00;
				$cmd[3]=64 if $conf{$dev}{PTZID} == 2;
				$cmd[3]=$cmd[3]+0x03;
				$cmd[4] = $respond{$attr}+10 if $param=~/up/i;
				$cmd[4] = $respond{$attr}-10 if $param=~/down/i;
				$cmd[5]=checkSum(5,@cmd);
				$cmd[6]=$ETX;
				$log->debug('GAIN---------------------------------->',$cmd[4]);
			} 

		} elsif($mode=~/hardware/i){   # mode=hardware  /Hardware reset,defaults/
			if($cmd=~/do/i && $param=~/defaults/i) { # init
				camReset($dev);
				return;
			}
		} elsif($mode=~/abs/i){                # mode=abs  /Absloute positioning/
			if($cmd=~/center/i) { # Center
				$cmd[2]=0x35; $cmd[3]=checkSum(3,@cmd); $cmd[4]=$ETX;
			} elsif($cmd=~/PT/i){            # Pan, Tilt
				$log->debug("ABS:  PT $param");
				$cmd[2]=0x33;
				$param=~/^(\-?.+)?\,(\-?.+)$/;
				my $p = int($1*100);
				my $t = int($2*100);
				$log->debug("ABS! pt=$1,$2 ($p,$t)");
				$p=99999 if $p>99999;
				#$p += 36000 if $p<0;
				#$t += 36000 if $t<0;
				my $p_hex = substr(sprintf("%06X",$p),-6);
				my $t_hex = substr(sprintf("%06X",$t),-6);
				$log->debug("==== P_HEX= $p_hex");
				$log->debug("==== T_HEX= $t_hex");
				$p_hex =~/(\w\w)(\w\w)(\w\w)$/;
				($cmd[3],$cmd[4],$cmd[5]) = (hex($3),hex($2),hex($1));
				$t_hex =~/^(\w\w)(\w\w)(\w\w)$/;
				($cmd[6],$cmd[7],$cmd[8]) = (hex($3),hex($2),hex($1));
				$cmd[9]=checkSum(9,@cmd); 
				$cmd[10]=$ETX;
				# command will be send later 
			} elsif ($cmd=~/Z/i){	# Absolute zoom
				# Nothing to do
				return;
			}
		} elsif($mode=~/preset/i){ # presets
			$cmd[3]= $param;
			if ( $cmd =~/save/i ) {
				$cmd[2]= 0x42;
			}
			if ( $cmd =~/clear/i ) {
				# Nothing to do
				return;
			}
			if ( $cmd =~/goto/i ) {
				$cmd[2]= 0x32;
			}
			$cmd[4]=checkSum(4,@cmd); 
			$cmd[5]=$ETX;
		} elsif($mode=~/smooth/i) { # Right/Left/Up/Down/Tele/Wide/focus_Near/focus_Far/iris_Open/irIs_close
			# set movement bits
			if ($cmd=~/move/i) { # pan/tilt
				# Nothing to do
				return;
			} elsif($cmd=~/zoom/i) { # zoom
				# Nothing to do
				return;
			} elsif($cmd=~/focus/i) { # focus
				# Nothing to do
				return;
			} elsif($cmd=~/iris/i) { # iris
				# Nothing to do
				return;
			}
			return;
		}
	}
	# calculate command check-sum
	cmdTransmit($dev,@cmd);
}


sub nonblock {
	my ($fd) = @_;
	my $flags = fcntl($fd, F_GETFL,0);
	fcntl($fd, F_SETFL, $flags|O_NONBLOCK);
}



# --------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {

	$log->debug('load_dev_conf');
	
	# fisrst, close everything
	foreach my $skt (keys %sock) {
		close($sock{$skt}{SOCK});
		delete $sock{$skt};
	}


	%conf = GetCfgs( eval("($query_str)") );     # Load configurations
	$log->debug("Config read as:");
	$log->debug('-------------------------------------');
	foreach my $dev (keys %conf) {
		$log->debug("[$dev]");
		$log->debug("DEVID=$conf{$dev}{DEVID}");
		$log->debug("CAMERAMODEL=$conf{$dev}{CAMERAMODEL}");
		$log->debug("POSITIONCTL=$conf{$dev}{POSITIONCTL}");
		$log->debug("PTZID=$conf{$dev}{PTZID}");
		$log->debug("PTZ_TCP_PORT=$conf{$dev}{PTZ_TCP_PORT}");
		$log->debug("-------------------------------------");

		$log->debug("PTZ_TCP_PORT=$conf{$dev}{PTZ_TCP_PORT}");
		if(not defined($sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}})) {
			$sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK} = IO::Socket::INET->new(
			                                        PeerAddr => $conf{$dev}{DEVIP},
			                                        PeerPort => $conf{$dev}{PTZ_TCP_PORT},
			                                        Proto    => "tcp",
			                                        Type     => SOCK_STREAM);
			if(! $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK} ) {
				$log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
				delete $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}};
				next;
			}
			$sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK}->autoflush(1);
			nonblock($sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK});
			$sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK});
		}
	} # foreach $dev
	# close unused ports
}
# --------------------------------------------------------- respond -----
sub respond {
	my $data=shift;
	my @tmp=un_Escape($data);
	my $command='';
#    for($b=0;$b<length($data);$b++) 
#    { 
#        $tmp[$b]=substr($data,$b,1); 
#    } Received:06 00 31 C5 FA FF 73 04 00 00 00 80 00 92 69 11 48 00 00 2D 09 00 00 00 00 00 00 00 00 11 05 00 00 00 94 03  PAN:167758.77
	if (hex($tmp[2]) == 0x31)
	{
		%respond='';
		$respond{PAN}=hex("$tmp[5]$tmp[4]$tmp[3]");
		if ($tmp[5] eq 'FF'){$respond{PAN}=((-1)*(hex('FFFFFF')-$respond{PAN}+1))/100}else{$respond{PAN}=$respond{PAN}/100;}
		$respond{TILT}=hex("$tmp[8]$tmp[7]$tmp[6]");
		if($tmp[8] eq 'FF'){$respond{TILT}=((-1)*(hex('FFFFFF')-$respond{TILT}+1))/100} else{$respond{TILT}=$respond{TILT}/100;}
		$respond{ZOOM1}=hex($tmp[12]);
		$respond{FOCUS1}=hex($tmp[13]);
		$respond{IRIS1}=hex($tmp[14]);
		$respond{GAIN1}=hex($tmp[15]);
		$respond{ZOOM2}=hex($tmp[22]);
		$respond{FOCUS2}=hex($tmp[23]);
		$respond{IRIS2}=hex($tmp[24]);
		$respond{GAIN2}=hex($tmp[25]);
	}
	$log->debug('Respond from camera:P=>',,$respond{PAN},' T=>',$respond{TILT},' Z1=>',$respond{ZOOM1},' F1=>',$respond{FOCUS1},' I1=>',$respond{IRIS1},' G1=>',$respond{GAIN1},' Z2=>',$respond{ZOOM2},' F2=>',$respond{FOCUS2},' I2=>',$respond{IRIS2},' G2=>',$respond{GAIN2});
}

sub CheckAntispoof
{
	#stage 1 1/2 - support antispoof stack
	if ($antispoof{cmd}!~/pt/i)
	{
#         $log->debug("sub CHCKantiSPOOF now is =>> $a");
		if ($antispoof{par})
		{
			my $newspoof='';         
			for(my $b=1;$b<length($antispoof{par});$b++) { $newspoof .= sprintf(substr($antispoof{par},$b,1)); }
			my $a= (substr($antispoof{par},0,1));
			$log->debug("sub CHCKantiSPOOF now is =>> $a");
			if ((substr($antispoof{par},0,1)) eq 'L')  
			{
				delete $commands{$antispoof{cam}};
				$commands{$antispoof{cam}}="$antispoof{cam} step move left $antispoof{options}";         
			}
			elsif ((substr($antispoof{par},0,1)) eq 'R')  
			{
				delete $commands{$antispoof{cam}};
				$commands{$antispoof{cam}}="$antispoof{cam} step move right $antispoof{options}";         
			}
			elsif ((substr($antispoof{par},0,1)) eq 'U')  
			{
				delete $commands{$antispoof{cam}};
				$commands{$antispoof{cam}}="$antispoof{cam} step move up $antispoof{options}";         
			}
			elsif ((substr($antispoof{par},0,1)) eq 'D')  
			{
				delete $commands{$antispoof{cam}};
				$commands{$antispoof{cam}}="$antispoof{cam} step move down $antispoof{options}";         
			}
			elsif ((substr($antispoof{par},0,1)) eq '1')  
			{
				delete $commands{$antispoof{cam}};
				$commands{$antispoof{cam}}="$antispoof{cam} preset goto 1 $antispoof{options}";         
			}
			elsif ((substr($antispoof{par},0,1)) eq '2')  
			{
				delete $commands{$antispoof{cam}};
				$commands{$antispoof{cam}}="$antispoof{cam} preset goto 2 $antispoof{options}";         
			}
			elsif ((substr($antispoof{par},0,1)) eq '3')  
			{
				delete $commands{$antispoof{cam}};
				$commands{$antispoof{cam}}="$antispoof{cam} preset goto 3 $antispoof{options}";         
			}
			elsif ((substr($antispoof{par},0,1)) eq '4')  
			{
				delete $commands{$antispoof{cam}};
				$commands{$antispoof{cam}}="$antispoof{cam} preset goto 4 $antispoof{options}";         
			}
			elsif ((substr($antispoof{par},0,1)) eq '5')  
			{
				delete $commands{$antispoof{cam}};
				$commands{$antispoof{cam}}="$antispoof{cam} preset goto 5 $antispoof{options}";         
			}
			elsif ((substr($antispoof{par},0,1)) eq '6')  
			{
				delete $commands{$antispoof{cam}};
				$commands{$antispoof{cam}}="$antispoof{cam} preset goto 6 $antispoof{options}";         
			}
			$antispoof{par}=$newspoof;
			if ($a =~/^\d/i)
			{
				select(undef,undef,undef,3);
			}
			else
			{
				select(undef,undef,undef,.2);
			}
		}
	}
	elsif ($antispoof{cmd}=~/pt/i)
	{
		$log->debug('sub CHCKantiSPOOF going to PT');
		my ($p,$t) = split(/,/,$antispoof{par});
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $antispoof{options} ");
		camCmd($antispoof{cam},'abs','pt','99999,0',\%options);
		$time_360=timegm(gmtime);
		select(undef,undef,undef,1);
		$log->debug('sub CHCKantiSPOOF =================> now we can send tour');
		$lastcmd{$antispoof{cam}}{CMD} = 'pt';
		$lastcmd{$antispoof{cam}}{MODE} = 'speed';
		$lastcmd{$antispoof{cam}}{PAR} = '5,0';
		$lastcmd{$antispoof{cam}}{OPTS} = \%options;
		camCmd($antispoof{cam},'speed','pt','5,0',\%options);
		#select(undef,undef,undef,5);
		%antispoof='';
	}

	return;
}
# ----------------------------------------------------------- un_Escape -----
sub un_Escape
{
	my $data=shift;
	
	my @arr;
	for(my $b=0;$b<length($data);$b++) { $arr[$b]= sprintf("%02X",ord(substr($data,$b,1)));}
	my @res;
	my $r = 0;
	for ( my $i = 0; $i <= $#arr; $i++) {
		if (hex($arr[$i]) == 0x1B)
		{
			$i++;
			$res[$r++] = hex($arr[$i])-128;
		} else {
			$res[$r++] = $arr[$i];
		}
	}
	return @res;
} # sub checkEscape
