#!/usr/bin/perl
#  $Id: ptz_ptcr20flir.pl 25370 2012-03-09 21:19:34Z teetov $
# -----------------------------------------------------------------------------
#  PTZ driver for QuickSet + FLIR
# -----------------------------------------------------------------------------
#  Author: Andrey Fomenko
#  Edited by: 
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT LLC, 2004-2005
# -----------------------------------------------------------------------------

use strict;

use POSIX;
use IO::Socket;
use IO::Select;
use Tie::RefHash;
use IO::File;
use Socket;
use Fcntl;
use Tie::RefHash;
use NextCAM::Init;

use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

my $log=get_logger('NEXTCAM::PTZ::PTCR20FLIR');

my $query_str = shift || "'POSITIONCTL'=>'ptcr20flir'";
$log->info("Starting PTCR20BAE PTZ DRIVER, query string: [$query_str]");


$SIG{HUP}=\&load_dev_conf;

my $APL=$ENV{APL};
my (%conf, %sock, $cam,$cmd,$par,$usrpsw,$last_cmd,$last_mode,$cmd_executed);
my $ETX = 0x03;
my $STX = 0x02;
my $high_speed = 0;   ## Flag -- 1 for high, 0, for low

load_dev_conf();

# -----------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# -----------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
                                   PeerPort => $TCP_PORT,
                                   Proto     => "tcp",
                                   Type      => SOCK_STREAM)
    or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';

my %commands;
my $ready;
my $trycount = 30;
my %lastcmd;
my $lastcmdtime = time;
my $polling = 0;


foreach my $dev (keys %conf) {
  my %options = {};
  camCmd($dev,'settings','init','',\%options);
}

while(1) {
	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/i;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/
				|| $commands{$cam} =~ /speed iris=0/
				|| $commands{$cam} =~ /speed gain=0/
				)
			{
				; # do not override stop command!
			}else{
				$commands{$cam} = $cmd;
			}
		} else {
			$commands{$cam} = $cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)
	
	$cmd = '';
	
	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		delete $commands{$camera};
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
		$usrpsw = "$conf{$cam}{USRNAME}:$conf{$cam}{PASSWD}" if $conf{$cam}{USRNAME} && $conf{$cam}{PASSWD};
		
		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam,$mode,"pt", $params[0].','.$params[1] ,\%options);
			camCmd($cam,$mode,"zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam,$mode,$cmd,$par,\%options);
		}
		
		$lastcmd{$camera}{CMD} = $cmd;
		$lastcmd{$camera}{MODE} = $mode;
		$lastcmd{$camera}{PAR} = $par;
		$lastcmd{$camera}{OPTS} = \%options;
		$polling = 0;
	}
	
	select(undef,undef,undef,.1) if not $cmd;


	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
		$lastcmd{$dev}{CMD} = 'goto';
		$lastcmd{$dev}{MODE} = 'preset';
		$lastcmd{$dev}{PAR} = 1;
		$lastcmd{$dev}{OPTS} = \%options;
		$polling = 0;
	}
	
	if(!$cmd) {
		foreach my $dev (keys %conf) {
			my %options = {};
			$polling = 1;
			if(defined($lastcmd{$dev})) {
				if($lastcmd{$dev}{MODE} eq 'speed') {
					if($lastcmd{$dev}{CMD} eq 'gain' and $lastcmd{$dev}{PAR} == 0) {
						camCmd($dev,'speed','pt','0,0',\%options);
					}
					else {
						camCmd($dev,$lastcmd{$dev}{MODE},$lastcmd{$dev}{CMD},$lastcmd{$dev}{PAR},$lastcmd{$dev}{OPTS});
					}
				}
				else {
					camCmd($dev,'speed','pt','0,0',\%options);
				}
				select(undef,undef,undef,.2);
			}
			else {
				camCmd($dev,'speed','pt','0,0',\%options);
				select(undef,undef,undef,.1);
			}
		}        
	}

} # while(1)




# ----------------------------------------------------------- cmdTransmit -----
sub cmdTransmit {
    my ($dev,@cmd)=@_;

    @cmd = checkEscape(@cmd);
    
    my $command='';
    my $b;
    my $skt;
    my $sel;
    foreach $b (@cmd) { $command .= sprintf("%02X ",$b); }
if ($cmd[2] != 0x31) {
    $log->debug('cmdTransmit( ',$dev, '=> ',$command,')');    
}

    ReconnectSocket($dev);
    $skt = $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK};
    $sel = $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SEL};

    my $bts = '';
    foreach my $b (@cmd) { $bts .= chr($b); }
    eval {
        local $SIG{PIPE} = sub { die "Error writing to socket" };
        print $skt $bts;
    };
    $log->error($@) if $@;
    select(undef,undef,undef,.15);
    foreach my $sss ( $sel->can_read(1) ) {
        my $data = '';
        eval {
            alarm 1;
            $sss->recv($data, POSIX::BUFSIZ, 0);
            $command='';
            for($b=0;$b<length($data);$b++) { $command .= sprintf("%02X ",ord(substr($data,$b,1))); }
        #    $log->debug('Received:',$command);
        };
        alarm 0;
        return $data;
    }  # foreach      

} # sub cmdTransmit

# -------------------------------------------------------------- CheckSum -----
sub checkSum
{
  my ($len,@arr) = @_;
  my $LRC = 0;
  for ( my $i = 2; $i < $len; $i++) {
    $LRC ^= $arr[$i];
  }
  return $LRC % 256;
  #return ($arr[1]+$arr[2]+$arr[3]+$arr[4]+$arr[5]) % 256;
} # sub checkSum

# ----------------------------------------------------------- checkEscape -----
sub checkEscape
{
  my @arr = @_;
  my @res = ($arr[0]);
  my $r = 1;
  for ( my $i = 1; $i <= $#arr; $i++) {
    if ( $#arr - 1 > $i && ( $arr[$i] == $STX || $arr[$i] == $ETX || $arr[$i] == 0x06 || $arr[$i] == 0x15 || $arr[$i] == 0x1B ) ) {
      $res[$r++] = 0x1B;
      $res[$r++] = $arr[$i] | 0x80;
    } else {
      $res[$r++] = $arr[$i];
    }
  }
  return @res;
} # sub checkEscape

# --------------------------------------------------------------- camStop -----
sub camStop
{
    my ($dev) = @_;
    my %options = {};
    camCmd($dev,'speed','pt','0,0',\%options);
}

# -------------------------------------------------------------- camReset -----
sub camReset
{
  my ($dev) = @_;
  my @cmd = ();
  
  camStop($dev); # Stop motion, if exist

  $cmd[0] = $STX;
  $cmd[1]=$conf{$dev}{PTZID};
  
  # Move to Absolute 0/0
  $cmd[2]=0x35; $cmd[3]=checkSum(3,@cmd); $cmd[4]=$ETX;
  cmdTransmit($dev,'PT',@cmd);

} # sub camReset

sub camFOV
{
  my ($dev,$narrowflag,$qptCamID) = @_;
  my @cmd = ();

  $cmd[0] = $STX;
  $cmd[1]=$conf{$dev}{PTZID};

  # Check zoom level -- make sure that it is appropriate

  my $zfile = '/opt/sarch/var/conf/' . $cam .'/zoom';
  my $zlevel=1;

  # If the file exists, read it in, otherwise, assume 1.
  if (-e $zfile) {
    open(zfilein, $zfile);
    $zlevel=<zfilein>;
    close(zfilein);
  }

  $log->debug($zlevel);
  if ($narrowflag == 1) {
    # @C(FOV_NARROW)<CR>
    $cmd[2]=0x62; 
    if ($qptCamID == 1) {
      $cmd[3]=0x0F;
    }
    else {
      $cmd[3]=0x8F;
    }

    my $idx = 4;
    foreach my $c (split(//,'@C(FOV_NARROW)')) {
      $cmd[$idx]=hex(sprintf("%x", ord($c)));
      $idx = $idx + 1;
    }
    $cmd[$idx]=0x0D; $idx = $idx + 1;
    $cmd[$idx]=checkSum($idx,@cmd); $idx = $idx + 1;
    $cmd[$idx]=$ETX;
    cmdTransmit($dev,@cmd);

    if ($zlevel < 3) {
      $zlevel = $zlevel + 3;
    }
  }
  else {
    # @C(FOV_WIDE)<CR>
    $cmd[2]=0x62; 
    if ($qptCamID == 1) {
      $cmd[3]=0x1D; # 8D
    }
    else {
      $cmd[3]=0x9D; # 8D
    }
    my $idx = 4;
    foreach my $c (split(//,'@C(FOV_WIDE)')) {
      $cmd[$idx]=hex(sprintf("%x", ord($c)));
      $idx = $idx + 1;
    }
    $cmd[$idx]=0x0D; $idx = $idx + 1;
    $cmd[$idx]=checkSum($idx,@cmd); $idx = $idx + 1;
    $cmd[$idx]=$ETX;
    cmdTransmit($dev,@cmd);
    if ($zlevel > 3) {
      $zlevel = $zlevel - 3;
    }
  }

  open(zfileout, ">$zfile");
  print zfileout $zlevel;
  close(zfileout);              
}

#----------------------------------- camFocus -----------------------
sub camFocus 
{
  my ($dev,$step,$nearflag,$qptCamID) = @_;
  my @cmd = ();

  $cmd[0] = $STX;
  $cmd[1]=$conf{$dev}{PTZID};

  if ($step == 0) {
    # Send focus stop command
    $cmd[2]=0x62; # pass-through to FLIR
    if ($qptCamID == 1) {
      $cmd[3]=0x0F; # Size of the packet
    }
    else { 
      $cmd[3]=0x8F; # Size of the packet
    }
    my $idx = 4;
    foreach my $c (split(//,'@C(FOCUS_STOP)')) {
      $cmd[$idx]=hex(sprintf("%x", ord($c)));
      $idx = $idx + 1;
    }
    $cmd[$idx]=0x0D; $idx = $idx + 1;
    $cmd[$idx]=checkSum($idx,@cmd); $idx = $idx + 1;
    $cmd[$idx]=$ETX;
    cmdTransmit($dev,@cmd);
  }
  else { 
      if ($nearflag == 1) {
        $cmd[2]=0x62; # pass-through to FLIR
      if ($qptCamID == 1) {
        $cmd[3]=0x18; # Size of the packet (0001 1000)
      } 
      else {
        $cmd[3]=0x98; # Size of the packet (1001 1000)
      }
      $cmd[4]=0x40;
      $cmd[5]=0x57;
      $cmd[6]=0x28;
      $cmd[7]=0x4D;
      $cmd[8]=0x49;
      $cmd[9]=0x4C;
      $cmd[10]=0x3A;
      $cmd[11]=0x46;
      $cmd[12]=0x4F;
      $cmd[13]=0x43;
      $cmd[14]=0x55;
      $cmd[15]=0x53;
      $cmd[16]=0x5F;
      $cmd[17]=0x4E;
      $cmd[18]=0x45;
      $cmd[19]=0x41;
      $cmd[20]=0x52;
      $cmd[21]=0x29;
      $cmd[22]=0x3D;
      my $f_hex = sprintf("%04X",$step);
      $f_hex =~/^(\w)(\w)(\w)(\w)$/;
      ($cmd[23],$cmd[24],$cmd[25],$cmd[26]) = (hex($1),hex($2),hex($3),hex($4));

      ## Add 48 to convert numbers to characters
      if ($cmd[23] < 10) {
        $cmd[23] = $cmd[23] + 48;
      }
      else {
        $cmd[23] = $cmd[23] + 55;
      }

      ## Add 48 to convert numbers to characters
      if ($cmd[24] < 10) {
        $cmd[24] = $cmd[24] + 48;
      }
      else {
        $cmd[24] = $cmd[24] + 55;
      }

      ## Add 48 to convert numbers to characters
      if ($cmd[25] < 10) {
        $cmd[25] = $cmd[25] + 48;
      }
      else {
        $cmd[25] = $cmd[25] + 55;
      }
  
      ## Add 48 to convert numbers to characters
      if ($cmd[26] < 10) {
        $cmd[26] = $cmd[26] + 48;
      }
      else {
        $cmd[26] = $cmd[26] + 55;
      }
  
      $cmd[27]=0x0D;
      $cmd[28]=checkSum(28,@cmd);
      $cmd[29]=$ETX;
      cmdTransmit($dev,@cmd);
    }
    else {
      $cmd[2]=0x62; # pass-through to FLIR
      if ($qptCamID == 1) {
        $cmd[3]=0x17; # Size of the packet  0001 0111
      }
      else {
        $cmd[3]=0x97; # Size of the packet  1001 0111
      }
      $cmd[4]=0x40;
      $cmd[5]=0x57;
      $cmd[6]=0x28;
      $cmd[7]=0x4D;
      $cmd[8]=0x49;
      $cmd[9]=0x4C;
      $cmd[10]=0x3A;
      $cmd[11]=0x46;
      $cmd[12]=0x4F;
      $cmd[13]=0x43;
      $cmd[14]=0x55;
      $cmd[15]=0x53;
      $cmd[16]=0x5F;
      $cmd[17]=0x46;
      $cmd[18]=0x41;
      $cmd[19]=0x52;
      $cmd[20]=0x29;
      $cmd[21]=0x3D;
      my $f_hex = sprintf("%04X",($step * -1));
      $f_hex =~/^(\w)(\w)(\w)(\w)$/;
      ($cmd[22],$cmd[23],$cmd[24],$cmd[25]) = (hex($1),hex($2),hex($3),hex($4));

      ## Add 48 to convert numbers to characters
      if ($cmd[23] < 10) {
        $cmd[23] = $cmd[23] + 48;
      }
      else {
        $cmd[23] = $cmd[23] + 55;
      }

      ## Add 48 to convert numbers to characters
      if ($cmd[24] < 10) {
        $cmd[24] = $cmd[24] + 48;
      }
      else {
        $cmd[24] = $cmd[24] + 55;
      }

      ## Add 48 to convert numbers to characters
      if ($cmd[25] < 10) {
        $cmd[25] = $cmd[25] + 48;
      }
      else {
        $cmd[25] = $cmd[25] + 55;
      }

      ## Add 48 to convert numbers to characters
      if ($cmd[22] < 10) {
        $cmd[22] = $cmd[22] + 48;
      }
      else {
        $cmd[22] = $cmd[22] + 55;
      }

      $cmd[26]=0x0D;
      $cmd[27]=checkSum(27,@cmd);
      $cmd[28]=$ETX;
      cmdTransmit($dev,@cmd);
    }
  }
}

#----------------------------------- camSensitivity -----------------------
sub camSensitivity
{
  my ($dev,$level,$qptCamID) = @_;
  my @cmd = ();

  $cmd[0] = $STX;
  $cmd[1]=$conf{$dev}{PTZID};

  $cmd[2]=0x62; # pass-through to FLIR 
  if ($qptCamID == 1) {
    $cmd[3]=0x1C; # Size of the packet
  }
  else {
    $cmd[3]=0x9C; # Size of the packet
  }

  my $idx=4;

  if ($level == 0) {
    foreach my $c (split(//,'@W(EQUALIZATION_TABLE)=0000')) {
      $cmd[$idx]=hex(sprintf("%x", ord($c)));
      $idx = $idx + 1;
    }
  }
  elsif ($level == 1) {
    foreach my $c (split(//,'@W(EQUALIZATION_TABLE)=0001')) {
      $cmd[$idx]=hex(sprintf("%x", ord($c))); 
      $idx = $idx + 1;
    }
  }
  else {
    foreach my $c (split(//,'@W(EQUALIZATION_TABLE)=0002')) {
      $cmd[$idx]=hex(sprintf("%x", ord($c))); 
      $idx = $idx + 1;
    }
  }
  $cmd[$idx]=0x0D;
  $idx = $idx + 1;
  $cmd[$idx]=checkSum($idx,@cmd); 
  $idx = $idx + 1;
  $cmd[$idx]=$ETX;
  cmdTransmit($dev,@cmd);
}

# ------------------------------------------------------------- camZoom -----
sub camZoom
{

  # ZOOM ZOOM:
  # 0 -- Wide 1x
  # 1 -- Wide 2x
  # 2 -- Wide 4x
  # 3 -- Narrow 1x
  # 4 -- Narrow 2x
  # 5 -- Narrow 4x

  my ($dev,$step,$speed,$qptCamID) = @_;
  my @cmd = ();

  $cmd[0] = $STX;
  $cmd[1]=$conf{$dev}{PTZID};

  my $zfile = '/opt/sarch/var/conf/' . $cam .'/zoom';
  my $zlevel=1;

  # If the file exists, read it in, otherwise, assume 1.
  if (-e $zfile) {
    open(zfilein, $zfile);
    $zlevel=<zfilein>;
    close(zfilein);
  }

  $log->debug($zlevel);

  if ($step != 0) {
    $zlevel = $zlevel + $step;
  }
  elsif ($speed != 0) {
    $zlevel = int($zlevel + int($speed) * (3 / 100));
  }

  ### Sanity check on zlevel first.
  if ($zlevel > 5) {
    $zlevel = 5;
  }
  if ($zlevel < 0) {
    $zlevel = 0;
  }

  if ($zlevel >= 3) {
    camFOV($dev, 1, $qptCamID);
  }
  else {
    camFOV($dev, 0, $qptCamID);
  }
  
  # Set communication parameters.
  # @W(EZOOM)<CR>=$zlevel
  $cmd[2]=0x62; # pass-through to FLIR 
  if ($qptCamID == 1) {
    $cmd[3]=0x0F; # (1000 1111) Size of the packet
  }
  else {
    $cmd[3]=0x8F; # (1000 1111) Size of the packet
  }
  $cmd[4]=0x40; #@
  $cmd[5]=0x57; #W
  $cmd[6]=0x28; #(
  $cmd[7]=0x45; #E
  $cmd[8]=0x5a; #Z
  $cmd[9]=0x4f; #O
  $cmd[10]=0x4f; #O
  $cmd[11]=0x4d; #M
  $cmd[12]=0x29; #)
  $cmd[13]=0x3d; #=
  $cmd[14]=0x24; 

  if (($zlevel == 0) || ($zlevel == 3)) {
    $cmd[15]=0x31; 
  }
  elsif (($zlevel == 1) || ($zlevel == 4)) {  
    $cmd[15]=0x32; 
  }
  elsif (($zlevel == 2) || ($zlevel == 5)) {
    $cmd[15]=0x34; 
  }
  $cmd[16]=0x30; 
  $cmd[17]=0x30; 
  $cmd[18]=0x0d; #<CR>
  $cmd[19]=checkSum(19,@cmd); 
  $cmd[20]=$ETX;
  cmdTransmit($dev,@cmd);

  open(zfileout, ">$zfile");
  print zfileout $zlevel;
  close(zfileout);              
} # sub camZoom

# ---------------------------------------------------------------- PTspeed ----
sub PTspeed {
  my ($dev,$speed)=@_;
  $speed = int( 1.27 * $speed * $conf{$dev}{PTZSPEED} / 100.0 );
  if($speed <= 0 ) {
       $speed = -127 if $speed < -127;
       $speed = -$speed * 2;
  } 
  else {
       $speed = 127 if $speed > 127;
       $speed = $speed * 2 + 1;
  }
  return $speed;
} # sub PTspeed
# ---------------------------------------------------------------- camCmd -----
sub camCmd {
    my ($dev,$mode,$cmd,$param,$options)=@_;
    my %options = %$options;
    my @cmd = ( $STX , $conf{$dev}{PTZID}); # camera hardware ID

    my $qptCamID=2;

#    $log->debug("camCmd: DEVID=$dev mode: $mode command: $cmd optional parameter: $param");

    # here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
    if($conf{$dev}{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
        if(not $polling and ($mode=~/speed/i || $mode=~/step/i || $mode=~/abs/i || $mode=~/rel/i || $mode=~/smooth/i)) {
#            $log->debug("Setting timeout +$conf{$dev}{PTZ_PRESET1TIMEOUT}");
            $conf{$dev}{TIMEOUT} = time + $conf{$dev}{PTZ_PRESET1TIMEOUT};
        }
        elsif(not $polling and ($mode=~/preset/i && $cmd=~/goto/i && $param!=1)) {
#            $log->debug("Setting timeout (preset) +$conf{$dev}{PTZ_PRESET1TIMEOUT}");
            $conf{$dev}{TIMEOUT} = time + $conf{$dev}{PTZ_PRESET1TIMEOUT};
        }
    }

    if(not $cmd=~/stop/) { # non-Stop
        ########################################################################
        if ($mode=~/speed/i) {        # mode speed
#            $log->debug("MODE->speed, cmd:$cmd  param: $param");
            $cmd[2]=0x31;
            if ($high_speed == 0) {
              $cmd[3]=0x10;
            }
            else {
              $cmd[3]=0x00;
            }

            if($cmd=~/PT/i) { # RPT
                my ($p,$t) = split(/,/,$param);
                $cmd[4]=PTspeed($dev,$p);
                $cmd[5]=PTspeed($dev,-$t);
                $cmd[6]=0;
                $cmd[7]=0;
                $cmd[8]=checkSum(8,@cmd);
                $cmd[9]=$ETX;
                cmdTransmit($dev,@cmd);
            }
            elsif($cmd=~/zoom/i) {
              camZoom($dev, 0, $param, $qptCamID);    
            }
            elsif($cmd=~/focus/i) {
              if ($param >= 0) {
                camFocus($dev, $param / 50, 1, $qptCamID);
              }
              else {
                camFocus($dev, -1 * $param / 50, 0, $qptCamID);
              }
            }
        }
        ########################################################################
        elsif($mode=~/rel/i) { # mode=rel: "<devid> rel size=640x480 xy=225,152"
            $param =~ /(\d+)x(\d+)/;
            my ($rel_size_x,$rel_size_y) = ($1,$2);
            $options{xy} =~ /(\d+),(\d+)/;
            my ($rel_click_x,$rel_click_y) = ($1,$2);
            my ($rel_p, $rel_t) = ( 220.01*($rel_click_x-$rel_size_x/2.01)/$rel_size_x, -220.01*($rel_click_y-$rel_size_y/2.01)/$rel_size_y );
#            $log->debug("REL [$rel_size_x] [$rel_size_y] [$rel_click_x] [$rel_click_y] [$rel_p] [$rel_t]");
            $cmd[2]=0x34;
            my $p_hex = substr(sprintf("%04X",int($rel_p)),-4);
            my $t_hex = substr(sprintf("%04X",int($rel_t)),-4);
#            $log->debug("REL [$rel_p] [$rel_t] {$p_hex} {$t_hex}");
            $p_hex =~/(\w\w)(\w\w)$/;
            ($cmd[3],$cmd[4]) = (hex($2),hex($1));
            $t_hex =~/^(\w\w)(\w\w)$/;
            ($cmd[5],$cmd[6]) = (hex($2),hex($1));
            $cmd[7]=checkSum(7,@cmd);
            $cmd[8]=$ETX;
            cmdTransmit($dev,@cmd);
        }
        ########################################################################
        elsif ($mode=~/settings/i){ # settings
#            $log->debug('settings',$cmd);
            if($cmd=~/temporal/i) {
              $cmd[2]=0x62; 
              if ($qptCamID == 1) {
                $cmd[3]=0x1D;  # 0001 1101
              }
              else {
                $cmd[3]=0x9D;  # 1001 1101
              }
              $cmd[4]=0x40;
              $cmd[5]=0x57;
              $cmd[6]=0x28;
              $cmd[7]=0x54;
              $cmd[8]=0x45;
              $cmd[9]=0x4d;
              $cmd[10]=0x50;
              $cmd[11]=0x4f;
              $cmd[12]=0x52;
              $cmd[13]=0x41;
              $cmd[14]=0x4c;
              $cmd[15]=0x5f;
              $cmd[16]=0x46;
              $cmd[17]=0x49;
              $cmd[18]=0x4c;
              $cmd[19]=0x54;
              $cmd[20]=0x45;
              $cmd[21]=0x52;
              $cmd[22]=0x5f;
              $cmd[23]=0x53;
              $cmd[24]=0x54;
              $cmd[25]=0x41;
              $cmd[26]=0x54;
              $cmd[27]=0x45;
              $cmd[28]=0x29;
              $cmd[29]=0x3d;
              $cmd[30]=0x24;

              if ($param=~/on/i) {
                #@W(TEMPORAL_FILTER_STATE)=$1
                $cmd[31]=0x31;
              }
              else {
                #@W(TEMPORAL_FILTER_STATE)=$0
                $cmd[31]=0x30;
              }
              $cmd[32]=0x0D;
              $cmd[33]=checkSum(33,@cmd);
              $cmd[34]=$ETX;
              cmdTransmit($dev,@cmd);
            }
            elsif($cmd=~/nuc/i) {
              if ($param=~/internal/i) {
                $cmd[2]=0x62; 
                if ($qptCamID == 1) {
                  $cmd[3]=0x11;  # 0001 0001
                }
                else {
                  $cmd[3]=0x91;  # 1001 0001
                }
                $cmd[4]=0x40;
                $cmd[5]=0x43;
                $cmd[6]=0x28;
                $cmd[7]=0x43;
                $cmd[8]=0x4f;
                $cmd[9]=0x4d;
                $cmd[10]=0x50;
                $cmd[11]=0x4c;
                $cmd[12]=0x45;
                $cmd[13]=0x54;
                $cmd[14]=0x45;
                $cmd[15]=0x5f;
                $cmd[16]=0x4e;
                $cmd[17]=0x55;
                $cmd[18]=0x43;
                $cmd[19]=0x29;
                $cmd[20]=0x0D;
                #@C(COMPLETE_NUC)
                $cmd[21]=checkSum(21,@cmd);
                $cmd[22]=$ETX;
                cmdTransmit($dev,@cmd);
              }
              else {
                #@C(NUC)
                $cmd[2]=0x62; 
                if ($qptCamID == 1) {
                  $cmd[3]=0x08;  # 0000 1000
                }
                else {
                  $cmd[3]=0x88;  # 1000 1000
                }
                $cmd[4]=0x40;
                $cmd[5]=0x43;
                $cmd[6]=0x28;
                $cmd[7]=0x4E;
                $cmd[8]=0x55;
                $cmd[9]=0x43;
                $cmd[10]=0x29;
                $cmd[11]=0x0D;
                $cmd[12]=checkSum(12,@cmd);
                $cmd[13]=$ETX;
                cmdTransmit($dev,@cmd);
              }
            }
            elsif($cmd=~/level/i) {
              if ($param=~/manual/i) {
                $cmd[2]=0x62; 
                if ($qptCamID == 1) {
                  $cmd[3]=0x17;  # 0001 0111
                }
                else {
                  $cmd[3]=0x97;  # 1001 0111
                }
                $cmd[4]=0x40;
                $cmd[5]=0x43;
                $cmd[6]=0x28;
                $cmd[7]=0x41;
                $cmd[8]=0x55;
                $cmd[9]=0x54;
                $cmd[10]=0x4f;
                $cmd[11]=0x5f;
                $cmd[12]=0x4c;
                $cmd[13]=0x45;
                $cmd[14]=0x56;
                $cmd[15]=0x45;
                $cmd[16]=0x4c;
                $cmd[17]=0x5f;
                $cmd[18]=0x44;
                $cmd[19]=0x49;
                $cmd[20]=0x53;
                $cmd[21]=0x41;
                $cmd[22]=0x42;
                $cmd[23]=0x4c;
                $cmd[24]=0x45;
                $cmd[25]=0x29;
                #@C(AUTO_LEVEL_DISABLE)

                $cmd[26]=0x0D;
                $cmd[27]=checkSum(27,@cmd);
                $cmd[28]=$ETX;
                cmdTransmit($dev,@cmd);
              }
              else {
                #@C(AUTO_LEVEL_ENABLE)
                $cmd[2]=0x62; 
                if ($qptCamID == 1) {
                  $cmd[3]=0x16;  # 0001 0110
                }
                else {
                  $cmd[3]=0x96;  # 1001 0110
                }
                $cmd[4]=0x40;
                $cmd[5]=0x43;
                $cmd[6]=0x28;
                $cmd[7]=0x41;
                $cmd[8]=0x55;
                $cmd[9]=0x54;
                $cmd[10]=0x4f;
                $cmd[11]=0x5f;
                $cmd[12]=0x4c;
                $cmd[13]=0x45;
                $cmd[14]=0x56;
                $cmd[15]=0x45;
                $cmd[16]=0x4c;
                $cmd[17]=0x5f;
                $cmd[18]=0x45;
                $cmd[19]=0x4e;
                $cmd[20]=0x41;
                $cmd[21]=0x42;
                $cmd[22]=0x4c;
                $cmd[23]=0x45;
                $cmd[24]=0x29;
                $cmd[25]=0x0D;
                $cmd[26]=checkSum(26,@cmd);
                $cmd[27]=$ETX;
                cmdTransmit($dev,@cmd);
              }
            } 
            elsif($cmd=~/gain/i) {
              if ($param=~/manual/i) {
                #@C(AUTO_GAIN_DISABLE)
                $cmd[2]=0x62; 
                if ($qptCamID == 1) {
                  $cmd[3]=0x16;  # 0001 0110
                }
                else {
                  $cmd[3]=0x96;  # 1001 0110
                }
                $cmd[4]=0x40;
                $cmd[5]=0x43;
                $cmd[6]=0x28;
                $cmd[7]=0x41;
                $cmd[8]=0x55;
                $cmd[9]=0x54;
                $cmd[10]=0x4f;
                $cmd[11]=0x5f;
                $cmd[12]=0x47;
                $cmd[13]=0x41;
                $cmd[14]=0x49;
                $cmd[15]=0x4e;
                $cmd[16]=0x5f;
                $cmd[17]=0x44;
                $cmd[18]=0x49;
                $cmd[19]=0x53;
                $cmd[20]=0x41;
                $cmd[21]=0x42;
                $cmd[22]=0x4c;
                $cmd[23]=0x45;
                $cmd[24]=0x29;
                $cmd[25]=0x0D;
                $cmd[26]=checkSum(26,@cmd);
                $cmd[27]=$ETX;
                cmdTransmit($dev,@cmd);
              }
              else {
                #@C(AUTO_GAIN_ENABLE)
                $cmd[2]=0x62; 
                if ($qptCamID == 1) {
                  $cmd[3]=0x15;  # 0001 0101
                }
                else {
                  $cmd[3]=0x95;  # 1001 0101
                }
                $cmd[4]=0x40;
                $cmd[5]=0x43;
                $cmd[6]=0x28;
                $cmd[7]=0x41;
                $cmd[8]=0x55;
                $cmd[9]=0x54;
                $cmd[10]=0x4f;
                $cmd[11]=0x5f;
                $cmd[12]=0x47;
                $cmd[13]=0x41;
                $cmd[14]=0x49;
                $cmd[15]=0x4e;
                $cmd[16]=0x5f;
                $cmd[17]=0x45;
                $cmd[18]=0x4e;
                $cmd[19]=0x41;
                $cmd[20]=0x42;
                $cmd[21]=0x4c;
                $cmd[22]=0x45;
                $cmd[23]=0x29;
                $cmd[24]=0x0D;
                $cmd[25]=checkSum(25,@cmd);
                $cmd[26]=$ETX;
                cmdTransmit($dev,@cmd);
              }
            } 
            elsif($cmd=~/aspan/i) {
              if ($param=~/lin/i) {
                $cmd[2]=0x62; 
                if ($qptCamID == 1) {
                  $cmd[3]=0x1C;  # 0001 1101
                }
                else {
                  $cmd[3]=0x9C;  # 1001 1101
                }

                my $idx=4;
              
                foreach my $c (split(//,'@W(AUTO_SPAN_LINEARITY)=0064')) {
                  $cmd[$idx]=hex(sprintf("%x", ord($c)));
                  $idx = $idx + 1;
                }

                $cmd[$idx]=0x0D; $idx = $idx + 1;
                $cmd[$idx]=checkSum($idx,@cmd); $idx = $idx + 1;
                $cmd[$idx]=$ETX;
                cmdTransmit($dev,@cmd);
              }
              else {
                $cmd[2]=0x62; 
                if ($qptCamID == 1) {
                  $cmd[3]=0x1C;  # 0001 1101
                }
                else {
                  $cmd[3]=0x9C;  # 1001 1101
                }

                my $idx=4;
              
                foreach my $c (split(//,'@W(AUTO_SPAN_LINEARITY)=0019')) {
                  $cmd[$idx]=hex(sprintf("%x", ord($c)));
                  $idx = $idx + 1;
                }

                $cmd[$idx]=0x0D; $idx = $idx + 1;
                $cmd[$idx]=checkSum($idx,@cmd); $idx = $idx + 1;
                $cmd[$idx]=$ETX;
                cmdTransmit($dev,@cmd);
              }
            } 
            elsif($cmd=~/fov/i) {
              if ($param=~/narrow/i) {
                camFOV($dev, 1, $qptCamID);
              }
              else {
                camFOV($dev, 0, $qptCamID);
              }
            }
            elsif($cmd=~/palette/i) {
                $cmd[2]=0x62; 
              if ($qptCamID == 1) {
                $cmd[3]=0x15; # 8D 0001 0101 
              }
              else {
                $cmd[3]=0x95; # 8D 1001 0101
              }
              $cmd[4]=0x40;
              $cmd[5]=0x57;
              $cmd[6]=0x28;
              $cmd[7]=0x56;
              $cmd[8]=0x49;
              $cmd[9]=0x44;
              $cmd[10]=0x45;
              $cmd[11]=0x4f;
              $cmd[12]=0x5f;
              $cmd[13]=0x50;
              $cmd[14]=0x41;
              $cmd[15]=0x4c;
              $cmd[16]=0x45;
              $cmd[17]=0x54;
              $cmd[18]=0x54;
              $cmd[19]=0x45;
              $cmd[20]=0x29;
              $cmd[21]=0x3d;
              $cmd[22]=0x24;
              $cmd[24]=0x0D; # <CR>
              if ($param=~/gray/i) {
                # Gray @W(VIDEO_PALETTE)=$0
                $cmd[23]=0x30;
              }
              elsif ($param=~/iron/i) {
                # Iron @W(VIDEO_PALETTE)=$2
                $cmd[23]=0x32;
              }
              elsif ($param=~/rainbow/i) {
              # Ramp @W(VIDEO_PALETTE)=$4
                $cmd[23]=0x33;
              }

              $cmd[25]=checkSum(25,@cmd); 
              $cmd[26]=$ETX;
              cmdTransmit($dev,@cmd);
            }
            elsif($cmd=~/ptspeed/i) {
               $high_speed=1 if $param=~/fast/i;
               $high_speed=0 if $param=~/slow/i;
            }
            elsif($cmd=~/polarity/i) {
              if ($param=~/on/i) {
                $cmd[2]=0x62; # pass-through to FLIR
                if ($qptCamID == 1) {
                  $cmd[3]=0x0F; # Size of the packet
                }
                else {
                  $cmd[3]=0x8F; # Size of the packet
                }
                $cmd[4]=0x40;
                $cmd[5]=0x57;
                $cmd[6]=0x28;
                $cmd[7]=0x50;
                $cmd[8]=0x4F;
                $cmd[9]=0x4C;
                $cmd[10]=0x41;
                $cmd[11]=0x52;
                $cmd[12]=0x49;
                $cmd[13]=0x54;
                $cmd[14]=0x59;
                $cmd[15]=0x29;
                $cmd[16]=0x3D;
                $cmd[17]=0x31;
                $cmd[18]=0x0D;
                $cmd[19]=checkSum(19,@cmd);
                $cmd[20]=$ETX;
                cmdTransmit($dev,@cmd);
              }
              else {
                $cmd[2]=0x62; # pass-through to FLIR
                if ($qptCamID == 1) {
                  $cmd[3]=0x0F; # Size of the packet
                }
                else {
                  $cmd[3]=0x8F; # Size of the packet
                }
                $cmd[4]=0x40;
                $cmd[5]=0x57;
                $cmd[6]=0x28;
                $cmd[7]=0x50;
                $cmd[8]=0x4F;
                $cmd[9]=0x4C;
                $cmd[10]=0x41;
                $cmd[11]=0x52;
                $cmd[12]=0x49;
                $cmd[13]=0x54;
                $cmd[14]=0x59;
                $cmd[15]=0x29;
                $cmd[16]=0x3D;
                $cmd[17]=0x30;
                $cmd[18]=0x0D;
                $cmd[19]=checkSum(19,@cmd);
                $cmd[20]=$ETX;
                cmdTransmit($dev,@cmd);
              }
            }
            elsif($cmd=~/focus/i) {
              if ($param=~/near/i) {
                $cmd[2]=0x62; # pass-through to FLIR
                if ($qptCamID == 1) {
                  $cmd[3]=0x17; # Size of the packet (0001 1000)
                } 
                else {
                  $cmd[3]=0x97; # Size of the packet (1001 1000)
                }
                $cmd[4]=0x40;
                $cmd[5]=0x57;
                $cmd[6]=0x28;
                $cmd[7]=0x4D;
                $cmd[8]=0x49;
                $cmd[9]=0x4C;
                $cmd[10]=0x3A;
                $cmd[11]=0x46;
                $cmd[12]=0x4F;
                $cmd[13]=0x43;
                $cmd[14]=0x55;
                $cmd[15]=0x53;
                $cmd[16]=0x5F;
                $cmd[17]=0x4E;
                $cmd[18]=0x45;
                $cmd[19]=0x41;
                $cmd[20]=0x52;
                $cmd[21]=0x29;
                $cmd[22]=0x3D;
                $cmd[23]=0x38;
                $cmd[24]=0x30;
                $cmd[25]=0x30;
                $cmd[26]=0x0D;
                $cmd[27]=checkSum(27,@cmd);
                $cmd[28]=$ETX;
                cmdTransmit($dev,@cmd);
              }
              elsif ($param=~/far/i) {
                $cmd[2]=0x62; # pass-through to FLIR
                if ($qptCamID == 1) {
                  $cmd[3]=0x17; # Size of the packet
                }
                else {
                  $cmd[3]=0x97; # Size of the packet
                }
                $cmd[4]=0x40;
                $cmd[5]=0x57;
                $cmd[6]=0x28;
                $cmd[7]=0x4D;
                $cmd[8]=0x49;
                $cmd[9]=0x4C;
                $cmd[10]=0x3A;
                $cmd[11]=0x46;
                $cmd[12]=0x4F;
                $cmd[13]=0x43;
                $cmd[14]=0x55;
                $cmd[15]=0x53;
                $cmd[16]=0x5F;
                $cmd[17]=0x4E;
                $cmd[18]=0x45;
                $cmd[19]=0x41;
                $cmd[20]=0x52;
                $cmd[21]=0x29;
                $cmd[22]=0x3D;
                $cmd[23]=0x38;
                $cmd[24]=0x30;
                $cmd[25]=0x30;
                $cmd[26]=0x0D;
                $cmd[27]=checkSum(27,@cmd);
                $cmd[28]=$ETX;
                cmdTransmit($dev,@cmd);
              }
              elsif ($param=~/stop/i) {
                $cmd[2]=0x62; # pass-through to FLIR
                if ($qptCamID == 1) {
                  $cmd[3]=0x0F; # Size of the packet
                }
                else { 
                  $cmd[3]=0x8F; # Size of the packet
                }
                $cmd[4]=0x40;
                $cmd[5]=0x43;
                $cmd[6]=0x28;
              $cmd[7]=0x46;
                $cmd[8]=0x4F;
                $cmd[9]=0x43;
                $cmd[10]=0x55;
                $cmd[11]=0x53;
                $cmd[12]=0x5F;
                $cmd[13]=0x53;
                $cmd[14]=0x54;
                $cmd[15]=0x4F;
                $cmd[16]=0x50;
                $cmd[17]=0x29;
                $cmd[18]=0x0D;
                $cmd[19]=checkSum(19,@cmd);
                $cmd[20]=$ETX;
                cmdTransmit($dev,@cmd);
              }
            }
            elsif($cmd=~/reset/i) {
                $cmd[2]=0x31;
                $cmd[3]=0x01;
                $cmd[4]=0;
                $cmd[5]=0;
                $cmd[6]=0;
                $cmd[7]=0;
                $cmd[8]=checkSum(8,@cmd);
                $cmd[9]=$ETX;
#                $log->debug("SETTINGS/RESET");
                cmdTransmit($dev,@cmd);
            }
            elsif($cmd=~/init/i) {
                $cmd[2]=0x62; # pass-through to FLIR
                if ($qptCamID == 1) {
                  $cmd[3]=0x08; # Size of the packet
                }
                else { 
                  $cmd[3]=0x88; # Size of the packet
                }
                $cmd[4]=0x52;
                $cmd[5]=0x45;
                $cmd[6]=0x53;
                $cmd[7]=0x50;
                $cmd[8]=0x4f;
                $cmd[9]=0x4e;
                $cmd[10]=0x44;
                $cmd[11]=0x0D;
                $cmd[12]=checkSum(12,@cmd);
                $cmd[13]=$ETX;
                cmdTransmit($dev,@cmd);
            }
            elsif($cmd=~/sensitivity/i) {
              if ($param=~/low/i) {
                camSensitivity($dev, 0, $qptCamID);
              }
              elsif ($param=~/med/i) {
                camSensitivity($dev, 1, $qptCamID);
              }
              elsif ($param=~/high/i) {
                camSensitivity($dev, 2, $qptCamID);
              }
            }
        } 
        ########################################################################
        elsif($mode=~/step/i){       # mode=step  /Step by step positioning/
            if($cmd=~/move/i) { # step pan/tilt
                $cmd[2]=0x34; $cmd[3]=$cmd[4]=$cmd[5]=$cmd[6]=0;
                if ( $param=~/right/i ) {
                    $cmd[3]=0x60;
                }
                elsif ( $param=~/left/i ) {
                    $cmd[3]=0x80; $cmd[4]=0xFF;
                }
                if ( $param=~/down/i ) {
                    $cmd[5]=0xBF; $cmd[6]=0xFF;
                }
                elsif ( $param=~/up/i ) {
                    $cmd[5]=0x40;
                }
                $cmd[7]=checkSum(7,@cmd); $cmd[8]=$ETX;
                cmdTransmit($dev,@cmd);
                $log->debug("STEP/MOVE:$param");
            } 
            elsif ($cmd=~/focus/i){
                if ($param=~/up/i){    
                  camFocus($dev, 2, 1, $qptCamID);
                  camFocus($dev, 0, 1, $qptCamID);
                }
                else {
                  camFocus($dev, 2, 0, $qptCamID);
                  camFocus($dev, 0, 0, $qptCamID);
		}
            } 
            elsif ($cmd=~/zoom/i){
                if ($param=~/in/i){    
                  camZoom($dev, 1, 0, $qptCamID);
                }
                else {
                  camZoom($dev, -1, 0, $qptCamID);
		}
            }
        }
        ########################################################################
        elsif($mode=~/hardware/i){   # mode=hardware  /Hardware reset,defaults/
            $cmd[2]=0x35;
            $cmd[3]=checkSum(3,@cmd); 
            $cmd[4]=$ETX;
            cmdTransmit($dev,@cmd);

#            if($cmd=~/do/i && $param=~/defaults/i) { # init
#                camReset($dev);
#                return;
#            }
        } 
        ########################################################################
        elsif($mode=~/abs/i){                # mode=abs  /Absloute positioning/
            if($cmd=~/center/i) { # Center
                $cmd[2]=0x35; 
                $cmd[3]=checkSum(3,@cmd); 
                $cmd[4]=$ETX;
                cmdTransmit($dev,@cmd);
            } 
            elsif($cmd=~/pt/i){            # Pan, Tilt
                $log->debug("ABS:  PT $param");
                $cmd[2]=0x33;
                $param=~/^(\-?.+)?\,(\-?.+)$/;
                my $p = int($1*10);
                my $t = int($2*10);
                $log->debug("ABS! pt=$1,$2 ($p,$t)");
                #$p += 36000 if $p<0;
                #$t += 36000 if $t<0;
                my $p_hex = sprintf("%04X",$p);
                my $t_hex = sprintf("%04X",$t);
                $log->debug("==== P_HEX= $p_hex");
                $log->debug("==== T_HEX= $t_hex");
                $p_hex =~/(\w\w)(\w\w)$/;
                ($cmd[3],$cmd[4]) = (hex($2),hex($1));
                $t_hex =~/(\w\w)(\w\w)$/;
                ($cmd[5],$cmd[6]) = (hex($2),hex($1));
                $cmd[7]=checkSum(7,@cmd);
                $cmd[8]=$ETX;
                cmdTransmit($dev,@cmd);
            } 
            elsif ($cmd=~/Z/i){	# Absolute zoom
                # Automatically choose the appropriate field of view
                my $angle = $1;
                $log->debug("ZOOM angle $angle degrees");
                return;
                my $narrow_fov = 2.2;
                my $wide_fov = 11.0;
                if ($angle < $narrow_fov) { # Narrow field of view
                  $cmd[2]=0x62; # pass-through: see PTCR20 protocol page 17
                  if ($qptCamID == 1) {
                    $cmd[3]=0x0F; # length: 15 bytes 
                  }
                  else {
                    $cmd[3]=0x8F; # length: 15 bytes 
                  }
       
                  # @C(FOV_NARROW)<cr>
                    $cmd[4]=0x40;
                    $cmd[5]=0x43;
                    $cmd[6]=0x28;
                    $cmd[7]=0x46;
                    $cmd[8]=0x4f;
                    $cmd[9]=0x56;
                    $cmd[10]=0x5f;
                    $cmd[11]=0x4e;
                    $cmd[12]=0x41;
                    $cmd[13]=0x52;
                    $cmd[14]=0x52;
                    $cmd[15]=0x4f;
                    $cmd[17]=0x57;
                    $cmd[18]=0x29;
                    $cmd[19]=0x0D;
                  $cmd[20]=checkSum(20,@cmd);
                  $cmd[21]=$ETX;        
                  cmdTransmit($dev,@cmd);

                  if ($angle < ($narrow_fov / 2)) { # 2x
                    $log->debug("narrow, 2x digital zoom");
                  }
                  elsif ($angle < ($narrow_fov / 4)) { # 4x
                    $log->debug("narrow, 4x digital zoom");
                  }
                  else { #1x
                    $log->debug("narrow, 1x digital zoom");
                  }
                }
                else { # Wide field of view
                  $log->debug("Wide");
                  # @C(FOV_WIDE)<CR>
                  $cmd[2]=0x62; # pass-through: see PTCR20 protocol page 17
                  if ($qptCamID == 1) {
                    $cmd[3]=0x0D; # length: 13 bytes 
                  }
                  else {
                    $cmd[3]=0x8D; # length: 13 bytes 
                  }
                    $cmd[4]=0x40;
                    $cmd[5]=0x43;
                    $cmd[6]=0x28;
                    $cmd[7]=0x46;
                    $cmd[8]=0x4f;
                    $cmd[9]=0x56;
                    $cmd[10]=0x5f;
                    $cmd[11]=0x57;
                    $cmd[12]=0x49;
                    $cmd[13]=0x44;
                    $cmd[14]=0x45;
                    $cmd[15]=0x29;
                    $cmd[16]=0x0D;
                  $cmd[17]=checkSum(17,@cmd);
                  $cmd[18]=$ETX;        
                  cmdTransmit($dev,@cmd);

                  if ($angle < ($wide_fov / 2)) { # 2x
                    $log->debug("wide, 2x digital zoom");
                  }
                  elsif ($angle < ($wide_fov / 4)) { #4x
                    $log->debug("wide, 4x digital zoom");
                  }
                  else { #1x
                    $log->debug("wide, 1x digital zoom");
                  }
                }
                return;

                $cmd[2]=0x62; # pass-through: see PTCR20 protocol page 17
                if ($qptCamID == 1) {
                  $cmd[3]=0x0A; 
                }
                else {
                  $cmd[3]=0x8A; 
                }
                    $cmd[4]=0x38;
                    $cmd[5]=0x30;
                    $cmd[6]=0x30;
                    $cmd[7]=0x31 if $param=~/up/i;
                    $cmd[7]=0x32 if $param=~/down/i;
                    $cmd[8]=0x0D; 
                    $cmd[9]=0x0A; 
                    $cmd[10]=0x56; 
                    $cmd[11]=0x53; 
                    $cmd[12]=0x50; 
                    $cmd[13]=0x3E;
                $cmd[14]=checkSum(14,@cmd);
                $cmd[15]=$ETX;        
                cmdTransmit($dev,@cmd);
            }
        } 
        ########################################################################
        elsif($mode=~/preset/i){ # presets
            $cmd[3]= hex($param);
            if ( $cmd =~/save/i ) {
                $cmd[2]= 0x42;
            }
            if ( $cmd =~/clear/i ) {
                # Nothing to do
                return;
            }
            if ( $cmd =~/goto/i ) {
                $cmd[2]= 0x32;
            }
            $cmd[4]=checkSum(4,@cmd);
            $cmd[5]=$ETX;
            cmdTransmit($dev,@cmd);
        } 
        ########################################################################
        elsif($mode=~/smooth/i) { # Right/Left/Up/Down/Tele/Wide/focus_Near/focus_Far/iris_Open/irIs_close
            # set movement bits
            if ($cmd=~/move/i) { # pan/tilt
                # Nothing to do
                return;
            } elsif($cmd=~/zoom/i) { # zoom
                # Nothing to do
                return;
            } elsif($cmd=~/focus/i) { # focus
                # Nothing to do
                return;
            } elsif($cmd=~/iris/i) { # iris
                # Nothing to do
                return;
            }
            return;
        } # modes
    }
}


sub nonblock {
    my ($fd) = @_;
    my $flags = fcntl($fd, F_GETFL,0);
    fcntl($fd, F_SETFL, $flags|O_NONBLOCK);
}

sub ReconnectSocket {
    my $dev = shift;
    
    if(not defined($sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}})) {
        $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK} = IO::Socket::INET->new(PeerAddr => $conf{$dev}{DEVIP},
                                                PeerPort => $conf{$dev}{PTZ_TCP_PORT},
                                                Proto    => "tcp",
                                                Type     => SOCK_STREAM);
        if(! $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK} ) {
            #$log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
            delete $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}};
            next;
        }
        $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK}->autoflush(1);
        nonblock($sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK});
        $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK});
    }
}


# --------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {

    $log->debug('load_dev_conf');
    
    # fisrst, close everything
    foreach my $skt (keys %sock) {
        close($sock{$skt}{SOCK});
        delete $sock{$skt};
    }


    %conf = GetCfgs( eval("($query_str)") );     # Load configurations
    $log->debug("Config read as:");
    $log->debug('-------------------------------------');
    foreach my $dev (keys %conf) {
        $log->debug("[$dev]");
        $log->debug("DEVID=$conf{$dev}{DEVID}");
        $log->debug("CAMERAMODEL=$conf{$dev}{CAMERAMODEL}");
        $log->debug("CAMERA NUMBER=$conf{$dev}{CAMERA}");
        $log->debug("POSITIONCTL=$conf{$dev}{POSITIONCTL}");
        $log->debug("PTZID=$conf{$dev}{PTZID}");
        $log->debug("PTZ_TCP_PORT=$conf{$dev}{PTZ_TCP_PORT}");
        $log->debug("-------------------------------------");
        ReconnectSocket($dev);
        $log->debug("PTZ_TCP_PORT=$conf{$dev}{PTZ_TCP_PORT}");
    } # foreach $dev
 # close unused ports

}

