#!/usr/bin/perl
#  $Id: ptz_ptcr20bae.pl 25370 2012-03-09 21:19:34Z teetov $
# -----------------------------------------------------------------------------
#  PTZ driver for QuickSet + BAE
# -----------------------------------------------------------------------------
#  Author: Andrey Fomenko
#  Edited by: 
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT LLC, 2004-2005
# -----------------------------------------------------------------------------

use strict;

use POSIX;
use IO::Socket;
use IO::Select;
use Tie::RefHash;
use IO::File;
use Socket;
use Fcntl;
use Tie::RefHash;
use NextCAM::Init;

use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

my $log=get_logger('NEXTCAM::PTZ::PTCR20BAE');

my $query_str = shift || "'POSITIONCTL'=>'ptcr20bae','CAMERAMODEL'=>'axis'";
$log->info("Starting PTCR20BAE PTZ DRIVER, query string: [$query_str]");


$SIG{HUP}=\&load_dev_conf;

my $APL=$ENV{APL};
my (%conf, %sock, $cam,$cmd,$par,$usrpsw,$last_cmd,$last_mode,$cmd_executed);
my $ETX = 0x03;
my $STX = 0x02;

load_dev_conf();

# -----------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# -----------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
                                   PeerPort => $TCP_PORT,
                                   Proto     => "tcp",
                                   Type      => SOCK_STREAM)
    or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';

my %commands;
my $ready;
my $trycount = 30;
my %lastcmd;
my $lastcmdtime = time;
my $polling = 0;

while(1) {

	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/i;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/
				|| $commands{$cam} =~ /speed iris=0/
				|| $commands{$cam} =~ /speed gain=0/
				)
			{
				; # do not override stop command!
			}else{
				$commands{$cam} = $cmd;
			}
		} else {
			$commands{$cam} = $cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)
	
	$cmd = '';
	
	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		delete $commands{$camera};
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
		$usrpsw = "$conf{$cam}{USRNAME}:$conf{$cam}{PASSWD}" if $conf{$cam}{USRNAME} && $conf{$cam}{PASSWD};
		
		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam,$mode,"pt", $params[0].','.$params[1] ,\%options);
			camCmd($cam,$mode,"zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam,$mode,$cmd,$par,\%options);
		}
		
		$lastcmd{$camera}{CMD} = $cmd;
		$lastcmd{$camera}{MODE} = $mode;
		$lastcmd{$camera}{PAR} = $par;
		$lastcmd{$camera}{OPTS} = \%options;
		$polling = 0;
	}
	
	select(undef,undef,undef,.1) if not $cmd;

	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
		$lastcmd{$dev}{CMD} = 'goto';
		$lastcmd{$dev}{MODE} = 'preset';
		$lastcmd{$dev}{PAR} = 1;
		$lastcmd{$dev}{OPTS} = \%options;
		$polling = 0;
	}
	
	if(!$cmd) {
		foreach my $dev (keys %conf) {
			my %options = {};
			$polling = 1;
			if(defined($lastcmd{$dev})) {
				if($lastcmd{$dev}{MODE} eq 'speed') {
					if($lastcmd{$dev}{CMD} eq 'gain' and $lastcmd{$dev}{PAR} == 0) {
						camCmd($dev,'speed','pt','0,0',\%options);
					}
					else {
						camCmd($dev,$lastcmd{$dev}{MODE},$lastcmd{$dev}{CMD},$lastcmd{$dev}{PAR},$lastcmd{$dev}{OPTS});
					}
				}
				else {
					camCmd($dev,'speed','pt','0,0',\%options);
				}
				select(undef,undef,undef,.2);
			}
			else {
				camCmd($dev,'speed','pt','0,0',\%options);
				select(undef,undef,undef,.1);
			}
		}        
	}

} # while(1)


# ----------------------------------------------------------- cmdTransmit -----
sub cmdTransmit {
	my ($dev,@cmd)=@_;
	
	@cmd = checkEscape(@cmd);
	
	my $command='';
	my $b;
	foreach $b (@cmd) { $command .= sprintf("%02X ",$b); }
	$log->debug('cmdTransmit( ',$dev, '=> ',$command,')');    
	
	if(not defined $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK}) {
		$log->debug('Error: socket connection is broken');
		$sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK} = IO::Socket::INET->new(
		                                        PeerAddr => $conf{$dev}{DEVIP},
		                                        PeerPort => $conf{$dev}{PTZ_TCP_PORT},
		                                        Proto    => "tcp",
		                                        Type     => SOCK_STREAM);
		if(! $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK} ) {
			$log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
			delete $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}};
			return;
		}
		$sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK}->autoflush(1);
		nonblock($sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK});
		$sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK});
		return;
	}

	my $bts = '';
	foreach my $b (@cmd) { $bts .= chr($b); }
	my $skt = $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK};
	eval {
		local $SIG{PIPE} = sub { die "Error writing to socket" };
		print $skt $bts;
	};
	$log->error($@) if $@;
	select(undef,undef,undef,.15);
	foreach my $sss ( $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SEL}->can_read(1) ) {
		my $data = '';
		eval {
			alarm 1;
			$sss->recv($data, POSIX::BUFSIZ, 0);
			$command='';
			for($b=0;$b<length($data);$b++) { $command .= sprintf("%02X ",ord(substr($data,$b,1))); }
			$log->debug('Received:',$command);
		};
		alarm 0;
		return $data;
	}  # foreach      

} # sub cmdTransmit

# -------------------------------------------------------------- CheckSum -----
sub checkSum
{
	my ($len,@arr) = @_;
	my $LRC = 0;
	for ( my $i = 2; $i < $len; $i++) {
		$LRC ^= $arr[$i];
	}
	return $LRC % 256;
	#return ($arr[1]+$arr[2]+$arr[3]+$arr[4]+$arr[5]) % 256;
} # sub checkSum

# ----------------------------------------------------------- checkEscape -----
sub checkEscape
{
	my @arr = @_;
	my @res = ($arr[0]);
	my $r = 1;
	for ( my $i = 1; $i <= $#arr; $i++) {
		if ( $#arr - 1 > $i && ( $arr[$i] == $STX || $arr[$i] == $ETX || $arr[$i] == 0x06 || $arr[$i] == 0x15 || $arr[$i] == 0x1B ) ) {
			$res[$r++] = 0x1B;
			$res[$r++] = $arr[$i] | 0x80;
		} else {
			$res[$r++] = $arr[$i];
		}
	}
	return @res;
} # sub checkEscape

# --------------------------------------------------------------- camStop -----
sub camStop
{
	my ($dev) = @_;
	my %options = {};
	camCmd($dev,'speed','pt','0,0',\%options);
}

# -------------------------------------------------------------- camReset -----
sub camReset
{
	my ($dev) = @_;
	my @cmd = ();
	
	camStop($dev); # Stop motion, if exist

	$cmd[0] = $STX;
	$cmd[1]=$conf{$dev}{PTZID};
	
	# Move to Absolute 0/0
	$cmd[2]=0x35; $cmd[3]=checkSum(3,@cmd); $cmd[4]=$ETX;
	cmdTransmit($dev,@cmd);
	# Auto Gain&Level - on
	@cmd = @cmd[0..1];
	$cmd[2]=0x62; $cmd[3]=2; $cmd[4]=0x80; $cmd[5]=0x04; 
	$cmd[6]=checkSum(6,@cmd); $cmd[7]=$ETX;
	cmdTransmit($dev,@cmd);
	# Auto Shutter - on
	@cmd = @cmd[0..1];
	$cmd[2]=0x62; $cmd[3]=2; $cmd[4]=0x80; $cmd[5]=0xF1; 
	$cmd[6]=checkSum(6,@cmd); $cmd[7]=$ETX;
	cmdTransmit($dev,@cmd);
	# Histogram box - off
	@cmd = @cmd[0..1];
	$cmd[2]=0x62; $cmd[3]=2; $cmd[4]=0x80; $cmd[5]=0x95; 
	$cmd[6]=checkSum(6,@cmd); $cmd[7]=$ETX;
	cmdTransmit($dev,@cmd);

} # sub camReset

# ------------------------------------------------------------- camFZspeed -----
sub camFZspeed
{
	# For +12VDC Bipolar Drive. Need to add to ini-file config param for this
	my ($dev,$FZ,$speed) = @_;
	$log->debug("camFZspeed($dev,$FZ,$speed)");
	my @cmd = ( $STX, 0, 0x61, 0x48, 0, 0, 0 , $ETX);
	$cmd[1]=$conf{$dev}{PTZID};
	# Default Enable Zoom/Focus Function
	$cmd[3] |= 0x20 if $FZ eq 'Z' && $speed < 0;
	$cmd[3] |= 0x04 if $FZ eq 'F' && $speed < 0;
	$cmd[4]=PTSpeed($dev,$speed,$conf{$dev}{PTZSPEED});
	$cmd[5]=PTSpeed($dev,$speed,$conf{$dev}{PTZSPEED});
	$cmd[6]=checkSum(6,@cmd);
	cmdTransmit($dev,@cmd);
} # sub camReset

# ---------------------------------------------------------------- PTspeed ----
sub PTspeed {
	my ($dev,$speed,$ptzspeed)=@_;
	$speed = int( 1.27 * $speed * $ptzspeed / 100.0 );
	if($speed <= 0 ) {
		$speed = -127 if $speed < -127;
		$speed = -$speed * 2;
	} 
	else {
		$speed = 127 if $speed > 127;
		$speed = $speed * 2 + 1;
	}
	return $speed;
} # sub PTspeed
# ---------------------------------------------------------------- camCmd -----
sub camCmd {
	my ($dev,$mode,$cmd,$param,$options)=@_;
	my %options = %$options;
	my @cmd = ( $STX , $conf{$dev}{PTZID}); # camera hardware ID
	$log->debug("camCmd: DEVID=$dev mode: $mode command: $cmd optional parameter: $param");

	# here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
	if($conf{$dev}{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
		if(not $polling and ($mode=~/speed/i || $mode=~/step/i || $mode=~/abs/i || $mode=~/rel/i || $mode=~/smooth/i)) {
			$log->debug("Setting timeout +$conf{$dev}{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}{TIMEOUT} = time + $conf{$dev}{PTZ_PRESET1TIMEOUT};
		}
		elsif(not $polling and ($mode=~/preset/i && $cmd=~/goto/i && $param!=1)) {
			$log->debug("Setting timeout (preset) +$conf{$dev}{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}{TIMEOUT} = time + $conf{$dev}{PTZ_PRESET1TIMEOUT};
		}
	}

	if(not $cmd=~/stop/) { # non-Stop
		if ($mode=~/speed/i){        # mode speed
			$log->debug("MODE->speed, cmd:$cmd  param: $param");
			$cmd[2]=0x31; 
			$cmd[3]=0;
			if($cmd=~/PT/i) { # RPT
				my ($p,$t) = split(/,/,$param);
				$cmd[4]=PTspeed($dev,$p,$conf{$dev}{PTZSPEED});
				$cmd[5]=PTspeed($dev,-$t,$conf{$dev}{PTZSPEED});
				$cmd[6]=0;
				$cmd[7]=0;
				$cmd[8]=checkSum(8,@cmd);
				$cmd[9]=$ETX;
				$log->debug("RPT $p $t");
			}
			elsif($cmd=~/Z/i) {
				$cmd[4]=0;
				$cmd[5]=0;
				$cmd[6]=PTspeed($dev,$param,$conf{$dev}{PTZZOOMSPEED});
				$cmd[7]=0;
				$cmd[8]=checkSum(8,@cmd);
				$cmd[9]=$ETX;
				$log->debug("Z $param");
				return;
			}
			elsif($cmd=~/focus/i) {
				$log->debug("FOCUS $param");
				$cmd[4]=0;
				$cmd[5]=0;
				$cmd[6]=0;
				$cmd[7]=PTspeed($dev,$param,$conf{$dev}{PTZSPEED});
				$cmd[8]=checkSum(8,@cmd);
				$cmd[9]=$ETX;
			}
			elsif($cmd=~/gain/i) {
				#return if !$param;
				# 02 00 62 A 38 30 30 31 D A 56 53 50 3E D 03 => "+"
				# 02 00 62 A 38 30 30 32 D A 56 53 50 3E E 03
				$cmd[2]=0x62; 
				$cmd[3]=0x0A;
				$cmd[4]=0x38;
				$cmd[5]=0x30;
				$cmd[6]=0x30;
				$cmd[7]=0x31 if $param >= 0;
				$cmd[7]=0x32 if $param < 0;
				$cmd[8]=0x0D; 
				$cmd[9]=0x0A; 
				$cmd[10]=0x56; 
				$cmd[11]=0x53; 
				$cmd[12]=0x50; 
				$cmd[13]=0x3E;
				$cmd[14]=checkSum(14,@cmd);
				$cmd[15]=$ETX;
				$log->debug("GAIN $param");
			}
		}elsif($mode=~/rel/i) { # mode=rel: "<devid> rel size=640x480 xy=225,152"
			$param =~ /(\d+)x(\d+)/;
			my ($rel_size_x,$rel_size_y) = ($1,$2);
			$options{xy} =~ /(\d+),(\d+)/;
			my ($rel_click_x,$rel_click_y) = ($1,$2);
			my ($rel_p, $rel_t) = ( 220.01*($rel_click_x-$rel_size_x/2.01)/$rel_size_x, -220.01*($rel_click_y-$rel_size_y/2.01)/$rel_size_y );
			$log->debug("REL [$rel_size_x] [$rel_size_y] [$rel_click_x] [$rel_click_y] [$rel_p] [$rel_t]");
			$cmd[2]=0x34;
			my $p_hex = substr(sprintf("%04X",int($rel_p)),-4);
			my $t_hex = substr(sprintf("%04X",int($rel_t)),-4);
			$log->debug("REL [$rel_p] [$rel_t] {$p_hex} {$t_hex}");
			$p_hex =~/(\w\w)(\w\w)$/;
			($cmd[3],$cmd[4]) = (hex($2),hex($1));
			$t_hex =~/^(\w\w)(\w\w)$/;
			($cmd[5],$cmd[6]) = (hex($2),hex($1));
			$cmd[7]=checkSum(7,@cmd);
			$cmd[8]=$ETX;
			#cmdTransmit($dev,@cmd);
			#select(undef,undef,undef,.2);
			#camStop($dev);
			#return;
		}elsif ($mode=~/settings/i){ # settings
			$log->debug('settings',$cmd);
			if($cmd=~/autolevel/i) { 
				# Auto Level
				# off => 02 00 62 A 38 30 30 39 D A 56 53 50 3E 05 03
				# on =>  02 00 62 A 38 30 30 35 D A 56 53 50 3E 09 03 
				$cmd[2]=0x62; 
				$cmd[3]=0x0A;
				$cmd[4]=0x38;
				$cmd[5]=0x30;
				$cmd[6]=0x30;
				$cmd[7]=0x35 if $param=~/on/i;
				$cmd[7]=0x39 if $param=~/off/i;
				$cmd[8]=0x0D; 
				$cmd[9]=0x0A; 
				$cmd[10]=0x56; 
				$cmd[11]=0x53; 
				$cmd[12]=0x50; 
				$cmd[13]=0x3E;
				$cmd[14]=checkSum(14,@cmd);
				$cmd[15]=$ETX;        
				
				# send command and prepare "SAVE" command ( 0x8039 )
				cmdTransmit($dev,@cmd); 
				select(undef,undef,undef,.1);
				$cmd[6]=0x33;
				$cmd[7]=0x39;
				$cmd[14]=checkSum(14,@cmd);
			}
			elsif ($cmd=~/shutter/i){
				#  scene => 02 00 62 A 38 30 33 31 D A 56 53 50 3E E 03
				# manual => 02 00 62 A 38 30 33 30 D A 56 53 50 3E F 03
				$cmd[2]=0x62; 
				$cmd[3]=0x0A;
				$cmd[4]=0x38;
				$cmd[5]=0x30;
				$cmd[6]=0x33;
				$cmd[7]=0x31 if $param=~/scene/i;
				$cmd[7]=0x30 if $param=~/manual/i;
				$cmd[8]=0x0D; 
				$cmd[9]=0x0A; 
				$cmd[10]=0x56; 
				$cmd[11]=0x53; 
				$cmd[12]=0x50; 
				$cmd[13]=0x3E;
				$cmd[14]=checkSum(14,@cmd);
				$cmd[15]=$ETX;        
			}
			elsif ($cmd=~/polarity/i){
				# white => 02 00 62 A 38 30 39 61 D A 56 53 50 3E 54 03
				# black => 02 00 62 A 38 30 64 61 D A 56 53 50 3E 09 03
				$cmd[2]=0x62; 
				$cmd[3]=0x0A;
				$cmd[4]=0x38;
				$cmd[5]=0x30;
				$cmd[6]=0x39 if $param=~/white/i;
				$cmd[6]=0x64 if $param=~/black/i;
				$cmd[7]=0x61;
				$cmd[8]=0x0D; 
				$cmd[9]=0x0A; 
				$cmd[10]=0x56; 
				$cmd[11]=0x53; 
				$cmd[12]=0x50; 
				$cmd[13]=0x3E;
				$cmd[14]=checkSum(14,@cmd);
				$cmd[15]=$ETX;      
			}
			elsif($cmd=~/autogain/i) { 
				# Auto Gain
				# off => 02 00 62 A 38 30 30 31 D A 56 53 50 3E D 03
				#  on => 02 00 62 A 38 30 30 34 D A 56 53 50 3E 08 03
				$cmd[2]=0x62; 
				$cmd[3]=0x0A;
				$cmd[4]=0x38;
				$cmd[5]=0x30;
				$cmd[6]=0x30;
				$cmd[7]=0x34 if $param=~/on/i;
				$cmd[7]=0x31 if $param=~/off/i;
				$cmd[8]=0x0D; 
				$cmd[9]=0x0A; 
				$cmd[10]=0x56; 
				$cmd[11]=0x53; 
				$cmd[12]=0x50; 
				$cmd[13]=0x3E;
				$cmd[14]=checkSum(14,@cmd);
				$cmd[15]=$ETX;        
				
				# send command and prepare "SAVE" command ( 0x8039 )
				cmdTransmit($dev,@cmd); 
				select(undef,undef,undef,.1);
				$cmd[6]=0x33;
				$cmd[7]=0x39;
				$cmd[14]=checkSum(14,@cmd);
			}
			elsif($cmd=~/autoshutter/i) { 
				# AutoShutter
				# off => 02 00 62 A 38 30 62 31 D A 56 53 50 3E 5F 03
				#  on => 02 00 62 A 38 30 66 31 D A 56 53 50 3E 5B 03
				$cmd[2]=0x62; 
				$cmd[3]=0x0A;
				$cmd[4]=0x38;
				$cmd[5]=0x30;
				$cmd[6]=0x62 if $param=~/off/i;
				$cmd[6]=0x66 if $param=~/on/i;
				$cmd[7]=0x31;
				$cmd[8]=0x0D; 
				$cmd[9]=0x0A; 
				$cmd[10]=0x56; 
				$cmd[11]=0x53; 
				$cmd[12]=0x50;
				$cmd[13]=0x3E;
				$cmd[14]=checkSum(14,@cmd);
				$cmd[15]=$ETX;        
				
				# send command and prepare "SAVE" command ( 0x8039 )
				cmdTransmit($dev,@cmd); 
				select(undef,undef,undef,.1);
				$cmd[6]=0x33;
				$cmd[7]=0x39;
				$cmd[14]=checkSum(14,@cmd);
				
			}
			elsif($cmd=~/reset/i) {
				$cmd[2]=0x31; 
				$cmd[3]=0x01;
				$cmd[4]=0;
				$cmd[5]=0;
				$cmd[6]=0;
				$cmd[7]=0;
				$cmd[8]=checkSum(8,@cmd);
				$cmd[9]=$ETX;
				$log->debug("SETTINGS/RESET");
			}
		} elsif($mode=~/step/i){       # mode=step  /Step by step positioning/
			if($cmd=~/move/i) { # step pan/tilt
				$cmd[2]=0x34; $cmd[3]=$cmd[4]=$cmd[5]=$cmd[6]=0;
				if ( $param=~/right/i ) {
					$cmd[3]=0x60;
				}
				elsif ( $param=~/left/i ) {
					$cmd[3]=0x80; $cmd[4]=0xFF;
				}
				if ( $param=~/down/i ) {
					$cmd[5]=0xBF; $cmd[6]=0xFF;
				}
				elsif ( $param=~/up/i ) {
					$cmd[5]=0x40;
				}
				$cmd[7]=checkSum(7,@cmd); $cmd[8]=$ETX;
			} elsif ($cmd=~/gain/i){
				# + => 02 00 62 A 38 30 30 31 D A 56 53 50 3E D 03
				# - => 02 00 62 A 38 30 30 32 D A 56 53 50 3E E 03
				$cmd[2]=0x62; 
				$cmd[3]=0x0A;
				$cmd[4]=0x38;
				$cmd[5]=0x30;
				$cmd[6]=0x30;
				$cmd[7]=0x31 if $param=~/up/i;
				$cmd[7]=0x32 if $param=~/down/i;
				$cmd[8]=0x0D; 
				$cmd[9]=0x0A; 
				$cmd[10]=0x56; 
				$cmd[11]=0x53; 
				$cmd[12]=0x50; 
				$cmd[13]=0x3E;
				$cmd[14]=checkSum(14,@cmd);
				$cmd[15]=$ETX;        
			} elsif ($cmd=~/level/i){
				# + => 02 00 62 A 38 30 30 39 D A 56 53 50 3E 05 03
				# - => 02 00 62 A 38 30 30 61 D A 56 53 50 3E 5D 03
				$cmd[2]=0x62; 
				$cmd[3]=0x0A;
				$cmd[4]=0x38;
				$cmd[5]=0x30;
				$cmd[6]=0x30;
				$cmd[7]=0x39 if $param=~/up/i;
				$cmd[7]=0x61 if $param=~/down/i;
				$cmd[8]=0x0D; 
				$cmd[9]=0x0A; 
				$cmd[10]=0x56; 
				$cmd[11]=0x53; 
				$cmd[12]=0x50; 
				$cmd[13]=0x3E;
				$cmd[14]=checkSum(14,@cmd);
				$cmd[15]=$ETX;
			}
		}elsif($mode=~/hardware/i){   # mode=hardware  /Hardware reset,defaults/
			if($cmd=~/do/i && $param=~/defaults/i) { # init
				camReset($dev);
				return;
			}
		} elsif($mode=~/abs/i){                # mode=abs  /Absloute positioning/
			if($cmd=~/center/i) { # Center
				$cmd[2]=0x35; $cmd[3]=checkSum(3,@cmd); $cmd[4]=$ETX;
			} elsif($cmd=~/PT/i){            # Pan, Tilt
				$log->debug("ABS:  PT $param");
				$cmd[2]=0x33;
				$param=~/^(\-?.+)?\,(\-?.+)$/;
				my $p = int($1*10);
				my $t = int($2*10);
				$log->debug("ABS! pt=$1,$2 ($p,$t)");
				#$p += 36000 if $p<0;
				#$t += 36000 if $t<0;
				my $p_hex = sprintf("%04X",$p);
				my $t_hex = sprintf("%04X",$t);
				$log->debug("==== P_HEX= $p_hex");
				$p_hex =~/(\w\w)(\w\w)$/;
				($cmd[4],$cmd[5]) = (hex($2),hex($1));
				$t_hex =~/^(\w\w)(\w\w)$/;
				($cmd[6],$cmd[7]) = (hex($2),hex($1));
				$cmd[8]=checkSum(8,@cmd); 
				$cmd[9]=$ETX;
				# command will be send later 
			} elsif ($cmd=~/Z/i){	# Absolute zoom
				# Nothing to do
				return;
			}
		} elsif($mode=~/preset/i){ # presets
			$cmd[3]= $param;
			if ( $cmd =~/save/i ) {
				$cmd[2]= 0x42;
			}
			if ( $cmd =~/clear/i ) {
				# Nothing to do
				return;
			}
			if ( $cmd =~/goto/i ) {
				$cmd[2]= 0x32;
			}
			$cmd[4]=checkSum(4,@cmd); 
			$cmd[5]=$ETX;
		} elsif($mode=~/smooth/i) { # Right/Left/Up/Down/Tele/Wide/focus_Near/focus_Far/iris_Open/irIs_close
			# set movement bits
			if ($cmd=~/move/i) { # pan/tilt
				# Nothing to do
				return;
			} elsif($cmd=~/zoom/i) { # zoom
				# Nothing to do
				return;
			} elsif($cmd=~/focus/i) { # focus
				# Nothing to do
				return;
			} elsif($cmd=~/iris/i) { # iris
				# Nothing to do
				return;
			}
			return;
		}
	}
	# calculate command check-sum
	cmdTransmit($dev,@cmd);
}


sub nonblock {
	my ($fd) = @_;
	my $flags = fcntl($fd, F_GETFL,0);
	fcntl($fd, F_SETFL, $flags|O_NONBLOCK);
}



# --------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {

	$log->debug('load_dev_conf');
	
	# fisrst, close everything
	foreach my $skt (keys %sock) {
		close($sock{$skt}{SOCK});
		delete $sock{$skt};
	}


	%conf = GetCfgs( eval("($query_str)") );     # Load configurations
	$log->debug("Config read as:");
	$log->debug('-------------------------------------');
	foreach my $dev (keys %conf) {
		$log->debug("[$dev]");
		$log->debug("DEVID=$conf{$dev}{DEVID}");
		$log->debug("CAMERAMODEL=$conf{$dev}{CAMERAMODEL}");
		$log->debug("POSITIONCTL=$conf{$dev}{POSITIONCTL}");
		$log->debug("PTZID=$conf{$dev}{PTZID}");
		$log->debug("PTZ_TCP_PORT=$conf{$dev}{PTZ_TCP_PORT}");
		$log->debug("-------------------------------------");

		$log->debug("PTZ_TCP_PORT=$conf{$dev}{PTZ_TCP_PORT}");
		if(not defined($sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}})) {
			$sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK} = IO::Socket::INET->new(
			                                        PeerAddr => $conf{$dev}{DEVIP},
			                                        PeerPort => $conf{$dev}{PTZ_TCP_PORT},
			                                        Proto    => "tcp",
			                                        Type     => SOCK_STREAM);
			if(! $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK} ) {
				$log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
				delete $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}};
				next;
			}
			$sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK}->autoflush(1);
			nonblock($sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK});
			$sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK});
		}
	} # foreach $dev
	# close unused ports

}