#!/usr/bin/perl
#  $Id: ptz_pelcov2.pl 27843 2013-02-06 16:21:01Z astarostin $
# ------------------------------------------------------------------------------
#  PTZ driver for Pelco D protocol
# ------------------------------------------------------------------------------
#  Author: Andrey Fomenko
#  Edited by: Vladimir Ryabovol
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT LLC, 2004-2005
# ------------------------------------------------------------------------------

use strict;
use POSIX;
use Socket;
use IO::File;
use IO::Select;
use IO::Socket;
use Fcntl;

use NextCAM::Init;
use NextCAM::PTZ;
use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

# Constants
my $CHECK_DANTE_TIMEOUT = 5;

my $log=get_logger('NEXTCAM::PTZ::PTZ_PELCOV2');

my $query_str = shift || "'POSITIONCTL'=>'Pelco'";
$log->info("Starting Pelco PTZ DRIVER, query string: [$query_str]");

$SIG{HUP}=\&load_dev_conf;

my $APL=$ENV{APL};
my (%conf, %sock, $cam, $cmd, $par, $last_cmd, $last_mode, $lastchecktime);

load_dev_conf();

# ------------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# ------------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
                                   PeerPort => $TCP_PORT,
                                   Proto     => "tcp",
                                   Type      => SOCK_STREAM)
    or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';
$lastchecktime = time;

my %commands;
my $trycount = 30;
my $c = 0;

while(1) {
	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/i;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/
				|| $commands{$cam} =~ /speed iris=0/
				|| $commands{$cam} =~ /speed gain=0/
				)
			{
				; # do not override stop command!
			} else {
				$commands{$cam} = $cmd;
			}
		} else {
			$commands{$cam} = $cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)

	$cmd = '';

	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		delete $commands{$camera};
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");

		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam, $mode, "pt", $params[0].','.$params[1], \%options);
			# select(undef,undef,undef,0.02);
			#camCmd($cam, $mode, "zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam, $mode, $cmd, $par, \%options);
		}
	}

	select(undef,undef,undef,.05) if not $cmd;

	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
	if($conf{$dev}{CAMERAMODEL} eq 'Dante' &&
	   (time-$lastchecktime > $CHECK_DANTE_TIMEOUT ||
		!$c))
	{
		$c++ if !$c;
		$lastchecktime = time;
		checkAndSetDanteSerialConfiguration($dev);
	}
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
	}

	if(!$cmd) {
		select(undef,undef,undef,.4);
		if(!$trycount--) {
			$trycount = 30;
			print $socket "test\0" or $log->logdie("Couldn't communicate to socket $TCP_PORT: $@");
		}
	}

} # while(1)


# ------------------------------------------------------------ cmdTransmit -----
sub cmdTransmit {
    my ($dev,@cmd)=@_;

    my $command='';
    my $b;
    foreach $b (@cmd) { $command .= sprintf("%02X ",$b); }
    #$command .= sprintf(' %d~byte CMD:',$cmd[5]);
    #for ($b=6; $b < $#cmd; $b++) { $command .= chr($cmd[$b]); }
    $log->debug('cmdTransmit( ',$dev, '=> ',$command,')');

    my $bts = '';
    if($conf{$dev}{PTZ_TRANSPORT_PROTO}=~/Bosch/) {
	log->warn("Bosch proprietary protocol is not implemented yet.");
	return;
    } # if($conf{$dev}{PTZ_TRANSPORT_PROTO}=~/Bosch/)
    elsif($conf{$dev}{PTZ_TRANSPORT_PROTO}=~/Dante/) {
	if($#cmd == 6) {
	    my @dante_specifics = (
		0x1f, 0x00, 0x00, 0x00, 0x5d, 0x03, 0x00, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, # 0x01 - RS485 interface
		0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
		@cmd);
	    foreach my $b (@dante_specifics) { $bts .= chr($b); }
	}
	else {
	    foreach my $b (@cmd) { $bts .= chr($b); }
	}
    } # if($conf{$dev}{PTZ_TRANSPORT_PROTO}=~/Dante/)
    else { # in case of TCP_SOCK
	foreach my $b (@cmd) { $bts .= chr($b); }
    }

    my $retries = 1;
    secondaryPass:
    my $skt = $sock{ $conf{$dev}{DEVIP} }{SOCK};
    eval {
        local $SIG{PIPE} = sub { die "Error writing to socket" };
        print $skt $bts;
    };
    if($@) {
        $log->error($@);
        # camera did not answer or socket error occured - try to close socket and reopen it
        eval{ $sock{$conf{$dev}{DEVIP}}{SOCK}->close(); };
        $log->error("Re-open socket connection: $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT}");
        $sock{$conf{$dev}{DEVIP}}{SEL} = undef;
        select(undef,undef,undef,.1);
        $sock{$conf{$dev}{DEVIP}}{SOCK} = IO::Socket::INET->new(PeerAddr => $conf{$dev}{DEVIP},
                                                PeerPort => $conf{$dev}{PTZ_TCP_PORT},
                                                Proto    => "tcp",
                                                Type     => SOCK_STREAM);
        if(! $sock{$conf{$dev}{DEVIP}}{SOCK} ) {
            $log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
            delete $sock{$conf{$dev}{DEVIP}};
            return;
        }
        $sock{$conf{$dev}{DEVIP}}{SOCK}->autoflush(1);
        nonblock($sock{$conf{$dev}{DEVIP}}{SOCK});
        $sock{$conf{$dev}{DEVIP}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}}{SOCK});
        $log->info('Socket was closed, reconnected');
        goto secondaryPass if $retries--;

        return;
    }

    select(undef,undef,undef,.05);
    foreach my $sss ( $sock{ $conf{$dev}{DEVIP} }{SEL}->can_read(0.1) ) {
	my $data = '';
        eval {
            alarm 1;
            $sss->recv($data, POSIX::BUFSIZ, 0);
            $command='';
            for($b=0;$b<length($data);$b++) { $command .= sprintf("%02X ",ord(substr($data,$b,1))); }
            $log->debug('Received:',$command);
        };
        alarm 0;
	if($conf{$dev}{PTZ_TRANSPORT_PROTO}=~/Dante/ &&
           ord(substr($data,0,1)) != 0x18 &&
	   ord(substr($data,0,1)) != 0x20) #check on acknowledge from the camera - reporting only
        {
    	    $log->error("No Acknowledge coming from the camera!");
        }
        return $data;
    }  # foreach

} # sub cmdTransmit

# --------------------------------------------------------------- CheckSum -----
sub checkSum
{
  my @arr = @_;
  return ($arr[1]+$arr[2]+$arr[3]+$arr[4]+$arr[5]) % 256;
} # sub checkSum

# ---------------------------------------------------------------- camStop -----
sub camStop
{
  my ($dev) = @_;
  my @cmd = ( 0xFF, 0, 0, 0, 0, 0, 0 );
  $cmd[1]=$conf{$dev}{PTZID}; # camera hardware ID
  $cmd[6]=checkSum(@cmd);
  cmdTransmit($dev,@cmd);
}

# --------------------------------------------------------------- camReset -----
sub camReset
{
  my ($dev) = @_;
  my @cmd = ( 0xFF, 0, 0, 0, 0, 0, 0 );
  $cmd[1]=$conf{$dev}{PTZID};
  #Load defaults
  $cmd[3]=0x29; $cmd[6]=checkSum(@cmd);
  cmdTransmit($dev,@cmd);
  # Auto-focus: byte 5: 0-auto/1-on/2-off
  $cmd[3]=0x2B; $cmd[5]=0; $cmd[6]=checkSum(@cmd);
  cmdTransmit($dev,@cmd);
  # Auto-iris: byte 5: 0-auto/1-on/2-off
  $cmd[3]=0x2D; $cmd[5]=0; $cmd[6]=checkSum(@cmd);
  cmdTransmit($dev,@cmd);
  # Auto white balance: byte 5: 1-on/2-off
  $cmd[3]=0x33; $cmd[5]=2; $cmd[6]=checkSum(@cmd);
  cmdTransmit($dev,@cmd);
  # Set focus speed: byte 5: 0..3
  $cmd[3]=0x27; $cmd[5]=1; $cmd[6]=checkSum(@cmd);
  cmdTransmit($dev,@cmd);
} # sub camReset

# ------------------------------------------------------------- camFZspeed -----
sub camFZspeed
{
  my ($dev,$FZ,$speed) = @_;
  $log->debug("camFZspeed($dev,$FZ,$speed)");
  my @cmd = ( 0xFF, 0, 0, 0, 0, 0, 0 );
  $cmd[1]=$conf{$dev}{PTZID};
  $cmd[3]=$FZ=='F'?0x27:0x25;
  $cmd[5]=sprintf("%d",abs($speed)/25);
  $cmd[6]=checkSum(@cmd);
  cmdTransmit($dev,@cmd);
} # sub camReset

# ----------------------------------------------------------------- PTspeed ----
sub PTspeed {
  my ($speed)=@_;
  #return 255 if(abs($speed)>99); # turbo speed - does not work with spectra !!!
  $speed=100 if abs($speed)>100;
  return sprintf("%2d",abs($speed)/100*63);
} # sub PTspeed

# ----------------------------------------------------------------- camCmd -----
sub camCmd {
  my ($dev,$mode,$cmd,$param,$options)=@_;
  my %options = %$options;
  my @cmd = ( 0xFF, 0, 0, 0, 0, 0, 0 );
  $log->debug("camCmd: DEVID=$dev mode: $mode command: $cmd optional parameter: $param");
  $cmd[1]=$conf{$dev}{PTZID}; # camera hardware ID

  # here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
  if($conf{$dev}->{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
     if($mode=~/speed/i || $mode=~/step/i || $mode=~/abs/i || $mode=~/rel/i || $mode=~/smooth/i) {
        $log->debug("Setting timeout +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
        $conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
     }
     elsif($mode=~/preset/i && $cmd=~/goto/i && $param!=1) {
        $log->debug("Setting timeout (preset) +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
        $conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
    }
  }

  if(not $cmd=~/stop/) { # non-Stop
    # set camera speed
    $cmd[4]=$cmd[5]=$conf{$dev}{PTZSPEED}?$conf{$dev}{PTZSPEED}:0x20;
    $cmd[4]=0x3F if $cmd[4]>0x3F;
    $cmd[5]=0x3F if $cmd[5]>0x3F;
   if ($mode=~/speed/i){        # mode speed
    if($cmd=~/PT/i) { # RPT
      my ($p,$t) = split(/,/,$param);
      $cmd[4]=PTspeed($p);
      $cmd[5]=PTspeed($t);
      $log->debug("RPT $p $t");
      $cmd[3] |= 0x02 if $p>0;
      $cmd[3] |= 0x04 if $p<0;
      $cmd[3] |= 0x08 if $t<0;
      $cmd[3] |= 0x10 if $t>0;
    }
    elsif($cmd=~/Z/i) {
      camFZspeed($dev,'Z',$param);
      $log->debug("ZOOM $param");
      $cmd[3] |= 0x20 if $param>0;
      $cmd[3] |= 0x40 if $param<0;
    }
    elsif($cmd=~/focus/i) {
      camFZspeed($dev,'F',$param);
      $log->debug("FOCUS $param");
      $cmd[2] |= 0x01 if $param>0;
      $cmd[3] |= 0x80 if $param<0;
    }
    elsif($cmd=~/iris/i) {
      camFZspeed($dev,'F',$param);
      $log->debug("IRIS $param");
      $cmd[2] |= 0x02 if $param>0;
      $cmd[2] |= 0x04 if $param<0;
    }
   }elsif($mode=~/rel/i) { # mode=rel: "<devid> rel size=640x480 xy=225,152"
      $param =~ /(\d+)x(\d+)/;
      my ($rel_size_x,$rel_size_y) = ($1,$2);
      $options{xy} =~ /(\d+),(\d+)/;
      my ($rel_click_x,$rel_click_y) = ($1,$2);
      my ($rel_p, $rel_t) = ( 200*($rel_click_x-$rel_size_x/2)/$rel_size_x, 200*($rel_click_y-$rel_size_y/2)/$rel_size_y );
      $log->debug("REL [$rel_size_x] [$rel_size_y] [$rel_click_x] [$rel_click_y] [$rel_p] [$rel_t]");
      $cmd[4]=PTspeed($rel_p);
      $cmd[5]=PTspeed($rel_t);
      $cmd[3] |= 0x02 if $rel_p>0;
      $cmd[3] |= 0x04 if $rel_p<0;
      $cmd[3] |= 0x08 if $rel_t<0;
      $cmd[3] |= 0x10 if $rel_t>0;
      $cmd[6]=checkSum(@cmd);
      cmdTransmit($dev,@cmd);
      select(undef,undef,undef,.2);
      camStop($dev);
      return;
   }elsif ($mode=~/settings/i){ # settings
    if($cmd=~/autoiris/i) {
      $cmd[3] = 0x2D;
      $cmd[4] = 0x00;
      $cmd[5] = 0x00 if $param=~/auto/i;
      $cmd[5] = 0x01 if $param=~/on/i;
      $cmd[5] = 0x02 if $param=~/off/i;
    }
    elsif($cmd=~/autofocus/i) {
      $cmd[3] = 0x2B;
      $cmd[4] = 0x00;
      $cmd[5] = 0x00 if $param=~/auto/i;
      $cmd[5] = 0x01 if $param=~/on/i;
      $cmd[5] = 0x02 if $param=~/off/i;
    }
    elsif($cmd=~/whitebalance/i) { #'awb_on','g', 'awb_off','b'
      $cmd[3] = 0x33;
      $cmd[4] = 0x00;
      $cmd[5] = 0x01 if $param=~/[auto|on]/i;
      $cmd[5] = 0x02 if $param=~/off/i;
    }
		elsif($cmd=~/aux1/i) {
  		#@cmd = ( 0xFF, 0, 0, 0, 0, 0, 0 );
			$cmd[3] = $param=~/set/ ? 0x09 : 0x0b;
			$cmd[5] = 0x01;
			$cmd[6] = $param=~/set/ ? 0x0b : 0x0d;
		}
   }elsif($mode=~/step/i){       # mode=step  /Step by step positioning/
      $cmd[4]=2;
      $cmd[5]=2;
      my $timeout = 0.2;
    if($cmd=~/move/i) { # step pan/tilt
      $cmd[3] |= 0x02 if $param=~/right/i;
      $cmd[3] |= 0x04 if $param=~/left/i;
      $cmd[3] |= 0x08 if $param=~/up/i;
      $cmd[3] |= 0x10 if $param=~/down/i;
    } elsif ($cmd=~/zoom/i){
      $cmd[3] |= 0x20 if $param=~/in/i;
      $cmd[3] |= 0x40 if $param=~/out/i;
    } elsif ($cmd=~/focus/i){
      $cmd[2] |= 0x01 if $param=~/near/i;
      $cmd[3] |= 0x80 if $param=~/far/i;
      $timeout = 0.02;          # reduce timeout for focusing
    } elsif ($cmd=~/iris/i){
      $cmd[2] |= 0x02 if $param=~/open/i;
      $cmd[2] |= 0x04 if $param=~/close/i;
    }
      $cmd[6]=checkSum(@cmd);
      cmdTransmit($dev,@cmd);
      select(undef,undef,undef,$timeout);
      camStop($dev);
      return;
   }elsif($mode=~/hardware/i){   # mode=hardware  /Hardware reset,defaults/
    if($cmd=~/do/i && $param=~/defaults/i) { # init
      camReset($dev);
      return;
    }
   } elsif($mode=~/abs/i){                # mode=abs  /Absloute positioning/
      if($cmd=~/center/i) { # Center
         camReset($dev);
         $cmd[3]=0x07;
         $cmd[5]=0x22;
      } elsif($cmd=~/PT/i){            # Pan, Tilt
         $log->debug("ABS:  PT $param");
         $cmd[2]=0x00;
         $param=~/^(\-?.+)?\,(\-?.+)$/;
         my $p = int($1*100);
         my $t = int($2*100);
         $log->debug("ABS! pt=$1,$2 ($p,$t)");
         $p += 36000 if $p<0;
         $t *= $conf{$dev}->{PTZ_INVERT} if $t<0 && defined($conf{$dev}->{PTZ_INVERT});
         $log->debug("ABS! ~ ($p,$t)");
         my $p_hex = sprintf("%04X",$p);
         my $t_hex = sprintf("%04X",$t);
         $log->debug("==== P_HEX= $p_hex");
         $p_hex =~/^(\w\w)(\w\w)$/;
         ($cmd[4],$cmd[5]) = (hex($1),hex($2));
         $cmd[3]=0x4B;	# For Pan Position
         $cmd[6]=checkSum(@cmd);
         cmdTransmit($dev,@cmd);	#sending Pan first
         $cmd[3]=0x4D;	# For Tilt Position
         $t_hex =~/^(\w\w)(\w\w)$/;
         ($cmd[4],$cmd[5]) = (hex($1),hex($2));
         # command will be send later
      } elsif ($cmd =~ /z/i){ # Absolute zoom
         $log->debug("ABS: Z $param");

         $param =~ /(\d+)/;

         my $max_zoom_step = hex '0xFFFF';
         my $abs_z = int($max_zoom_step * $1 / 100);

         $cmd[2] = 0x00;
         $cmd[3] = 0x4F;

         my $abs_z_hex = sprintf("%04X", $abs_z);
         $log->debug("ABS: ZOOM $abs_z_hex");

         $abs_z_hex =~ /^(\w\w)(\w\w)$/;
         ($cmd[4], $cmd[5]) = (hex($1), hex($2));
      } elsif($cmd =~ /hfov/i) {
         $log->debug("ABS: HFOV $param");

         if ($conf{$dev}->{PTZ_HFOV})
         {
             $param =~ /(\d+(\.\d+)?)/;

             my $max_zoom_step = hex '0xFFFF';
             my $step = hfov2step(\$conf{$dev}, $1, 1);
             my $abs_z = int($max_zoom_step * $step);

             $cmd[2] = 0x00;
             $cmd[3] = 0x4F;

             my $abs_z_hex = sprintf("%04X", $abs_z);
             $log->debug("ABS: ZOOM $abs_z_hex");

             $abs_z_hex =~ /^(\w\w)(\w\w)$/;
             ($cmd[4], $cmd[5]) = (hex($1), hex($2));
         }
      }
   } elsif($mode=~/preset/i){ # presets
      $cmd[3]= 0x03 if $cmd =~/save/i;
      $cmd[3]= 0x05 if $cmd =~/clear/i;
      $cmd[3]= 0x07 if $cmd =~/goto/i;
      $cmd[5]= $param;
   } elsif($mode=~/smooth/i) { # Right/Left/Up/Down/Tele/Wide/focus_Near/focus_Far/iris_Open/irIs_close
      # set movement bits
      if ($cmd=~/move/i) { # pan/tilt
        $cmd[3] |= 0x02 if $param=~/right/i;
        $cmd[3] |= 0x04 if $param=~/left/i;
        $cmd[3] |= 0x08 if $param=~/up/i;
        $cmd[3] |= 0x10 if $param=~/down/i;
      } elsif($cmd=~/zoom/i) { # zoom
        $cmd[3] |= 0x20 if $param=~/in/i;
        $cmd[3] |= 0x40 if $param=~/out/i;
      } elsif($cmd=~/focus/i) { # focus
        $cmd[2] |= 0x01 if $param=~/near/i; # reduce speed of focusing
        $cmd[3] |= 0x80 if $param=~/far/i;
      } elsif($cmd=~/iris/i) { # iris
        $cmd[2] |= 0x02,$cmd[4]=$cmd[5]=0 if $param=~/open/i;
        $cmd[2] |= 0x04,$cmd[4]=$cmd[5]=0 if $param=~/close/i;
      }
    }
  }
  # calculate command check-sum
  $cmd[6]=checkSum(@cmd);
  cmdTransmit($dev,@cmd);
}

# --------------------------------------------------------------- nonblock -----
#   nonblock($socket) puts socket into nonblocking mode
sub nonblock {
    my $socket = shift;
    my $flags;

    $flags = fcntl($socket, F_GETFL, 0)
            or die "Can't get flags for socket: $!\n";
    fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
            or die "Can't make socket nonblocking: $!\n";
}

# ---------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {

    $log->info('Configurations refreshed: ',$query_str);

    # fisrst, close everything
    foreach my $skt (keys %sock) {
        close($sock{$skt}{SOCK});
        delete $sock{$skt};
    }

    %conf = GetCfgs( eval("($query_str)") );     # Load configurations
    foreach my $dev (keys %conf) {
	# in a case when PTZ works on another server (e.g. 3ETI COMTROL Serial Server)
	$conf{$dev}{DEVIP} = $conf{$dev}{PTZ_TCP_IP} if $conf{$dev}{PTZ_TCP_IP};

	$log->debug("[$dev]");
	$log->debug("DEVID=$conf{$dev}{DEVID}");
	$log->debug("PTZID=$conf{$dev}{PTZID}");
	$log->debug("PTGROUP=$conf{$dev}{PTZGROUP}");
	$log->debug("PTZSPEED=$conf{$dev}{PTZSPEED}");
	$log->debug("DEVIP=$conf{$dev}{DEVIP}");
	$log->debug("PTZ_TRANSPORT_PROTO=$conf{$dev}{PTZ_TRANSPORT_PROTO}");
	$log->debug("PTZ_TCP_PORT=$conf{$dev}{PTZ_TCP_PORT}");

	$log->info("Open socket connection: $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT}");
        if(not defined($sock{$conf{$dev}{DEVIP}})) {
	    $sock{$conf{$dev}{DEVIP}}{SOCK} = IO::Socket::INET->new(PeerAddr => $conf{$dev}{DEVIP},
    	                            		PeerPort => $conf{$dev}{PTZ_TCP_PORT},
                                                Proto    => "tcp",
                                                Type     => SOCK_STREAM);
	    if(! $sock{$conf{$dev}{DEVIP}}{SOCK} ) {
                $log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
                delete $sock{$conf{$dev}{DEVIP}};
                next;
	    }

	    $sock{$conf{$dev}{DEVIP}}{SOCK}->autoflush(1);
	    nonblock($sock{$conf{$dev}{DEVIP}}{SOCK});
	    $sock{$conf{$dev}{DEVIP}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}}{SOCK});
        }
    } # foreach $dev
} # sub load_dev_conf

# ------------------------------------ checkAndSetDanteSerialConfiguration -----
sub checkAndSetDanteSerialConfiguration {
    my ($dev) = @_;
    my @req = (
	0x18, 0x00, 0x00, 0x00, 0x5a, 0x03, 0x00, 0x00,
	0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 0x01 - RS485 interface
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    );

    my $resp = cmdTransmit($dev, @req);

    my @serial_port_cfg = (
	0xff,
	0x03, # data length: 8 bits
	0x01, # baud rate:   2400
	0x00,
	0x00, # stop bits:   1 bit
	0x00, # parity:      disable
	0x00, # parity mode: even
	0x00
    );

    my $cfg = '';
    foreach my $b (@serial_port_cfg) { $cfg .= chr($b); }

    if(substr($resp, 24) ne $cfg) {
	$log->error("$conf{$dev}{DEVIP}: Dante serial port configuration is wrong. Sending correct one...");
	my @req = (
    	    0x20, 0x00, 0x00, 0x00, 0x5c, 0x03, 0x00, 0x00,
	    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # 0x01 - RS485 interface
	    0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
	    @serial_port_cfg
	);
	cmdTransmit($dev, @req);
    }
} # sub checkAndSetDanteSerialConfiguration
