#!/usr/bin/perl
# -----------------------------------------------------------------------------
#  PTZ driver for Pelco cameras
# -----------------------------------------------------------------------------
#  Author: Alexey Tsibulnik
#  Edited by: 
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT LLC, 2004-2009
# -----------------------------------------------------------------------------

use strict;
use Socket;
use MIME::Base64();
use IO::File;
use IO::Select;
use IO::Socket;
use Fcntl;
use LWP::UserAgent;
use HTTP::Request::Common;

use NextCAM::Init;
use NextCAM::PTZ;
use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

my $ua = LWP::UserAgent->new();


sub my_log
{
	my $msg = shift;
	#`echo $msg >> /tmp/log1.log`;
	open(F, ">> /tmp/log.log");
	print(F "$msg\n");
	close(F);
	
	#my_log("-= 111111111 =-");
}

my $log=get_logger('NEXTCAM::PTZ::PTZ_PELCO');

my $query_str = shift || "'POSITIONCTL'=>'Pelco','CAMERAMODEL'=>'Pelco'";
$log->info("Starting PELCO PTZ DRIVER, query string: [$query_str]");
                                                                     
my $ua = LWP::UserAgent->new();
                                                                     
$SIG{HUP}=\&load_dev_conf;
my $APL=$ENV{APL};
my $APL_CONF=$ENV{APL_CONF};
my (%conf,$cam,$cmd,$usrpsw,$par,$last_cmd,$last_mode);
load_dev_conf();

# -----------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# -----------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
                                   PeerPort => $TCP_PORT,
                                   Proto     => "tcp",
                                   Type      => SOCK_STREAM)
    or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';

my %commands;
my $ready;
my $trycount = 30;

# Pelco camera default parameters
my $min_x_velocity = 0;
my $min_y_velocity = 0;
my $max_x_velocity = 12500;
my $max_y_velocity = 4000;
my $max_mag = 3200;

while(1) {
	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/i;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/
				|| $commands{$cam} =~ /speed iris=0/
				|| $commands{$cam} =~ /speed gain=0/
				)
			{
					; # do not override stop command!
			}else{
				$commands{$cam} = $cmd;
			}
		} else {
			$commands{$cam} = $cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)
	
	$cmd = '';
	
	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		delete $commands{$camera};
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
		$usrpsw = '';
		$usrpsw = "$conf{$cam}{USRNAME}:$conf{$cam}{PASSWD}" if $conf{$cam}{USRNAME} && $conf{$cam}{PASSWD};
		
		httpAuth($conf{$cam}{USRNAME}, $conf{$cam}{PASSWD}) if $conf{$cam}{USRNAME} && $conf{$cam}{PASSWD};
		
		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam,$mode,"pt", $params[0].','.$params[1] ,\%options);
			camCmd($cam,$mode,"zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam,$mode,$cmd,$par,\%options);
		}
	}

	select(undef,undef,undef,.1) if not $cmd;

	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
	}
	
	if(!$cmd) {
		select(undef,undef,undef,.4);
		if(!$trycount--) {
			$trycount = 30;
			print $socket "test\0" or $log->logdie("Couldn't communicate to socket $TCP_PORT: $@");
		}
	}

} # while(1)




# =============================================================================

# ---------------------------------------------------------------- camCmd -----
sub camCmd
{
  my ($dev,$mode,$cmd,$param,$options)=@_;
  my %options = %$options;
  $log->debug("camCmd: DEVID=[$dev] mode=[$mode] command:[$cmd] param=[$param] ");
  my $posURL="http://$conf{$dev}{DEVIP}:$conf{$dev}{HTTP_PORT}/api/positioningControl.php";
  my $lensURL="http://$conf{$dev}{DEVIP}:$conf{$dev}{HTTP_PORT}/api/lensControl.php";
  my $scriptURL="http://$conf{$dev}{DEVIP}:$conf{$dev}{HTTP_PORT}/api/scriptControl.php";
  my $spd=($options{speed}=~/^\d+/) ? $options{speed}:$conf{$dev}->{PTZSPEED};

  # we need to "terminate" each command because it can move forever if used with no STOP !
  if ($last_mode=~/smooth/i) {
  	# stop if MODELID=TXB-N???
		if($conf{$dev}->{MODELID}=~/^(TXB-N|TXB-N1|D5|ESTI|TI)/) {
			checkSOAP($conf{$dev}, "PositioningControl", "SetVelocity", "SetVelocity1", "<velocity><rotation><x>0</x><y>0</y></rotation></velocity>") if $last_cmd=~/move/i;
			checkSOAP($conf{$dev}, "LensControl", "Zoom", "Zoom", "<inOut>0</inOut>") if $last_cmd=~/zoom/i;
			checkSOAP($conf{$dev}, "LensControl", "Iris", "Iris", "<openClose>0</openClose>") if $last_cmd=~/iris/i;
			checkSOAP($conf{$dev}, "LensControl", "Focus", "Focus", "<nearFar>0</nearFar>") if $last_cmd=~/focus/i;
		}else{
			checkURL("$posURL?action=setVelocity&x=0&y=0") if $last_cmd=~/move/i;
			checkURL("$lensURL?action=zoom&zoom=0")	    if $last_cmd=~/zoom/i;
			checkURL("$lensURL?action=iris&iris=0")	    if $last_cmd=~/iris/i;
			checkURL("$lensURL?action=focus&focus=0")	    if $last_cmd=~/focus/i;
		}
  }

  # here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
  if($conf{$dev}->{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
     if($mode=~/speed/i || $mode=~/step/i || $mode=~/abs/i || $mode=~/rel/i || $mode=~/smooth/i) {
        $log->debug("Setting timeout +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
        $conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
     }
     elsif($mode=~/preset/i && $cmd=~/goto/i && $param!=1) {
        $log->debug("Setting timeout (preset) +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
        $conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
    }
  }

  if ($mode=~/speed/i){
     $spd=$conf{$dev}->{PTZSPEED};
     $spd=1 if $spd < 1;
     $spd=100 if $spd > 100;
     if($cmd=~/pt/i) { #  pan-tilt
        my ($p,$t) = split(/,/,$param);
        $p = int($p*$spd/100*($max_x_velocity-$min_x_velocity)/100);
        $t = -int($t*$spd/100*($max_y_velocity-$min_y_velocity)/100);

		if($conf{$dev}->{MODELID}=~/^(TXB-N|TXB-N1|D5|ESTI|TI)/) {
			my $respond1 = checkSOAP($conf{$dev}, "PositioningControl", "SetVelocity", "SetVelocity1", "<velocity><rotation><x>$p</x><y>$t</y></rotation></velocity>");
		}
		else{
        	checkURL("$posURL?action=setVelocity&x=$p&y=$t");
        }
     }
     elsif($cmd=~/z/i) { # zoom
        my $zm=$param>0?2:$param<0?1:0;
		if($conf{$dev}->{MODELID}=~/^(TXB-N|TXB-N1|D5|ESTI|TI)/) {
			my $respond = checkSOAP($conf{$dev}, "LensControl", "Zoom", "Zoom", "<inOut>$zm</inOut>");
		}
		else{
	        checkURL("$lensURL?action=zoom&zoom=$zm");
	    }
     }

     elsif($cmd=~/focus/i) { # focus
        my $fc=$param>0?2:$param<0?1:0;
		if($conf{$dev}->{MODELID}=~/^(TXB-N|TXB-N1|D5|ESTI|TI)/) {
			my $respond = checkSOAP($conf{$dev}, "LensControl", "Focus", "Focus", "<nearFar>$fc</nearFar>");
		}
		else{
	        checkURL("$lensURL?action=focus&focus=$fc");
	    }
     }
     elsif($cmd=~/iris/i) { # iris
        my $ir=$param>0?2:$param<0?1:0;
		if($conf{$dev}->{MODELID}=~/^(TXB-N|TXB-N1|D5|ESTI|TI)/) {
			my $respond = checkSOAP($conf{$dev}, "LensControl", "Iris", "Iris", "<openClose>$ir</openClose>");
		}
		else{
	        checkURL("$lensURL?action=iris&iris=$ir");
	    }    
     }
  } elsif($mode=~/abs/i){    # mode=ABS
    if ($cmd=~/pt/i) {
      $param=~/(\-?\d+)\s*,\s*(\-?\d+)\s*$/;
      my $x = $1*100;
      my $y = $2*100;
		if($conf{$dev}->{MODELID}=~/^(TXB-N|TXB-N1|D5|ESTI|TI)/) {
			my $respond = checkSOAP($conf{$dev}, "PositioningControl", "SetPosition", "SetPosition", "<position><translation><x>$x</x></translation><rotation><z>$y</z></rotation></position>");
		}
		else{
	      checkURL("$posURL?action=setPosition&x=$x&y=$y");
	    }
    }
    elsif($cmd=~/Z/i) {
      $param=~/(\d+)/;
      my $abs_z = int($1*$max_mag/100);
      checkURL("$lensURL?action=setMag&mag=");
    }
    elsif($cmd=~/center/i) { # center
      checkURL("$posURL?action=setPosition?x=0&y=0");
      checkURL("$lensURL?action=setMag&mag=100");
    }
  } elsif($mode=~/rel/i){ # relative positioning (recentering)
      $param =~ /(\d+)x(\d+)/;
      my ($rel_size_x,$rel_size_y) = ($1,$2);
      $options{xy} =~ /(\d+),(\d+)/;
      my ($rel_click_x,$rel_click_y) = ($1,$2);
	  
      my $x = -100 + int($rel_click_x * 200 / $rel_size_x);
      my $y =  100 - int($rel_click_y * 200 / $rel_size_y);
		if($conf{$dev}->{MODELID}=~/^(TXB-N|TXB-N1|D5|ESTI|TI)/) {
			my $ax = int(14000 * (sin(($rel_click_x-$rel_size_x/2) * 3.14 / $rel_size_x)));
			my $ay = int(3600 * (sin((-$rel_click_y+$rel_size_y/2) * 3.14 / $rel_size_y)));
			my $respond = checkSOAP($conf{$dev}, "PositioningControl", "SetVelocity", "SetVelocity1", "<velocity><rotation><x>$ax</x><y>$ay</y></rotation></velocity>");
			my $respond = checkSOAP($conf{$dev}, "PositioningControl", "SetVelocity", "SetVelocity1", "<velocity><rotation><x>0</x><y>$ay</y></rotation></velocity>");
			my $respond = checkSOAP($conf{$dev}, "PositioningControl", "SetVelocity", "SetVelocity1", "<velocity><rotation><x>0</x><y>0</y></rotation></velocity>");
		}
		else{
			checkURL("$posURL?action=viewObject&x=$x&y=$y");
	    }

  # mode=step  /Step by step positioning/
  } elsif($mode=~/step/i) {       # mode=step  /Step by step positioning/
     if($cmd=~/move/i) { # step pan/tilt
        my ($x,$y) = (0,0);
        $x = $param=~/right/i ? 15 : $param=~/left/i ? -15 : 0;
        $y = $param=~/up/i ? 15 : $param=~/down/i ? -15 : 0;

		if($conf{$dev}->{MODELID}=~/^(TXB-N|TXB-N1|D5|ESTI|TI)/) {
			$spd=$conf{$dev}->{PTZSPEED};
			$x = $x * 5 * $spd;
			$y = $y * $spd;
			my $respond = checkSOAP($conf{$dev}, "PositioningControl", "SetVelocity", "SetVelocity1", "<velocity><rotation><x>$x</x><y>$y</y></rotation></velocity>");
			$respond = checkSOAP($conf{$dev}, "PositioningControl", "SetVelocity", "SetVelocity1", "<velocity><rotation><x>0</x><y>0</y></rotation></velocity>");
		}
		else{
			checkURL("$posURL?action=viewObject&x=$x&y=$y");
		}
     }
     elsif($cmd=~/zoom/i) {
	my $zm = $param=~/out/i? 90 : 110;
	if($conf{$dev}{MODELID} eq 'SpectraMiniIP' && $param=~/out/i) {
	    $zm = 80;	# DE1991: Workaround for SpectraMiniIP
	}
	$zm=int($zm);
		if($conf{$dev}->{MODELID}=~/^(TXB-N|TXB-N1|D5|ESTI|TI)/) {
			my $zm = $param=~/out/i? 1 : 2;
			my $respond = checkSOAP($conf{$dev}, "LensControl", "Zoom", "Zoom", "<inOut>$zm</inOut>");
			$respond = checkSOAP($conf{$dev}, "LensControl", "Zoom", "Zoom", "<inOut>0</inOut>");
		}
		else{
			checkURL("$posURL?action=viewObject&z=$zm");
		}
     }
     elsif($cmd=~/focus/i) {
		my $focus = $param=~/far/i? 1 : 2;
		if($conf{$dev}->{MODELID}=~/^(TXB-N|TXB-N1|D5|ESTI|TI)/) {
			my $respond = checkSOAP($conf{$dev}, "LensControl", "Focus", "Focus", "<nearFar>$focus</nearFar>");
			$respond = checkSOAP($conf{$dev}, "LensControl", "Focus", "Focus", "<nearFar>0</nearFar>");
		}else{
			checkURL("$lensURL?action=focus&focus=$focus");
			checkURL("$lensURL?action=focus&focus=0");
		}	
     }
     elsif($cmd=~/iris/i) {
		my $iris = $param=~/open/i? 2 : 1;
		if($conf{$dev}->{MODELID}=~/^(TXB-N|TXB-N1|D5|ESTI|TI)/) {
			my $respond = checkSOAP($conf{$dev}, "LensControl", "Iris", "Iris", "<openClose>$iris</openClose>");
			$respond = checkSOAP($conf{$dev}, "LensControl", "Iris", "Iris", "<openClose>0</openClose>");
		}else{
			checkURL("$lensURL?action=iris&iris=$iris");
			checkURL("$lensURL?action=iris&iris=0");
		}	
     }

  } elsif($mode=~/smooth/i){  # mode=smooth  /Old style PTZ. Goes to direction
                              # untill stop (or any other command is sent)/
     if($cmd=~/move/i) { # pan/tilt
	my $x = $param=~/left/i? -6000 : $param=~/right/i? 6000 : 0;
	my $y = $param=~/down/i? -1000 : $param=~/up/i? 1000 : 0;
	checkURL("$posURL?action=setVelocity&x=$x&y=$y");
     }
     elsif($cmd=~/zoom/i) { # zoom
	my $zm=$param=~/in/i ? 2 : $param=~/out/i ? 1 : 0;
        checkURL("$lensURL?action=zoom&zoom=$zm");
     }
     elsif($cmd=~/iris/i) { # iris
	my $ir=$param=~/open/i ? 2 : $param=~/close/i ? 1 : 0;
        checkURL("$lensURL?action=iris&iris=$ir");
     }
     elsif($cmd=~/focus/i) { # focus
	my $fc=$param=~/near/i ? 2 : $param=~/far/i ? 1 : 0;
        checkURL("$lensURL?action=focus&focus=$fc");
     }
  } elsif($mode=~/preset/i){ # presets
		my $action = $cmd=~/save/ ? 'setPreset' : 'goPreset';
		if($conf{$dev}->{MODELID}=~/^(TXB-N|TXB-N1|D5|ESTI|TI)/) {
			my $presetname = "PRESET". $param;
			if($cmd=~/save/){
				my $respond = checkSOAP($conf{$dev}, "ScriptControl", "EndScript", "EndScript", "<scriptName>$presetname</scriptName>");
			}else{
				my $respond = checkSOAP($conf{$dev}, "ScriptControl", "ExecuteScript", "ExecuteScript", "<scriptName>$presetname</scriptName>");
			}
		}
		else{
			checkURL("$scriptURL?action=$action&preset=$param");
		}
  } elsif($mode=~/settings/i){ # settings
     my $code = $param=~/on/ ? 1 : 0;
     if ($cmd=~/timeout/i) {
        $log->debug("Setting timer: [$param]");
        $conf{$dev}->{TIMEOUT} = time + $param;
     }
     elsif ($cmd=~/autofocus/i) {
     	if($conf{$dev}->{MODELID}=~/^(TXB-N|TXB-N1|D5|ESTI|TI)/) {
			my $respond = checkSOAP($conf{$dev}, "LensControl", "AutoFocus", "AutoFocus", "<onOff>$code</onOff>");
		}
		else{
        	checkURL("$lensURL?action=autoFocus&auto=$code");
        }
     } elsif ($cmd=~/autoiris/i) {
     	if($conf{$dev}->{MODELID}=~/^(TXB-N|TXB-N1|D5|ESTI|TI)/) {
			my $respond = checkSOAP($conf{$dev}, "LensControl", "AutoIris", "AutoIris", "<onOff>$code</onOff>");
		}
		else{
	        checkURL("$lensURL?action=autoIris&auto=$code");
	    }
     }
  }
  $last_mode= $mode;
  $last_cmd = $cmd;
}


# ------------------------------------------------------------- UpdatePID -----
sub UpdatePID {
  open(PID,"> $APL/var/ptz/ptz_axis.pid");
  print PID $$;
  close PID;
} # sub UpdatePID

# --------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {
 %conf = GetCfgs( eval("($query_str)") );     # Load configurations
 my $ids='';
 foreach my $dev (keys  %conf) {
  next if not $conf{$dev}->{PTZID} =~/[12]/;
  next if not $conf{$dev}->{DEVIP};
  $ids.=" $dev";
 }
 $log->info("Loaded configurations for cameras:$ids");
}

# -------------------------------------------------------------- checkSOAP -----
sub checkSOAP {
  my @answ;
#  my ($path,$control,$func,$actionfunc,$params)=@_;
  my ($confdev,$control,$func,$actionfunc,$params)=@_;
#  my ($control,$func,$actionfunc,$params,$param,$options)=@_;
my $soapcontent=<<END;
<?xml version="1.0" encoding="UTF-8"?>
<s:Envelope xmlns:s="http://schemas.xmlsoap.org/soap/envelope/"><s:Body><$func xmlns:="urn:schemas-pelco-com:service:$control:1">$params</$func></s:Body></s:Envelope>
END

  my $path = "$confdev->{DEVIP}:$confdev->{HTTP_PORT}";

			my $request =
				POST "http://$path/control/" . $control . "-1",
				"SOAPAction"   => "urn:schemas-pelco-com:service:" . $control . ":1#" . $actionfunc,
				"Content-Type" => "application/xml; charset=UTF-8",
				Content => $soapcontent
			;

			$ua->credentials($path, "Sarix", $confdev->{USRNAME}, $confdev->{PASSWD});
			my $respond=$ua->request($request);
			$log->debug($respond->content);
			return $respond->content;
}

# ----------------------------------------------------------- truncDigits -----
sub truncDigits{
  my($val)=@_;
  return sprintf("%5.5f",$val)
} # sub truncDigits

# -------------------------------------------------------------- nonblock -----
#   nonblock($socket) puts socket into nonblocking mode
sub nonblock {
    my $socket = shift;
    my $flags;
    
    $flags = fcntl($socket, F_GETFL, 0)
            or die "Can't get flags for socket: $!\n";
    fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
            or die "Can't make socket nonblocking: $!\n";
}
