#!/usr/bin/perl
# -----------------------------------------------------------------------------
#  PTZ driver for Panasonic WV-NS324 cameras, Panasonic HTTP API
# -----------------------------------------------------------------------------
#  Author: Serg Pososhenko
#  Edited by: 
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT LLC, 2004-2005
# -----------------------------------------------------------------------------

use strict;
use Socket;
use MIME::Base64();
use IO::File;
use IO::Select;
use IO::Socket;
use Fcntl;

use NextCAM::Init;
use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";


my $log=get_logger('NEXTCAM::PTZ::PTZ_PANASONIC_WV');
my $uid ='';
my $query_str = shift || "'POSITIONCTL'=>'PanasonicWV','CAMERAMODEL'=>'PanasonicPTZ'";
$log->info("Starting PANASONIC_WV PTZ DRIVER, query string: [$query_str]");

$SIG{HUP}=\&load_dev_conf;
my $APL=$ENV{APL};
my (%conf,$cam,$cmd,$usrpsw,$par,$last_cmd,$last_mode);
my ($topend, $bottomend, $leftend, $rightend);
load_dev_conf();



# -----------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# -----------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
                                   PeerPort => $TCP_PORT,
                                   Proto     => "tcp",
                                   Type      => SOCK_STREAM)
    or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';

my %commands;
my $ready;
my $trycount = 30;
				$log->debug( rdUID(4));
while(1) {
	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/i;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/
				|| $commands{$cam} =~ /speed iris=0/
				|| $commands{$cam} =~ /speed gain=0/
				)
			{
				; # do not override stop command!
			}else{
				$commands{$cam} = $cmd;
			}
		} else {
			$commands{$cam} = $cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)
	
	$cmd = '';
	
	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		delete $commands{$camera};
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
		$usrpsw = "$conf{$cam}{USRNAME}:$conf{$cam}{PASSWD}" if $conf{$cam}{USRNAME} && $conf{$cam}{PASSWD};
		
		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam,$mode,"pt", $params[0].','.$params[1] ,\%options);
			camCmd($cam,$mode,"zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam,$mode,$cmd,$par,\%options);
		}
	}

	select(undef,undef,undef,.1) if not $cmd;

	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
		if ($conf{$dev}->{MOUNT} eq 'CELING')
		{
			$topend =9000;
			$bottomend =0;
			$leftend = 18000;
			$rightend =-18000;
		}
		else
		{
			$topend = 0;
			$bottomend = -12000;
			$leftend = -18000;
			$rightend = 18000;
		}
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
	}
	
	if(!$cmd) {
		select(undef,undef,undef,.4);
		if(!$trycount--) {
			$trycount = 30;
			print $socket "test\0" or $log->logdie("Couldn't communicate to socket $TCP_PORT: $@");
		}
	}

} # while(1)




# =============================================================================

# ---------------------------------------------------------------- camCmd -----
sub camCmd
{
  my ($dev,$mode,$cmd,$param,$options)=@_;
	#$log->debug( rdUID($dev));
  my %options = %$options;
  $log->debug("camCmd: DEVID=[$dev] mode=[$mode] command:[$cmd] param=[$param] ");
  my $URL="http://$conf{$dev}{DEVIP}:$conf{$dev}{HTTP_PORT}/cgi-bin/camctrl?";
  my $spd=($options{speed}=~/^\d+/) ? $options{speed}:$conf{$dev}->{PTZSPEED};

  # we need to "terminate" each command because it can move forever if used with no STOP !
  if ($last_mode=~/smooth/i) {
     checkURL("$URL&continuouspantiltmove=0,0") if $last_cmd=~/move/i;
     checkURL("$URL&continuouszoommove=0")	if $last_cmd=~/zoom/i; 
     checkURL("$URL&continuousirismove=0")	if $last_cmd=~/iris/i; 
     checkURL("$URL&continuousfocusmove=0")	if $last_cmd=~/focus/i; 
  }

  # here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
  if($conf{$dev}->{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
     if($mode=~/speed/i || $mode=~/step/i || $mode=~/abs/i || $mode=~/rel/i || $mode=~/smooth/i) {
        $log->debug("Setting timeout +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
        $conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
     }
     elsif($mode=~/preset/i && $cmd=~/goto/i && $param!=1) {
        $log->debug("Setting timeout (preset) +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
        $conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
    }
  }

  if ($mode=~/speed/i){
	my $myURL = "http://$conf{$dev}{DEVIP}/"; #substr ($URL,0,23);
     if($cmd=~/pt/i) { #  pan-tilt
	my ($p,$t) = split(/,/,$param);
	$p = int($p);
	$t = -int($t);
	if ($p ==0 && $t == 0)
		{
		#	checkURL("$myURL"."nphPtzf?Func=Set&PanAng=32768&PanAngRate=1000&TiltAng=32768&TiltAngRate=1000&Sync=0");
		}
		else
		{
			if ($conf{$dev}->{MOUNT} eq 'CELING')
			{	
  	 #    checkURL("$myURL"."nphPtzf?Func=Set&PanAng=".($p*($leftend/100))."&PanAngRate=".(abs($p)*10)."&TiltAng=".(abs($t+100)*45)."&TiltAngRate=".(abs($t)*10)."&Sync=0");
			}
			else
			{
  	 #    checkURL("$myURL"."nphPtzf?Func=Set&PanAng=".($p*($leftend/100))."&PanAngRate=".(abs($p)*10)."&TiltAng=".(($t-90)*64-3200)."&TiltAngRate=".(abs($t)*10)."&Sync=0");	
			}
		}
     }
     elsif($cmd=~/z/i) { # zoom
	my $zm=int($param);
	if ($zm == 0)
		{
		#checkURL("$myURL"."nphPtzf?Func=Set&ZoomMag=32768&ZoomSpeed=362&Sync=0");
    }
	elsif($zm > 0)
		{
		#checkURL("$myURL"."nphPtzf?Func=Set&ZoomMag=4200&ZoomSpeed=".($zm/2)."&Sync=0");
    }
	elsif($zm < 0)
		{
	#	checkURL("$myURL"."nphPtzf?Func=Set&ZoomMag=100&ZoomSpeed=".($zm/(-2))."&Sync=0");
    }
		 }
     elsif($cmd=~/focus/i) { # focus
	my $fc=int($param);
	if ($fc == 0)
		{
	#	checkURL("$myURL"."nphPtzf?Func=Set&FocusPos=32768&Sync=0");
    }else{
	# checkURL("$myURL"."nphPtzf?Func=Set&FocusPos=".(($fc+100)*5)."&Sync=0")
		}
     }
  } elsif($mode=~/abs/i){    # mode=ABS
    if ($cmd=~/pt/i) {
      $param=~/(\-?\d+)\s*,\s*(\-?\d+)\s*$/;
      my $p = $1;
      my $t = $2;
    #  checkURL("$URL&pan=$p&tilt=$t");
    }
    elsif($cmd=~/Z/i) {
      $param=~/(\d+)/;
      my $abs_z = ($1 + 1)*99.98;
    #  checkURL("$URL&zoom=$abs_z");
    }
    elsif($cmd=~/center/i) { # center
    #  checkURL("$URL&pan=0&tilt=0&zoom=1");
    }
  } elsif($mode=~/rel/i){ # relative positioning (recentering)
			my $myURL = "http://$conf{$dev}{DEVIP}/";
      $param =~ /(\d+)x(\d+)/;
      my ($rel_size_x,$rel_size_y) = ($1,$2);
      $options{xy} =~ /(\d+),(\d+)/;
      my ($rel_click_x,$rel_click_y) = ($1,$2);
      my ($rel_p, $rel_t) = ( int(3.5* ($rel_click_x-$rel_size_x/2)), int(3.5* ($rel_click_y-$rel_size_y/2)));
        if ($rel_p >0){
					$rel_p = "&PanAng=".$leftend."&PanAngRate=".(abs($rel_p)); 
				}
				elsif($rel_p <0){
        	$rel_p = "&PanAng=".$rightend."&PanAngRate=".(abs($rel_p)); 
				}elsif($rel_p == 0){
        	$rel_p = '';      # no movement in this direction
				}
				if ($rel_t >0){
        	$rel_t ="&TiltAng=".$bottomend."&TiltAngRate=".(abs($rel_t)); 
				}elsif($rel_t <0){
        	$rel_t ="&TiltAng=".$topend."&TiltAngRate=".(abs($rel_t)); 
				}elsif($rel_t == 0){
        	$rel_t = '' ;      # no movement in this direction
				}

			#checkURL("$myURL"."nphPtzf?Func=Set".$rel_p.$rel_t."&Sync=0");
			select(undef,undef,undef,.1);
			#checkURL("$myURL"."nphPtzf?Func=Set&PanAng=32768&PanAngRate=1000&TiltAng=32768&TiltAngRate=1000&Sync=0");
			
      
  } elsif($mode=~/step/i) {       # mode=step  /Step by step positioning/
     if($cmd=~/move/i) { # step pan/tilt
		 	my $direction = 'HomePosition';
		 	if ($param=~/^left\b/i){
				$direction='PanLeft';
				checkURL("$URL"."PAN=4"."&TILT=0");
			}
			elsif($param=~/^right\b/i){
				$direction='PanRight';
				checkURL("$URL"."PAN=-4"."&TILT=0");
			}
			elsif($param=~/^up\b/i){
				$direction='TiltUp';
				checkURL("$URL"."TILT=-4"."&PAN=0");
			}
			elsif($param=~/^down\b/i){
				$direction='TiltDown';
				checkURL("$URL"."TILT=4"."&PAN=0");
			}
	    elsif($param=~/^downright$/i){ 
				checkURL("$URL"."PAN=-4&TILT=4");
			}
  	  elsif($param=~/^downleft$/i) { 
				checkURL("$URL"."PAN=4&TILT=4");
			}
    	elsif($param=~/^upright$/i)	{ 
				checkURL("$URL"."PAN=-4=&TILT=-4");
			}
    	elsif($param=~/^upleft$/i)	{ 
				checkURL("$URL"."PAN=4&TILT=-4");
			};
#			checkURL("$URL"."Direction=$direction")
     }
     elsif($cmd=~/zoom/i) {
			my $zm = 'ZOOM=-3' ;
			$zm='ZOOM=3' if $param=~/in/i;	
			checkURL("$URL".$zm);
     }
     elsif($cmd=~/focus/i) {
			my $focus ='FocusNear';
			$focus ='FocusFar' if $param=~/far/i;
			checkURL("$URL"."ZOOM=3");
     }
     elsif($cmd=~/iris/i) { 
			my $iris ='IRIS=-2';
			$iris ='IRIS=2' if $param=~/up/i;
			checkURL("$URL"."Direction=$iris");
     }

  } elsif($mode=~/smooth/i){  # mode=smooth  /Old style PTZ. Goes to direction
                              # untill stop (or any other command is sent)/
     if($cmd=~/move/i) { # pan/tilt
	$spd=3 if not $spd;
	my $pan = $param=~/left/i? -$spd : $param=~/right/i? $spd : 0;
	my $tlt = $param=~/down/i? -$spd : $param=~/up/i? $spd : 0;
	checkURL("$URL&continuouspantiltmove=$pan,$tlt");
     }
     elsif($cmd=~/zoom/i) { # zoom
	$spd=1 if not $spd;
	my $zm = $param=~/out/? -$spd : $spd;
	checkURL("$URL&continuouszoommove=$zm")
     }
     elsif($cmd=~/iris/i) { # iris
	$spd=1 if not $spd;
	my $iris = $param=~/close/i? -$spd : $spd;
	checkURL("$URL&continuousirismove=$iris")
     }
     elsif($cmd=~/focus/i) { # focus
	$spd=1 if not $spd;
	my $focus = $param=~/far/i? -$spd : $spd;
	checkURL("$URL&continuousfocusmove=$focus")
     }
  } elsif($mode=~/preset/i){ # presets
	#rdUID($dev);
	my $code = 'gotodevicepreset';	# default is goto
	my $myURL = "http://$conf{$dev}{DEVIP}/"; #substr($URL,0,23);
	$code="/cgi-bin/camctrlid?UID=$uid"."$param&Data"	if $cmd=~/goto/i;
	$code="/cgi-bin/camposiset?PRESETSET=$param&Data"		if $cmd=~/save/i;
	$code=""	if $cmd=~/clear/i;
	print $code;
	#checkURL("$myURL$code=$param");
  } elsif($mode=~/settings/i){ # settings
     my $code = '';
     if ($cmd=~/timeout/i) {
        $log->debug("Setting timer: [$param]");
        $conf{$dev}->{TIMEOUT} = time + $param;
     }
     elsif ($cmd=~/autofocus/i) {
	$code = "FocusAuto";
     } elsif ($cmd=~/autoiris/i) {
	$code ="DefaultBrightness";
     } elsif ($cmd=~/whitebalance/i) {
	$code ='';
	my $myURL = "http://$conf{$dev}{DEVIP}/"; #substr ($URL,0,23);
	if ($param=~/auto/i){
	#checkURL("$myURL"."nphPtzf?Func=Set&WBMode=0&Sync=0");
     }
	elsif($param=~/outdoor/i)
	{
	#checkURL("$myURL"."nphPtzf?Func=Set&WBMode=80&Sync=0");
	}
	elsif($param=~/indoor/i)
	{
	#checkURL("$myURL"."nphPtzf?Func=Set&WBMode=32&Sync=0");
	}

	}
    #checkURL("$URL"."Direction=$code") if $code;
  }
  $last_mode= $mode;
  $last_cmd = $cmd;
}

# ------------------------------------------------------------- UpdatePID -----
sub UpdatePID {
  open(PID,"> $APL/var/ptz/ptz_panasonic.pid");
  print PID $$;
  close PID;
} # sub UpdatePID

# --------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {
 %conf = GetCfgs( eval("($query_str)") );     # Load configurations
 my $ids='';
 foreach my $dev (keys  %conf) {
  next if not $conf{$dev}->{PTZID} =~/[12]/;
  next if not $conf{$dev}->{DEVIP};
  $ids.=" $dev";
  camParams($dev);
 }
 $log->info("Loaded configurations for cameras:$ids");
}

# ------------------------------------------------------------- camParams -----
sub camParams {
  my ($_dev)=@_;
  if($conf{$_dev}{PTZPARAMS}=~/(.+):(.+):(.+):(.+)/) {
    $conf{$_dev}{pan}=$1;
    $conf{$_dev}{tilt}=$2;
    $conf{$_dev}{zoom}=$3;
    $conf{$_dev}{speed}=$4;
  }
  elsif($conf{$_dev}{PTZPARAMS} eq 'SONY EVI-D30/31') {
    $conf{$_dev}{pan}=1.66;
    $conf{$_dev}{tilt}=0.54;
    $conf{$_dev}{zoom}=625;
    $conf{$_dev}{speed}=20;
  }
  elsif($conf{$_dev}{PTZPARAMS} eq 'SONY EVI-D100/D100P') {
    $conf{$_dev}{pan}=2.78;
    $conf{$_dev}{tilt}=0.69;
    $conf{$_dev}{zoom}=10000;
    $conf{$_dev}{speed}=20;
  }
  else {
    $conf{$_dev}{pan}=1;
    $conf{$_dev}{tilt}=1;
    $conf{$_dev}{zoom}=1000;
    $conf{$_dev}{speed}=100;
  }

  if($conf{$_dev}{PTZSTEPS}=~/(.+):(.+):(.+)/) {
    $conf{$_dev}{span}=$1;
    $conf{$_dev}{stilt}=$2;
    $conf{$_dev}{szoom}=$3;
  }
  else {
    $conf{$_dev}{span}=50;
    $conf{$_dev}{stilt}=50;
    $conf{$_dev}{szoom}=50;
  }
  $conf{$_dev}{p_pos}=0;
  $conf{$_dev}{t_pos}=0;
  $conf{$_dev}{z_pos}=0;
} # sub camParams


# -------------------------------------------------------------- checkURL -----
sub checkURL {
  my @answ;
  my $val=-2;
  my($url)=@_;  #print "URL: $url\n";
  $log->debug("checkURL($url)");
	$log->debug("checkURL($usrpsw)");
  my($server,$port,$path)=($url=~/^http:\/\/([\w\.]+):?(\d+)?\/(.*)/)? ($1,$2,$3):('','');
  $port = 80 if not $port;
	`curl "http://$usrpsw\@$server/$path"`;  
  $val = 1;
  return $val;
} # sub checkURL

# ----------------------------------------------------------- truncDigits -----
sub truncDigits{
  my($val)=@_;
  return sprintf("%5.5f",$val)
} # sub truncDigits

# -------------------------------------------------------------- nonblock -----
#   nonblock($socket) puts socket into nonblocking mode
sub nonblock {
    my $socket = shift;
    my $flags;
    
    $flags = fcntl($socket, F_GETFL, 0)
            or die "Can't get flags for socket: $!\n";
    fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
            or die "Can't make socket nonblocking: $!\n";
}
# -------------------------------------------------------------- rdUID -----
#   read UID - for preset command
sub rdUID {
		my $dev = $_;
		open (UID,"$APL/store/live/$conf{$dev}->{DEVID}.uid");
    while ($uid = <UID>)
		{
		chomp ($uid);
		}
		close (UID);
}