#!/usr/bin/perl
#  $Id: ptz_odis.pl 25370 2012-03-09 21:19:34Z teetov $
# -----------------------------------------------------------------------------
#  PTZ driver for Odis
# -----------------------------------------------------------------------------
#  Author: Jeff Scott
#  Edited by: 
#  Copyright: (c) videoNEXT LLC, 2010-2011
# -----------------------------------------------------------------------------

use strict;

use Switch;
use POSIX;
use IO::Socket;
use IO::Select;
use Tie::RefHash;
use IO::File;
use Socket;
use Fcntl;
use Tie::RefHash;
use NextCAM::Init;
use POSIX "fmod";

use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

use XML::Simple;
use NextCAM::Messenger;
use NextCAM::Messenger::Shared;
use NextCAM::Messenger::Flags;

my $log=get_logger('NEXTCAM::PTZ::PTCR20FLIR');

my $query_str = shift || "'POSITIONCTL'=>'odis'";
$log->info("Starting ODIS PTZ DRIVER, query string: [$query_str]");


$SIG{HUP}=\&load_dev_conf;

my $APL=$ENV{APL};
my (%conf, %sock, $cam,$cmd,$par,$usrpsw,$last_cmd,$last_mode,$cmd_executed);
my $ETX = 0x03;
my $STX = 0x02;
my $high_speed = 0;   ## Flag -- 1 for high, 0, for low
my $speedScale = 0x00; #used for incremental speed settings

load_dev_conf();

# -----------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# -----------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
                                   PeerPort => $TCP_PORT,
                                   Proto     => "tcp",
                                   Type      => SOCK_STREAM)
    or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';

my %commands;
my $ready;
my $trycount = 30;
my %lastcmd;
my $lastcmdtime = time;
my $polling = 0;


foreach my $dev (keys %conf) {
  my %options = {};
  camCmd($dev,'settings','init','',\%options);
}

while(1) {
    # stage 1 - read from socket everything
    if (defined($cmd=<$socket>)) {
        chomp $cmd;
        $log->debug("COMMAND:[$cmd]");
        next if not $cmd;
        load_dev_conf(),next if $cmd=~/HUP/i;
        next if not $cmd=~/^(\d+)/;
        $cam=$1;
        next if not defined $conf{$cam}; # ignore cameras not belonging this engine
        if(defined($commands{$cam})){
	        if($commands{$cam}=~/speed pt=0,0/ || $commands{$cam}=~/speed zoom=0/ || $commands{$cam}=~/speed focus=0/ || $commands{$cam}=~/speed iris=0/ || $commands{$cam}=~/speed gain=0/)
			{
	            ; # do not override stop command!
    	    }else{
	            $commands{$cam} = $cmd;
	        }
        } else {
	        $commands{$cam} = $cmd;
        }
        next; # suck all the commands from queue before going to the rest of loop
    } # if ($cmd=<$socket>)
    
    $cmd = '';
    
    # stage 2 - send commands to cameras
    foreach my $camera (keys %commands){
        $cmd=$commands{$camera};
        delete $commands{$camera};
        next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
        my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
        my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
        $usrpsw = "$conf{$cam}{USRNAME}:$conf{$cam}{PASSWD}" if $conf{$cam}{USRNAME} && $conf{$cam}{PASSWD};
        camCmd($cam,$mode,$cmd,$par,\%options);
        $lastcmd{$camera}{CMD} = $cmd;
        $lastcmd{$camera}{MODE} = $mode;
        $lastcmd{$camera}{PAR} = $par;
        $lastcmd{$camera}{OPTS} = \%options;
        $polling = 0;
    }
    
    select(undef,undef,undef,.1) if not $cmd;

    if(!$cmd) {
        foreach my $dev (keys %conf) {
            my %options = {};
            $polling = 1;
            if(defined($lastcmd{$dev})) {
                if($lastcmd{$dev}{MODE} eq 'speed') {
                    if($lastcmd{$dev}{CMD} eq 'gain' and $lastcmd{$dev}{PAR} == 0) {
                        camCmd($dev,'speed','pt','0,0',\%options);
                    }
                    else {
                        camCmd($dev,$lastcmd{$dev}{MODE},$lastcmd{$dev}{CMD},$lastcmd{$dev}{PAR},$lastcmd{$dev}{OPTS});
                    }
                }
                else {
                    camCmd($dev,'speed','pt','0,0',\%options);
                }
                select(undef,undef,undef,.1);
            }
            else {
                camCmd($dev,'speed','pt','0,0',\%options);
                select(undef,undef,undef,.1);
            }
        }        
    }

} # while(1)




# ----------------------------------------------------------- cmdTransmit -----
sub Idle {
	my @a=();
	$a[0]=0x24;$a[1]=0x00;$a[2]=0x00;$a[3]=0x2E;$a[4]=0x00;$a[5]=0x1E;
	$a[6]=0x00;$a[7]=0x00;$a[8]=0x00;$a[9]=0x00;$a[10]=0x00;$a[11]=0x00;
	$a[12]=0x00;$a[13]=0x00;$a[14]=0x00;$a[15]=0x00;#xDot
	$a[16]=0x00;$a[17]=0x00;$a[18]=0x00;$a[19]=0x00;#yDot
	$a[20]=0x00;$a[21]=0x00;$a[22]=0x00;$a[23]=0x00;#yawDot
	$a[24]=0x00;$a[25]=0x00;$a[26]=0x00;$a[27]=0x00;$a[28]=0x00;$a[29]=0x01;$a[30]=0x00;$a[31]=0x01;
	$a[32]=0x00;$a[33]=0x00;$a[34]=0x00;$a[35]=0x00;#Tilt
	$a[36]=0x00;$a[37]=0x00;$a[38]=0x00;$a[39]=0x00;#Pan
	$a[40]=0x00;$a[41]=0x00;$a[42]=0x00;$a[43]=0x00;#Zoom
	$a[44]=0;
	$a[45]=0x5D;
	$a[44]=chkSum(@a);
	return @a;
}
sub Left {
	#16-19 are for left/right
	my (@a)= @_;
	$a[16]=0xBF;#y
	$a[17]=$speedScale;#scalar factor range 00-ff
	$a[44]=0;
	$a[44]=chkSum(@a);
	return @a;
}
sub Right {
	#16-19 are for left/right
	my (@a)= @_;
	$a[16]=0x3F;#y
	$a[17]=$speedScale;#scalar factor range 00-ff
	$a[44]=0;
	$a[44]=chkSum(@a);
	return @a;
}
sub Forward {
	#12-15 are for left/right
	my (@a)= @_;
	$a[12]=0x3F;#x
	$a[13]=$speedScale;#scalar factor range 00-ff
	$a[44]=0;
	$a[44]=chkSum(@a);
	return @a;
}
sub Reverse {
	#12-15 are for left/right
	my (@a)= @_;
	$a[12]=0xBF;#x
	$a[13]=$speedScale;#scalar factor range 00-ff
	$a[44]=0;
	$a[44]=chkSum(@a);
	return @a;
}
sub CW {
	#20-23 are for left/right
	my (@a)= @_;
	$a[20]=0x3F;#yaw
	$a[21]=$speedScale;#scalar factor range 00-ff
	$a[44]=0;
	$a[44]=chkSum(@a);
	return @a;
}
sub CCW {
	#20-23 are for left/right
	my (@a)= @_;
	$a[20]=0xBF;#yaw
	$a[21]=$speedScale;#scalar factor range 00-ff
	$a[44]=0;
	$a[44]=chkSum(@a);
	return @a;
}
sub cmd2robot {
    my ($dev,@cmd)=@_;
    #my $command=pack("W46",@cmd);
    my $command=pack("C46",@cmd);
    ReconnectSocket($dev);
    my $skt = $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK};
    my $sel = $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SEL};
  
    eval {
        local $SIG{PIPE} = sub { die "Error writing to socket" };
        print $skt $command;
    };
    $log->error($@) if $@;
    foreach my $sss ( $sel->can_read(1) ) {
        my $data = '';
        eval {
            alarm 1;
            $sss->recv($data, POSIX::BUFSIZ, 0);
            if ($data =~ /GPGGA(.+)\*/ ) {
		geoPush($1,$dev);
            }
            $command='';
	    
        };
        alarm 0;
    }  # foreach      
}

sub geoPush{
    my ($data,$dev) = @_;
    # If all we care about is position data
    # then the prefix will be $GPGGA
    
    # Each value is comma separated
    
    # $GPGGA,<Time>,<Lat>,<LatH>,<Lon>,<LonH>,<Qual>,<Sats>,<hper>,<alt>,<gAlt>
    
    my @values = split(',',$data);
    
    my (undef,$time,$latitude,$latHemisphere,$longitude,$longHemisphere,$qual,$satellites,$hDilution,$alt,undef,$altG,undef,undef,undef) =     @values;


    $log->debug($data);
    # Convert lat/long to decimal format
    my $lat_part=floor($latitude/100);
    my $lat_dec=fmod($latitude,100)/60;
    $latitude=$lat_part+$lat_dec;

    my $long_part=floor($longitude/100);
    my $long_dec=fmod($longitude,100)/60;
    $longitude=$long_part+$long_dec;
   
    if ($latHemisphere=~/S/){
	$latitude*=-1;
    }

    if ($longHemisphere=~/W/){
	$longitude*=-1;
    }

	$log->debug("Latitude: ", $latitude, " Longitude: ", $longitude, " Alt: ", $altG, " Satellites: ", $satellites);

    my $geo="                                                      \
    <ptz topic=\"status\">                                                  \
	<camera id=\"%s\">                                                  \
	    <status>                                                        \
		<GEO>                                                       \
		    <position lat=\"%f\" long=\"%f\" alt=\"%f\" alt_mode=\"relativeToGround\"/> \
		    <direction azim=\"0.0\" tilt=\"0.0\" hfov=\"90\"/>      \
                </GEO>                                                      \
	    </status>                                                       \
	</camera>                                                           \
    </ptz>";

												    
    my $messenger = NextCAM::Messenger->new(
	    host=>"localhost",
	    port=>"10000",
	    conn_timeout=>1000,
	    );
    
    unless (defined $messenger){ die "Failed to connect!\n";}

    my $sid=$messenger->sid;

    my $msg = {
	    device_IP=>127.0.0.1,
	    device_ID=>$dev,
	    ctxType=>0x0010,
	    evtType=>0x0004,
	    procLvl=>0x0002,
	    message_ID=>0,
	    data=>""
	    };

    $msg->{data} = sprintf($geo, $dev, $latitude, $longitude, 0);

    my $ret = $messenger->send($msg);

    if(not defined $ret){
	$log->error('Could not send geo message');    
    }
}

sub cmdTransmit {
    my ($dev,@cmd)=@_;

    @cmd = checkEscape(@cmd);
   
    my $command='';
    my $b;
    my $skt;
    my $sel;
    foreach $b (@cmd) { $command .= sprintf("%02X ",$b); }
    if ($cmd[2] != 0x31) {
	$log->debug('cmdTransmit( ',$dev, '=> ',$command,')');    
    }
    ReconnectSocket($dev);
    $skt = $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK};
    $sel = $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SEL};

    my $bts = '';
    foreach my $b (@cmd) { $bts .= chr($b); }
    eval {
        local $SIG{PIPE} = sub { die "Error writing to socket" };
        print $skt $bts;
    };
    $log->error($@) if $@;
    #select(undef,undef,undef,.15);
    foreach my $sss ( $sel->can_read(1) ) {
        my $data = '';
        eval {
            alarm 1;
            $sss->recv($data, POSIX::BUFSIZ, 0);
            $command='';
            for($b=0;$b<length($data);$b++) { $command .= sprintf("%02X ",ord(substr($data,$b,1))); }
            $log->debug('Received:',$command);
        };
        alarm 0;
        return $data;
    }  # foreach      

} # sub cmdTransmit


#--------------------------------------------------- Robot Check Sum ----------
sub chkSum {
#Entire command minus chksum byte (55 bytes
	(my @cmd )= @_;
	my $sum=0;
	foreach (@cmd) { $sum += $_; }
	$sum = $sum & 0x000000FF;
	$sum = 0x100 - $sum;
	return $sum;
}

# ----------------------------------------------------------- checkEscape -----
sub checkEscape
{
  my @arr = @_;
  my @res = ($arr[0]);
  my $r = 1;
  for ( my $i = 1; $i <= $#arr; $i++) {
    if ( $#arr - 1 > $i && ( $arr[$i] == $STX || $arr[$i] == $ETX || $arr[$i] == 0x06 || $arr[$i] == 0x15 || $arr[$i] == 0x1B ) ) {
      $res[$r++] = 0x1B;
      $res[$r++] = $arr[$i] | 0x80;
    } else {
      $res[$r++] = $arr[$i];
    }
  }
  return @res;
} # sub checkEscape

# --------------------------------------------------------------- camStop -----
sub camStop
{
    my ($dev) = @_;
    my %options = {};
#    camCmd($dev,'speed','pt','0,0',\%options);
}

# -------------------------------------------------------------- camReset -----
sub camReset
{
} # sub camReset

sub camFOV
{
}

# ------------------------------------------------------------- camZoom -----
sub camZoom
{
#Hijacked for Yaw Control
#
# Zoom Wide = Clockwise?
# Zoom Narrow = Counter-Clockwise?
  # ZOOM ZOOM:
  # 0 -- Wide 1x >>> slowest
  # 1 -- Wide 2x >>> med
  # 2 -- Wide 4x >>> fastest
  # 3 -- Narrow 1x
  # 4 -- Narrow 2x
  # 5 -- Narrow 4x

  my ($dev,$step,$speed,$qptCamID) = @_;
  my @cmd = ();

  my $zfile = '/opt/sarch/var/conf/' . $cam .'/zoom';
  my $zlevel=1;

  # If the file exists, read it in, otherwise, assume 1.
  if (-e $zfile) {
    open(zfilein, $zfile);
    $zlevel=<zfilein>;
    close(zfilein);
  }

  $log->debug($zlevel);

  if ($step != 0) {
    $zlevel = $zlevel + $step;
  }
  elsif ($speed != 0) {
    $zlevel = int($zlevel + int($speed) * (3 / 100));
  }

  ### Sanity check on zlevel first.
  if ($zlevel > 5) {
    $zlevel = 5;
  }
  if ($zlevel < 0) {
    $zlevel = 0;
  }

  my $s; 
  $cmd[0]=0x24;
  $cmd[1]=0x00;
  $cmd[2]=0x00;
  $cmd[3]=0x2E;
  $cmd[4]=0x00;
  $cmd[5]=0x1E;
  foreach $s (6..10) {$cmd[$s]=0x00;}
  $cmd[11]=0x01;
  foreach $s (12..19) {$cmd[$s]=0x00;}
  $cmd[20]=($zlevel<3)?0xBF:0x3F;
  if($zlevel == 1 || $zlevel == 4){
	$cmd[21]=0x0F;
  }elsif($zlevel == 2 || $zlevel == 5) {
	$cmd[21]=0xF0;
  }else{
	$cmd[21]=0x00;
  }
  foreach $s (22..28) {$cmd[$s]=0x00;}
  $cmd[29]=0x01;
  $cmd[30]=0x00;
  $cmd[31]=0x01;
  foreach $s (32..43) {$cmd[$s]=0x00;}
  $cmd[44]=chkSum(@cmd); 
  $cmd[45]=0x5D;
  cmd2robot($dev,@cmd);

} # sub camZoom

# ---------------------------------------------------------------- PTspeed ----
sub PTspeed {
  my ($dev,$speed)=@_;
  $speed = int( 1.27 * $speed * $conf{$dev}{PTZSPEED} / 100.0 );
  if($speed <= 0 ) {
       $speed = -127 if $speed < -127;
       $speed = -$speed * 1;
  } 
  else {
       $speed = 127 if $speed > 127;
       $speed = $speed * 1 + 1;
  }
  return $speed;
} # sub PTspeed
# ---------------------------------------------------------------- camCmd -----
sub camCmd {
    my ($dev,$mode,$cmd,$param,$options)=@_;
    my %options = %$options;
    my @cmd = ( $STX , $conf{$dev}{PTZID}); # camera hardware ID

    my $qptCamID=2;
#    $log->debug("camCmd: DEVID=$dev mode: $mode command: $cmd optional parameter: $param");
#    open LOG, ">>/tmp/serial" || `echo ERROR >>/tmp/serial`;
#    print LOG "camCmd: mode: $mode command: $cmd param:$param\n";
#    close LOG;

    # here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
    if($conf{$dev}{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
        if(not $polling and ($mode=~/speed/i || $mode=~/step/i || $mode=~/abs/i || $mode=~/rel/i || $mode=~/smooth/i)) {
#            $log->debug("Setting timeout +$conf{$dev}{PTZ_PRESET1TIMEOUT}");
            $conf{$dev}{TIMEOUT} = time + $conf{$dev}{PTZ_PRESET1TIMEOUT};
        }
        elsif(not $polling and ($mode=~/preset/i && $cmd=~/goto/i && $param!=1)) {
#            $log->debug("Setting timeout (preset) +$conf{$dev}{PTZ_PRESET1TIMEOUT}");
            $conf{$dev}{TIMEOUT} = time + $conf{$dev}{PTZ_PRESET1TIMEOUT};
        }
    }

    if(not $cmd=~/stop/) { # non-Stop
        ########################################################################
        if ($mode=~/speed/i) {        # mode speed
            if($cmd=~/PT/i) { # RPT		 
                my ($p,$t) = split(/,/,$param);#pan=y t=x
		my @a=Idle(); # create an idle frame
			      # if no input happens, idle gets sent
		if($t>15) {
			$a[12]=0xBF;
			$a[13]=PTspeed($dev,$t);
		}elsif($t<-15) {
			$a[12]=0x3F;
			$a[13]=PTspeed($dev,$t);
		}
		
                if($p<-15) {
			$a[16]=0xBF;
			$a[17]=PTspeed($dev,$p);
		}elsif($p>15) {
			$a[16]=0x3F;
			$a[17]=PTspeed($dev,$p);
		}
		$a[44]=0;
		$a[44]=chkSum(@a);
		cmd2robot($dev,@a);
		#select(undef,undef,undef,0.1);#driver seems to freeze from being flooded with commands, so slow them down a bit. 
            }
            elsif($cmd=~/zoom/i) {
              camZoom($dev, 0, $param, $qptCamID);    
            }
            elsif($cmd=~/focus/i) {
		my @a=Idle();
              if ($param < 0) {
		$a[32]=0xC1;
		$a[33]=0xF0;
		$a[44]=0;
		$a[44]=chkSum(@a);
		cmd2robot($dev,@a);
		#select(undef,undef,undef,.1);
		cmd2robot($dev,Idle());
#                camFocus($dev, $param / 50, 1, $qptCamID);
              }
              elsif($param > 0) {
		$a[32]=0x41;
		$a[33]=0xF0;
		$a[44]=0;
		$a[44]=chkSum(@a);
		cmd2robot($dev,@a);
		#select(undef,undef,undef,.1);
		cmd2robot($dev,Idle());
#                camFocus($dev, -1 * $param / 50, 0, $qptCamID);
              }
            }
        }
        ########################################################################
        elsif($mode=~/rel/i) { # mode=rel: "<devid> rel size=640x480 xy=225,152"
        }
        ########################################################################
        elsif ($mode=~/settings/i){ # settings
	    if($cmd=~/zoom/i) {
		my @a=Idle();
              if ($param=~/in/i) {
		$a[32]=0x41;
		$a[33]=0xF0;
		$a[44]=0;
		$a[44]=chkSum(@a);
		cmd2robot($dev,@a);
		#select(undef,undef,undef,.2);
		cmd2robot($dev,Idle());
              }
              elsif($param=~/out/i) {
		$a[32]=0xC1;
		$a[33]=0xF0;
		$a[44]=0;
		$a[44]=chkSum(@a);
		cmd2robot($dev,@a);
		#select(undef,undef,undef,.2);
		cmd2robot($dev,Idle());
	      }
	    }
        } 
        ########################################################################
        elsif($mode=~/step/i){       # mode=step  /Step by step positioning/
            if($cmd=~/move/i) { # step pan/tilt
		my @a=Idle();
                if ( $param=~/right/i ) {
			#@a=Right(@a);
			@a=CW(@a);
                }
                elsif ( $param=~/left/i ) {
			#@a=Left(@a);
			@a=CCW(@a);
                }
                if ( $param=~/down/i ) {
			@a=Reverse(@a);
                }
                elsif ( $param=~/up/i ) {
			@a=Forward(@a);
                }
		cmd2robot($dev,@a);
		#select(undef,undef,undef,.2);
                $log->debug("STEP/MOVE:$param");
            } 
            elsif ($cmd=~/focus/i){
            } 
	    elsif($cmd=~/zoom/i) {
		my @a=Idle();
              if ($param=~/in/i) {
		$a[32]=0x41;
		$a[33]=0xF0;
		$a[44]=0;
		$a[44]=chkSum(@a);
		cmd2robot($dev,@a);
		#select(undef,undef,undef,.2);
		cmd2robot($dev,Idle());
              }
              elsif($param=~/out/i) {
		$a[32]=0xC1;
		$a[33]=0xF0;
		$a[44]=0;
		$a[44]=chkSum(@a);
		cmd2robot($dev,@a);
		#select(undef,undef,undef,.2);
		cmd2robot($dev,Idle());
	      }
            }
        }
        ########################################################################
        elsif($mode=~/preset/i){ # presets
	    switch($param){
		case 1 { $speedScale=0x00}
		case 2 { $speedScale=0x0F}
		case 3 { $speedScale=0x30}
		case 4 { $speedScale=0xFF}
	    }
        } 
        ########################################################################
        elsif($mode=~/smooth/i) { # Right/Left/Up/Down/Tele/Wide/focus_Near/focus_Far/iris_Open/irIs_close
            return;
        } # modes
    }
}


sub nonblock {
    my ($fd) = @_;
    my $flags = fcntl($fd, F_GETFL,0);
    fcntl($fd, F_SETFL, $flags|O_NONBLOCK);
}

sub ReconnectSocket {
    my $dev = shift;
    
    if(not defined($sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}})) {
        $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK} = IO::Socket::INET->new(PeerAddr => $conf{$dev}{DEVIP},
                                                PeerPort => $conf{$dev}{PTZ_TCP_PORT},
                                                Proto    => "tcp",
                                                Type     => SOCK_STREAM);
        if(! $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK} ) {
            #$log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
            delete $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}};
            next;
        }
        $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK}->autoflush(1);
        nonblock($sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK});
        $sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}.$conf{$dev}{PTZ_TCP_PORT}}{SOCK});
    }
}


# --------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {

    $log->debug('load_dev_conf');
    
    # fisrst, close everything
    foreach my $skt (keys %sock) {
        close($sock{$skt}{SOCK});
        delete $sock{$skt};
    }


    %conf = GetCfgs( eval("($query_str)") );     # Load configurations
    $log->debug("Config read as:");
    $log->debug('-------------------------------------');
    foreach my $dev (keys %conf) {
        $log->debug("[$dev]");
        $log->debug("DEVID=$conf{$dev}{DEVID}");
        $log->debug("CAMERAMODEL=$conf{$dev}{CAMERAMODEL}");
        $log->debug("CAMERA NUMBER=$conf{$dev}{CAMERA}");
        $log->debug("POSITIONCTL=$conf{$dev}{POSITIONCTL}");
        $log->debug("PTZID=$conf{$dev}{PTZID}");
        $log->debug("PTZ_TCP_PORT=$conf{$dev}{PTZ_TCP_PORT}");
        $log->debug("-------------------------------------");
        ReconnectSocket($dev);
        $log->debug("PTZ_TCP_PORT=$conf{$dev}{PTZ_TCP_PORT}");
    } # foreach $dev
 # close unused ports

}

