#!/usr/bin/perl
#  $Id: ptz_nve.pl
# -----------------------------------------------------------------------------
#  PTZ driver for UDP NVE cameras, NVE HTTP PTZ2 API
# -----------------------------------------------------------------------------
#  Author: Vladimir Ryabovol
#  Edited by: 
#  QA by: 
#  Copyright: (c) videoNEXT LLC, 2008
# -----------------------------------------------------------------------------

use strict;
use Socket;
use MIME::Base64();
use IO::File;
use IO::Select;
use IO::Socket;
use Fcntl;
use LWP::UserAgent;
use HTTP::Request::Common;

use NextCAM::Init;
use NextCAM::PTZ;
use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

my $log=get_logger('NEXTCAM::PTZ::PTZ_NVE');

my $query_str = shift || "'POSITIONCTL'=>'NVE','CAMERAMODEL'=>'NVE'";
$log->info("Starting NVE PTZ DRIVER, query string: [$query_str]");
                                                                     
my $ua = LWP::UserAgent->new();
                                                                     
$SIG{HUP}=\&load_dev_conf;
my $APL=$ENV{APL};
my $APL_CONF=$ENV{APL_CONF};
my (%conf,$cam,$cmd,$usrpsw,$par,$last_cmd,$last_mode);
load_dev_conf();

# -----------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# -----------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
                                   PeerPort => $TCP_PORT,
                                   Proto     => "tcp",
                                   Type      => SOCK_STREAM)
    or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';

my %commands;
my $ready;
my $trycount = 30;

while(1) {
	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/i;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/
				|| $commands{$cam} =~ /speed iris=0/
				|| $commands{$cam} =~ /speed gain=0/
				)
			{
				; # do not override stop command!
			}else{
				$commands{$cam} = $cmd;
			}
		} else {
			$commands{$cam} = $cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)
	
	$cmd = '';
	
	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		delete $commands{$camera};
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
		httpAuth($conf{$cam}{USRNAME}, $conf{$cam}{PASSWD});
		
		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam,$mode,"pt", $params[0].','.$params[1] ,\%options);
			camCmd($cam,$mode,"zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam,$mode,$cmd,$par,\%options);
		}
	}

	select(undef,undef,undef,.1) if not $cmd;

	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
	}
	
	if(!$cmd) {
		select(undef,undef,undef,.4);
		if(!$trycount--) {
			$trycount = 30;
			print $socket "test\0" or $log->logdie("Couldn't communicate to socket $TCP_PORT: $@");
		}
	}

} # while(1)




# =============================================================================

# ---------------------------------------------------------------- PTspeed ----
sub PTspeed {
	my ($speed)=@_;
	$speed=100 if $speed>100;
	$speed=-100 if $speed<-100;
	return sprintf("%d", $speed);
} # sub PTspeed

# ---------------------------------------------------------------- camCmd -----
sub camCmd
{
	my ($dev,$mode,$cmd,$param,$options)=@_;
	my %options = %$options;
	$log->debug("camCmd: DEVID=[$dev] mode=[$mode] command:[$cmd] param=[$param] ");
	my $URL="http://$conf{$dev}{DEVIP}:$conf{$dev}{HTTP_PORT}/enc-cgi/ptz/ptz2.cgi?ptzid=$conf{$dev}{PTZID}";
	my $spd=($options{speed}=~/^\d+/) ? $options{speed}:$conf{$dev}->{PTZSPEED};
	my $zoomspd=$conf{$dev}->{PTZZOOMSPEED};

	# we need to "terminate" each command because it can move forever if used with no STOP !
	if ($last_mode=~/smooth/i) {
		checkURL("$URL&cpantiltmove=0,0")	if $last_cmd=~/move/i;
		checkURL("$URL&czoommove=0")	if $last_cmd=~/zoom/i;
		checkURL("$URL&cirismove=0")	if $last_cmd=~/iris/i;
		checkURL("$URL&cfocusmove=0")	if $last_cmd=~/focus/i;
	}

	# here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
	if($conf{$dev}->{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
		if($mode=~/speed/i || $mode=~/step/i || $mode=~/abs/i || $mode=~/rel/i || $mode=~/smooth/i) {
			$log->debug("Setting timeout +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
		elsif($mode=~/preset/i && $cmd=~/goto/i && $param!=1) {
			$log->debug("Setting timeout (preset) +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
	}
	
	if ($mode=~/speed/i){
		$spd=$conf{$dev}->{PTZSPEED};
		$spd=1 if $spd < 1;
		$spd=100 if $spd > 100;
		if($cmd=~/pt/i) { #  pan-tilt
			my ($p,$t) = split(/,/,$param);
			$p = int($p*$spd/100);
			$t = -int($t*$spd/100);
			checkURL("$URL&cpantiltmove=$p,$t");
		}
		elsif($cmd=~/z/i) { # zoom
			my $zoom=int($param*$zoomspd/100);
			checkURL("$URL&czoommove=$zoom")
		}
		elsif($cmd=~/focus/i) { # focus
			my $focus=int($param*$spd/100);
			checkURL("$URL&cfocusmove=$focus")
		}
	} elsif($mode=~/abs/i){    # mode=ABS
		if ($cmd=~/pt/i) {
			$param=~/(\-?\d+)\s*,\s*(\-?\d+)\s*$/;
			my $p = $1>0 ? 0.0 + $1 : 360.0 + $1;
			my $t = $2>0 ? 0.0 + $2 : 360.0 + $2;
			checkURL("$URL&apantiltmove=$p,$t");
		}
		elsif($cmd=~/Z/i) {
			$param=~/(\d+)/;
			my $abs_z = ($1 + 1)*99.98;
			checkURL("$URL&azoommove=$abs_z");
		}
		elsif($cmd=~/center/i) { # center
			checkURL("$URL&apantiltzoommove=0,0,1");
		}
	} elsif($mode=~/rel/i){ # relative positioning (recentering)
		$param =~ /(\d+)x(\d+)/;
		my ($rel_size_x,$rel_size_y) = ($1,$2);
		$options{xy} =~ /(\d+),(\d+)/;
		my ($rel_click_x,$rel_click_y) = ($1,$2);
		
		my ($rel_p, $rel_t) = ( 200*($rel_click_x-$rel_size_x/2)/$rel_size_x, -200*($rel_click_y-$rel_size_y/2)/$rel_size_y );
		$log->debug("REL [$rel_size_x] [$rel_size_y] [$rel_click_x] [$rel_click_y] [$rel_p] [$rel_t]");
		
		$rel_p = PTspeed($rel_p);
		$rel_t = PTspeed($rel_t);
		
		checkURL("$URL&cpantiltmove=$rel_p,$rel_t");
		select(undef,undef,undef,0.2);
		checkURL("$URL&cpantiltmove=0,0");
	} elsif($mode=~/step/i) {       # mode=step  /Step by step positioning/
		my $step_speed = 2;
		if($cmd=~/move/i) { # step pan/tilt
			my ($pan, $tilt) = (0, 0);
			$pan += $step_speed if $param =~ /right/i;
			$pan -= $step_speed if $param =~ /left/i;
			$tilt += $step_speed if $param =~ /up/i;
			$tilt -= $step_speed if $param =~ /down/i;
			
			checkURL("$URL&cpantiltmove=$pan,$tilt");
			select(undef,undef,undef,0.2);
			checkURL("$URL&cpantiltmove=0,0");
			#checkURL("$URL&stepmove=$param");	# don't supported by NVE-4000
		}
		elsif($cmd=~/zoom/i) {
			my $zoom = $param=~/in/i ? $step_speed : -$step_speed;
			checkURL("$URL&czoommove=$zoom");
			select(undef,undef,undef,0.2);
			checkURL("$URL&czoommove=0");
		}
		elsif($cmd=~/focus/i) {
			my $focus = $param=~/far/i ? -$step_speed : $step_speed;
			checkURL("$URL&cfocusmove=$focus");
			select(undef,undef,undef,0.2);
			checkURL("$URL&cfocusmove=0");
		}
		elsif($cmd=~/iris/i) { 
			my $iris = $param=~/open/i ? $step_speed : -$step_speed;
			checkURL("$URL&cirismove=$iris");
			select(undef,undef,undef,0.2);
			checkURL("$URL&cirismove=0");
		}
	} elsif($mode=~/smooth/i){  # mode=smooth  /Old style PTZ. Goes to direction
                              # untill stop (or any other command is sent)/
		if($cmd=~/move/i) { # pan/tilt
			$spd=3 if not $spd;
			my $pan = $param=~/left/i? -$spd : $param=~/right/i? $spd : 0;
			my $tlt = $param=~/down/i? -$spd : $param=~/up/i? $spd : 0;
			checkURL("$URL&cpantiltmove=$pan,$tlt");
		}
		elsif($cmd=~/zoom/i) { # zoom
			$zoomspd=1 if not $zoomspd;
			my $zm = $param=~/in/? $zoomspd : -$zoomspd;
			checkURL("$URL&czoommove=$zm")
		}
		elsif($cmd=~/iris/i) { # iris
			$spd=1 if not $spd;
			my $iris = $param=~/open/i? $spd : -$spd;
			checkURL("$URL&cirismove=$iris")
		}
		elsif($cmd=~/focus/i) { # focus
			$spd=1 if not $spd;
			my $focus = $param=~/far/i? -$spd : $spd;
			checkURL("$URL&cfocusmove=$focus")
		}
	} elsif($mode=~/preset/i){ # presets
		my $code = 'gotodevicepreset';	# default is goto
		$code=$conf{$dev}{SAVE_PRESET}	if $cmd=~/save/i;
		$code="removedevicepreset"	if $cmd=~/clear/i;
		checkURL("$URL&$code=$param");
	} elsif($mode=~/settings/i){ # settings
		my $code = '';
		if ($cmd=~/timeout/i) {
			$log->debug("Setting timer: [$param]");
			$conf{$dev}->{TIMEOUT} = time + $param;
		}
		elsif ($cmd=~/autofocus/i) {
			$code = "autofocus";
		} elsif ($cmd=~/autoiris/i) {
			$code ="autoiris";
		}
		if($code) {
			my $value = $param=~/on/i? 1 : 0;
			checkURL("$URL&$code=$value");
		}
	}
	$last_mode= $mode;
	$last_cmd = $cmd;
}

# ------------------------------------------------------------- UpdatePID -----
sub UpdatePID {
	open(PID,"> $APL/var/ptz/ptz_nve.pid");
	print PID $$;
	close PID;
} # sub UpdatePID

# ------------------------------------------------------------- camParams -----
sub camParams {
	my ($_dev)=@_;

	my $URL="http://$conf{$_dev}{DEVIP}:$conf{$_dev}{HTTP_PORT}/enc-cgi/ptz/ptz2.cgi?ptzid=$conf{$_dev}{PTZID}";
	my $info = checkURL("$URL&query=allinfo");
	my $protocollist = checkURL("$URL&query=protocollist");

	# Check and store in %conf 'Save preset' command name
	$conf{$_dev}{SAVE_PRESET} = $info=~/^(storedevicepreset)/m ? $1 : $info=~/^(setdevicepreset)/m ? $1 : '';

	# Setting up PTZ protocol
	my $proto = $conf{$_dev}{PTZ_PROTOCOL};
	if($protocollist=~/($proto\.ptzs)[,\s]/i) {
		checkURL("$URL&protocol=$1");
	} elsif($protocollist=~/($proto[(]?[\w\.]*[)]?[\w\.]*)[,\s]/i) {
		checkURL("$URL&protocol=$1");
	} else {
		$log->error("DEVID=[$_dev]: protocol $proto is not supported");
	}

	# Is PTZ enabled?
	my $ptz_enabled = $info=~/^enable=(\d+)/m ? $1 : 0;
	checkURL("$URL&enable=1") if !$ptz_enabled; # turn PTZ on if disabled
} # sub camParams

# --------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {
	%conf = GetCfgs( eval("($query_str)") );     # Load configurations
	my $ids='';
	foreach my $dev (keys  %conf) {
		next if not $conf{$dev}->{PTZID} =~/[12]/;
		next if not $conf{$dev}->{DEVIP};
		$ids.=" $dev";
		camParams($dev);
	}
	$log->info("Loaded configurations for cameras:$ids");
}

# -------------------------------------------------------------- nonblock -----
#   nonblock($socket) puts socket into nonblocking mode
sub nonblock {
	my $socket = shift;
	my $flags;
	
	$flags = fcntl($socket, F_GETFL, 0)
		or die "Can't get flags for socket: $!\n";
	fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
		or die "Can't make socket nonblocking: $!\n";
}
