#!/usr/bin/perl
#  $Id: ptz_axisv2.pl 21985 2011-02-23 18:18:15Z teetov $
# -----------------------------------------------------------------------------
#  PTZ driver for Moxa cameras, Moxa HTTP API
# -----------------------------------------------------------------------------
#  Author: Andrey Fomenko
#  Edited by: 
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT LLC, 2004-2005
# -----------------------------------------------------------------------------

use strict;
use Socket;
use MIME::Base64();
use IO::File;
use IO::Select;
use IO::Socket;
use Fcntl;
use LWP::UserAgent;
use HTTP::Request::Common;

use NextCAM::Init;
use NextCAM::PTZ;
use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

my $log=get_logger('NEXTCAM::PTZ::PTZ_MOXA');

my $query_str = shift || "'POSITIONCTL'=>'Moxa','CAMERAMODEL'=>'moxa'";
$log->info("Starting AXIS PTZ DRIVER, query string: [$query_str]");

my $ua = LWP::UserAgent->new();


$SIG{HUP}=\&load_dev_conf;
my $APL=$ENV{APL};
my $APL_CONF=$ENV{APL_CONF};
my (%conf,$cam,$cmd,$usrpsw,$par,$last_cmd,$last_mode);
load_dev_conf();

# -----------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# -----------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
                                   PeerPort => $TCP_PORT,
                                   Proto    => "tcp",
                                   Type     => SOCK_STREAM)
    or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';

my %commands;
my $ready;
my $trycount = 30;

while(1) {
	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/i;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/
				|| $commands{$cam} =~ /speed iris=0/
				|| $commands{$cam} =~ /speed gain=0/
				)
			{
				; # do not override stop command!
			}else{
				$commands{$cam} = $cmd;
			}
		} else {
			$commands{$cam} = $cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)
	
	$cmd = '';
	
	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		delete $commands{$camera};
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
		$usrpsw = '';
		$usrpsw = "$conf{$cam}{USRNAME}:$conf{$cam}{PASSWD}" if $conf{$cam}{USRNAME} && $conf{$cam}{PASSWD};
		
		httpAuth($conf{$cam}{USRNAME}, $conf{$cam}{PASSWD}) if $conf{$cam}{USRNAME} && $conf{$cam}{PASSWD};
		
		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam,$mode,"pt", $params[0].','.$params[1] ,\%options);
			camCmd($cam,$mode,"zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam,$mode,$cmd,$par,\%options);
		}
	}

	select(undef,undef,undef,.1) if not $cmd;

	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
	}
	
	if(!$cmd) {
		select(undef,undef,undef,.4);
		if(!$trycount--) {
			$trycount = 30;
			print $socket "test\0" or $log->logdie("Couldn't communicate to socket $TCP_PORT: $@");
		}
	}

} # while(1)


# ---------------------------------------------------------------- camCmd -----
sub camCmd
{
	my ($dev,$mode,$cmd,$param,$options)=@_;
	my %options = %$options;
	$log->debug("camCmd: DEVID=[$dev] mode=[$mode] command:[$cmd] param=[$param] ");
	my $URL="http://$conf{$dev}{DEVIP}:$conf{$dev}{HTTP_PORT}/moxa-cgi/-cgi/setptzctrl.cgi?channel=$conf{$dev}{CAMERA}";
	my $spd=($options{speed}=~/^\d+/) ? $options{speed}:$conf{$dev}->{PTZSPEED};
	my $zoomspd = $conf{$dev}->{PTZZOOMSPEED};

	# we need to "terminate" each command because it can move forever if used with no STOP !
	if ($last_mode=~/smooth/i) {
		checkURL("$URL&move=stop") if $last_cmd=~/move/i;
	}

	# here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
	if($conf{$dev}->{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
		if($mode=~/speed/i || $mode=~/step/i || $mode=~/abs/i || $mode=~/rel/i || $mode=~/smooth/i) {
			$log->debug("Setting timeout +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
		elsif($mode=~/preset/i && $cmd=~/goto/i && $param!=1) {
			$log->debug("Setting timeout (preset) +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
	}

	if ($mode=~/speed/i){
		$spd=$conf{$dev}->{PTZSPEED};
		$spd=1 if $spd < 1;
		$spd=100 if $spd > 100;
		if($cmd=~/^pt$/i) { #  pan-tilt
			
			my ($p,$t) = split(/,/,$param);
			my $pm = abs(int(16 * $p*$spd/10000));
			my $tm = abs(int(16 * $t*$spd/10000));
			checkURL("$URL&move=right&speedpan=$pm") if $p > 0;
			checkURL("$URL&move=left&speedpan=$pm") if $p < 0;
			checkURL("$URL&move=up&speedtilt=$tm") if $t < 0;
			checkURL("$URL&move=down&speedtilt=$tm") if $t > 0;
		}
		elsif($cmd=~/^zoom$/i) { # zoom
			my $zm=abs(int(16 * $param*$zoomspd/10000));
			checkURL("$URL&move=zin&speedzoom=$zm") if $param > 0;
			checkURL("$URL&move=zout&speedzoom=$zm") if $param < 0;
		}
		elsif($cmd=~/focus/i) { # focus
			checkURL("$URL&move=focusfar") if $param > 0;
			checkURL("$URL&move=focusnear") if $param < 0;
		}
		
		elsif($cmd=~/iris/i) { # focus
			checkURL("$URL&move=irisopen") if $param > 0;
			checkURL("$URL&move=irisclose") if $param < 0;
		}
		
	} elsif($mode=~/abs/i){    # mode=ABS
	
		if ($cmd=~/pt/i) {
			$param=~/(\-?\d+)\s*,\s*(\-?\d+)\s*$/;
			my $p = $1;
			my $t = $2;
			checkURL("$URL&pan=$p&tilt=$t");
		}
		elsif($cmd=~/Z/i) {
			$param=~/(\d+)/;
			my $abs_z = ($1 + 1)*99.98;
			checkURL("$URL&zoom=$abs_z");
		}
		elsif($cmd=~/center/i) { # center
			checkURL("$URL&pan=0&tilt=0&zoom=1");
		}
		
	} elsif($mode=~/rel/i){ # relative positioning (recentering)
		
		$param =~ /(\d+)x(\d+)/;
		my ($rel_size_x,$rel_size_y) = ($1,$2);
		$options{xy} =~ /(\d+),(\d+)/;
		my ($rel_click_x,$rel_click_y) = ($1,$2);
		checkURL("$URL&imagewidth=$rel_size_x&imageheight=$rel_size_y&center=$rel_click_x,$rel_click_y");
	} elsif($mode=~/step/i) {       # mode=step  /Step by step positioning/
		if($cmd=~/move/i) { # step pan/tilt
			checkURL("$URL&move=$param")
		}
		elsif($cmd=~/zoom/i) {
			my $zm = $param=~/out/i? "zout" : "zin";
			checkURL("$URL&move=$zm");
		}
		elsif($cmd=~/focus/i) {
			my $focus = $param=~/far/i? "focusnear" : "focusfar";
			checkURL("$URL&move=$focus");
		}
		elsif($cmd=~/iris/i) { 
			my $iris = $param=~/open/i? "irisopen" : "irisclose";
			checkURL("$URL&move=$iris");
		}
	} elsif($mode=~/smooth/i){
		# mode=smooth  /Old style PTZ. Goes to direction
		# untill stop (or any other command is sent)/
		if($cmd=~/move/i) { # pan/tilt
			$spd=3 if not $spd;
			my $pan = $param=~/left/i? -$spd : $param=~/right/i? $spd : 0;
			my $tlt = $param=~/down/i? -$spd : $param=~/up/i? $spd : 0;
			checkURL("$URL&continuouspantiltmove=$pan,$tlt");
		}
		elsif($cmd=~/zoom/i) { # zoom
			$zoomspd=1 if not $zoomspd;
			my $zm = $param=~/out/? -$zoomspd : $zoomspd;
			checkURL("$URL&continuouszoommove=$zm")
		}
		elsif($cmd=~/iris/i) { # iris
			$spd=1 if not $spd;
			my $iris = $param=~/close/i? -$spd : $spd;
			checkURL("$URL&continuousirismove=$iris")
		}
		elsif($cmd=~/focus/i) { # focus
			$spd=1 if not $spd;
			my $focus = $param=~/far/i? -$spd : $spd;
			checkURL("$URL&continuousfocusmove=$focus")
		}
	} elsif($mode=~/preset/i){ # presets
		my $code = 'gotodevicepreset';	# default is goto
		$code="gopreset"	if $cmd=~/goto/i;
		$code="addpreset"	if $cmd=~/save/i;
		$code="clearpreset"	if $cmd=~/clear/i;
		checkURL("http://$conf{$dev}{DEVIP}:$conf{$dev}{HTTP_PORT}/moxa-cgi/setpreset.cgi?channel=$conf{$dev}{CAMERA}&move=$code&presetindex=$param");
	} elsif($mode=~/settings/i){ # settings
		my $code = '';
		if ($cmd=~/autofocus/i) {
			$code = "autofocus";
		} elsif ($cmd=~/autoiris/i) {
			$code ="autoiris";
		}
		checkURL("$URL&move=$code") if $code;
	}
	$last_mode= $mode;
	$last_cmd = $cmd;
}

# ------------------------------------------------------------- UpdatePID -----
sub UpdatePID {
	open(PID,"> $APL/var/ptz/ptz_axis.pid");
	print PID $$;
	close PID;
} # sub UpdatePID

# --------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {
	%conf = GetCfgs( eval("($query_str)") );     # Load configurations
	my $ids='';
	foreach my $dev (keys  %conf) {
		next if not $conf{$dev}->{PTZID} =~/[12]/;
		next if not $conf{$dev}->{DEVIP};
		$ids.=" $dev";
		camParams($dev);
	}
	$log->info("Loaded configurations for cameras:$ids");
}

# ------------------------------------------------------------- camParams -----
sub camParams {

	my ($_dev)=@_;
	
	# here we load Axis firmware revision number, so we can analyze PTZ dialect to talk on
	if(open(F, "$APL_CONF/$_dev/CAMERAMODEL")) {
		$conf{$_dev}{CAMERAMODEL} = <F>;
		chomp $conf{$_dev}{CAMERAMODEL};
		close F;
	}
	else {
		$conf{$_dev}{CAMERAMODEL} = 'axis241S[4.00]';
	}
	
	if($conf{$_dev}{PTZPARAMS}=~/(.+):(.+):(.+):(.+)/) {
		$conf{$_dev}{pan}=$1;
		$conf{$_dev}{tilt}=$2;
		$conf{$_dev}{zoom}=$3;
		$conf{$_dev}{speed}=$4;
	}
	elsif($conf{$_dev}{PTZPARAMS} eq 'SONY EVI-D30/31') {
		$conf{$_dev}{pan}=1.66;
		$conf{$_dev}{tilt}=0.54;
		$conf{$_dev}{zoom}=625;
		$conf{$_dev}{speed}=20;
	}
	elsif($conf{$_dev}{PTZPARAMS} eq 'SONY EVI-D100/D100P') {
		$conf{$_dev}{pan}=2.78;
		$conf{$_dev}{tilt}=0.69;
		$conf{$_dev}{zoom}=10000;
		$conf{$_dev}{speed}=20;
	}
	else {
		$conf{$_dev}{pan}=1;
		$conf{$_dev}{tilt}=1;
		$conf{$_dev}{zoom}=1000;
		$conf{$_dev}{speed}=100;
	}

	if($conf{$_dev}{PTZSTEPS}=~/(.+):(.+):(.+)/) {
		$conf{$_dev}{span}=$1;
		$conf{$_dev}{stilt}=$2;
		$conf{$_dev}{szoom}=$3;
	}
	else {
		$conf{$_dev}{span}=50;
		$conf{$_dev}{stilt}=50;
		$conf{$_dev}{szoom}=50;
	}
	$conf{$_dev}{p_pos}=0;
	$conf{$_dev}{t_pos}=0;
	$conf{$_dev}{z_pos}=0;
} # sub camParams

# ----------------------------------------------------------- truncDigits -----
sub truncDigits{
	my($val)=@_;
	return sprintf("%5.5f",$val)
} # sub truncDigits

# -------------------------------------------------------------- nonblock -----
#   nonblock($socket) puts socket into nonblocking mode
sub nonblock {
	my $socket = shift;
	my $flags;
	
	$flags = fcntl($socket, F_GETFL, 0)
		or die "Can't get flags for socket: $!\n";
	fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
		or die "Can't make socket nonblocking: $!\n";
}
