#!/usr/bin/perl
#  $Id: ptz_icd001.pl 25370 2012-03-09 21:19:34Z teetov $
# -----------------------------------------------------------------------------
#  PTZ engine for ICD-001 protocol, TCP and SERIAL transports
# -----------------------------------------------------------------------------
#  Unlike other engines this one handles different transport types in the same 
#  time (TCP sockets, URL and Serial )
# -----------------------------------------------------------------------------
#  Author: Andriy Fomenko
#  Edited by: Serg Pososhenko
#  QA by:  Christopher C Gettings
#  Copyright: videoNEXT LLC 2004,2005
# -----------------------------------------------------------------------------

use strict;

# -----------------------------------------------------------------------------
# Constants section
# -----------------------------------------------------------------------------
my $STEP_P = 0x150;  # step mode: step in absolute position for PAN
my $STEP_T = 0x350;  # step mode: step in absolute position for TILT
my $STEP_Z = 0x100;  # step mode: step in absolute value for ZOOM
my $STEP_F = 0x30;   # step mode: step in absolute value for FOCUS
my $STEP_B = 0x10;   # step mode: step in absolute value for BRIGHTNESS
my $STEP_C = 0x10;   # step mode: step in absolute value for CONTRAST
my $STEP_TMIX = 0x2B; # step for Thermal MIX - 6 steps total from min to max

# poll devices every ?? seconds to prevent disconnects
my $POLL_TIMEOUT = 60;        # forsed poll - every 60 seconds
my $INACTIVITY_TIMEOUT = 5;  # user inactivity poll - every 5 seconds
# -----------------------------------------------------------------------------

# PTZ_CAMERATYPE := {COHU_iDome,COHU_iView,WSTI,MRTI}

my $NN = 0;

use POSIX;
use IO::Socket;
use IO::Select;
use Tie::RefHash;
use IO::File;
use Socket;
use Fcntl;
use MIME::Base64();
use Tie::RefHash;
use NextCAM::Init;
use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

my $log=get_logger('NEXTCAM::PTZ::PTZ_ICD001');

my $query_str = shift || "'POSITIONCTL'=>'ICD001'";
$log->info("Starting ICD001 PTZ DRIVER, query string: [$query_str]");

$SIG{HUP}=\&load_dev_conf;

my (%conf, %sock, %port, $cam, $cmd, $par, $usrpsw, $last_cmd, $last_mode, $cmd_executed);

my $APL=$ENV{APL} || $log->logdie('Variable APL is not set');
my $APL_CONF=$ENV{APL_CONF} || $log->logdie('Variable APL_CONF is not set');

load_dev_conf();

my $lastcmdtime = time;
my $lastPoll = time;

# -----------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# -----------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
                                   PeerPort => $TCP_PORT,
                                   Proto     => "tcp",
                                   Type      => SOCK_STREAM)
    or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';

my %commands;
my $ready;
my $trycount = 30;
my $lastcommand;

# AF: here is an ugly thing - ICD cameras loose "STOP" periodically if it is sent 
# in 10-20 mS from another command, and then spin forever.
# So, it is a fix - we create %TODO hash which will hold DevID, time to send 
# command and comamnd itself, and in the loop time will be checked and commands will be sent

my %TODO;

while(1) {
	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/i;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/
				|| $commands{$cam} =~ /speed iris=0/
				|| $commands{$cam} =~ /speed gain=0/
				)
			{
			  $log->debug("STOP COMMAND coming!lastcommand:$lastcommand !");              
			  if($lastcommand !~/speed pt=0,0/)
			  {
				addTODO($cam,time+1,"$cam speed pt=0,0");
				$lastcommand=$cmd;
			  }
				; # do not override stop command!
			}else{
				$commands{$cam} = $cmd;
				$lastcommand=$cmd;
			}
		} else {
			$commands{$cam} = $cmd;
			$log->debug("COMMAND coming!lastcommand:$lastcommand !");
			if($cmd =~/speed pt=0,0/)
			{
			  if($lastcommand !~/speed pt=0,0/)
			  {
				addTODO($cam,time+2,"$cam speed pt=0,0");
				$lastcommand=$cmd;
			  }              
			}
			$lastcommand=$cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)
	
	$cmd = '';
	
	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		delete $commands{$camera};
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
		$usrpsw = "$conf{$cam}{USRNAME}:$conf{$cam}{PASSWD}" if $conf{$cam}{USRNAME} && $conf{$cam}{PASSWD};
		
		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam,$mode,"pt", $params[0].','.$params[1] ,\%options);
			camCmd($cam,$mode,"zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam,$mode,$cmd,$par,\%options);
		}
	}
	
	select(undef,undef,undef,.1) if not $cmd;

	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
		$lastcmdtime = time;
	}
	
	processTODO();
	
	next; # do not poll per Tim DiMeglio request
	# check that cameras health only if no user activity was detected or it's absolutelly necessary
	if( ((time-$lastcmdtime > $INACTIVITY_TIMEOUT)) or ((time-$lastPoll) > $POLL_TIMEOUT)) {
		$lastPoll = time;
		$lastcmdtime = time;
		SendAliveSignal();
	}

} # while(1)


# --------------------------------------------------------------- addTODO -----
sub addTODO {
	my $devid = shift;
	$TODO{$devid}{TIME} = shift;
	$TODO{$devid}{CMD} = shift; # array here!
	$log->debug("Preserving STOP in TODO $devid ->> $TODO{$devid}{CMD}");
}

# ----------------------------------------------------------- processTODO -----
sub processTODO {
	foreach my $dev ( keys %TODO ) {
		next if $TODO{$dev}{TIME} > time;
		$log->debug("Running TODO for $dev -> $TODO{$dev}{CMD}");        
		$commands{$dev} = $TODO{$dev}{CMD};
#        cmdTransmit($dev,$TODO{$dev}{CMD});
		delete $TODO{$dev};
	}
}

# ---------------------------------------------------------------- camCmd -----
sub camCmd {
	my ($dev,$mode,$cmd,$param,$options)=@_;
	my %options = %$options;    
	my $resp;
	my @cmd;                                # Byte | Description
	$cmd[0] = 0xF8;                         #   0  | Autorate character
	$cmd[1] = $conf{$dev}{PTZID};           #   1  | Message destination address
	$cmd[2] = 0x2A;                         #   2  | '*' (asterisk) constant
	$cmd[3] = $conf{$dev}{PTZGROUP};        #   3  | Group address
	$cmd[4] = 0x1F;                         #   4  | Message source address (0x1F is master control unit)
                                            #   5  | Length of message data
                                            #   .. | Command data
                                            # 6+len| Checksum + 0x80 (0x80..0x8F)

	# we have to "terminate" each command in smooth mode because it can move forever if used with no STOP !
	if ($last_mode=~/smooth/i) {
		$log->debug("Last mode was <$last_mode>. Last command <$last_cmd>");
		#checkURL("$URL=8101060100000303FF") if $last_cmd=~/move/i;
		#checkURL("$URL=8101060100000303FF") if $last_cmd=~/zoom/i;  # zoom
		#checkURL("$URL=8101040B00FF") if $last_cmd=~/iris/i;  # iris
		#checkURL("$URL=8101040800FF") if $last_cmd=~/focus/i;  # focus
	}
	
	# here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
	if($conf{$dev}->{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
		if($mode=~/speed/i || $mode=~/step/i || $mode=~/abs/i || $mode=~/rel/i || $mode=~/smooth/i) {
			$log->debug("Setting timeout +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
		elsif($mode=~/preset/i && $cmd=~/goto/i && $param!=1) {
			$log->debug("Setting timeout (preset) +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
	}
	
	if($mode=~/step/i) { # ==================================== step =====
		if($cmd=~/move/i) { # --------------------------------- move -----
			#commented out - so COHU will follow standard ICD procedure
			#if( camType($dev) =~ /COHU/i ) {
			#    # here is "work-around" to use P*/T*/S*/E* commands instead proper controls
			#    my $st_p = $param=~/left/i? 'L' : $param=~/right/i? 'R' : 'S';
			#    my $st_t = $param=~/up/i? 'U' : $param=~/down/i? 'D' : 'S';
			#    $cmd[5] = 0x08; # Command length
			#    $cmd[6] = ord('S'); # PAN speed S0..SF
			#    $cmd[7] = ord('0') + 3;
			#    $cmd[8] = ord('E'); # TILT speed E0..EF
			#    $cmd[9] = ord('0') + 3;
			#    $cmd[10] = ord('P'); # PAN
			#    $cmd[11] = ord( $st_p );
			#    $cmd[12] = ord('T'); # TILT
			#    $cmd[13] = ord( $st_t );
			#    $cmd[14] = checkSum(@cmd);
			#    for(my $i=3; --$i;) {
			#        $resp = cmdTransmit($dev,@cmd);
			#        last if ord(substr($resp,6,1)) == 0x06; # ACK=0x06 / NACK=0x15
			#    }
			#    if(ord(substr($resp,6,1)) != 0x06) {
			#        $log->warn('Camera DEVID=',$dev,' did not acknowledged command');
			#        return;
			#    }
			#    select(undef,undef,undef,.05);
			#    $#cmd = 9;
			#    $cmd[5] = 0x04; # Command length
			#    $cmd[6] = ord('P'); # PAN
			#    $cmd[7] = ord('S');
			#    $cmd[8] = ord('T'); # TILT
			#    $cmd[9] = ord('S');
			#    $cmd[10] = checkSum(@cmd);
			#}
			#else { # standard ICD commands for MRTI/WSTI and others
				$cmd[5] = 2;
				if($param=~/left|right/i) {
					$cmd[6] = ord('S'); # PAN speed S0..SF
					$cmd[7] = ord('0') + 3;
					$cmd[8] = checkSum(@cmd);
					cmdTransmit($dev,@cmd);
					$#cmd = 6;
					$cmd[6] = ord('P');
					$cmd[7] = ord( $param=~/left/i? 'L' : $param=~/right/i? 'R' : 'S' );
					$cmd[8] = checkSum(@cmd);
					cmdTransmit($dev,@cmd);
					select(undef,undef,undef,.02);
					$#cmd = 7;
					$cmd[6] = ord('P');
					$cmd[7] = ord( 'S' );
					$cmd[8] = checkSum(@cmd);
					cmdTransmit($dev,@cmd);
				}
				select(undef,undef,undef,.2);
				if($param=~/up|down/) {
					$#cmd = 7;
					$cmd[6] = ord('E'); # TILT speed E0..EF
					$cmd[7] = ord('0') + 3;
					$cmd[8] = checkSum(@cmd);
					cmdTransmit($dev,@cmd);
					$#cmd = 6;
					$cmd[6] = ord('T');
					$cmd[7] = ord( $param=~/up/i? 'U' : $param=~/down/i? 'D' : 'S' );
					$cmd[8] = checkSum(@cmd);
					cmdTransmit($dev,@cmd);
					select(undef,undef,undef,.02);
					$#cmd = 7;
					$cmd[6] = ord('T');
					$cmd[7] = ord( 'S' );
					$cmd[8] = checkSum(@cmd);
					cmdTransmit($dev,@cmd);
				}
				addTODO($cam,time+2,"$cam speed pt=0,0");
				return;
			#}
		}
		elsif($cmd=~/zoom/i) { # ------------------------------------ zoom -----
			if( camType($dev) =~ /WSTI/i ) {
				$cmd[5] = 4;
				$cmd[6] = ord('B');
				$cmd[7] = ord('0');
				$cmd[8] = ord($param=~/in/i? '7' : '5');
				$cmd[9] = ord('P'); # Press
				$cmd[10] = checkSum(@cmd);
				cmdTransmit($dev,@cmd);
				$#cmd = 9; 
				$cmd[9] = ord('R'); # Release
				$cmd[10] = checkSum(@cmd);
				cmdTransmit($dev,@cmd);
				return;                
			}
			elsif ( camType($dev) =~ /COHU|MRTI/i ) {
				# work-around for COHU
				my $st_z = $param=~/in/i? 'I' : $param=~/out/i? 'O' : 'S';
				$cmd[5] = 0x02; # Command length
				$cmd[6] = ord('Z'); # Zoom
				$cmd[7] = ord($st_z);
				$cmd[8] = checkSum(@cmd);
				cmdTransmit($dev,@cmd);
				select(undef,undef,undef,.05);
				$#cmd = 7;
				$cmd[5] = 0x02; # Command length
				$cmd[6] = ord('Z'); 
				$cmd[7] = ord('S');
				$cmd[8] = checkSum(@cmd);
			}
		}
		elsif($cmd=~/focus/i) { # ---------------------------------- focus -----
			# work-around
			my $st_f = $param=~/near/i? 'N' : $param=~/far/i? 'F' : 'S';
			$cmd[5] = 0x02; # Command length
			$cmd[6] = ord('F'); # Focus
			$cmd[7] = ord($st_f);
			$cmd[8] = checkSum(@cmd);
			cmdTransmit($dev,@cmd);
			select(undef,undef,undef,.01);
			$#cmd = 7;
			$cmd[5] = 0x02; # Command length
			$cmd[6] = ord('F');
			$cmd[7] = ord('S');
			$cmd[8] = checkSum(@cmd);
			# at this point let standard handler do the job
		}
		elsif($cmd=~/iris|gain/i) { # ------------------------- iris/gain -----
			# work-around
			my $st_f = $param=~/open/i? 'O' : $param=~/close/i? 'C' : 'S';
			$cmd[5] = 0x02; # Command length
			$cmd[6] = ord('I'); # Focus
			$cmd[7] = ord($st_f);
			$cmd[8] = checkSum(@cmd);
			cmdTransmit($dev,@cmd);
			select(undef,undef,undef,.01);
			$#cmd = 7;
			$cmd[5] = 0x02; # Command length
			$cmd[6] = ord('I');
			$cmd[7] = ord('S');
			$cmd[8] = checkSum(@cmd);
			# at this point let standard handler do the job
		}
		elsif($cmd=~/tmix/i) { # ------------------------------------ tmix -----
			# Thermal MIX
			# FIRST - request current TMIX value
			my $tmix_retries = 3;
			tmix_retry1:
			$cmd[5] = 0x0B; # Command length
			$cmd[6] = ord('E');
			$cmd[7] = ord('M');
			$cmd[8] = ord('M');
			$cmd[9] = ord('?');
			$cmd[10] = $cmd[11] = $cmd[12] = $cmd[13] = $cmd[14] = $cmd[15] = $cmd[16] = ord('0');
			$cmd[12] =  $cmd[15] = ord('1');
			$cmd[17] = checkSum(@cmd);
			$resp = cmdTransmit($dev,@cmd);
			$log->debug(substr($resp,6,10));
			if(substr($resp,6,10) ne 'EMMX001002') {
				my $command='';
				for(my $b=0;$b<length($resp);$b++) { $command .= sprintf("%02X ",ord(substr($resp,$b,1))); }
				$log->warn('Camera DEVID=',$dev,' did not return TMIX values, but acknowledged request, responce: ',$command);
				goto tmix_retry1 if $tmix_retries--;
				return;
			}
			# second - acknowledge feedback
			$#cmd = 5;
			$cmd[5] = 1;
			$cmd[6] = 6; # ACK
			$cmd[7] = checkSum(@cmd);
			cmdTransmit($dev,@cmd);
			# THIRD - adjust values
			my $tmix = hex(substr($resp,16,2));
			$log->debug('Current TMIX=',$tmix);
			$tmix += $param=~/up/i ? $STEP_TMIX : -$STEP_TMIX;
			$tmix = $tmix < 0 ? 0 : $tmix > 0xFF? 0xFF : $tmix;
			$log->debug('New TMIX=',$tmix);
			$tmix = sprintf('%02X',$tmix);
			# THIRD - send new position to lens
			$#cmd = 9; 
			$cmd[5] = 12 ; # command length
			$cmd[6] = ord('E');
			$cmd[7] = ord('M');
			$cmd[8] = ord('m');
			$cmd[9] = ord('x');
			$cmd[10] = $cmd[11] = $cmd[13] = $cmd[14] = ord('0');
			$cmd[12] = ord('1');
			$cmd[15] = ord('2');
			$cmd[16] = ord(substr($tmix,0,1));
			$cmd[17] = ord(substr($tmix,1,1));
			$cmd[18] = checkSum(@cmd);
			# let standard handler transmit command
		}
	}
	elsif($mode=~/preset/i) { # ============================= preset =====
		if($cmd=~/goto/i) {
			# Return if $param <0 or $param > 9;
			$cmd[5] = 0x02; # Command length
			$cmd[6] = ord('A');  # HOME
			$cmd[7] = ord('F');
			$cmd[8] = checkSum(@cmd);
			cmdTransmit($dev,@cmd);
			$#cmd = 7;
			$cmd[5] = 0x02; # Command length
			$cmd[6] = ord('H');  # HOME
			$cmd[7] = ord(sprintf('%01d',$param-1));
			$cmd[8] = checkSum(@cmd);            
		}
		elsif($cmd=~/save/i) {
			return if $param <0 or $param > 9;
			$cmd[5] = 0x02; # Command length
			$cmd[6] = ord('P');  # Preset
			$cmd[7] = ord(sprintf('%01d',$param-1));
			$cmd[8] = checkSum(@cmd);            
		}
		elsif($cmd=~/clear/i) {
		}
	}
	elsif($mode=~/speed/i) { # =============================== speed =====
		if($cmd=~/pt/i) { # ------------------------------------- pt -----
			my ($p,$t) = split(/,/,$param); 
			$log->debug("PT: [$p,$t]");
			if( camType($dev) =~ /WSTI|MRTI/i ) {
				$p = $p>99? 99: $p<-99? -99: $p; # put into range of 0..99
				$t = $t>99? 99: $t<-99? -99: $t;
				$p *= $conf{$dev}->{PTZSPEED}/100;
				$t *= $conf{$dev}->{PTZSPEED}/100;
				$cmd[5] = 0x07; # Command length
				$cmd[6] = ord('J'); # Joystick commands
				$cmd[7] = $p>0? ord('R') : ord('L'); # pan direction
				$cmd[8] = ord(sprintf('%01d', abs($p/10)));
				$cmd[9] = ord(sprintf('%01d', abs($p%10)));
				$cmd[10] = $t<0? ord('U') : ord('D'); # pan direction
				$cmd[11] = ord(sprintf('%01d', abs($t/10)));
				$cmd[12] = ord(sprintf('%01d', abs($t%10)));
				$cmd[13] = checkSum(@cmd);
			}
			elsif( camType($dev) =~ /COHU/i ) {
				# here is "work-around" to use P*/T*/S*/E* commands instead of one J command
				$p = $p / 100.0 * 15; $p = sprintf("%d", $p>15? 15 : $p<-15? -15 : $p);
				$t = $t / 100.0 * 15; $t = sprintf("%d", $t>15? 15 : $t<-15? -15 : $t);
				$p *= $conf{$dev}->{PTZSPEED}/100;
				$t *= $conf{$dev}->{PTZSPEED}/100;
				$log->debug("$p $t");
				$cmd[5] = 0x08; # Command length
				$cmd[6] = ord('S'); # PAN speed S0..SF
				$cmd[7] = ord(sprintf("%01X",abs($p))); 
				$cmd[8] = ord('E'); # TILT speed E0..EF
				$cmd[9] = ord(sprintf("%01X",abs($t)));
				$cmd[10] = ord('P'); # PAN
				$cmd[11] = ord( $p==0? 'S' : $p<0? 'L' : 'R' );
				$cmd[12] = ord('T'); # TILT
				$cmd[13] = ord( $t==0? 'S' : $t<0? 'U' : 'D' );
				$cmd[14] = checkSum(@cmd);
			}
		}
		elsif($cmd=~/z/i) { # ------------------------------------ z -----
			$cmd[5] = 0x02; # Command length
			$cmd[6] = ord('L');  # Lens speed: LO - slow / LT - fast
			$cmd[7] = ord( abs($param)<50? 'O' : 'T');
			$cmd[8] = checkSum(@cmd);
			cmdTransmit($dev,@cmd);
			$#cmd = 5;
			$cmd[5] = 0x02; # Command length
			$cmd[6] = ord('Z');  # ZOOM
			$cmd[7] = ord($param>0 ? 'I' : $param<0? 'O' : 'S' );
			$cmd[8] = checkSum(@cmd);
		}
		elsif($cmd=~/focus/i) { # ---------------------------- focus -----
			$cmd[5] = 0x02; # Command length
			$cmd[6] = ord('L');  # Lens speed: LO - slow / LT - fast
			$cmd[7] = ord( abs($param)<50? 'O' : 'T');
			$cmd[8] = checkSum(@cmd);
			cmdTransmit($dev,@cmd);
			$#cmd = 5;
			$cmd[5] = 0x02; # Command length
			$cmd[6] = ord('F');  # FOCUS
			$cmd[7] = ord($param>0 ? 'N' : $param<0? 'F' : 'S' );
			$cmd[8] = checkSum(@cmd);

		}
		elsif($cmd=~/iris|gain/i) { # ------------------------- iris|gain -----
			$cmd[5] = 0x02; # Command length
			$cmd[6] = ord('I');  # IRIS
			$cmd[7] = ord($param>0 ? 'O' : $param<0? 'C' : 'S' );
			$cmd[8] = checkSum(@cmd);
		}
		elsif($cmd=~/tmix/i) { # ------------------------------- tmix -----
			# Thermal MIX
			# FIRST - request current TMIX value
			my $tmix_retries2 = 3;
			tmix_retry2:
			$cmd[5] = 0x0B; # Command length
			$cmd[6] = ord('E');
			$cmd[7] = ord('M');
			$cmd[8] = ord('M');
			$cmd[9] = ord('?');
			$cmd[10] = $cmd[11] = $cmd[12] = $cmd[13] = $cmd[14] = $cmd[15] = $cmd[16] = ord('0');
			$cmd[12] =  $cmd[15] = ord('1');
			$cmd[17] = checkSum(@cmd);
			$resp = cmdTransmit($dev,@cmd);
			return if not $resp;
			$log->debug(substr($resp,6,10));
			if(substr($resp,6,10) ne 'EMMX001002') {
				my $command='';
				for(my $b=0;$b<length($resp);$b++) { $command .= sprintf("%02X ",ord(substr($resp,$b,1))); }
				$log->warn('Camera DEVID=',$dev,' did not return TMIX values, but acknowledged request, responce: ',$command);
				goto tmix_retry2 if $tmix_retries2--;
				return;
			}
			# second - acknowledge feedback
			$#cmd = 5;
			$cmd[5] = 1;
			$cmd[6] = 6; # ACK
			$cmd[7] = checkSum(@cmd);
			cmdTransmit($dev,@cmd);
			# THIRD - adjust values
			my $tmix = hex(substr($resp,16,2));
			$log->debug('Current TMIX=',$tmix);
			$tmix += ($param/4);
			$tmix = $tmix < 0 ? 0 : $tmix > 0xFF? 0xFF : $tmix;
			$log->debug('New TMIX=',$tmix);
			$tmix = sprintf('%02X',$tmix);
			# THIRD - send new position to lens
			$#cmd = 9; 
			$cmd[5] = 12 ; # command length
			$cmd[6] = ord('E');
			$cmd[7] = ord('M');
			$cmd[8] = ord('m');
			$cmd[9] = ord('x');
			$cmd[10] = $cmd[11] = $cmd[13] = $cmd[14] = ord('0');
			$cmd[12] = ord('1');
			$cmd[15] = ord('2');
			$cmd[16] = ord(substr($tmix,0,1));
			$cmd[17] = ord(substr($tmix,1,1));
			$cmd[18] = checkSum(@cmd);
			# let standard handler transmit command
		}
	}
	elsif($mode=~/rel/i) { # =================================== rel =====
		$param =~ /(\d+)x(\d+)/;
		my ($rel_size_x,$rel_size_y) = ($1,$2);
		$options{xy} =~ /(\d+),(\d+)/;
		my ($rel_click_x,$rel_click_y) = ($1,$2);
		my ($rel_p, $rel_t) = ( 200*($rel_click_x-$rel_size_x/2)/$rel_size_x, 200*($rel_click_y-$rel_size_y/2)/$rel_size_y );
		$log->debug('Relative movement: [',$rel_p,'] [',$rel_t,']');
		# first, request current position
		$cmd[5] = 0x02; # Command length
		$cmd[6] = ord('P');
		$cmd[7] = ord('?');
		$cmd[8] = checkSum(@cmd);
		for(my $i=3; --$i;) {
			$resp = cmdTransmit($dev,@cmd);
			last if ord(substr($resp,6,1)) == 0x06; # ACK=0x06 / NACK=0x15
		}
		if(ord(substr($resp,6,1)) != 0x06) {
			$log->warn('Camera DEVID=',$dev,' did not return pan/tilt position');
			return;
		}
		if(ord(substr($resp,8,1))!=0xF8) {
			my $command='';
			for(my $b=8;$b<length($resp);$b++) { $command .= sprintf("%02X ",ord(substr($resp,$b,1))); }
			$log->warn('Camera DEVID=',$dev,' did not return pan/tilt position, but acknowledged request, responce: ',$command);
			return
		}

		my ($rel_step_p, $rel_step_t) = (hex(substr($resp,15,3)), hex(substr($resp,18,3)));
		$log->debug('Current positions: [',$rel_step_p,'] [',$rel_step_t,']');
		return if not $rel_step_p and not $rel_step_t; # sometimes we get trash here, disregard command then
		$rel_step_p += $rel_p;
		$rel_step_t -= $rel_t;
		$rel_step_p = $rel_step_p<0? 0 : $rel_step_p > 0xFFF? 0xFFF : $rel_step_p;
		$rel_step_t = $rel_step_t<0? 0 : $rel_step_t > 0xFFF? 0xFFF : $rel_step_t;
		$rel_step_p = sprintf('%03X',$rel_step_p);
		$rel_step_t = sprintf('%03X',$rel_step_t);
		$log->debug('New calculated positions: [',hex($rel_step_p),'] [',hex($rel_step_t),']');
		# acknowledge feedback
		$#cmd = 5;
		$cmd[5] = 1;
		$cmd[6] = 6; # ACK
		$cmd[7] = checkSum(@cmd);
		$resp = cmdTransmit($dev,@cmd);
		# now form actual reposition command
		$cmd[5] = 0x07; # Command length
		$cmd[6] = ord('p'); # 'go-to-position' command
		$cmd[7] = ord(substr($rel_step_p,0,1));
		$cmd[8] = ord(substr($rel_step_p,1,1));
		$cmd[9] = ord(substr($rel_step_p,2,1));
		$cmd[10] = ord(substr($rel_step_t,0,1));
		$cmd[11] = ord(substr($rel_step_t,1,1));
		$cmd[12] = ord(substr($rel_step_t,2,1));
		$cmd[13] = checkSum(@cmd);
		# at this point let standard handler do the job        
	}
	elsif($mode=~/settings/i) { # ========================= settings =====
		if($cmd=~/whitebalance/i) {
		}
		elsif($cmd=~/power/i) {
			$cmd[5] = 0x02; # Command length
			$cmd[6] = ord('P');
			$cmd[7] = ord( $param=~/on/i? 'N' : 'F');
			$cmd[8] = checkSum(@cmd);
		}
		elsif($cmd=~/direct/i) {
			my $bytestr = hex_to_ascii($param);
			# send direct bytecode coded in HEX
			for(my $ii=0; $ii < length($bytestr); $ii++) {
				$cmd[$ii] = ord(substr($bytestr,$ii,1));
			}
		}
		elsif($cmd=~/payload/i) {
			$cmd[5] = length($param); # Command length
			for(my $ii=0; $ii < $cmd[5]; $ii++) {
				$cmd[$ii+6] = ord(substr($param,$ii,1));
			}
			$cmd[$cmd[5]+6] = checksum(@cmd);
		}
		elsif($cmd=~/TVIR/i) { # TV/IR flip
			if( camType($dev) =~ /WSTI/i ) {
				$cmd[5] = 4;
				$cmd[6] = ord('B');
				$cmd[7] = ord('1');
				$cmd[8] = ord('5');
				$cmd[9] = ord('P'); # Press
				$cmd[10] = checkSum(@cmd);
				cmdTransmit($dev,@cmd);
				$#cmd = 9; 
				$cmd[9] = ord('R'); # Release
				$cmd[10] = checkSum(@cmd);
				cmdTransmit($dev,@cmd);
				return;
			}
			elsif( camType($dev) =~ /MRTI/i ) {
				$cmd[5] = 12 ; # command length
				$cmd[6] = ord('E');
				$cmd[7] = ord('M');
				$cmd[8] = ord('m');
				$cmd[9] = ord('x');
				$cmd[10] = $cmd[11] = $cmd[13] = $cmd[14] = ord('0');
				$cmd[12] = ord('1');
				$cmd[15] = ord('2');
				$cmd[16] = ord(substr($param,0,1));
				$cmd[17] = ord(substr($param,1,1));
				$cmd[18] = checkSum(@cmd);
				# let standard handler transmit command
			}
		}
		elsif($cmd=~/gain/i || $cmd=~/autoiris/i) {
			$cmd[5] = 0x02; # Command length
			$cmd[6] = ord('L');
			$cmd[7] = ord( $param=~/auto|on/i? 'A' : 'B');
			$cmd[8] = checkSum(@cmd);
			if( camType($dev) =~ /WSTI|MRTI/i ) { # for thermal - issue IA / IM
				cmdTransmit($dev,@cmd);
				$#cmd = 5;
				$cmd[5] = 0x02; # Command length
				$cmd[6] = ord('I');
				$cmd[7] = ord($param=~/auto|on/i? 'A' : 'M');
				$cmd[8] = checkSum(@cmd);
			}
		}
		elsif($cmd=~/autofocus/i) {
			if ( camType($dev) =~ /COHU/i ) { # actually will work only for COHU
				$cmd[5] = 0x02; # Command length
				$cmd[6] = ord('L'); 
				$cmd[7] = ord('?');
				$cmd[8] = checkSum(@cmd);
				$resp = cmdTransmit($dev,@cmd);
				select(undef,undef,undef,.25);
				# if returned L1A3 -> manual / L1A0 -> auto
				$log->debug('Resp AUTOFOCUS code:',substr($resp,14,4));                 #posox   
				# check byte 14 is  'L' and byte 16 is 'A', if no - return
				#(byte 17 - 0x30)%2  return 1-st bit    
				if((substr($resp,14,1) eq 'L')&&(substr($resp,16,1) eq 'A'))       
				{
				  my $checkAutoFocus=substr($resp,17,1) % 2;
					  $log->debug('Resp checkAutoFocus :',$checkAutoFocus);                 #posox                                    
				  
				  if((($checkAutoFocus!=0) and ($param=~/on/)) or (($checkAutoFocus==0) and ($param=~/off/)))
					#  ((substr($resp,14,4) ne 'L1A0') and ($param=~/on/))
					# or
					#  ((substr($resp,14,4) eq 'L1A0') and ($param=~/off/))
					#) 
				  { # do only if have to
					  $#cmd = 5;
					  $cmd[5] = 0x02; # Command length
					  $cmd[6] = ord('L'); 
					  $cmd[7] = ord('2');
					  $cmd[8] = checkSum(@cmd);
					  $resp = cmdTransmit($dev,@cmd);
					  $log->debug('Resp AUTOFOCUS toggle code:',substr($resp,14,4));                 #posox                                    
					  return;
				  }
				}
				else {
					return; # do nothing - it's perfect already
				}
			}
		}
		elsif($cmd=~/hot/i) {
			$cmd[5] = 0x02; # Command length
			$cmd[6] = ord('H');  # HB - hot black / HW - hot white
			$cmd[7] = ord( $param=~/black/i? 'B' : 'W');
			$cmd[8] = checkSum(@cmd);
		}
		elsif($cmd=~/fov/i) {
			$cmd[5] = 0x02; # Command length
			$cmd[6] = ord('L');  # LW - wide / LN - narrow
			$cmd[7] = ord( $param=~/wide/i? 'W' : 'N');
			$cmd[8] = checkSum(@cmd);
		}
	}
	# default behavior - send command to devices up to 3 times until acknowledged
	for(my $i=3; --$i;) {
		$resp = cmdTransmit($dev,@cmd);
		$log->debug('Resp code:',ord(substr($resp,6,1)));
		last if ord(substr($resp,6,1)) == 0x06; # ACK=0x06 / NACK=0x15
	}
} # sub camCmd

# -------------------------------------------------------------- checkSum -----
sub checkSum
{
	my (@arr) = @_;
	my $LRC = 0;
	for ( my $i = 1; $i <= $#arr; $i++) {
		$LRC ^= $arr[$i];
		# $log->debug(sprintf('checksum byte:%02X LRC:%02X',$arr[$i],$LRC));
	}
	return ($LRC & 0x0F) + 0x80;
} # sub checkSum

# ------------------------------------------------------- SendAliveSignal -----

sub SendAliveSignal {
	$log->debug('---===*** SendAliveSignal ***===---');
	foreach my $dev ( keys %conf ) {
		next if not $dev;
		my @cmd;                                # Byte | Description
		$cmd[0] = 0xF8;                         #   0  | Autorate character
		$cmd[1] = $conf{$dev}{PTZID};           #   1  | Message destination address
		$cmd[2] = 0x2A;                         #   2  | '*' (asterisk) constant
		$cmd[3] = $conf{$dev}{PTZGROUP};        #   3  | Group address
		$cmd[4] = 0x1F;                         #   4  | Message source address (0x1F is master control unit)
		$cmd[5] = 0x02;                         #   5  |Command length
		$cmd[6] = ord('P');
		$cmd[7] = ord('?');
		$cmd[8] = checkSum(@cmd);
		$log->debug("DEVICE:$dev $conf{$dev}{PTZ_TRANSPORT}");
		cmdTransmit($dev,@cmd);
	}
} # sub SendAliveSignal

# ----------------------------------------------------------- cmdTransmit -----
sub cmdTransmit {
	my ($dev,@cmd)=@_;
	
	$log->debug("NNNNNN=",$NN++);
	
	my $command='';
	my $b;
	foreach $b (@cmd) { $command .= sprintf("%02X ",$b); }
	$command .= sprintf(' %d~byte CMD:',$cmd[5]);
	for ($b=6; $b < $#cmd; $b++) { $command .= chr($cmd[$b]); }
	$log->debug('cmdTransmit( ',$dev, '=> ',$command,')');    
	if($conf{$dev}{PTZ_TRANSPORT}=~/HTTP/) {
		my $sock1 = IO::Socket::INET->new(PeerAddr => $conf{$dev}{DEVIP},
                                                    PeerPort => 80,
                                                    Proto    => "tcp",
                                                    Type     => SOCK_STREAM);
		if(!$sock1) {
			$log->debug('Can not open socket for port 80');
			return
		}
		$sock1->autoflush(1);
		nonblock($sock1);
		my $ss1 = IO::Select->new($sock1);
		
		my $usrpsw = "$conf{$cam}{USRNAME}:$conf{$cam}{PASSWD}" if $conf{$cam}{USRNAME} && $conf{$cam}{PASSWD};
		my $auth=($usrpsw)?"\nAuthorization: Basic ". MIME::Base64::encode($usrpsw,''):'';
		$command = '';
		foreach $b (@cmd) { $command .= sprintf("%02X",$b); }
		my $path = "/axis-cgi/com/serial.cgi?port=2&write=$command&timeout=120&read=50";
		#$log->debug($path);
		my $get=qq(GET /$path HTTP/1.0$auth
Accept: image/gif, image/x-xbitmap, image/jpeg, image/pjpeg, image/png
Accept-Charset: iso-8859-1,*,utf-8
Accept-Encoding: gzip
Accept-Language: en
User-Agent: Mozilla/4.76 [en] (X11; U; Linux 2.2.16 i686)

);
		$log->debug("SEND: $get");
		print $sock1 $get;
		
		
		select(undef,undef,undef,.115);
		foreach my $sss ( $ss1->can_read(1) ) {
			my $data = '';
			eval {
				alarm 1;
				$sss->recv($data, POSIX::BUFSIZ, 0);
				#$command='';
				#for($b=0;$b<length($data);$b++) { $command .= sprintf("%02X ",ord(substr($data,$b,1))); }
				#$log->debug('Received:',$command);
				#$log->debug('Received:',$data);
			};
			alarm 0;
			$data =~ /#(.+)#/g;
			$log->debug("Feedback: $1");
			my $resp= $1;
			if (ord(substr($resp,6,1)) != 0x06) #check on acknowledge from the camera - reporting only
			{
			  $log->error("NO Acknowledge coming from the camera!");
			}
			return '' if ! $1;
			return hex_to_ascii($1);
		}  # foreach                                                          
	}
	elsif($conf{$dev}{PTZ_TRANSPORT}=~/TCP/) {
		my $bts = '';
		foreach my $b (@cmd) { $bts .= chr($b); }
		my $retries = 1;
		secondaryPass:
		my $skt = $sock{ $conf{$dev}{DEVIP} }{SOCK};
		eval {
			local $SIG{PIPE} = sub { die "Error writing to socket" };
			print $skt $bts;
		};
		if($@) {
			$log->error($@);
			# camera did not answer or socket error occured - try to close socket and reopen it
			eval{ $sock{$conf{$dev}{DEVIP}}{SOCK}->close(); };
			$log->error("Re-open socket connection: $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT}");
			$sock{$conf{$dev}{DEVIP}}{SEL} = undef;
			select(undef,undef,undef,.1);
			$sock{$conf{$dev}{DEVIP}}{SOCK} = IO::Socket::INET->new(PeerAddr => $conf{$dev}{DEVIP},
                                                    PeerPort => $conf{$dev}{PTZ_TCP_PORT},
                                                    Proto    => "tcp",
                                                    Type     => SOCK_STREAM);
			if(! $sock{$conf{$dev}{DEVIP}}{SOCK} ) {
				$log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
				delete $sock{$conf{$dev}{DEVIP}};
				return;
			}
			$sock{$conf{$dev}{DEVIP}}{SOCK}->autoflush(1);
			nonblock($sock{$conf{$dev}{DEVIP}}{SOCK});
			$sock{$conf{$dev}{DEVIP}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}}{SOCK});
			$log->info('Socket was closed, reconnected');
			goto secondaryPass if $retries--;

		return;
		}
		select(undef,undef,undef,.15);
		foreach my $sss ( $sock{ $conf{$dev}{DEVIP} }{SEL}->can_read(1) ) {
			my $data = '';
			eval {
				alarm 1;
				$sss->recv($data, POSIX::BUFSIZ, 0);
				$command='';
				for($b=0;$b<length($data);$b++) { $command .= sprintf("%02X ",ord(substr($data,$b,1))); }
				$log->debug('Received:',$command);
			};
			alarm 0;
			if (ord(substr($data,6,1)) != 0x06) #check on acknowledge from the camera - reporting only
			{
			  $log->error("NO Acknowledge coming from the camera!");
			}            
			return $data;
		}  # foreach      
	}
	else { # serial connection
		foreach $b (@cmd) { $port{$conf{$dev}{PTZHWPORT}}->write(chr($b)); }
		$port{$conf{$dev}{PTZHWPORT}}->write_drain;
		my ($count_in, $string_in) = $port{$conf{$dev}{PTZHWPORT}}->read(50);
		my $resp= substr($string_in,0,$count_in);
		if (ord(substr($resp,6,1)) != 0x06)#check on acknowledge from the camera - reporting only
		{
		  $log->error("NO Acknowledge coming from the camera!");
		}        
		return $resp;
	} # if($conf{$dev}{PTZ_TRANSPORT}=~/TCP/)
} # sub cmdTransmit


# ---------------------------------------------------------- purgeBuffers -----
# go other devices and purge buffers to prevent chaotic movements due to
# messages stuck in buffers
sub purgeBuffers {
	foreach my $dev ( keys %conf ) {
		next if ! ($conf{$dev}{PTZ_TRANSPORT}=~/TCP/);
		my $skt = $sock{ $conf{$dev}{DEVIP} }{SOCK};
		foreach my $sss ( $sock{$conf{$dev}{DEVIP}}{SEL}->can_read(1) ) {
			my $data = '';
			eval {
				alarm 1;
				$sss->recv($data, POSIX::BUFSIZ, 0);
				if( length($data)) {
					my $command='';
					for($b=0;$b<length($data);$b++) { $command .= sprintf("%02X ",ord(substr($data,$b,1))); }
					$log->debug('PURGED:',$command);
				}
			};
			alarm 0;
		}
	}
} # sub purgeBuffers

# --------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {

	$log->info('Configurations refreshed: ',$query_str);
	
	# fisrst, close everything
	foreach my $skt (keys %sock) {
		close($sock{$skt}{SOCK});
		delete $sock{$skt};
	}
	foreach my $p ( keys %port ) {
		$port{$p}->close;
		delete $port{$p};
	}

	%conf = GetCfgs( eval("($query_str)") );     # Load configurations
	foreach my $dev (keys %conf) {
		$log->debug("[$dev]");
		$log->debug("DEVID=$conf{$dev}{DEVID}");
		$log->debug("PTZID=$conf{$dev}{PTZID}");
		$log->debug("PTGROUP=$conf{$dev}{PTZGROUP}");
		$log->debug("DEVIP=$conf{$dev}{DEVIP}");
		$log->debug("PTZ_TRANSPORT=$conf{$dev}{PTZ_TRANSPORT}");
		if($conf{$dev}{PTZ_TRANSPORT}=~/HTTP/) {
			$log->debug('!!!HTTP!!!');
		}
		elsif($conf{$dev}{PTZ_TRANSPORT}=~/TCP/) {
			$log->info("Open socket connecftion: $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT}");
			if(not defined($sock{$conf{$dev}{DEVIP}})) {
                $sock{$conf{$dev}{DEVIP}}{SOCK} = IO::Socket::INET->new(PeerAddr => $conf{$dev}{DEVIP},
                                                        PeerPort => $conf{$dev}{PTZ_TCP_PORT},
                                                        Proto    => "tcp",
                                                        Type     => SOCK_STREAM);
				if(! $sock{$conf{$dev}{DEVIP}}{SOCK} ) {
					$log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
					delete $sock{$conf{$dev}{DEVIP}};
					next;
				}
				$sock{$conf{$dev}{DEVIP}}{SOCK}->autoflush(1);
				nonblock($sock{$conf{$dev}{DEVIP}}{SOCK});
				$sock{$conf{$dev}{DEVIP}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}}{SOCK});
			}
		}
		else { # serial
			$log->debug("PTZHWPORT=$conf{$dev}{PTZHWPORT}");
			$log->debug("PTZSPEED=$conf{$dev}{PTZSPEED}");
			$log->debug("PTZHWPORT_SPEED=$conf{$dev}{PTZHWPORT_SPEED}");
			$log->debug("PTZHWPORT_BITS=$conf{$dev}{PTZHWPORT_BITS}");
			$log->debug("PTZHWPORT_STOPBITS=$conf{$dev}{PTZHWPORT_STOPBITS}");
			$log->debug("PTZHWPORT_PARITY=$conf{$dev}{PTZHWPORT_PARITY}");
			if(not defined($port{$conf{$dev}{PTZHWPORT}})) {
				$log->debug("Initialise serial port: $conf{$dev}{PTZHWPORT}");
				$port{$conf{$dev}{PTZHWPORT}} = new Device::SerialPort($conf{$dev}{PTZHWPORT}) || $log->error("Can't open port $conf{$dev}{PTZHWPORT}: $!\n"),next;
				$port{$conf{$dev}{PTZHWPORT}}->handshake("none"); # none / rts / xoff
				$port{$conf{$dev}{PTZHWPORT}}->baudrate($conf{$dev}{PTZHWPORT_SPEED});
				$port{$conf{$dev}{PTZHWPORT}}->parity($conf{$dev}{PTZHWPORT_PARITY});
				$port{$conf{$dev}{PTZHWPORT}}->databits($conf{$dev}{PTZHWPORT_BITS});
				$port{$conf{$dev}{PTZHWPORT}}->stopbits($conf{$dev}{PTZHWPORT_STOPBITS});

				$port{$conf{$dev}{PTZHWPORT}}->buffers(4096, 4096);
				$port{$conf{$dev}{PTZHWPORT}}->read_const_time(20);  # milliseconds
				$port{$conf{$dev}{PTZHWPORT}}->read_char_time(5);
			}
		} # if($conf{$dev}{PTZ_TRANSPORT}=~/TCP/)
		camIdentify($dev);
	} # foreach $dev
} # sub load_dev_conf

# -------------------------------------------------------------- nonblock -----
sub nonblock {
	my ($fd) = @_;
	my $flags = fcntl($fd, F_GETFL,0);
	fcntl($fd, F_SETFL, $flags|O_NONBLOCK);
}

# -------------------------------------------------------------- icdDigit -----
sub icdDigit {
	my ($d) = @_;
	$d = ord($d) - 0x30;
	return $d<10? $d+0x30 : $d+0x41;
} # sub icdDigit

# ---------------------------------------------------------------- camType ----
sub camType {
	my $dev = shift;
	return '' if not defined $conf{$dev}{PTZ_CAMERATYPE};
	return $conf{$dev}{PTZ_CAMERATYPE};
} # sub camType

# ------------------------------------------------------------ camIdentify ----
sub camIdentify {
	my ($dev) = @_;
	my @cmd;                                # Byte | Description
	$cmd[0] = 0xF8;                         #   0  | Autorate character
	$cmd[1] = $conf{$dev}{PTZID};           #   1  | Message destination address
	$cmd[2] = 0x2A;                         #   2  | '*' (asterisk) constant
	$cmd[3] = $conf{$dev}{PTZGROUP};        #   3  | Group address
	$cmd[4] = 0x1F;                         #   4  | Message source address (0x1F is master control unit)
											#   5  | Length of message data
											#   .. | Command data
											# 6+len| Checksum + 0x80 (0x80..0x8F)
	$cmd[5] = 0x02; # Command length
	$cmd[6] = ord('I');
	$cmd[7] = ord('?');
	$cmd[8] = checkSum(@cmd);
	
  
	my $resp = cmdTransmit($dev,@cmd);
	return if length($resp) < 10;
	$log->debug('Identify:',substr($resp,10));    
} # sub camIdentify

sub ascii_to_hex ($)
{
	## Convert each ASCII character to a two-digit hex number.
	(my $str = shift) =~ s/(.|\n)/sprintf("%02lx", ord $1)/eg;
	return $str;
}

sub hex_to_ascii ($)
{
	## Convert each two-digit hex number back to an ASCII character.
	(my $str = shift) =~ s/([a-fA-F0-9]{2})/chr(hex $1)/eg;
	return $str;
}


# ----------===== Changes log =====----------
# $Log$
# Revision 1.44  2006/06/19 15:10:58  posox
# Report ERROR on NO acknowledge from the camera on cmdTransmit()
#
# Revision 1.43  2006/05/22 12:16:35  posox
# After getting STOP command - preserving and sending one more
#
# Revision 1.42  2006/05/19 14:27:17  posox
# AutoFOCUS bug fix
#
# Revision 1.41  2006/05/19 09:18:25  afomenko
# intermediate version - Sergey will take care of COHU focusing and commit soon
#
# Revision 1.40  2005/12/19 20:46:26  afomenko
# use PTZSPEED when working with JOYSTICK commands - COHU added
#
# Revision 1.39  2005/12/19 20:45:21  afomenko
# use PTZSPEED when working with JOYSTICK commands
#
# Revision 1.38  2005/12/19 20:01:05  afomenko
# shift preset number
#
# Revision 1.37  2005/10/18 15:46:59  afomenko
# COHU got same type of STEP/MOVE as the rest
#
# Revision 1.36  2005/10/17 17:54:38  afomenko
# logger include file changed
#
# Revision 1.35  2005/08/04 20:11:10  afomenko
# extra debug messages, one small bug fixed in reconnect
#
# Revision 1.34  2005/06/28 17:12:34  afomenko
# reduced timeout waiting for feedback
#
# Revision 1.33  2005/06/23 14:49:36  afomenko
# added direct TVIR command for MRTI
#
# Revision 1.32  2005/06/23 13:50:46  afomenko
# step mode - commands now go in go-stop-go-stop sequence
#
# Revision 1.31  2005/06/16 15:58:27  afomenko
# fixed TMIX broken earlier
#
# Revision 1.30  2005/06/15 18:38:14  afomenko
# added ACK to REL comamnd
#
# Revision 1.29  2005/06/15 17:52:51  afomenko
# removed cameras polling, added ACK back to cameras on TMIX, increased delay betwee go-stop for step commands for MRTI
#
# Revision 1.28  2005/06/09 14:59:55  afomenko
# added "sleep" into main loop to reduce CPU load
#
# Revision 1.27  2005/06/02 15:10:38  afomenko
# polling command changed to p?
#
# Revision 1.26  2005/06/02 14:09:48  afomenko
# added delay in "step" commands for MRTI
#
# Revision 1.25  2005/05/20 13:38:22  afomenko
# reduced polling activity. NOT TESTED on real cameras yet
#
# Revision 1.24  2005/04/21 13:00:43  afomenko
# PERFORMANCE IMPROVEMENT: added possibility to "SPLIT" driver processes, so each driver serves one resource
#
# Revision 1.23  2005/04/19 15:18:52  afomenko
# all the drivers corrected to work with "TCP broadcast" from PTZ_server.pl
#
# Revision 1.22  2005/03/25 19:36:37  afomenko
# added direct pass-through mode, NOT TESTED
#
# Revision 1.21  2005/03/22 22:34:25  afomenko
# twicked timing for HTTP protocol
#
# Revision 1.20  2005/03/21 20:44:59  afomenko
# HTTP protocol support added
#
# Revision 1.19  2005/02/08 21:09:45  afomenko
# settings&power=on/off added
#
# Revision 1.18  2005/02/04 22:45:13  afomenko
# fix to reconnect quickly on "knocked down" connection to Axis
#
# Revision 1.17  2005/01/31 22:36:59  afomenko
# MIX fixed in Step mode
#
# Revision 1.16  2005/01/24 15:06:21  afomenko
# step commands for MRTI/WSTI/etc.. - chnaged from query/go to regular lft/right/up/down
#
# Revision 1.15  2005/01/18 04:49:48  afomenko
# fixed syntax errors
#
# Revision 1.14  2005/01/17 19:25:34  afomenko
# zoom for WSTI through button emulation
#
# Revision 1.13  2005/01/17 16:53:53  afomenko
#  Added device poll to allow "time-out" Axis servers and easy reconnects
#
# Revision 1.12  2005/01/14 21:28:29  afomenko
# working version after corrections on L3 site
#
# Revision 1.10  2004/12/17 15:02:53  afomenko
# WSTI stuff corrected, still need MRTI
#
# Revision 1.9  2004/12/15 22:43:00  afomenko
# added different commands for different camera types
#
# Revision 1.8  2004/12/10 20:34:24  afomenko
# Thermal MIX command added
#
# Revision 1.7  2004/12/08 03:08:40  afomenko
# changed autofocus command per COHU explanations
#
# Revision 1.6  2004/12/03 20:58:58  afomenko
# fixes in step zoom and autofocus
#
# Revision 1.5  2004/11/30 20:57:49  afomenko
# most of core commands work fine
#
# Revision 1.4  2004/11/30 15:10:31  afomenko
# partially working, still in process
#
# Revision 1.3  2004/10/26 15:50:04  afomenko
# The most of impoirtant commands are implemented. Nothing tested yet.
#
# Revision 1.2  2004/10/22 22:13:44  afomenko
# intermediate edition, just to keep track on what's goin on
#
# Revision 1.1  2004/10/20 21:16:18  afomenko
# initial check-in. should be great sceleton for other PTZ engines, as it has no specific functionality on this point, just structure
#
