#!/usr/bin/perl
#  $Id: ptz_pelcov2.pl 25092 2012-01-31 23:57:51Z astarostin $
# ------------------------------------------------------------------------------
#  PTZ driver for Honeywell Aux Control protocol
# ------------------------------------------------------------------------------
#  Author: Andrey Fomenko
#  Edited by: Vladimir Ryabovol
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT LLC, 2004-2005
# ------------------------------------------------------------------------------

use strict;
use POSIX;
use Socket;
use IO::File;
use IO::Select;
use IO::Socket;
use Fcntl;

use NextCAM::Init;
use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

# Constants
my $CHECK_DANTE_TIMEOUT = 5;

my $log=get_logger('NEXTCAM::PTZ::PTZ_HONEYWELL');

my $query_str = shift || "'POSITIONCTL'=>'honeywell'";
$log->info("Starting Honeywell PTZ DRIVER, query string: [$query_str]");

$SIG{HUP}=\&load_dev_conf;

my $APL=$ENV{APL};
my (%conf, %sock, $cam, $cmd, $par, $last_cmd, $last_mode, $lastchecktime);

load_dev_conf();

# ------------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# ------------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
                                   PeerPort => $TCP_PORT,
                                   Proto     => "tcp",
                                   Type      => SOCK_STREAM)
    or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';
$lastchecktime = time;

my %commands;
my $trycount = 30;
my $c = 0;

while(1) {
	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/i;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/ 
				|| $commands{$cam} =~ /speed iris=0/ 
				|| $commands{$cam} =~ /speed gain=0/ 
				)
			{
				; # do not override stop command!
			} else {
				$commands{$cam} = $cmd;
			}
		} else {
			$commands{$cam} = $cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)
	
	$cmd = '';
	
	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		delete $commands{$camera};
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
		
		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam, $mode, "pt", $params[0].','.$params[1], \%options);
			# select(undef,undef,undef,0.02);
			#camCmd($cam, $mode, "zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam, $mode, $cmd, $par, \%options);
		}
	}

	select(undef,undef,undef,.05) if not $cmd;

	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
	if($conf{$dev}{CAMERAMODEL} eq 'Dante' &&
	   (time-$lastchecktime > $CHECK_DANTE_TIMEOUT ||
		!$c))
	{
		$c++ if !$c;
		$lastchecktime = time;
		checkAndSetDanteSerialConfiguration($dev);
	}
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
	}
	
	if(!$cmd) {
		select(undef,undef,undef,.4);
		if(!$trycount--) {
			$trycount = 30;
			print $socket "test\0" or $log->logdie("Couldn't communicate to socket $TCP_PORT: $@");
		}
	}

} # while(1)


# ------------------------------------------------------------ cmdTransmit -----
sub cmdTransmit {
    my ($dev,$cmd)=@_;


    my $retries = 1;
    secondaryPass:
    my $skt = $sock{ $conf{$dev}{DEVIP} }{SOCK};
    eval {
        local $SIG{PIPE} = sub { die "Error writing to socket" };
        print $skt $cmd;
    };
    if($@) {
        $log->error($@);
        # camera did not answer or socket error occured - try to close socket and reopen it
        eval{ $sock{$conf{$dev}{DEVIP}}{SOCK}->close(); };
        $log->error("Re-open socket connection: $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT}");
        $sock{$conf{$dev}{DEVIP}}{SEL} = undef;
        select(undef,undef,undef,.1);
        $sock{$conf{$dev}{DEVIP}}{SOCK} = IO::Socket::INET->new(PeerAddr => $conf{$dev}{DEVIP},
                                                PeerPort => $conf{$dev}{PTZ_TCP_PORT},
                                                Proto    => "tcp",
                                                Type     => SOCK_STREAM);
        if(! $sock{$conf{$dev}{DEVIP}}{SOCK} ) {
            $log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
            delete $sock{$conf{$dev}{DEVIP}};
            return;
        }
        $sock{$conf{$dev}{DEVIP}}{SOCK}->autoflush(1);
        nonblock($sock{$conf{$dev}{DEVIP}}{SOCK});
        $sock{$conf{$dev}{DEVIP}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}}{SOCK});
        $log->info('Socket was closed, reconnected');
        goto secondaryPass if $retries--;

        return;
    }

    select(undef,undef,undef,.05);
    foreach my $sss ( $sock{ $conf{$dev}{DEVIP} }{SEL}->can_read(0.1) ) {
	my $data = '';
        eval {
            alarm 1;
            $sss->recv($data, POSIX::BUFSIZ, 0);
        };
        alarm 0;
        if($data=~/\?\?\?/){
    	    $log->error("No Acknowledge coming from the camera!");
        }
        return $data;
    }  # foreach
    
} # sub cmdTransmit


# ----------------------------------------------------------------- camStop ----
sub camStop {
  my ($devid)=@_;
  return "PTZC $devid,0,0\r"; 
} # sub PTspeed
# ----------------------------------------------------------------- PTspeed ----
sub PTspeed {
  my ($speed)=@_;
  return int($speed/100 * 127); 
} # sub PTspeed

# ----------------------------------------------------------------- camCmd -----
sub camCmd {
  my ($dev,$mode,$cmd,$param,$options)=@_;
  my %options = %$options;
  $log->debug("camCmd: DEVID=$dev mode: $mode command: $cmd optional parameter: $param");
  my $devid=$conf{$dev}{PTZID}; # camera hardware ID
  
  # here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
  if($conf{$dev}->{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
     if($mode=~/speed/i || $mode=~/step/i || $mode=~/abs/i || $mode=~/rel/i || $mode=~/smooth/i) {
        $log->debug("Setting timeout +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
        $conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
     }
     elsif($mode=~/preset/i && $cmd=~/goto/i && $param!=1) {
        $log->debug("Setting timeout (preset) +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
        $conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
    }
  }
  
  my $cd = '';
  if(not $cmd=~/stop/) { # non-Stop
    # set camera speed
   if ($mode=~/speed/i){        # mode speed
    if($cmd=~/PT/i) { # RPT
      my ($p,$t) = split(/,/,$param);
      my $ps=PTspeed($p);
      my $ts=PTspeed($t);
      $log->debug("RPT $p $t");
      $cd .= "PTZC $devid,1,$ps\r";
      $cd .= "PTZC $devid,2,$ts\r";
    }
    elsif($cmd=~/Z/i) { 
      $log->debug("ZOOM $param");
      my $zs = PTspeed($param);
      $cd .= "PTZC $devid,3,$zs\r";
    }
    elsif($cmd=~/focus/i) {
      $log->debug("FOCUS $param");
      my $fs = PTspeed($param);
      $cd .= "PTZC $devid,4,$fs\r";
    }
    elsif($cmd=~/iris/i) {
      $log->debug("IRIS $param");
      my $is = PTspeed($param);
      $cd .= "PTZC $devid,5,$is\r";
    }
   }elsif($mode=~/rel/i) { # mode=rel: "<devid> rel size=640x480 xy=225,152"
      #cmdTransmit($dev,@cmd);
      return;
   }elsif ($mode=~/settings/i){ # settings
    if($cmd=~/irpwr/i) { 
      $cd .= "AUX $devid,6";
    }
    elsif($cmd=~/irswitch/i) { 
      $cd .= "AUX $devid,1"; 
    }
    elsif($cmd=~/autopan/i) { #'awb_on','g', 'awb_off','b'
      $cd .= "AUX $devid,5"; 
    }
    cmdTransmit($dev,$cd);
   }elsif($mode=~/step/i){       # mode=step  /Step by step positioning/
      my $timeout = 0.2;
    if($cmd=~/move/i) { # step pan/tilt
      $cd .= "PTZC $devid,1,63\r" if $param=~/right/;
      $cd .= "PTZC $devid,1,-63\r" if $param=~/left/;
      $cd .= "PTZC $devid,2,63\r" if $param=~/up/;
      $cd .= "PTZC $devid,2,-63\r" if $param=~/down/;
    } elsif ($cmd=~/zoom/i){
      $cd .= "PTZC $devid,3,63\r" if $param=~/in/;
      $cd .= "PTZC $devid,3,-63\r" if $param=~/out/;
    } elsif ($cmd=~/focus/i){    
      $cd .= "PTZC $devid,4,63\r" if $param=~/near/;
      $cd .= "PTZC $devid,4,-63\r" if $param=~/far/;
    } elsif ($cmd=~/iris/i){
      $cd .= "PTZC $devid,5,63\r" if $param=~/open/;
      $cd .= "PTZC $devid,5,-63\r" if $param=~/close/;
    }
      cmdTransmit($dev,$cd);
      select(undef,undef,undef,$timeout);
      cmdTransmit($dev,camStop($devid));
      return;   
   }elsif($mode=~/hardware/i){   # mode=hardware  /Hardware reset,defaults/
    }
   } elsif($mode=~/abs/i){                # mode=abs  /Absloute positioning/
      if($cmd=~/center/i) { # Center
      } elsif($cmd=~/PT/i){            # Pan, Tilt
      } elsif ($cmd=~/Z/i){	# Absolute zoom
      }
   } elsif($mode=~/sequence/i){ # presets
      if($param=~/stop,(\d*)i/i){ 
	$cd.="STOPSEQ $1\r";
      }else{
	$cd.="RUNSEQ $param\r";
      }
   } elsif($mode=~/preset/i){ # presets
      $cd.="PTZSTOR $devid,$param\r" if $cmd =~/save/i; 
      $cd.="PTZRCL $devid,$param\r" if $cmd =~/goto/i; 
   } elsif($mode=~/smooth/i) { # Right/Left/Up/Down/Tele/Wide/focus_Near/focus_Far/iris_Open/irIs_close
      # set movement bits
      if ($cmd=~/move/i) { # pan/tilt
      } elsif($cmd=~/zoom/i) { # zoom
      } elsif($cmd=~/focus/i) { # focus
      } elsif($cmd=~/iris/i) { # iris
      }
  } else{ # Stop
    $cd = camStop($devid);
  }

  cmdTransmit($dev,$cd);
}

# --------------------------------------------------------------- nonblock -----
#   nonblock($socket) puts socket into nonblocking mode
sub nonblock {
    my $socket = shift;
    my $flags;
    
    $flags = fcntl($socket, F_GETFL, 0)
            or die "Can't get flags for socket: $!\n";
    fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
            or die "Can't make socket nonblocking: $!\n";
}

# ---------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {

    $log->info('Configurations refreshed: ',$query_str);
    
    # fisrst, close everything
    foreach my $skt (keys %sock) {
        close($sock{$skt}{SOCK});
        delete $sock{$skt};
    }

    %conf = GetCfgs( eval("($query_str)") );     # Load configurations
    foreach my $dev (keys %conf) {
	# in a case when PTZ works on another server (e.g. 3ETI COMTROL Serial Server)
	$conf{$dev}{DEVIP} = $conf{$dev}{PTZ_TCP_IP} if $conf{$dev}{PTZ_TCP_IP};

	$log->debug("[$dev]");
	$log->debug("DEVID=$conf{$dev}{DEVID}");
	$log->debug("PTZID=$conf{$dev}{PTZID}");
	$log->debug("PTGROUP=$conf{$dev}{PTZGROUP}");
	$log->debug("PTZSPEED=$conf{$dev}{PTZSPEED}");
	$log->debug("DEVIP=$conf{$dev}{DEVIP}");
	$log->debug("PTZ_TRANSPORT_PROTO=$conf{$dev}{PTZ_TRANSPORT_PROTO}");
	$log->debug("PTZ_TCP_PORT=$conf{$dev}{PTZ_TCP_PORT}");

	$log->info("Open socket connection: $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT}");
        if(not defined($sock{$conf{$dev}{DEVIP}})) {
	    $sock{$conf{$dev}{DEVIP}}{SOCK} = IO::Socket::INET->new(PeerAddr => $conf{$dev}{DEVIP},
    	                            		PeerPort => $conf{$dev}{PTZ_TCP_PORT},
                                                Proto    => "tcp",
                                                Type     => SOCK_STREAM);
	    if(! $sock{$conf{$dev}{DEVIP}}{SOCK} ) {
                $log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
                delete $sock{$conf{$dev}{DEVIP}};
                next;
	    }

	    $sock{$conf{$dev}{DEVIP}}{SOCK}->autoflush(1);
	    nonblock($sock{$conf{$dev}{DEVIP}}{SOCK});
	    $sock{$conf{$dev}{DEVIP}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}}{SOCK});
        }
    } # foreach $dev
} # sub load_dev_conf

