#!/usr/bin/perl
# -----------------------------------------------------------------------------
#  PTZ driver for HikVision cameras
# -----------------------------------------------------------------------------
#  Author: Andrey Fomenko
#  Edited by: Alexey Tsibulnik
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT LLC, 2004-2005
# -----------------------------------------------------------------------------

use strict;
use Socket;
use MIME::Base64();
use IO::File;
use IO::Select;
use IO::Socket;
use Fcntl;
use XML::Simple;
use HTTP::Request::Common qw(GET POST PUT DELETE);
use SKM::Agent;

use NextCAM::Conf "GetCfgs";
use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

my $log=get_logger('NEXTCAM::PTZ');

my $query_str = shift || "'POSITIONCTL'=>'HikVision','CAMERAMODEL'=>'HikVision'";
$log->info("Starting HikVision PTZ DRIVER, query string: [$query_str]");

my $ua = new SKM::Agent;
$ua->timeout(5);

$SIG{HUP}=\&load_dev_conf;
my $APL=$ENV{APL};
my $APL_CONF=$ENV{APL_CONF};
my (%conf,$cam,$cmd,$usrname,$passwd,$par,$last_cmd,$last_mode);
load_dev_conf();

# -----------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# -----------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
                                   PeerPort => $TCP_PORT,
                                   Proto    => "tcp",
                                   Type     => SOCK_STREAM)
    or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';

my %commands;
my $ready;
my $trycount = 30;

while(1) {
	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/i;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/
				|| $commands{$cam} =~ /speed iris=0/
				|| $commands{$cam} =~ /speed gain=0/
				)
			{
				; # do not override stop command!
			}else{
				$commands{$cam} = $cmd;
			}
		} else {
			$commands{$cam} = $cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)

	$cmd = '';

	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		delete $commands{$camera};
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options = map {/(\w+)=(.*)/} split(/\s/," $options ");
		($usrname, $passwd) = ('', '');
		$usrname = $conf{$cam}{USRNAME};
		$passwd = $conf{$cam}{PASSWD};

		$ua->set_creds($conf{$cam}{USRNAME}, $conf{$cam}{PASSWD});

		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam,$mode,"pt", $params[0].','.$params[1] ,\%options);
			camCmd($cam,$mode,"zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam,$mode,$cmd,$par,\%options);
		}
	}

	select(undef,undef,undef,.1) if not $cmd;

	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
	}

	if(!$cmd) {
		select(undef,undef,undef,.4);
		if(!$trycount--) {
			$trycount = 30;
			print $socket "test\0" or $log->logdie("Couldn't communicate to socket $TCP_PORT: $@");
		}
	}
} # while(1)

# ---------------------------------------------------------------- camCmd -----
sub camCmd
{
	my ($dev,$mode,$cmd,$param,$options)=@_;
	my $cfg = $conf{$dev};
	my %options = %$options;
	$log->debug("camCmd: DEVID=[$dev] mode=[$mode] command:[$cmd] param=[$param] ");
	my $spd=($options{speed}=~/^\d+/) ? $options{speed}:$cfg->{PTZSPEED};
	my $zoomspd = $cfg->{PTZZOOMSPEED} || 50;
	
	# Normalize speed
	$spd=1 if $spd < 1;
        $spd=100 if $spd > 100;


	# we need to "terminate" each command because it can move forever if used with no STOP !
	if ($last_mode=~/smooth/i) {
		hikContCmd($cfg, 0, 0, 0);
	}

	# here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
	if($conf{$dev}->{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
		if($mode=~/speed/i || $mode=~/step/i || $mode=~/abs/i || $mode=~/rel/i || $mode=~/smooth/i) {
			$log->debug("Setting timeout +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
		elsif($mode=~/preset/i && $cmd=~/goto/i && $param!=1) {
			$log->debug("Setting timeout (preset) +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
	}

	if ($mode=~/speed/i){
		if($cmd=~/^pt$/i) { # pan-tilt
			my ($p,$t) = split(/,/,$param);
			$p = int($p*$spd/100);
			$t = -int($t*$spd/100);
			hikContCmd($cfg, $p, $t, 0);
		}
		elsif($cmd=~/^zoom$/i) { # zoom
			my $zm=int($param * $zoomspd/100);
			hikContCmd($cfg, 0, 0, $zm);
		}
		elsif($cmd=~/focus/i) { # focus
			my $fc=int($param*$spd/100);
			hikFocusCmd($cfg, $fc);
		}
		elsif($cmd=~/iris/i) { # focus
			my $iris=int($param*$spd/50);
			hikIrisCmd($cfg, $iris);
		}
	} elsif($mode=~/abs/i){ # mode=ABS
		# Absolute position function
		# <AbsoluteHigh> is high precision positioning which is accurate to a bit after the decimal point;
		# For example elevation -900..2700 is corresponding to vertical -90.0-270.0 degree, and azimuth
		# 0..3600 is corresponding to horizontal 0.0-360.0 degree, absoluteZoom is corresponding to zoom
		# 0.0..100.0;
		if ($cmd=~/pt/i) {
			$param=~/(\-?\d+)\s*,\s*(\-?\d+)\s*$/;
			my $p = $1;
			my $t = $2;
			# Convert pan and tilt angles from -180..180 to 0..360.0 
			# horizontal degree and -90.0..270.0 vertical degree
			$p = ($p + 180) * 10;
			$t = ($t + 90) + 10;
			hikAbsCmd($cfg, $p, $t, undef);
		}
		elsif($cmd=~/z/i) {
			$param =~ /(\d+)/;
			$param = 0 unless $param;
			my $absZoom = int($param * 10); # value should be in range [0, 1000]
			$absZoom = 1000 if $absZoom > 1000;
			$absZoom = 0 if $absZoom < 0;
			hikAbsCmd($cfg, undef, undef, $absZoom);
		}
		elsif($cmd=~/center/i) { # center
			hikAbsCmd($cfg, 0, 0, 0);
		}
	} elsif($mode=~/rel/i){ # relative positioning (recentering)
		$param =~ /(\d+)x(\d+)/;
		my ($width, $height) = ($1,$2);
		$options{xy} =~ /(\d+),(\d+)/;
		my ($x, $y) = ($1,$2);
		my ($center_x, $center_y) = ($width/2, $height/2);
		my ($diff_x, $diff_y) = ($x - $center_x, $y - $center_y);
		my ($abs_diff_x, $abs_diff_y) = ($diff_x/$width, $diff_y/$width);
		
		my $status = hikGetPTZStatus($cfg);
		if (defined $status) {
			my ($elev, $azim, $zoom) = @$status;
			# Convert screen coord to absolute position, using current value of absolute zoom
			my $hfov = 59.5/(0.79 * ($zoom / 10) + 0.21);
			$hfov = int($hfov * 10);
			my $new_azim = $azim + int($abs_diff_x * $hfov);
			my $new_elev = $elev + int($abs_diff_y * $hfov);
			
			hikAbsCmd($cfg, $new_elev, $new_azim, $zoom);
		}
	} elsif($mode=~/step/i) { # mode=step /Step by step positioning/
		if($cmd=~/move/i) { # step pan/tilt
			my ($pan, $tilt) = (0, 0);
			for ($param) {
			    /^right$/i and do { $pan = $spd;   last };
			    /^left$/i  and do { $pan = -$spd;  last };
			    /^up$/i    and do { $tilt = $spd;  last };
			    /^down$/i  and do { $tilt = -$spd; last };
			}
			hikMomentaryCmd($cfg, $pan, $tilt, 0, 300);
		}
		elsif($cmd=~/zoom/i) {
			my $zm = $param=~/out/i? -$zoomspd : $zoomspd;
			$zm = int($zm);
			hikMomentaryCmd($cfg, 0, 0, $zm, 300);
		}
		elsif($cmd=~/focus/i) {
			$spd=100;
			my $focus = $param=~/far/i? -$spd : $spd;
			hikFocusCmd($cfg, $focus);
			select(undef, undef, undef, 0.2);
			hikFocusCmd($cfg, 0);
		}
		elsif($cmd=~/iris/i) {
			$spd=100;
			my $iris = $param=~/open/i? $spd : -$spd;
			hikFocusCmd($cfg, $iris);
			select(undef, undef, undef, 0.2);
			hikFocusCmd($cfg, 0);
		}
	} elsif($mode=~/smooth/i){
		# mode=smooth /Old style PTZ. Goes to direction
		# untill stop (or any other command is sent)/
		if($cmd=~/move/i) { # pan/tilt
			$spd=10 if not $spd;
			my $pan = $param=~/left/i? -$spd : $param=~/right/i? $spd : 0;
			my $tilt = $param=~/down/i? -$spd : $param=~/up/i? $spd : 0;
			hikContCmd($cfg, $pan, $tilt, 0);
		}
		elsif($cmd=~/zoom/i) { # zoom
			$zoomspd=1 if not $zoomspd;
			my $zm = $param=~/out/? -$zoomspd : $zoomspd;
			hikContCmd($cfg, 0, 0, $zm);
		}
		elsif($cmd=~/iris/i) { # iris
			$spd=1 if not $spd;
			my $iris = $param=~/close/i? -$spd : $spd;
			hikIrisCmd($cfg, $iris);
		}
		elsif($cmd=~/focus/i) { # focus
			$spd=1 if not $spd;
			my $focus = $param=~/far/i? -$spd : $spd;
			hikFocusCmd($cfg, $focus);
		}
	} elsif($mode=~/preset/i){ # presets
		hikPresetCmd($cfg, $cmd, $param);
	} elsif($mode=~/settings/i){ # settings
		if ($cmd=~/timeout/i) {
			$log->debug("Setting timer: [$param]");
			$conf{$dev}->{TIMEOUT} = time + $param;
		}
		elsif ($cmd=~/autofocus/i) {
		} elsif ($cmd=~/autoiris/i) {
		}
	}
	$last_mode= $mode;
	$last_cmd = $cmd;
}

sub hikCmd {
	my ($cfg, $url, $data, $rootName, $reqType) = @_;
	$reqType = 'PUT' if not defined $reqType;
	my $xml = "";
	if ($data) {
	    $xml = XMLout($data, RootName => $rootName, NoAttr => '1');
	    $xml =~ s|<$rootName>|<$rootName version="2.0" xmlns="http://www.isapi.org/ver20/XMLSchema">|;
	}
	my $req;
	my $fullURL = "http://$cfg->{DEVIP}/$url";
	for ($reqType) {
		/PUT/ and do { $req = PUT $fullURL, Content => $xml; last };
		/POST/ and do { $req = POST $fullURL, Content => $xml; last };
		/GET/ and do { $req = GET $fullURL; last };
		/DELETE/ and do { $req = DELETE $fullURL; last };
	}
	
	my $rsp = $ua->request($req);
	
	if ($rsp->is_error) {
		my $parsed = eval { XMLin $rsp->content };
		if (not $@ and ref $parsed and defined $parsed->{statusCode}) {
			$log->error(
			"$parsed->{subStatusCode} ($parsed->{statusCode}): $parsed->{statusString}; ".
			"URL=$parsed->{requestURL}");
		} else {
			$log->error("HTTP error: ".$rsp->status_line);
		}
	}
	elsif ($reqType eq 'GET') {	# return parsed response
		my $parsed = eval { XMLin($rsp->content, NoAttr => 1) };
		return undef if $@;
		return $parsed;
	}
}

sub hikGetPTZStatus {
	my $cfg = shift;
	my $status = hikCmd($cfg, "/PTZCtrl/channels/$cfg->{PTZID}/status", undef, undef, "GET");
	if (defined $status) {
		my $elev = $status->{AbsoluteHigh}{elevation};
		my $azim = $status->{AbsoluteHigh}{azimuth};
		my $zoom = $status->{AbsoluteHigh}{absoluteZoom};
		
		return [ $elev, $azim, $zoom ];
	}
	else {
		return undef;
	}
}

sub hikContCmd {
	my ($cfg, $pan, $tilt, $zoom) = @_;
	my $PTZData = {
		pan => $pan,
		tilt => $tilt,
		zoom => $zoom
	};

	hikCmd($cfg, "/PTZCtrl/channels/$cfg->{PTZID}/continuous", $PTZData, "PTZData");
}

sub hikMomentaryCmd {
	my ($cfg, $pan, $tilt, $zoom, $duration) = @_;
	my $PTZData = {
		pan => $pan,
		tilt => $tilt,
		zoom => $zoom,
		Momentary => {
			duration => $duration
		}
	};

	hikCmd($cfg, "/PTZCtrl/channels/$cfg->{PTZID}/momentary", $PTZData, "PTZData");
}

sub hikRelCmd {
	my ($cfg, $x, $y, $zoom) = @_;
	my $PTZData = {
		Relative => {
			positionX => $x,
			positionY => $y,
			relativeZoom => $zoom
		}
	};

	hikCmd($cfg, "/PTZCtrl/channels/$cfg->{PTZID}/relative", $PTZData, "PTZData");
}

sub hikAbsCmd {
	my ($cfg, $elev, $azim, $zoom) = @_;
	my $PTZData = {};
	$PTZData->{AbsoluteHigh}{elevation} = $elev if defined $elev;
	$PTZData->{AbsoluteHigh}{azimuth} = $azim if defined $azim;
	$PTZData->{AbsoluteHigh}{absoluteZoom} = $zoom if defined $zoom;

	hikCmd($cfg, "/PTZCtrl/channels/$cfg->{PTZID}/absolute", $PTZData, "PTZData");
}

sub hikPresetCmd {
	my ($cfg, $cmd, $id, $name) = @_;
	if ($cmd eq 'goto') {
		hikCmd($cfg, "/PTZCtrl/channels/$cfg->{PTZID}/presets/$id/goto");
	}
	elsif ($cmd eq 'save') {
		$name = $id if not defined $name;
		my $PTZPreset = {
			enabled => 'true',
			id => $id,
			presetName => $name
		};
		hikCmd($cfg, "/PTZCtrl/channels/$cfg->{PTZID}/presets/$id", $PTZPreset, "PTZPreset");
	}
	elsif ($cmd eq 'clear') {
		hikCmd($cfg, "/PTZCtrl/channels/$cfg->{PTZID}/presets/$id", undef, undef, "DELETE");
	}
}

sub hikFocusCmd {
	my ($cfg, $focus) = @_;
	my $FocusData = {
		focus => $focus
	};
	
	hikCmd($cfg, "/ISAPI/System/Video/inputs/channels/1/focus", $FocusData, "FocusData");
}

sub hikIrisCmd {
	my ($cfg, $iris) = @_;
	my $IrisData = {
		iris => $iris
	};
	
	hikCmd($cfg, "/ISAPI/System/Video/inputs/channels/1/iris", $IrisData, "IrisData");
}

# --------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {
	%conf = GetCfgs( eval("($query_str)") ); # Load configurations
	my $ids='';
	foreach my $dev (keys %conf) {
		next if not $conf{$dev}->{PTZID} =~/\d+/;
		next if not $conf{$dev}->{DEVIP};
		$ids.=" $dev";
		camParams($dev);
	}
	$log->info("Loaded configurations for cameras:$ids");
}

# ------------------------------------------------------------- camParams -----
sub camParams {

	my ($_dev)=@_;

	if($conf{$_dev}{PTZPARAMS}=~/(.+):(.+):(.+):(.+)/) {
		$conf{$_dev}{pan}=$1;
		$conf{$_dev}{tilt}=$2;
		$conf{$_dev}{zoom}=$3;
		$conf{$_dev}{speed}=$4;
	}
	elsif($conf{$_dev}{PTZPARAMS} eq 'SONY EVI-D30/31') {
		$conf{$_dev}{pan}=1.66;
		$conf{$_dev}{tilt}=0.54;
		$conf{$_dev}{zoom}=625;
		$conf{$_dev}{speed}=20;
	}
	elsif($conf{$_dev}{PTZPARAMS} eq 'SONY EVI-D100/D100P') {
		$conf{$_dev}{pan}=2.78;
		$conf{$_dev}{tilt}=0.69;
		$conf{$_dev}{zoom}=10000;
		$conf{$_dev}{speed}=20;
	}
	else {
		$conf{$_dev}{pan}=1;
		$conf{$_dev}{tilt}=1;
		$conf{$_dev}{zoom}=1000;
		$conf{$_dev}{speed}=100;
	}

	if($conf{$_dev}{PTZSTEPS}=~/(.+):(.+):(.+)/) {
		$conf{$_dev}{span}=$1;
		$conf{$_dev}{stilt}=$2;
		$conf{$_dev}{szoom}=$3;
	}
	else {
		$conf{$_dev}{span}=50;
		$conf{$_dev}{stilt}=50;
		$conf{$_dev}{szoom}=50;
	}
	$conf{$_dev}{p_pos}=0;
	$conf{$_dev}{t_pos}=0;
	$conf{$_dev}{z_pos}=0;
} # sub camParams

# ----------------------------------------------------------- truncDigits -----
sub truncDigits{
	my($val)=@_;
	return sprintf("%5.5f",$val)
} # sub truncDigits

# -------------------------------------------------------------- nonblock -----
# nonblock($socket) puts socket into nonblocking mode
sub nonblock {
	my $socket = shift;
	my $flags;

	$flags = fcntl($socket, F_GETFL, 0)
		or die "Can't get flags for socket: $!\n";
	fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
		or die "Can't make socket nonblocking: $!\n";
}
