#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#  PTZ driver for GEMINEYE protocol - Modified from VISCA
# ------------------------------------------------------------------------------
#  Author: Jeffrey Scott
#  Edited by: Andriy Fomenko
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT Network Solutions, Inc, 2012
# ------------------------------------------------------------------------------
#  VISCA PTZ driver
#  
# Requires STOMP
#

use strict;
use POSIX;
use Socket;
use IO::File;
use IO::Select;
use IO::Socket;
use Fcntl;
#use JSON;
use XML::Simple;
use Net::STOMP::Client;
use Data::Dumper;
#use Math::Trig;

use NextCAM::Init;
use Log::Log4perl "get_logger";

require "$ENV{APL}/common/bin/logger.engine";

# Constants

my $log=get_logger('NEXTCAM::PTZ');

my $query_str = shift || "'POSITIONCTL'=>'gemineye'";
$log->info("Starting GEMINEYE PTZ DRIVER, query string: [$query_str]");

$SIG{HUP}=\&load_dev_conf;

my $APL=$ENV{APL};
my (%conf, %sock, $cam, $cmd, $par, $last_cmd, $last_mode, $lastchecktime);

load_dev_conf();

# ------------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where PTZ server communicates
# ------------------------------------------------------------------------------

my $socket = IO::Socket::INET->new(PeerAddr => '127.0.0.1',
			PeerPort => $TCP_PORT,
			Proto     => "tcp",
			Type      => SOCK_STREAM)
		or $log->logdie("Couldn't connect to socket $TCP_PORT: $@");

my $stomp = Net::STOMP::Client->new( uri=>"stomp://localhost:61613" );
$stomp->message_callback(sub{return(1) } );
eval{
	$stomp->connect();

	my $sid = $stomp->uuid();
	$stomp->subscribe(
		destination	=> "/topic/amq.gis.topic",
		id		=> $sid,
	);
	sub msg_cb($$) {
		my ($self, $frame) = @_;
		return(0) unless $frame->command() eq "MESSAGE";
		return($frame->body());
	}
};

nonblock($socket);

print $socket "PTZ DRIVER\n";

$last_mode='smooth';
$last_cmd='left';

my %commands;
my $trycount = 30;
my $c = 0;
my $lastpostime = time;

my %pos;
my ($deg_p, $deg_t);
my $deg_cal = 0;
my $vID=1;


while(1) {
	# stage 1 - read from socket everything
	if (defined($cmd=<$socket>)) {
		chomp $cmd;
		$log->debug("COMMAND:[$cmd]");
		next if not $cmd;
		load_dev_conf(),next if $cmd=~/HUP/;
		next if not $cmd=~/^(\d+)/;
		$cam=$1;
		next if not defined $conf{$cam}; # ignore cameras not belonging this engine
		if(defined($commands{$cam})){
			if($commands{$cam} =~ /speed pt=0,0/ || $commands{$cam} =~ /speed ptz=0,0,\d+/
				|| $commands{$cam} =~ /speed zoom=0/ || $commands{$cam} =~ /speed ptz=\d+,\d+,0/
				|| $commands{$cam} =~ /speed focus=0/
				|| $commands{$cam} =~ /speed iris=0/
				|| $commands{$cam} =~ /speed gain=0/
				)
			{
				; # do not override stop command!
			}else{
				$commands{$cam} = $cmd;
			}
		} else {
			$commands{$cam} = $cmd;
		}
		next; # suck all the commands from queue before going to the rest of loop
	} # if ($cmd=<$socket>)
	
	$cmd = '';
	
	# STOMP stuff
	eval{
		my $frame = $stomp->wait_for_frames(callback => \&msg_cb, timeout=>0);
		writePos($cam,1,$1) if ($frame=~/gisPtzFbPosGetRequest.*usid="(.*)"/); 
		my $ref = XMLin($frame);
		goPos($cam, $ref) if ($frame=~/gisPtzFbPosSet/);
	};
	# stage 2 - send commands to cameras
	foreach my $camera (keys %commands){
		$cmd=$commands{$camera};
		delete $commands{$camera};
		next if not $cmd=~/^(\d+)\s(\w+)\s(\w+)[=\s]?([^\s]+)?(\s?.*)$/;
		my ($cam,$mode,$cmd,$par,$options) = ($1,$2,$3,$4,$5);
		my %options =  map {/(\w+)=(.*)/} split(/\s/," $options ");
		
		if ($cmd eq "ptz") {
			# handle special ptz command
			my @params = split(',', $par);
			camCmd($cam,$mode,"pt", $params[0].','.$params[1] ,\%options);
			camCmd($cam,$mode,"zoom", $params[2] ,\%options);
		} else {
			# ordinary command
			camCmd($cam,$mode,$cmd,$par,\%options);
		}
	}
	
	select(undef,undef,undef,.1) if not $cmd;

	# keepalive
	#foreach my $dev (keys %conf) {
	#	my %options = ();
	#	if ($lastchecktime != time){
	#		camCmd($dev,'speed','keepalive',1,\%options);
	#		$lastchecktime = time;
	#	}
	#}

	# stage 3 - here we check for timeouts PTZ_PRESET1TIMEOUT
	foreach my $dev (keys %conf) {
		next if not $conf{$dev}->{TIMEOUT};
		next if $conf{$dev}->{TIMEOUT} > time;
		$log->debug("TIMEOUT EXPIRED!");
		$conf{$dev}->{TIMEOUT} = 0;
		my %options = {};
		camCmd($dev,'preset','goto',1,\%options);
	}
	
	if(!$cmd) {
		select(undef,undef,undef,.4);
		if(!$trycount--) {
			$trycount = 30;
			print $socket "test\0" or $log->logdie("Couldn't communicate to socket $TCP_PORT: $@");
		}
	}
	
} # while(1)


# ------------------------------------------------------------ cmdTransmit -----
sub cmdTransmit {
    my ($dev,$delay, @cmd)=@_;
	
    my $command='';
    my $b;
	my $ret;
    foreach $b (@cmd) { 
		$command .= sprintf("%02X ",$b); 
	}
	
	# add ID, 00 is broadcast
	@cmd = (0x00, @cmd);

	my $bts = '';

	# prepend STX
	$bts .= chr(0x02);
	
	# add checksum LRC
	@cmd = (@cmd, checkSum(@cmd));
	foreach my $b (@cmd) {
		if($b == 0x02 || $b == 0x03 || $b == 0x06 || $b == 0x15 || $b == 0x1b){
			# escape things that look like control characters, set 0x80 flag on data
			$bts .= chr(0x1b);
			$b |= 0x80;
		}
		$bts .= chr($b);
	}
	
	# append EXT
	$bts .= chr(0x03);

    $log->debug("cmdTransmit ", $dev, " => ", unpack("H*",$bts));    
    my $retries = 1;
secondaryPass:
    my $skt = $sock{ $conf{$dev}{DEVIP} }{SOCK};
    eval {
        local $SIG{PIPE} = sub { die "Error writing to socket" };
        print $skt $bts;
    };
    if($@) {
        $log->error($@);
        # camera did not answer or socket error occured - try to close socket and reopen it
        eval{ $sock{$conf{$dev}{DEVIP}}{SOCK}->close(); };
        $log->error("Re-open socket connection: $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT}");
        $sock{$conf{$dev}{DEVIP}}{SEL} = undef;
        select(undef,undef,undef,.1);
        $sock{$conf{$dev}{DEVIP}}{SOCK} = IO::Socket::INET->new(PeerAddr => $conf{$dev}{DEVIP},
		PeerPort => $conf{$dev}{PTZ_TCP_PORT},
		Proto    => "tcp",
		Type     => SOCK_STREAM);
        if(! $sock{$conf{$dev}{DEVIP}}{SOCK} ) {
            $log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
            delete $sock{$conf{$dev}{DEVIP}};
            return;
        }
        $sock{$conf{$dev}{DEVIP}}{SOCK}->autoflush(1);
        nonblock($sock{$conf{$dev}{DEVIP}}{SOCK});
        $sock{$conf{$dev}{DEVIP}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}}{SOCK});
        $log->info('Socket was closed, reconnected');
        goto secondaryPass if $retries--;
		
        return;
    }
	
	# Recieve Loop
    select(undef,undef,undef,$delay+0.1);
    foreach my $sss ( $sock{ $conf{$dev}{DEVIP} }{SEL}->can_read(.1) ) {
		my $data = '';
		my $ret = '';
		my $tmp = '';
		my $escd = 0;
		eval {
			alarm 1;
			$sss->recv($data, POSIX::BUFSIZ, 0);	
			for($b=0;$b<length($data);$b++) { 
				$tmp = sprintf("%02X",ord(substr($data,$b,1))); # escape char
				if ($tmp =~/1B/){ # escape char
					$escd = 1; 
					next;
				}elsif ($escd == 1){
					$tmp = sprintf("%02X", hex($tmp) & 0x7f); # unescape
					$escd = 0;
				}
				$b = length($data) if $tmp =~/03/;
				$ret .= $tmp;
			}
		};
		$ret = pack("H*",$ret);
		alarm 0;
		$log->debug('Received:',unpack("H*",$ret));
		return $ret;
	}  # foreach

} # sub cmdTransmit

# --------------------------------------------------------------- CheckSum -----
sub checkSum
{
	my @cmd = @_;
	my $sum = 0;
	foreach my $a (@cmd){
		$sum ^= $a;
	}
	return $sum;
}
#checkSum


# --------------------------------------------------------------- camKeepAlive--
sub camKeepAlive
{
	my ($dev) = @_;
}

# ---------------------------------------------------------------- camStop--
sub camStop
{
	my ($dev) = @_;
}

# --------------------------------------------------------------- camReset -----
sub camReset
{
	my ($dev) = @_;

} # sub camReset

# ----------------------------------------------------------------- PTspeed ----
sub PTspeed {
	my ($speed)=@_;
	my $val = int(abs($speed/100*255));
	$val=255 if ($val > 255);
	return $val;
} # sub PTspeed

# ----------------------------------------------------------------- FZspeed ----
sub FZspeed {
	my ($speed)=@_;
	$speed=100 if abs($speed)>100;
	return 0 if $speed == 0;
	return (abs($speed));
} # sub FZspeed

# ----------------------------------------------------------------- checkReg ----
sub checkReg {
	my ($data, $mask, @cmd) = @_;
	
	my $reg;
	foreach my $a (@cmd) { $reg .= sprintf("%02X ", $a); }
	$data=~/$reg$mask/i;
	#$log->debug("Matched:: $1 :: with '$reg($mask)'");
	return $1;
}

# ----------------------------------------------------------------- goPos ------

sub goPos{
	my ($dev, $ref) = @_;
	my ($range, $az, $el, $obji) = ($ref->{range}, $ref->{az}, $ref->{el}, $ref->{objid});
	my $range_scale = 375.5070; # This should be set by some form of calibration later. Calculated from 30.1752 meters for a zoom level of 0x2c43
	
	#return if not $obji == $dev;
	
	# camera is accurate for tenth of a degree
	my $p = $az;
	my $t = $el;

	# negative angles are 360degrees - angle
	$p += 360 if $p<0;
	$p = pack("V", int($p*100));
	$t = pack("V", int($t*100));
	$log->debug(sprintf("P T (%02X, %02X)", $p, $t));

	my @cmd = (0x33, (pack("C C C", $p)), (pack("C C C", $t)));
	parsePos2(cmdTransmit($dev,0,@cmd));	# Sending Pan / Tilt

	$log->debug("Going to: az:$az el:$el range:$range");

	#scale zoom, will do digital zoom if needed
	my $zoom = int($range * $range_scale);
	$zoom = 0x7000 if $zoom > 0x7000;

	# send both zoom and PT at once.
	cmdTransmit($dev,0,(@cmd));

}

# ----------------------------------------------------------------- writePos -----
sub writePos {
	my ($dev, $target, $USID) = @_;	
	my $mTime = time();
	my $dTime = sprintf("%s_%s",$dev,$mTime);

	$USID = 0 unless defined $USID;
	eval{
	
		my $range = -1;

		$stomp->send(
			destination	=>"/topic/amq.gis.topic",
			type		=>"gisPtzFbPosGetResponse",
			time		=>time(), 
			id			=>"$dTime",
			usid		=>"$USID",
			az			=>"$pos{pan}",
		#	fov 		=>"$posUpdate{FOV}",
			el			=>"$pos{tilt}",
			zoom		=>"$pos{z1}",
			focus		=>"$pos{f1}",
			body		=>".",
			range		=>"$range",
		);
		

	};
}

sub getFov {
	return 0;
}

sub parsePos2 {
	my ($rsp) = @_;


	#$rsp=~s/\s*//g;
	# sample response:  06 00 31 BA BC FF EC FD FF 00 00 80 01 8F 00 00 96 03
	#  06 - ACK
	#  00 - Cam ID
	#  BA BC FF - Pan Coord 24bit integer, 2's complement
	#  EC FD FF - Tilt Coord 24bin integer, 2's complement
	#  00 - Pan Status bitfield: 0(CW Soft Limit), 0(CCW Soft Limit), 0, 0, 0(Timeouts), 0(Directional Error), 0, 0
	#  00 - Tilt Status bitfield: 0(Up Soft Limit), 0(Down Soft Limit), 0, 0, 0(Timeouts), 0(Directional error), 0, 0
	#  80 - Gen Status bitfield: 1(Continuous rotation), 0(Exec), 0(Destination), 0(OSLR), 0(CW Motor), 0(CCW Motor), 0(UP Motor), 0(DWN Motor)
	#  01 - Zoom 1 Pos
	#  8F - Focus 1 Pos
	#  00 - Zoom 2 Pos
	#  00 - Focus 2 Pos
	#  96 - LRC
	#  03 - End of TeXt ETX

   eval{
	my ($pan, $tilt, $flags, $z1, $f1, $z2, $f2) = unpack("x3a3a3C3CCCC", $rsp);
	$pan = unpack("V", $pan."\x00");
	$tilt = unpack("V", $tilt."\x00");

	#$pan = ($pan | ($pan & 0x00800000) << 8) & 0xFF7FFFFF;
	
	if (defined $tilt && defined $pan){

		# convert 24bit unsigned int to signed int by 2's complement
		if ($tilt & 0x00800000) {
			$tilt &= 0x007FFFFF;
			$tilt ^= 0x007FFFFF;
			$tilt *= -1;
		}
		if ($pan & 0x00800000) {
			$pan  &= 0x007FFFFF;
			$pan  ^= 0x007FFFFF;
			$pan  *= -1;
		}
		$pan  /= 100;
		$tilt /= 100;
	}
	my %coords = (
		pan => $pan,
		tilt => $tilt,
		z1 => $z1,
		z2 => $z2,
		f1 => $f1,
		f2 => $f2,
	);
	%pos = %coords;
	$log->debug("Unpacked pan: ", $pan);
   };
}
	

sub parsePos {
	my ($rsp) = @_;

	# sample response:  06 00 31 BA BC FF EC FD FF 00 00 80 01 8F FF 77 60 00 00 2F 49 01 00 00 7F 7F 00 00 00 11 21 01 00 00 96 03
	#  06 - ACK
	#  00 - Cam ID
	#  BA BC FF - Pan Coord 24bit integer, 2's complement
	#  EC FD FF - Tilt Coord 24bin integer, 2's complement
	#  00 - Pan Status bitfield: 0(CW Soft Limit), 0(CCW Soft Limit), 0, 0, 0(Timeouts), 0(Directional Error), 0, 0
	#  00 - Tilt Status bitfield: 0(Up Soft Limit), 0(Down Soft Limit), 0, 0, 0(Timeouts), 0(Directional error), 0, 0
	#  80 - Gen Status bitfield: 1(Continuous rotation), 0(Exec), 0(Destination), 0(OSLR), 0(CW Motor), 0(CCW Motor), 0(UP Motor), 0(DWN Motor)
	#  01 - Zoom 1 Pos
	#  8F - Focus 1 Pos
	#  FF - Iris 1 level
	#  77 - Gain 1 level
	#  60 - Shutter 1 level
	#  00 - White Balance 1 level
	#  00 - Digital Zoom 1 level
	#  2F - Cam 1 Stat 1 bitfield: Zoom MV, Focus MV, AutoFocus, AutoGain, AutoIris, Auto WhiteBalance, AIR, PWR
	#  49 - Cam 1 Stat 2 bitfield: PF, STA, POL, IR, DZ, PAL, 0, VS
	#  01 - Cam 1 Stat 3 bitfield: 0, EC, HR, WD, LF, LA, HT, CI
	#  00 - Zoom 2 Pos
	#  00 - Focus 2 Pos
	#  7F - Iris 2 level
	#  7F - Gain 2 level
	#  00 - Shutter 2 level
	#  00 - White Balance 2 level
	#  00 - Digital Zoom 2 level
	#  11 - Cam 2 Stat 1 bitfield: Zoom MV, Focus MV, AutoFocus, AutoGain, AutoIris, Auto WhiteBalance, AIR, PWR
	#  21 - Cam 2 Stat 2 bitfield: PF, STA, POL, IR, DZ, PAL, 0, VS
	#  01 - Cam 2 Stat 3 bitfield: 0, EC, HR, WD, LF, LA, HT, CI
	#  00 - Cam 1 Count (up to 80 bytes extra data from cam 1)
	#  00 - Cam 2 Count (up to 80 bytes extra data from cam 2)
	#  96 - LRC
	#  03 - End of TeXt ETX

	eval{
		my ($pan, $tilt, $flags, $z1, $f1, $dz1, $z2, $f2, $dz2) = unpack("x3a3a3C3CCx4Cx3CCx4C", $rsp);
		$pan = unpack("V", $pan."\x00");
		$tilt = unpack("V", $tilt."\x00");
		if (defined $tilt && defined $pan){

			# convert 24bit unsigned int to signed int by 2's complement
			if ($tilt & 0x00800000) {
				$tilt &= 0x007FFFFF;
				$tilt ^= 0x007FFFFF;
				$tilt *= -1;
			}
			if ($pan & 0x00800000) {
				$pan  &= 0x007FFFFF;
				$pan  ^= 0x007FFFFF;
				$pan  *= -1;
			}
			$pan  /= 100;
			$tilt /= 100;
		};

		my %coords = (
			pan => $pan,
			tilt => $tilt,
			z1 => $z1,
			z2 => $z2,
			f1 => $f1,
			f2 => $f2,
		);
		%pos = %coords;
		$log->debug("Unpacked pan: ", $pan);
	};

}

# ----------------------------------------------------------------- camCmd -----

sub camCmd {
	my ($dev,$mode,$cmd,$param,$options)=@_;
	my %options = %$options;
	my @cmd;
	$log->debug("camCmd: DEVID=$dev mode: $mode command: $cmd optional parameter: $param");
	
	# here is logic to start a timer if PTZ_PRESET1TIMEOUT parameter present
	if($conf{$dev}->{PTZ_PRESET1TIMEOUT} && $conf{$dev}->{PTZ_PRESET1TIMEOUT} > 0) {
		if($mode=~/speed/ || $mode=~/step/ || $mode=~/abs/ || $mode=~/rel/ || $mode=~/smooth/) {
			$log->debug("Setting timeout +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
		elsif($mode=~/preset/ && $cmd=~/goto/ && $param!=1) {
			$log->debug("Setting timeout (preset) +$conf{$dev}->{PTZ_PRESET1TIMEOUT}");
			$conf{$dev}->{TIMEOUT} = time + $conf{$dev}->{PTZ_PRESET1TIMEOUT};
		}
	}


	# Basic command structure:
	# 1: 0x02 - STX (Start-of-text)
	# 2: 0x?? - Identity (0x00 is broadcast)
	# 3: 0x?? - Command
	# 4 to N: 0x00 - Data
	# N+1: LRC - Checksum, XOR of bytes 2 to N
	# N+2: 0x03 - ETX (End-of-text)
	if ( not $cmd=~/stop/){
		# set camera speed
		if ($mode=~/speed/){        # mode speed
			@cmd = ( 0x31, 0x00, 0,0,0,0,0,0 );
			if($cmd=~/pt/i) { # RPT
				my ($p,$t) = split(/,/,$param);
				$cmd[2]=PTspeed($p);
				$cmd[3]=PTspeed($t);
				$log->debug(sprintf("RPT $p(%02X) $t(%02X)",$cmd[4],$cmd[5]));
				$cmd[1] |= 0x80 if $p>0;
				$cmd[1] |= 0x40 if $t<0;
			}
			elsif($cmd=~/z/i) { # Zoom 
				$log->debug("ZOOM $param");
				$cmd[4] |= FZspeed($param);
				$cmd[4] = $cmd[4] << 1;
				$cmd[4] |= 0x01 if $param < 0;
				$cmd[6] = $cmd[4]
			}
			elsif($cmd=~/focus/) {
				$log->debug("Focus $param");
				$cmd[5] |= FZspeed($param);
				$cmd[5] = $cmd[5] << 1;
				$cmd[5] |= 0x01 if $param < 0;
				$cmd[7] = $cmd[5];
			}
			parsePos(cmdTransmit($dev,0,@cmd));
			return;
		}
		elsif($mode=~/rel/){ # mode=rel: "<devid> rel size=640x480 xy=225,152"
			$param =~ /(\d+)x(\d+)/;
			my ($rel_size_x,$rel_size_y) = ($1,$2);
			$options{xy} =~ /(\d+),(\d+)/;
			my ($rel_click_x,$rel_click_y) = ($1,$2);
			@cmd = ( 0x34, (0,0,0), (0,0,0));
			eval{
				
				my $fov = $conf{$dev}->{CAM_GEO_HFOV};
				# Calculate % offset from screen center, adjusted for current FoV
				my ($rel_p, $rel_t) = ( $fov*($rel_click_x-$rel_size_x/2)/$rel_size_x, $fov*($rel_click_y-$rel_size_y/2)/$rel_size_y ); 
				$log->debug("REL [$rel_size_x] [$rel_size_y] [$rel_click_x] [$rel_click_y] [$rel_p] [$rel_t]");
				$cmd[4]=PTspeed($rel_p); # % difference from the middle of the screen, +-50%
				$cmd[5]=PTspeed($rel_t);
				
				# Convert to whole (hex) numbers
				$rel_p = pack("V", int($rel_p *  100));
				$rel_t = pack("V", int($rel_t * -100)); 
					
				parsePos2(@cmd = ( 0x34, (unpack("C C C", $rel_p)), (unpack("C C C", $rel_t))));
				
			};
			cmdTransmit($dev,0,@cmd);
			return;
		}
		elsif($mode=~/abs/){    # mode=ABS
			if ($cmd=~/pt/) {
				$log->debug("ABS:  PT $param");
				
				$param=~/^(\-?.+)?\,(\-?.+)$/;
				# camera is accurate for tenth of a degree
				my $p = $1;
				my $t = $2;
				# negative angles are 360degrees - angle
				$p = pack("V", int($p*100));
				$t = pack("V", int($t*100));

				# pan
				@cmd = (0x33, (unpack("C C C", $p)), (unpack("C C C", $t)));
				
				# tilt
				parsePos2(cmdTransmit($dev,0,@cmd));	# Sending Pan / Tilt
				return;
			} 
			elsif ($cmd=~/z/){	# Absolute zoom
				$log->debug("ABS:  Z $param");
				my $zoom = int($param * 2.55); #assuming ABS zoom is 0-100
				my $p = pack("V", 99999); # no move
				my $t = pack("V", 99999);

				@cmd = ( 0x3a, unpack("C C C", $p), unpack("C C C", $t), pack("V", $zoom), 0, pack("V", $zoom),0);

				cmdTransmit($dev,0,@cmd);	# Sending Pan / Tilt
				
			}
			
		}
		elsif($mode=~/settings/){ # settings
			if($cmd=~/autofocus/) { 
				@cmd = ( 0x63, 0x00 ) if $param=~/on/;
				@cmd = ( 0x63, 0x06 ) if $param=~/off/;
				cmdTransmit($dev,0,@cmd);
				return;
			}
			elsif($cmd=~/gain/){
				@cmd = (0x98, 0x03) if $param=~/reset/;
				cmdTransmit($dev,0,@cmd);
				return;
			}
			
			elsif($cmd=~/mh/){
				@cmd = ( 0x35 );
				cmdTransmit($dev, 0, @cmd);
			}
			elsif($cmd=~/reset/){
				@cmd = (0xff, 0, 0x32, 0x77, 0x01, 0x11);
			}
			elsif($cmd=~/home/){
				@cmd = (0x80+$vID, 0x01, 0x06, 0x04, 0x02, 0xff) if $param=~/set/;
				@cmd = (0x80+$vID,0x01,0x06,0x04,0x03,0xff) if $param=~/goto/;
				cmdTransmit($dev,0, @cmd);
				return;
			}
		}
		elsif($mode=~/step/){       # mode=step  /Step by step positioning/
			my $timeout = 0.2;
			@cmd = ( 0x31, 0, 0, 0, 0, 0, 0, 0,);
			if($cmd=~/move/) { # step pan/tilt
				$cmd[1] |= 0x80 if $param=~/right/;
				$cmd[1] |= 0x40 if $param=~/up/;
				$cmd[2] = 0x0F if( $param=~/right/ || $param=~/left/);
				$cmd[3] = 0x0F if ($param=~/up/ || $param=~/down/);
			} elsif ($cmd=~/zoom/){
				$cmd[4] = 64;
				$cmd[4] = $cmd[4] << 1;
				$cmd[4] |= 0x01 if $param=~/out/;
				$cmd[5] = $cmd[4];
			} elsif ($cmd=~/focus/){    
				$cmd[6] = 64;
				$cmd[6] = $cmd[6] << 1;
				$cmd[6] |= 0x01 if $param=~/near/;
				$cmd[7] = $cmd[6];
			}
			cmdTransmit($dev, 0, @cmd);
			select(undef,undef,undef,$timeout);
			parsePos(cmdTransmit($dev, 0, (0x31, 0, 0, 0, 0, 0, 0, 0)));
		}
		elsif($mode=~/hardware/){   # mode=hardware  /Hardware reset,defaults/
			if($cmd=~/do/ && $param=~/defaults/) { # init
				cmdTransmit($dev,0,(0x80+$vID, 0x01, 0x06, 0x05, 0xff));
				return;
			}
		} elsif($mode=~/preset/){ # presets
			$param = $param - 1;
			@cmd = (0x42, $param) if $cmd =~/save/; 
			@cmd = (0x32, $param) if $cmd =~/goto/;
			$cmd[1]= 31 if $param > 32;
			
			parsePos2(cmdTransmit($dev,0,@cmd));
		}
	}
}

# --------------------------------------------------------------- nonblock -----
#   nonblock($socket) puts socket into nonblocking mode
sub nonblock {
	my $socket = shift;
	my $flags;
	
	$flags = fcntl($socket, F_GETFL, 0)
	or die "Can't get flags for socket: $!\n";
	fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
	or die "Can't make socket nonblocking: $!\n";
}

# ---------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {
	
	$log->info('Configurations refreshed: ',$query_str);
	
	# fisrst, close everything
	foreach my $skt (keys %sock) {
		close($sock{$skt}{SOCK});
		delete $sock{$skt};
	}
	
	%conf = GetCfgs( eval("($query_str)") );     # Load configurations
	foreach my $dev (keys %conf) {
		# in a case when PTZ works on another server (e.g. 3ETI COMTROL Serial Server)
		$conf{$dev}{DEVIP} = $conf{$dev}{PTZ_TCP_IP} if $conf{$dev}{PTZ_TCP_IP};
		
		$log->debug("[$dev]");
		$log->debug("DEVID=$conf{$dev}{DEVID}");
		$log->debug("PTZID=$conf{$dev}{PTZID}");
		$log->debug("PTGROUP=$conf{$dev}{PTZGROUP}");
		$log->debug("PTZSPEED=$conf{$dev}{PTZSPEED}");
		$log->debug("DEVIP=$conf{$dev}{DEVIP}");
		$log->debug("PTZ_TRANSPORT_PROTO=$conf{$dev}{PTZ_TRANSPORT_PROTO}");
		$log->debug("PTZ_TCP_PORT=$conf{$dev}{PTZ_TCP_PORT}");
		
		$log->info("Open socket connection: $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT}");
		if(not defined($sock{$conf{$dev}{DEVIP}})) {
			$sock{$conf{$dev}{DEVIP}}{SOCK} = IO::Socket::INET->new(PeerAddr => $conf{$dev}{DEVIP},
			PeerPort => $conf{$dev}{PTZ_TCP_PORT},
			Proto    => "tcp",
			Type     => SOCK_STREAM);
			if(! $sock{$conf{$dev}{DEVIP}}{SOCK} ) {
				$log->error("Couldn't connect to $conf{$dev}{DEVIP}:$conf{$dev}{PTZ_TCP_PORT} : $@\n");
				delete $sock{$conf{$dev}{DEVIP}};
				next;
			}
			
			$sock{$conf{$dev}{DEVIP}}{SOCK}->autoflush(1);
			nonblock($sock{$conf{$dev}{DEVIP}}{SOCK});
			$sock{$conf{$dev}{DEVIP}}{SEL} = IO::Select->new($sock{$conf{$dev}{DEVIP}}{SOCK});
		}
	} # foreach $dev
} # sub load_dev_conf

