#!/usr/bin/perl
#  $Id: PTZ_server.pl 33593 2016-02-15 09:40:06Z atsybulnik $
# -----------------------------------------------------------------------------
#  Socket server for PTZ (socket 7766 from the clients / socket 7768 to drivers)
# -----------------------------------------------------------------------------
#  Author: Andrey Fomenko
#  Edited by: Serg Pososhenko
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT LLC, 2005
# -----------------------------------------------------------------------------
# AF: Changes relevant to Release 2.4: "MBus" is introduced to carry mesages
# to PTZ_server.pl, but internally it uses same interface as before to drivers.
# ------------------------------------------------------------------------------

use strict;
use POSIX;
use IO::Socket;
use IO::Select;
use Socket;
use Fcntl qw(:DEFAULT :flock);
use Tie::RefHash;
use CGI qw/escape unescape/;
use XML::Simple;
use LWP::UserAgent;
use Data::Dumper;
use Time::Local;
use JSON;
use NextCAM::Conf "GetCfgs";
use SKM::Common;
use SKM::DB;

#require "$ENV{APL}/ptz/bin/send2pipe.pl";

system("kill -9 `ps ax|grep bin/ptz_|grep -v grep|grep -v ptz_patrol|cut -c 1-6` 2>/dev/null");

# -----------------------------------------------------------------------------
my $TCP_PORT = 7766; # TCP port where server communicates
# -----------------------------------------------------------------------------

use Log::Log4perl "get_logger";
my $log;

my $APL = $ENV{APL};
my $APL_VAR = $ENV{APL_VAR};
my $APL_CONF = $ENV{APL_CONF};

my $UNI;

my %Cfgs;
my %asrv;
my $Low_P_timeout = 10;
my %Last_Command_time = ();

my $MBUS_CONN_TRIES = 30;
# Consider lock expired if its mtime wasn't updated for 1 min
my $LOCK_AUTOEXPIRE_TIMEOUT = 60; # sec
my $LOCKDIR = "$APL_VAR/ptz/locks";
my $SESSDIR = $^O=~/darwin/i ? '/tmp' : '/var/lib/php/session';
my $LastLockCheck = 0;

# Check if we are running on Avatar
my $IS_AVATAR = $0 =~ m{/opt/avatar};

my $ANALYTICS_PTZ_RESET_TIMEOUT = 10; # Wait after goto preset1 cmd and then unlock analytics for camera

# PTZ_PROPAGATE2ELOG impl
my $PTZ_PROPAGATE2ELOG = 'info';
#my $USER_LOCK_TIMEOUT = 10;
my $EVENTSOURCE_USER = 4;
my $EVENTTYPE_INFO = 0;
my $EVENTTYPE_ALERT = 1;
my %UserLocks;

# Close filehandles
#close(STDIN); close(STDOUT); close(STDERR);

$/ = "\0"; # Set the input terminator to a zero byte string, pursuant to the protocol in the Flash documentation.

my %inbuffer  = ();
my %outbuffer = ();
my %ready     = ();
my %clients   = ();
my %lastwill  = ();
my $t_hb=timegm(gmtime);
my $master=checkMaster();
my $current_time = time();
my $t_asrv = time;

my %devices   = ();

my %ptz_tours = ();

my %preset1timeouts = ();

my %anl_locks = ();

my $dbm;
my %dbs;

# Audit
my $AUDIT_PTZ_CAT = 9;

my $messenger;
my $server;
my $select;
my $ELClient;

my $MBusReconnect = 0;

# SIG
$SIG{HUP}=\&reload_dev_conf;

###############################################################
# PTZ Tour, executes once a second,
# send commands to change position for cameras in ptz tour
#
# Info about currently running tours is stored in %ptz_tours
# Structure of %ptz_tours:
#
# ptz_tours{ camera_id }{'tour'}       - array of steps of ptz tour, each step contains camera preset number and number of seconds to stay on that preset
# ptz_tours{ camera_id }{'curr'}       - current step in ptz_tour; after reaching last step we return to step number 0.
# ptz_tours{ camera_id }{'time'}       - estimated time to stay on current step
# This variable must be shared!
#
###############################################################
sub ptz_tour {

	foreach my $cam_id (keys %ptz_tours) {

		$ptz_tours{$cam_id}{'time'}--;

		# Logic updated!
		# PTZ tour should check if any other user(including GUI video tours) has obtained
		# lock on target device. If lock exists, server tour should wait untill it is released
		if ($ptz_tours{$cam_id}{'time'} <= 0 and not $UserLocks{$cam_id}) {
			#if time is up, go to next step of tour
			my $curr = $ptz_tours{$cam_id}{'curr'};

			if ($curr == $#{$ptz_tours{$cam_id}{'tour'}}) {
				$curr = 0;
			} else {
				$curr++;
			}

			my $msg;
			$msg->{device_ID} = $cam_id;
			$msg->{data} = "<PTZ_Command>do.ptz?dev=$cam_id&mode=preset&goto=".$ptz_tours{$cam_id}{'tour'}[$curr]{'preset'}."</PTZ_Command>";
			$msg->{ptz_tour} = 1; # set this flag to indicate that this command came from ptz tour, not user, so we dont need to set "penalty" time
			#$log->warn($msg->{data});
			on_message($msg);

			#$log->warn("go on preset ".$ptz_tours{$cam_id}{'tour'}[$curr]{'preset'}." on time ".$ptz_tours{$cam_id}{'tour'}[$curr]{'time'}."\n");
			$ptz_tours{$cam_id}{'curr'} = $curr;
			$ptz_tours{$cam_id}{'time'} = $ptz_tours{$cam_id}{'tour'}[$curr]{'time'};

		}
	}

}


########################################
# Add ptz tour for a device;
# For starting tour we send command dev=<camera id>&mode=setptztuor&override=5&tour=<ptz tour description>
#
# Override - number of seconds to stop ptz tour when user take control of camera
#
# Structure of ptz tour description: pipe-separated pairs of numbers, which contains preset number and time to stay on this preset (in seconds).
# '1:3|5:4|3:2' means stay 3 seconds on 1 preset, then stay 4 seconds on preset 5, then stay 2 seconds on preset 3
# stopptztour command stops ptz tour for device
########################################
sub ptz_tour_manage {

	my $msg = shift;
	my $serialize = shift;

	$serialize = 1 unless defined $serialize;

	# stop ptz tour
	if ($msg->{data} =~ /mode=stopptztour/) {
		delete $ptz_tours{$msg->{device_ID}};
		$log->debug($msg->{device_ID}.': Stopping PTZ tour');

		# serialize tour data to DB
		ptz_tour_serialize($msg->{device_ID}, '') if $serialize;

		return;
	}


	if ($msg->{data} =~ /mode=setptztour/) {

		$msg->{data} =~ /tour=(.*)(&|<)/;
		my $tour_data = $1;

		$log->debug($msg->{device_ID}.': Starting PTZ tour: '.$tour_data);

		# Clear previous ptz tour
		delete $ptz_tours{$msg->{device_ID}};

		# We must share contents of %ptz_tours
		my %h;
		my @h;

		$ptz_tours{$msg->{device_ID}} = \%h;
		$ptz_tours{$msg->{device_ID}}{'tour'} = \@h;

		# Adding presets into tour
		my $i = 0;
		foreach my $pair (split(/\|/, $tour_data)) {

			my ($preset, $sec) = split(/:/, $pair);

			next if ($preset >= 100 || $preset <=0 || $sec <= 0);

			my %h;
			$ptz_tours{$msg->{device_ID}}{'tour'}[$i] = \%h;
			$ptz_tours{$msg->{device_ID}}{'tour'}[$i]{'preset'} = $preset;
			$ptz_tours{$msg->{device_ID}}{'tour'}[$i]{'time'} = $sec;
			$i++;
		}

		my $tour_size = $#{$ptz_tours{$msg->{device_ID}}{'tour'}};

		# If presets were not added, clear tour and exit
		if ($tour_size == -1) {
			delete $ptz_tours{$msg->{device_ID}};
			$log->warn($msg->{device_ID}.": Tour is incorrect, aborting");
			return;
		}

		# set curr to last element and time to zero,
		# so first call of ptz_tour() will position camera to first preset in list
		$ptz_tours{$msg->{device_ID}}{'curr'} = $tour_size;
		$ptz_tours{$msg->{device_ID}}{'time'} = 0;

		# serialize tour data to DB
		ptz_tour_serialize($msg->{device_ID}, $tour_data) if $serialize;

		return;
	}

}

sub ptz_tour_serialize {
    my ($dev, $tour_data) = @_;

    my $dbh = DBMaster({RaiseError=>0,PrintError=>0});
    if ($dbh) {
       my $rows = $dbh->do("update _obj_attr set val='$tour_data' where obj=$dev and attr='STAT_PTZ_TOUR'");
       if ($rows < 1) {
           $dbh->do("insert into _obj_attr (obj,attr,val) values ($dev,'STAT_PTZ_TOUR','$tour_data')");
       }
       $dbh->disconnect;
    }
}

# ----------------- handler for messages from MBus --------------------
sub on_message {
	my $msg = shift;
	my $devid;
	my $priority = 1;
	$log->debug('new message from MBus:[',$msg->{data},']',$msg->{device_ID});
	if ( $msg->{data} =~ /\?dev=(\d+)/i ) {
		$devid = $1;
		$priority = lc($1) if $msg->{data} =~ /priority=(\w+)/i;
		$priority = pri($priority) if $priority!~/^\d+$/;
		if ($priority eq 'low') {
			return if time - $Last_Command_time{$devid} < $Low_P_timeout;
			$msg->{data} =~ s/\&priority=low//i;
			$msg->{data} =~ s/priority=low\&//i;
		}
		else {
			$Last_Command_time{$devid} = time ;
		}
	}

	# if we have command for ptz touring (start / stop)
	if (-d "$APL_CONF/$devid" && $msg->{data} =~ /mode=(setptztour|stopptztour)/) {
		ptz_tour_manage($msg);
		return;
	}

	# Check for User locks for device
	# If device is locked, must compare call priority with lock priority
	# If current user call has priority higher than the lock has, then
	# the lock should pass to new user. If priorities are equal
	# old lock should be preserved
	my ($sid, $userid, $username, $action);
	if ($msg->{att_data}) {
	    foreach my $d (@{$msg->{att_data}}) {
		$sid      = $1 if $d=~/^sid=(.+)$/i;
		$userid   = $1 if $d=~/^userid=(.*)$/i;
		$username = $1 if $d=~/^username=(.*)$/i;
		$action   = $1 if $d=~/^action=(.*)$/i;
	    }
	}

	if (-d "$APL_CONF/$devid" && $userid && $userid > 0) { # Locks are created on the node that owns camera
	    $username = fetch_user_name($userid) unless defined $username;
	    my $curLock = $UserLocks{$devid};

	    my %h = ();
	    my $newLock = \%h;
	    $newLock->{devid}    = $devid;
	    $newLock->{objid}    = dev2obj($devid);
	    $newLock->{obtained} = time;
	    $newLock->{updated}  = time;
	    $newLock->{sid}      = $sid;
	    $newLock->{userid}   = $userid;
	    $newLock->{username} = $username;
	    $newLock->{priority} = $priority;
	    $newLock->{submited} = 0;
	    $newLock->{numtries} = 0;
	    # If no lock for this camera, obtain the lock
	    # Do the same if new priority is higher then the old one

	    if ($curLock) {
		if ($action eq 'release') { # release lock
	    	    $log->debug("User '$username' released lock for device $devid");
		    $curLock->{released} = time;
		    submit_lock_event($curLock, 0);
		}
		elsif ($action eq "override") { # PTZ override
		    $log->debug("User '$username' obtained lock for device $devid with PTZ override");
		    $curLock->{overriden} = $newLock->{username};
		    $curLock->{released} = 1;
		    submit_lock_event($curLock, 0, 1);
		    $UserLocks{$devid} = $newLock;
		}
		elsif ($action=~/^lock|override/ && $newLock->{sid} == $curLock->{sid}) {
		    # prolongate lock
		    $log->debug("Lock on device $devid updated by user '$username'");
		    $curLock->{updated} = $newLock->{updated};
		}
	    }
	    elsif ($action =~ /^lock|override$/) {
		$log->debug("User '$username' obtained lock for device $devid");
		$UserLocks{$devid} = $newLock;
	    }

	    # If PTZ command is a 'lock' command, do not broadcast it to engines
	    return if $msg->{data} =~ /\Wmode=lock/;
	}

	Broadcast_MSG($msg->{device_ID}, $msg->{data});
}

sub on_disconnect {
	$MBusReconnect = 1;
}

sub fetch_user_name {
    my $userid = shift;
    my $username = '[unknown]';

    db_master() unless $dbm;

    if ($dbm) {
	my $ra = eval { $dbm->selectrow_arrayref(
	    "SELECT name FROM _objs WHERE obj=? AND otype='U' AND deleted=0",
	    undef,
	    $userid
	    ) };
	$username = $ra->[0] if $ra and @$ra;
	db_close() if $@;
    }
    return $username;
}

sub submit_lock_event {
    my ($lock, $obtained, $overriden) = @_;

    if ($PTZ_PROPAGATE2ELOG eq 'disabled') {
	$lock->{submited} = 1;
	return;
    }

    $lock->{submited} = 0;
    my $action = 'create';
    my $msg = $overriden ?
	"User '$lock->{overriden}' took PTZ controls via PTZ override" :
	"PTZ control by '$lock->{username}'";
    my $params = {
	objid => $lock->{objid},
	msg   => $msg,
	when  => $obtained ? $lock->{obtained} : time,
    };
    unless ($obtained) {
	$params->{to} = $lock->{updated} + 15;
	$params->{eventid} = $lock->{eventid};
	$action = 'update';
    }
    $params->{source} = $EVENTSOURCE_USER;
    $params->{eventtype} = $PTZ_PROPAGATE2ELOG eq 'alert' ? $EVENTTYPE_ALERT : $EVENTTYPE_INFO;

    my $ok = 0;
    my $rsp = $action eq 'create' ? $ELClient->createEvent($params) : $ELClient->updateEvent($params);

    if (!$rsp) {
	$log->error("Create event failed: ".$ELClient->{error});
    }
    else {
	$ok = 1;
	$lock->{eventid} = $rsp if $action eq 'create';
    }

    $lock->{submited} = time if $ok;
    $lock->{numtries}++;
}

sub ptz_lock_obtain_release {
    my $time = time;
    foreach my $lock (values %UserLocks) {
	# Remove events that cannot be submitted
	if ($lock->{numtries} >= 10) {
	    delete $UserLocks{$lock->{devid}};
	    next;
	}

	# Mark lock as released if timeout expired
	if (! $lock->{released} && $time - $lock->{updated} > $Low_P_timeout) {
	    $log->debug("Timeout for user lock (dev=$lock->{devid}, user=$lock->{username},".
			" prio=$lock->{priority}) has expired"
	    );
	    $lock->{released} = $time;
	    submit_lock_event($lock, 0);
	    audit(
		$lock->{userid},
		$lock->{objid},
		$AUDIT_PTZ_CAT,
		"Lock expired"
	    );
	    next;
	}

	# Delete expired locks
	if ($lock->{released} and $lock->{submited}) {
	    delete $UserLocks{$lock->{devid}};
	}
	elsif (! $lock->{submited}) {
	    submit_lock_event($lock, $lock->{released} ? 0 : 1);
	}
    }

    # Check lock files and release expired ones
    if ($master && time-$LastLockCheck>=5) {
	my %locks;
	if (opendir DH, $LOCKDIR) {
	    my @files = grep {/^\d+$/} readdir DH;
	    closedir DH;
	    foreach my $objid (@files) {
		my $lck = "$LOCKDIR/$objid";
		if (open F, $lck) {
		    local $/ = "\n";
		    my %props = map {/^(\w+)=(.+)$/} grep {/^\w+=.+/} <F>;
		    close F;

		    $locks{$objid}         = \%props;
		    $locks{$objid}{_objid} = $objid;
		    $locks{$objid}{_path}  = $lck;
		    $locks{$objid}{_mtime} = (stat $lck)[9];

		    if (
		      time - $locks{$objid}{_mtime} > $LOCK_AUTOEXPIRE_TIMEOUT ||
		      ! -f "$SESSDIR/sess_$locks{$objid}{SID}")
		    {
			$log->info("Remove expired lock for OBJID=$objid. mtime=$locks{$objid}{_mtime}");
			unlink $lck;
			# Release PTZServer lock if exists
			if (defined $UserLocks{$objid}) {
			    my $ulock = $UserLocks{$objid};
			    $ulock->{released} = time;
			    submit_lock_event($ulock, 0);
			    audit(
				$ulock->{userid},
				$ulock->{objid},
				$AUDIT_PTZ_CAT,
				"Lock expired"
			    );

			}
		    }
		}
	    }
	}

	$LastLockCheck = time;
    }
}

sub pri {
    for ($_[0]) {
	/low/    and return 1;
	/normal/ and return 5;
	/high/   and return 9;
    }
    return 1;
}

sub dev2obj {
    $1 if shift=~/^\D*(\d+)$/;
}
# ==============================================================================



# ---------------------------------------------------------------- handle -----
#   handle($socket) deals with all pending requests for $client
sub handle {
    # requests are in $ready{$client}
    # send output to $outbuffer{$client}
    my $client = shift;
    my $request;
    $log->debug("$client - now handeling");

    foreach $request (@{$ready{$client}})
	{
		# if flash client request permission (policy file)
		if ($request =~ /\<policy-file-request\/\>/)
		{
			# allow acces flash client to socket
			$outbuffer{$client} = "<?xml version=\"1.0\"?><cross-domain-policy><allow-access-from domain=\"*\" to-ports=\"7766,80\" /></cross-domain-policy>\0";
			next;
		}

        # $request is the text of the request
        # put text of reply into $outbuffer{$client}
        $log->debug("[$request]"); # [<PTZ_Command>do%2Eptz%3Fdev%3D3%26mode%3Dspeed%26pt%3D%2D7%2C%2D11%26undefined</PTZ_Command>]
        $request =~ /(.+)[\n,\0,\r]?/m;
        $request = $1;
        if(not defined($clients{$client})) { # first line is a name
            $log->debug("[$request] it is not in \$clients");
            $request =~ /^PTZ\s+(\w+)/;
            if(($1 eq 'CLIENT') or ($1 eq 'DRIVER') or ($1 eq 'MONITOR')) {
                $clients{$client} = $1;
                #$outbuffer{$client} = 'Accepted: '.$clients{$client}."\n";
                $log->debug("Accepted: $clients{$client}");
				if ($request =~ /\s+(LASTWILL\s+)/)
                {
                  $log->debug("$1$'");
                  $lastwill{$client}=$';
                }
            }
            else {
                $log->debug("Wrong registration");
                $outbuffer{$client} .= "Wrong registration\n\0";
            }
        }
        else {
            $log->debug('INPUT FROM:',$clients{$client},' DATA=',$request); # INPUT FROM:CLIENT DATA=<PTZ_Command>do%2Eptz%3Fdev%3D3%26mode%3Dspeed%26pt%3D%2D7%2C%2D11%26undefined</PTZ_Command>
            if(defined($clients{$client})) {
              if (($clients{$client} eq 'CLIENT')) {
                if( $request =~/<PTZ_Command>(.*)<\/PTZ_Command>/i) {
    				my $ttt = unescape($1);
                    $log->debug('BROADCAST:',$ttt); # BROADCAST:do.ptz?dev=3&mode=speed&pt=-7,-11&undefined
                    my $cmd = PrepareCommand($ttt);
                    foreach my $destination ($select->handles) {
                        next if not defined $clients{$destination};
                        next if $clients{$destination} ne 'DRIVER';
                        $outbuffer{$destination} .= $cmd."\n";
                    }
                }
                elsif($request =~ /HUP/ ) {
                    read_configs();
                    $log->debug('BROADCAST:',$request);
                    foreach my $destination ($select->handles) {
                        next if not defined $clients{$destination};
                        next if $clients{$destination} ne 'DRIVER';
                        $outbuffer{$destination} .= "$request\n";
                    }
                }
              }
              elsif (($clients{$client} eq 'MONITOR')) {
                  # Process hearbeat request
            	  if ($request =~ /HEARBEATREQ: (\d+)/) {
            	      $outbuffer{$client} = "HEARTBEATRSP: $1\n";
            	  }

              }
            }
        }
    }
    delete $ready{$client};
    select(undef,undef,undef,0.05);
}
# -------------------------------------------------------------- Broadcast_MSG -----
#   Broadcasting message bus incoming messages to all socket clients
sub Broadcast_MSG {
    my ($id,$income)= @_;

    if (! $master)
    {
        if (($id==0)&&((timegm(gmtime)-$t_hb) < 15))
        {
            $log->debug($income,'===============================>',(timegm(gmtime)-$t_hb));
            $t_hb=timegm(gmtime);
        }
    }

    if($income =~ /<PTZ_Command>(.*)<\/PTZ_Command>/i)
    {
        my $str_4_trl=$1;
        $str_4_trl =~/(do\.ptz\?dev\=)(\d+)(.*)/i;
		my $dev = $2;
		#$str_4_trl ="$1".$dev."$3";

		my $ttt = unescape($str_4_trl);
        $log->debug('BROADCAST:',$ttt); # BROADCAST:do.ptz?dev=3&mode=speed&pt=-7,-11&undefined
        my $cmd = PrepareCommand($ttt);
        send2drivers($cmd);

		# save data about last command and time from PTZ client
		if ($ttt =~ /mode=(.*?)&/)
		{
			my $mode = $1;
			if ($mode eq "speed")
			{
				# we need to save maby missed in future stop command
				my $ptz_command = "do.ptz?dev=$dev&mode=speed";
				my %ptz_commands = (
					"pt" => {
						"speed" => "pt=((-?[1-9]+\\d*,-?[1-9]+\\d*)|(-?[1-9]+\\d*,-?\\d+)|(-?\\d+,-?[1-9]+\\d*))",
						"stop" => "pt=0,0",
						"stop_command" => "$ptz_command&pt=0,0"
					},
					"ptz_zoom" => {
						"speed" => "ptz=-?\\d+,-?\\d+,-?[1-9]+\\d*",
						"stop" => "ptz=-?\\d+,-?\\d+,0",
						"stop_command" => "$ptz_command&zoom=0"
					},
					"ptz_pt" => {
						"speed" => "ptz=((-?[1-9]+\\d*,-?[1-9]+\\d*)|(-?[1-9]+\\d*,-?\\d+)|(-?\\d+,-?[1-9]+\\d*)),-?\\d+",
						"stop" => "ptz=0,0,\\d+",
						"stop_command" => "$ptz_command&pt=0,0"
					},
					"zoom" => {
						"speed" => "zoom=-?[1-9]+\\d*",
						"stop" => "zoom=0" ,
						"stop_command" => "$ptz_command&zoom=0"
					},
					"focus" => {
						"speed" => "focus=-?[1-9]+\\d*",
						"stop" => "focus=0",
						"stop_command" => "$ptz_command&focus=0"
					},
					"iris" => {
						"speed" => "iris=-?[1-9]+\\d*",
						"stop" => "iris=0",
						"stop_command" => "$ptz_command&iris=0"
					},
					"gain" => {
						"speed" => "gain=-?[1-9]+\\d*",
						"stop" => "gain=0",
						"stop_command" => "$ptz_command&gain=0"
					}
				);

				$devices{"DEVID"} = {};
				$devices{"DEVID"}{$id} = {};

				foreach (keys(%ptz_commands))
				{
					if (!defined($devices{"DEVID"}{$id}{$_}))
					{
						$devices{"DEVID"}{$id}{$_} = {};
					}

					my $command = $ptz_commands{$_};

					if ($ttt =~ /$command->{speed}/i)
					{
						$devices{"DEVID"}{$id}{$_}{"ptz_command"} = $command->{stop_command};
						$devices{"DEVID"}{$id}{$_}{"time"} = time;
					}
					elsif ($ttt =~ /$command->{stop}/i)
					{
						$devices{"DEVID"}{$id}{$_}{"ptz_command"} = "";
						$devices{"DEVID"}{$id}{$_}{"time"} = 0;
					}
				}
			}
		}
    }
    elsif($income =~ /HUP/ )
    {
        $log->debug('BROADCAST:',$income);
        send2drivers($income);
    }
}

# -------------------------------------------------------------- nonblock -----
#   nonblock($socket) puts socket into nonblocking mode
sub nonblock {
    my $socket = shift;
    my $flags;

    $flags = fcntl($socket, F_GETFL, 0)
            or die "Can't get flags for socket: $!\n";
    fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
            or die "Can't make socket nonblocking: $!\n";
}

# -------------------------------------------------------- PrepareCommand -----
sub PrepareCommand{
	my $form_info = shift;
	$form_info = $1 if $form_info =~/^do.ptz\?(.*)/;
	my %args = ();
	my @args;	# to have parameters in order
	foreach(split /&/,$form_info){
	   if (/(\S+)=(\S+)/) {
		next if $1 eq 'imagewidth';
		next if $1 eq 'imageheight';
		next if $2 eq '';
		$args{lc($1)}=unescape($2);
		if ( lc($1) eq 'center') {
			$args{xy} = $args{center};
			delete $args{center};
			push(@args,'xy');
			next;
		}
	    if ( lc($1) eq 'xy') {
	        $args{xy} =~s/\?//;
	    }
		push(@args,$1);
	   }
	}

	my $topipe = "$args{dev} $args{mode}";
	foreach (@args){
	    next if /dev|mode/;
	    $topipe .=" $_=$args{$_}";
	}
	
        # TP6767: Copy PRESET1_TIMEOUT logic to PTZ_Server
        # Place to handle commands scheduled for sending to client
        # Manage PRESET1TIMEOUT here
        {
            my ($dev, $mode) = ($args{dev}, $args{mode});
            last if not $dev or not $mode;
            my $cfg = $Cfgs{$dev};
            last if not $cfg; # Device belongs to another node
            my $tmt = $cfg->{PTZ_PRESET1TIMEOUT};
            last if not $tmt or $tmt <= 0;
            if ($mode=~/speed/i || $mode=~/step/i || $mode=~/abs/i || $mode=~/rel/i || $mode=~/smooth/i) {
                    $log->debug("Setting timeout +$tmt");
                    $preset1timeouts{$dev} = time + $tmt;
            }
            elsif ($mode=~/preset/i && defined $args{goto} && $args{goto} ne '1') {
                    $log->debug("Setting timeout (preset) +$tmt");
                    $preset1timeouts{$dev} = time + $tmt;
            }
        }
        
        # TP6761: Add analytics suppression logic to PTZ server
        {
            my ($dev, $mode) = ($args{dev}, $args{mode});
            last if not $dev or not $mode; 
            if ($mode=~/preset/i && defined $args{goto} && $args{goto} eq '1') {
                # Unlock analytics if lock exists
                if ($anl_locks{$dev}{lock}) {
                    $anl_locks{$dev}{unlock} = time + $ANALYTICS_PTZ_RESET_TIMEOUT;
                }
            }
            else {
                # Lock analytics for camera $dev
                my $lockfile = "$APL_CONF/$dev/ptz/lock";
                if (not -f $lockfile) {
                    $log->warn("Analytics lock file for DEVID=$dev is missing");
                }
                else {
                    if (not open($anl_locks{$dev}{fh}, '>', $lockfile)) {
                        $log->error("Cannot open $lockfile for writing: $!");
                        last;
                    }
                    if (not flock($anl_locks{$dev}{fh}, LOCK_EX)) {
                        $log->error("Cannot flock() $lockfile: $!");
                        last;
                    }
                    $log->info("Established analytics lock on DEVID=$dev") if $anl_locks{$dev}{lock} == 0;
                    $anl_locks{$dev}{lock} = time;
                    $anl_locks{$dev}{unlock} = 0;
                }
            }

        }
	return $topipe;
}

sub Mess_suscr {

    #foreach (keys(%device_ID))
    #{
    #    my $msg = {
    #            device_ID => $_,
    #    	    ctxType   => Context_Class_POSITIONER,#65535,
    #    	    evtType   => Event_Class_CMD,#65535,
    #    	    procLvl   => Data_Processing_Level_LVL0,#65535,
    #    	    };
    #    my $subid = $messenger->subscribe($msg);
    #    $log->info ("subscribe for obj=$_ id=$subid hash=>$device_ID{$_}");
    #}
    ##subscr to ID = 0 as system
    my $msg = {
            device_ID => 0,
    	    ctxType   => Context_Class_OBJ3D(), #Context_Class_POSITIONER,#65535,
    	    evtType   => Event_Class_CMD(),#65535,
    	    procLvl   => Data_Processing_Level_LVL0(),#65535,
    	    };
    my $subid = $messenger->subscribe($msg);
    $log->info ("subscribe for obj=0 id=$subid");

}

sub reload_dev_conf {
    #Mess_suscr();
    system("kill -9 `ps ax|grep bin/PTZ_|grep -v grep|cut -c 1-6` 2>/dev/null");
    #kill -9 =>$$;
}

sub checkMaster
{
    if (! -f "$APL_CONF/master/s_master")
    {
        return 0;
    }
    return 1;
}

sub mbus_connect {
    my $tries = 0;
    while (1) {
	last if $messenger->connect;
	$tries++;
	if ($tries >= $MBUS_CONN_TRIES) {
	    $log->logdie("MBus connection failed. Stop trying after $tries attempts");
	}
	sleep 1;
    }
    $log->info("Connected to MBus successfully. sid=".$messenger->sid);
    $MBusReconnect = 0;
    Mess_suscr;
}

sub update_asrv {
    %asrv = GetAsrv();
    $Low_P_timeout = exists $asrv{PTZ_LOCK_TIMEOUT} ? $asrv{PTZ_LOCK_TIMEOUT} : 15;
    $PTZ_PROPAGATE2ELOG = $asrv{PTZ_PROPAGATE2ELOG} || 'disabled'; # info, alert or disabled
}

sub db_close {
    eval { $dbm->disconnect() if $dbm; $dbm=''; }; # disconnect if defined
}

sub db_master {
    db_close;
    eval {
    	$dbm=DBMaster({PrintError=>0,RaiseError => 1});
    	$dbm->{FetchHashKeyName} = 'NAME_uc';
    };
    if($@) {
	db_close;
    }
    eval {
	$dbs{INSERT_AUDIT}=$dbm->prepare(
	    "INSERT INTO audit (category,userid,objid,parameters)
	    VALUES (?, ?, ?, ?)"
	);
    };
    db_close if $@;
}

sub audit {
    my ($userid, $objid, $cat, $action, $attr) = @_;

    db_master() unless $dbm;

    my $params = [[$action],[]];
    if ($attr) {
	foreach my $key (keys %$attr) {
	    push ( @{ $params->[1] }, ['@AV', $key, $attr->{$key}] );
	}
    }

    eval {
	$dbs{INSERT_AUDIT}->execute($cat, $userid, $objid, to_json($params));
    };
    if ($@) {
	$log->error("Error storing audit data: $@");
	db_close();
    }
}

sub load_ptz_tours {
    my $dbh = DBMaster({RaiseError=>0,PrintError=>0});
    if ($dbh) {
       my $arr = $dbh->selectall_arrayref(
           "select o.obj,a.val from _objs o inner join _obj_attr a on o.obj=a.obj
           where o.deleted=0 and o.node_ip='$UNI' and o.otype='D' and o.subtype='C'
           and a.attr='STAT_PTZ_TOUR' and a.val != ''"
       );
       foreach my $row (@$arr) {
           my ($obj, $tour_data) = ($row->[0], $row->[1]);
           my $msg = {
               device_ID => $obj,
               data => "mode=setptztour&tour=$tour_data&"
           };
           $log->info("Loaded active tour for DEV=$obj: $tour_data");
           ptz_tour_manage($msg, 0);
       }
       $dbh->disconnect;
    }
}

sub send2drivers {
    my $cmd = shift;
    
    foreach (keys  %clients) {
        next if not defined $clients{$_};
        next if $clients{$_} ne 'DRIVER';
        $outbuffer{$_} .= $cmd."\n";
    }
}

sub init_mbus {
    $messenger= NextCAM::Messenger->new(
        host => 's_master',
        port => 10000,
        conn_timeout => 1000,
        autoconnect => 0
    );
    mbus_connect;
    $log->debug('sid = ', $messenger->sid);
}

sub start_server {
    $server = new IO::Socket::INET(
                    LocalPort => $TCP_PORT,
                    Listen => 5,
                    Reuse => 1,
                    Proto => 'tcp'
                ) or $log->logdie("Couldn't open port $TCP_PORT: $!");


    nonblock($server);
    $select = new IO::Select( $server );
    tie %ready, 'Tie::RefHash';

    $log->warn("PTZ socket switch is ready. Waiting for connections on port $TCP_PORT ...");
}

sub read_configs {
    $/ = "\n";
    %Cfgs = GetCfgs();
    local $/ = "\0";
    
    foreach my $dev (%preset1timeouts) {
        delete $preset1timeouts{$dev} if not exists $Cfgs{$dev};
    }
    foreach my $dev (%anl_locks) {
        if (not exists $Cfgs{$dev}) {
            close $anl_locks{$dev}{fh};
            delete $anl_locks{$dev};
        }
    }
}

sub av_init {
    # Init Logger
    Log::Log4perl::init("$APL/etc/logger/logger_ptz.conf");
    $log = Log::Log4perl->get_logger("ptz_server");
    
    # Read device configs
    read_configs;
}


sub va_init {
    # Init Stratus modules
    require NextCAM::Init;
    require NextCAM::Messenger;
    require NextCAM::Messenger::Flags;
    require Node::Conf;
    require NextCAM::ELClient;
    NextCAM::Init->import("GetAsrv");
    Node::Conf->import("UNI");
    SKM::DB->import;
    NextCAM::Messenger::Flags->import;
    
    # Pid management
    WritePid;
    
    # Init logger
    require "$ENV{APL}/common/bin/logger.engine";
    $log=get_logger('NEXTCAM::PTZ::PTZ_SERVER');
    
    # Init globals
    $UNI = UNI();
    update_asrv();
    $ELClient = new ELClient;
    
    # Init MBus client
    init_mbus;
    
    # Load server tours from DB
    load_ptz_tours;
    
    # Read device configs
    read_configs;
}

sub main_loop {
    while(1) {
        my $client;
        my $rv;
        my $data;
        
        if (not $IS_AVATAR) {
            if ($MBusReconnect) {
                $log->warn("MBus connection was lost. Trying to reconnect");
                mbus_connect();
            }

            # Synchronous MBus communication
            my $msg = $messenger->recv(50);
            on_disconnect unless defined $msg;
            on_message($msg) if $msg;
        }

	if (defined($devices{"DEVID"}))
	{
	    my $DEVID = $devices{"DEVID"};
	    foreach my $id (keys %{$DEVID})
		{
			foreach (keys(%{$devices{"DEVID"}{$id}}))
			{
				# if last command was 2 seconds ago
				# we need to send missed stop command
				if ($DEVID->{$id}{$_}{"time"} != 0 && time - $DEVID->{$id}{$_}{"time"} > 2)
				{
					$DEVID->{$id}{$_}{"time"} = 0;

					my $command = $DEVID->{$id}{$_}{"ptz_command"};
					if ($command ne "")
					{
						# send command to all clients
						$log->debug('BROADCAST:', $command); # BROADCAST:do.ptz?dev=3&mode=speed&pt=0,0&undefined
						my $cmd = PrepareCommand($DEVID->{$id}{$_}{"ptz_command"});
						send2drivers($cmd);
					}
				}
			}
		}
	}

        $log->debug('tick! ',$master,(timegm(gmtime)-$t_hb));
#        if((!$master)&&((timegm(gmtime)-$t_hb) >15))
#        {
#            $log->debug("I need reconnect HERE====================>");
#             $t_hb=timegm(gmtime);
#             reload_dev_conf();
#        }

        foreach $client ($select->can_read(0.05)) {
            UpdatePid() if not $IS_AVATAR;
            if($client == $server) { # ACCEPT NEW CONNECTION
                $client = $server->accept();
                $log->debug("Accepted new GUI client: $client");
                $select->add($client);
                nonblock($client);
            }
            else { # READ DATA
                # read data
                $data = '';
                $rv   = $client->recv($data, POSIX::BUFSIZ, 0);

                unless (defined($rv) && length $data) {
                    if($lastwill{$client})
                    {
                        $log->debug("$client LASTWILL:$lastwill{$client}");
                        my $cmd = PrepareCommand($lastwill{$client});
                        foreach my $destination ($select->handles) {
                            next if not defined $clients{$destination};
                            next if $clients{$destination} ne 'DRIVER';
                            $outbuffer{$destination} .= $cmd."\n";
                        }
                    }
                    # This would be the end of file, so close the client
                    delete $inbuffer{$client};
                    delete $outbuffer{$client};
                    delete $ready{$client};
                    delete $clients{$client};
                    $log->debug("removing client from table: $client");
                    $select->remove($client);
                    close $client;
                    next;
                }
                $inbuffer{$client} .= $data;

                # test whether the data in the buffer or the data we
                # just read means there is a complete request waiting
                # to be fulfilled.  If there is, set $ready{$client}
                # to the requests waiting to be fulfilled.
                while ($inbuffer{$client} =~ s/(.*)[\n,\0,\r]//m) {

                    push( @{$ready{$client}}, $1 );
                }
            }
        } # foreach CLIENT

        # Any complete requests to process?
        foreach $client (keys %ready) {
            $log->debug("$client in READY queue");
            handle($client);
        }

        # Buffers to flush?
        foreach $client ($select->can_write(0.05)) {
            # Skip this client if we have nothing to say
            next unless exists $outbuffer{$client};

            $rv = $client->send($outbuffer{$client}, 0);
            unless (defined $rv) {
                # Whine, but move on.
                $log->warn("I was told I could write, but I can't.");
                next;
            }
            if ($rv == length $outbuffer{$client} ||
                $! == POSIX::EWOULDBLOCK) {
                substr($outbuffer{$client}, 0, $rv) = '';
                delete $outbuffer{$client} unless length $outbuffer{$client};
            } else {
                # Couldn't write all the data, and it wasn't because
                # it would have blocked.  Shutdown and move on.
                delete $inbuffer{$client};
                delete $outbuffer{$client};
                delete $ready{$client};

                $select->remove($client);
                close($client);
                next;
            }
        }

        # Out of band data?
        foreach $client ($select->has_exception(0)) {  # arg is timeout
            # Deal with out-of-band data here, if you want to.
        }

        if (not $IS_AVATAR) {
            # providing "still alive" signal
            if (((timegm(gmtime)-$t_hb)>5)&& ($master))
            {
                my $msg = {
        	    device_ID => 0,
        	    ctxType   => Context_Class_OBJ3D(), #Context_Class_POSITIONER,
        	    evtType   => Event_Class_CMD(),
        	    procLvl   => Data_Processing_Level_LVL0()
        	    };
                $t_hb=timegm(gmtime);
                $msg->{data} =$t_hb;
                my $res = $messenger->send($msg);
                $log->debug('\"STILL ALIVE\" Message sended res:',$res);
            }

            if ($current_time != time()) {
                $current_time = time();
                ptz_tour();
                ptz_lock_obtain_release();
            }

            # Periodically update asrv
            if (time - $t_asrv > 10) {
                $t_asrv = time;
                update_asrv();
            }
        }
        # TP6767 Copy PRESET1_TIMEOUT logic to PTZ_Server
        # Apply to both Avatar and Stratus
        foreach my $dev (keys %preset1timeouts) {
            my $tmt = $preset1timeouts{$dev};
            next if not $tmt;
            next if $tmt > time;
            $log->debug("TIMEOUT EXPIRED!");
            $preset1timeouts{$dev} = 0;
            # Schedule unlock analytics if lock exists
            if ($anl_locks{$dev}{lock}) {
                $anl_locks{$dev}{unlock} = time + $ANALYTICS_PTZ_RESET_TIMEOUT;
            }
            my $cmd = "$dev preset goto=1";
            send2drivers($cmd);
        }
        
        # TP6761: Add analytics suppression logic to PTZ server
        # Check and release analytics locks here
        foreach my $dev (keys %anl_locks) {
            my $lock = $anl_locks{$dev};
            next if not $lock->{unlock};
            next if $lock->{unlock} > time;
            $log->info("Unlock analytics on DEVID=$dev");
            flock($anl_locks{$dev}{fh}, LOCK_UN) or $log->error("Cannot unlock analytics for DEVID=$dev: $!");
            $anl_locks{$dev}{lock} = $anl_locks{$dev}{unlock} = 0;
        }
    } # while(1)
}

# MAIN
#
sub main {
    if ($IS_AVATAR) {
        av_init;
    }
    else { # Stratus
        va_init;
    }
    
    start_server;
    
    main_loop;
}

main;
