#!/usr/bin/perl
# -----------------------------------------------------------------------------
#  Author: Alexey Tsibulnik
#  Edited by: 
#  QA by:  
#  Copyright: (c) videoNEXT LLC, 2005
# -----------------------------------------------------------------------------
# ------------------------------------------------------------------------------
use strict;
use IO::Socket;
use IO::Select;
use Socket qw(IPPROTO_TCP);
use HTTP::Request::Common;
use LWP::UserAgent;
use Net::Ping;
use POSIX qw(_exit WNOHANG);
use Log::Log4perl "get_logger";
use NextCAM::Conf;

# CONS
my $APL = $ENV{APL} || '/opt/sarch';
my $LOCAL_SOCK = "/tmp/local_udp.soc";
my $CAMSTAT_CONF = "$APL/mgears/etc/cam_stat.conf";
my $SLEEP = 15;
my $LISTEN_PORT = 2555;	# Default listen port on UDP encoders for TCP notifications
my $TCP_TIMEOUT   = 10;
my $PING_TIMEOUT  = 30;
my $RETRY_TIMEOUT = 60; # Wait before next try

# VAR
my $SocketName;
my $Outsock;
my $Outpeer;
my %Cams;

Log::Log4perl::init_and_watch("$ENV{APL}/common/etc/logger.camstat", 60);
my $log = get_logger('HM::CAMSTAT');

# SIG
$SIG{TERM} = $SIG{INT} = sub { exit 1 };
$SIG{HUP} = 'IGNORE';
$SIG{CHLD} = \&on_listener_exit;

# SUB
sub prepare_socket
{
	my %camstat_conf;
	if (open CONF, $CAMSTAT_CONF) {
		%camstat_conf = map {/^(\w+)=(.+)$/} grep {/^\w+=/} <CONF>;
		close CONF;
		
		$SocketName = $camstat_conf{SOCKET_NAME};
		die "SOCKET_NAME param is missing in camstat_conf\n" unless $SocketName;
	}
	else {
		die "Cannot open $CAMSTAT_CONF for reading: $!\n";
	}
	
	# Create unconnected UNIX socket
	$Outsock = IO::Socket::UNIX->new(Local => $LOCAL_SOCK, Type  => SOCK_DGRAM)
		or die "Couldn't open socket for path [$SocketName]: $!";

	$Outpeer = sockaddr_un($SocketName);
}

sub send_state
{
	my $msg = shift;
	
	$log->debug("Sending to socket -> [$msg]");
	my $res = send ($Outsock, $msg."\n", 0, $Outpeer);
	$log->error("Send failed : $!") unless $res;
	return $res;
}

sub stop_listener
{
	my $dev = shift;
	
	$log->info("Stop listener for dev=$dev");
	my $pid = $Cams{$dev}{pid};
	if ($pid and kill 0 => $pid) {
		kill 15 => $pid;
	}
	
	delete $Cams{$dev};
}

sub listener
{
	my $dev = shift;
	my $cam = $Cams{$dev};
	my $prefix = "[$dev]";
	
	# Prepare user agent
	my $base = "http://$cam->{devip}:$cam->{http_port}";
	my $ua = LWP::UserAgent->new;
	$ua->timeout($TCP_TIMEOUT);
	$ua->agent("health monitor");
	
	# Ping camera
	my $p = Net::Ping->new("tcp",$TCP_TIMEOUT);
	$p->port_number($cam->{http_port});
	$log->error("$prefix Ping failed"),return if not $p->ping($cam->{devip});
	my $last_ping_time = time;
	$log->debug("$prefix Ping OK");
	
	# Enable tcp event notification on camera
	my $req = GET "$base//nvc-cgi/admin/param.fcgi?".
		      "action=update&group=Event.Notify.tcp&".
		      "enable=yes&name=hm&listenport=$LISTEN_PORT";
	$req->authorization_basic($cam->{usrname}, $cam->{passwd});
	my $rsp = $ua->request($req);
	$log->debug("$prefix Response: ".$rsp->content);
	$log->error("$prefix Error updating Event.Notify: [".$rsp->code."] ".$rsp->content ),return 
		if $rsp->is_error or $rsp->content !~ /^#200/;
	
	# Event rule for video loss
	$req = GET "$base/nvc-cgi/admin/param.fcgi?action=update&group=Event.Rule.video&tcp=yes&http=no";
	$req->authorization_basic($cam->{usrname}, $cam->{passwd});
	my $rsp = $ua->request($req);
	$log->debug("$prefix Response: ".$rsp->content);
	$log->error("$prefix Error updating Event.Rule: [".$rsp->code."] ".$rsp->content ),return 
		if $rsp->is_error or $rsp->content !~ /^#200/;
	
	# Open persistent connection
	my $sock = IO::Socket::INET->new(
		PeerAddr => $cam->{devip},
		PeerPort => $LISTEN_PORT,
		Proto    => 'tcp',
		Timeout  => $TCP_TIMEOUT,
		KeepALive => 1
	);
	$log->error("$prefix Create socket failed: $!"),return unless $sock;
	
	# Tune up socket
	$sock->autoflush(1);
	setsockopt($sock, IPPROTO_TCP, SOL_SOCKET, SO_KEEPALIVE) or $log->warn("$prefix setsockopt: $!");
	my $select = IO::Select->new($sock);
	
	# Read data
	while (1) {
	
		# Ping camera
		if (time - $last_ping_time > $PING_TIMEOUT) {
			$log->debug("$prefix Time to ping");
			$log->error("$prefix Ping failed!"),last unless $p->ping($cam->{devip});
			$log->debug("$prefix Ping OK");
			$last_ping_time = time;
		}
		
		next unless $select->can_read($TCP_TIMEOUT);
	
		my $buf = "";
		# First goes packet header
		my $rb = $sock->sysread($buf, 8);
		if ($buf ne "DOOFTEN\0") {
			$log->error("$prefix Bad header: $buf!");
			last;
		}
		
		# Then read decimal string of packet length
		$rb = $sock->sysread($buf, 8);
		my ($pkt_len) = $buf =~ /^(\d+)\x{00}/;
		$log->debug("$prefix Packet length: $pkt_len");
		
		# Finally read rest of the packet
		$rb = $sock->sysread($buf, $pkt_len - 16);
		$log->debug("$prefix Received: $rb bytes");
		if (not defined $rb) {
			$log->error("$prefix Socket error: $!");
			last;
		}
		elsif (not $rb) {
			$log->error("$prefix Server closed connection");
			last;
		}
		
		# Process packet
		process_packet($dev, $buf);
	}

	
	$sock->close;
}

sub process_packet
{
	my ($dev, $pkt) = @_;
	
	my %fld = map {/^(\w+)=(.*)$/} grep {/^\w+=/} split /\n/, $pkt;
	
	if ($fld{type} eq 'vsignal') {
	    my $info = $fld{info};
	    my %msg = map {/^(\w+)=(.*)$/} grep {/^\w+=/} split /&/, $info;
	    my ($currmask, $prevmask) = @msg{"currmask", "prevmask"};
	    
	    # Consider 'count' value
	    # It should correspond to value of video input for target device
	    if ($msg{count} eq $Cams{$dev}{camera} and defined $currmask and defined $prevmask) {
		    if ($currmask == 0) {
			    $Cams{$dev}{state} = 0;
			    send_state("udp_video_lost objid=$dev");
		    }
		    elsif ($currmask == 1 and $Cams{$dev}{state} != 1) {
			    $Cams{$dev}{state} = 1;
			    send_state("udp_video_restored objid=$dev");
		    }
	    }
	}
}

sub start_listener
{
	my $dev = shift;
	
	$log->info("Start listener for dev=$dev");
	my $pid = fork;
	if ($pid == 0) {
		$SIG{HUP} = 'IGNORE';
		$SIG{TERM} = $SIG{INT} = undef;
		$SIG{PIPE} = sub { $log->warn("[$dev] PIPE caught"); _exit(1) };
		
		listener $dev;
		
		_exit 0;
	}
	elsif ($pid < 0) {
		die "Cannot fork: $!";
	}
	else {
		$Cams{$dev}{pid} = $pid;
		$Cams{$dev}{started} = time;
	}
}

sub on_listener_exit
{
	my $kid;
	while ( ($kid = waitpid(-1, &WNOHANG)) > 0) {
		foreach my $cam (values %Cams) {
			if ($cam->{pid} == $kid) {
				$log->info("Listener for dev=$cam->{devid} pid=$kid has exited");
				$cam->{pid} = 0;
				$cam->{exited} = time;
			}
		}
	}
	$SIG{CHLD} = \&on_listener_exit;
}

sub read_conf
{
    my %conf = GetCfgs('DEVICETYPE'=>'CAMERA', 'CAMERAMODEL' => 'UDP');
    
    # Update camera cache
    foreach my $dev (keys %conf) {
	    my $cfg = $conf{$dev};
	    # Skip everything except UDP cameras
	    next unless $cfg->{CAMERAMODEL} =~ /^UDP$/i;
	    # Skip deleted devices
	    next if $cfg->{LOCATION} eq '@garbage';
	
	    $Cams{$dev} = {
		    devid     => $dev,
		    devip     => $cfg->{DEVIP},
		    http_port => $cfg->{HTTP_PORT},
		    usrname   => $cfg->{USRNAME},
		    passwd    => $cfg->{PASSWD},
		    camera    => $cfg->{CAMERA},
		    state     => 0,
		    started   => 0,
		    exited    => 0,
		    pid       => 0
	    } unless $Cams{$dev};
    }
    
    # Wipe deleted devices from cache and stop corresponding processes
    foreach my $dev (keys %Cams) {
	    stop_listener $dev if 
		    not exists $conf{$dev} or 
		    $conf{$dev}{CAMERAMODEL} ne 'UDP' or
	    	    $conf{$dev}{LOCATION} eq '@garbage';
    }
    
    # Start listeners
    foreach my $dev (keys %Cams) {
	    my $cam = $Cams{$dev};
	    unless ($cam->{pid}) {
		    next if time - $cam->{exited} < $RETRY_TIMEOUT;
		    start_listener $dev;
	    }
    }
}

sub finalize
{
	foreach my $dev (keys %Cams) {
		stop_listener $dev;
	}
	
	unlink $LOCAL_SOCK;
	
	$log->info("UDP health monitor finished");
}

# =============================================================================
# Main
sub main
{
	$log->info("UDP health monitor started");
	
	prepare_socket;
	
	while(1) {
		read_conf;
		sleep $SLEEP;
	}
}

main;

END {
	finalize;
}
