#!/usr/bin/perl
#  $Id: camerahealth_d.pl 12484 2008-10-28 15:52:07Z starostin $
# -----------------------------------------------------------------------------
#  Socket server receives camera health states from clients
# -----------------------------------------------------------------------------
#  Author: Sinyatkin Yaroslav (yas@videonext.net)
#  Edited by: Alexey Tsibulnik
#  QA by:
#  Copyright: (c) videoNEXT LLC, 2005
# -----------------------------------------------------------------------------
# ------------------------------------------------------------------------------

use strict;
use POSIX;
use IO::Socket;
use IO::Select;
use Socket;
use Fcntl;
use Tie::RefHash;
use Data::Dumper;
use Time::Local;
use NextCAM::Conf "GetCfgs";
use NextCAM::OMClient;

# -----------------------------------------------------------------------------

# device states to store in OM
use constant STATE_UNKNOWN      => 0; #
use constant STATE_CONNECTED    => 1; # connected but not streaming yet
use constant STATE_STREAMING    => 2; # streaming OK
use constant STATE_OFFLINE      => 3; # disconnected

use constant CONF_FILE          => $ENV{APL}.'/mgears/etc/cam_stat.conf'; # config file

use constant OM_REPORT_FREQ     => 7; # seconds to accumulate status before call OM
use constant NO_REPORT_TIMEOUT  => 5; # check alive reports frequency (after this seconds seems camera's retriever dies)
use constant READ_CONF_FREQ	=> 5; # seconds to wait before device list update

use constant SENDER_AXIS        => 'axis';  # set as flag in report queue
use constant SENDER_UDP         => 'udp';   # set as flag in report queue
use constant SENDER_OTHER       => 'other'; # set as flag in report queue

# -----------------------------------------------------------------------------
my $DEBUG = shift || 0; # any argument sets DEBUG mode

use Log::Log4perl "get_logger";
Log::Log4perl::init_and_watch("$ENV{APL}/common/etc/logger.camstat", 60);
my $log = get_logger('HM::CAMSTAT');
# ==============================================================================
sub log_debug { $log->debug(@_); print "@_\n" if $DEBUG; }
sub log_fatal { $log->fatal(@_); die @_; }
# ==============================================================================

my %device_ID;

# Close filehandles
#close(STDIN); close(STDOUT); close(STDERR);

my %inbuffer            = ();
my %outbuffer		= ();
my %ready               = ();
my %clients	        = ();
my $t_hb	        = timegm(gmtime);
my $master              = checkMaster();
my %devices	        = ();
my %device_list         = ();
tie %ready, 'Tie::RefHash';
my $socket_path 		= undef;
my $last_reported 		= timegm(gmtime);
my $last_readconf;
my %report_list 		= ();

$SIG{TERM} = $SIG{INT} = sub { exit 0 };
$SIG{HUP}=\&reinit;

$log->info("Starting camera health monitor daemon");

my $om_client = new OMClient() or log_fatal ('unable to init OM client');

load_dev_conf();

my $server = prepare_socket();
my $select = new IO::Select( $server );


process_requests($server, $select);

# --------------------------------------------------- process_requests ----
sub process_requests
{
    $server = shift;
    $select = shift;
    while(1) {
        my $client;
        my $rv;
        my $data;
        my $time_curr;
        #$rv   = recv($server, $data, POSIX::BUFSIZ, 0);
   	    #log_debug("rv=$rv, data=[$data]\n");
		#} while (! defined($rv));

        foreach $client ($select->can_read(0.05))
        {
    		local $/ = "\0"; # Set the input terminator to a zero byte string, pursuant to the protocol in the Flash documentation.
                # read data
                $data = '';
                $rv   = recv($client, $data, POSIX::BUFSIZ, 0);
                #log_debug("recved data=[$data], defined rv=".(defined $rv ? 1 : 0));
                unless (defined($rv) && length $data) {
                    # This would be the end of file, so close the client
                    delete $inbuffer{$client};
                    delete $outbuffer{$client};
                    delete $ready{$client};
                    delete $clients{$client};
                    #log_debug("removing client from table: $client");
                    $select->remove($client);
                    close $client;
                    next;
                }
                $inbuffer{$client} .= $data;

                # test whether the data in the buffer or the data we
                # just read means there is a complete request waiting
                # to be fulfilled.  If there is, set $ready{$client}
                # to the requests waiting to be fulfilled.
                while ($inbuffer{$client} =~ s/(.*)[\n,\0,\r]//m) {

                    push( @{$ready{$client}}, $1 );
                }
        } # foreach CLIENT

        # Any complete requests to process?
        foreach $client (keys %ready) {
            #log_debug("$client in READY queue");
            handle($client);
        }

    	$time_curr = timegm(gmtime);
    	load_dev_conf() if $time_curr-$last_readconf > READ_CONF_FREQ;

	foreach my $id (keys %device_list)
	{
	    if ( (timegm(gmtime) - $device_list{$id}{'last_updated'}) > NO_REPORT_TIMEOUT)
	    {
		unless ($device_list{$id}{av_status}) {
		    if ($device_list{$id}{'last_state'} ne STATE_UNKNOWN) {
        		$log->warn("NO_REPORT_TIMEOUT($id) -> force STATE_UNKNOWN(0) for OBJID=$id");
        	    }
            	    set_state (STATE_UNKNOWN, $id, '', SENDER_OTHER);
            	}
            	else {	# Update status for Avatar cameras
            	    my $state = STATE_UNKNOWN;
            	    my $status = $device_list{$id}{av_status};
            	    if ($status =~ /^ON$/i) {
            		$state = STATE_STREAMING;
            	    }
            	    elsif ($status =~ /^(OFF|DOWN)$/i) {
            		$state = STATE_OFFLINE;
            	    }
            	    elsif ($status =~ /^(STARTING|RESETTING)$/i) {
            		$state = STATE_CONNECTED;
            	    }

            	    set_state ($state, $id, '', SENDER_OTHER);
            	}
		$device_list{$id}{'last_updated'} = timegm(gmtime);
	    }
	}

	my $rin = '';
	vec($rin,fileno($server),1) = 1;
	select($rin, undef, undef, 0.5);
    } # while(1)
} #     process_requests



# ---------------------------------------------------------------- handle -----
#   handle($socket) deals with all pending requests for $client
sub handle {
    my $client = shift;
    my $request;
    #log_debug("$client - now handeling");

    foreach $request (@{$ready{$client}})
    {
        #log_debug("handle request [$request]");
        $request =~ /(.+)[\n,\0,\r]?/m;
        $request = $1;

        if      ($request =~ /streaming pos=(\d+\.?\d+?), objid=(\d+)/i) {
            set_state (STATE_STREAMING, $2, "streaming pos=$1", SENDER_OTHER);
        }
        elsif ($request =~ /stream_lost objid=(\d+), code=(\d+)/i) {
        	log_debug("handle stream_lost($1, $2)");
            set_state (STATE_OFFLINE, $1, ''.$2, SENDER_OTHER);
        }
        elsif ($request =~ /axis_video_lost objid=(\d+)/i) {
        	log_debug("handle axis_video_lost($1)");
            set_state (STATE_OFFLINE, $1, 'axis_video_lost', SENDER_AXIS);
        }
        elsif ($request =~ /axis_video_restored objid=(\d+)/i) {
        	log_debug("handle axis_video_restored($1)");
            set_state (STATE_STREAMING, $1, 'axis_video_restored', SENDER_AXIS);
        }
        elsif ($request =~ /udp_video_lost objid=(\d+)/i) {
        	log_debug("handle udp_video_lost($1)");
            set_state (STATE_OFFLINE, $1, 'udp_video_lost', SENDER_UDP);
        }
        elsif ($request =~ /udp_video_restored objid=(\d+)/i) {
        	log_debug("handle udp_video_restored($1)");
            set_state (STATE_STREAMING, $1, 'udp_video_restored', SENDER_UDP);
        }
        elsif ($request =~ /connecting objid=(\d+)/i) {
        	log_debug("handle connecting($1)");
            set_state (STATE_UNKNOWN, $1, '', SENDER_OTHER);
        }
        elsif ($request =~ /unknown objid=(\d+)/i) {
        	log_debug("handle unknown($1)");
            set_state (STATE_UNKNOWN, $1, '', SENDER_OTHER);
        }
        elsif ($request =~ /startup objid=(\d+)/i) {
        	log_debug("handle startup($1)");
            set_state (STATE_UNKNOWN, $1, '', SENDER_OTHER);
        }
        else { log_debug ("unknown request [$request] ignored"); }
    }
    delete $ready{$client};
#    select(undef,undef,undef,0.1);
}


sub reinit {
    $log->info("Reinit daemon - clear device cache");
	%device_list = ();
	%report_list = ();
}

# -------------------------------------------------------- load_dev_conf -----
sub load_dev_conf {
    $log->debug("Reading configuration");
    my $time = timegm gmtime;
    my $objid;
    my %conf = GetCfgs;
    $device_list{$_}{deleted} = 1 foreach keys %device_list;
    foreach my $dev (keys %conf) {
	$objid = $conf{$dev}{OBJID};
	next unless defined $objid;
	next if $conf{$dev}{LOCATION} eq '@garbage';
	next unless $conf{$dev}{DEVICETYPE} eq 'CAMERA';
	unless (exists $device_list{$objid}) {
	    $device_list{$objid} = {
		last_updated => $time,
		deleted => 0
	    };
	}
	else {
	    $device_list{$objid}{deleted} = 0;
	}
	# Read status for Avatar cameras
	if ($conf{$dev}{AVATARID}) {
	    if (open(STAT, "$ENV{APL_CONF}/$dev/stat")) {
		my %stat = map {/^(\w+)=(.*)$/} grep {/^\w+=/} <STAT>;
		close STAT;
		$device_list{$objid}{av_status} = $stat{STATUS};
	    }
	}
    }
    # Remove deleted devices from cache
    foreach $objid (keys %device_list) {
	delete $device_list{$objid} if $device_list{$objid}{deleted};
    }
    # Update global
    $last_readconf = $time;
}


# ------------------------------------------------------ set_state --------
# update device state  when $objid=0 set state for all devices
sub set_state
{
    my ($state, $objid, $text, $sender)  = @_;

    return if (!defined $objid || $objid eq 0);
	return unless defined $state;

    $device_list{$objid} = {} unless $device_list{$objid};
    $device_list{$objid}{'last_updated'} = timegm(gmtime);
    $device_list{$objid}{'last_state'} = $state;

    eval {
	state2OM ($state, $objid, $text, $sender);
    };
    if ($@) {
	# Handle OM failures
	if ($@ =~ /^FwIface::FwException/) {
	    # Perhaps non-critical error, just ignore it
	}
	elsif ($@ =~ /^Bad: (.+)$/i) {
	    my $ex = $1;
	    if ($ex =~ /Thrift::TException/) {
		# Seems like MBus connection was lost.
		# Try to reconnect
		#
		$log->warn("Got Thrift::TException. Try to reinit OM client");
		my $tries = 0;
		until($om_client = new OMClient) {
		    $tries++;
		    $log->logdie("Exiting after $tries tries") if $tries > 10;
		    $log->warn("Unable to reconnect to MBus after $tries tries") if $tries % 5 == 0;
		    sleep 5;
		}
		$log->info("Reconnected successfully");
	    }
	    else {
		$log->logdie("Exiting due to unknown exception raised by OM: $ex");
	    }
	}
    }
}
# ------------------------------------------------------ state2OM --------
# writing data to OM
sub state2OM
{
    my ($state, $objid, $text, $sender)  = @_;

    return if (!defined $objid || $objid eq 0 || !defined $state);

    if ($report_list{$objid}{old} ne $state && $state ne STATE_STREAMING ) # force report: state changed and state isn't OK
    {
    	log_debug ("state2OM: force $objid:$report_list{$objid}{old}->$state, sender=$sender");
	my $ret_str = $om_client->send_update($objid, {'health' => $state});
	#log_fatal ("state2OM failed: $ret_str") if $ret_str;
    	$log->error ("state2OM: force $objid:$report_list{$objid}{old}->$state FAILED: [$ret_str]") if ($ret_str);
	$report_list{$objid}{old} = $state;
	$report_list{$objid}{new} = $state;
	$report_list{$objid}{sender} = $sender;
	die $ret_str;
	#return; # dont queue this state
    }
    #log_debug ("($objid): prev_sender=".$report_list{$objid}{sender}.", new_sender=$sender");

    if ($report_list{$objid}{old} ne $state
	&& ($report_list{$objid}{old} eq STATE_STREAMING || $state eq STATE_STREAMING)
	) # dont owerwrite state from AXIS daemon
    {
		if ( ($report_list{$objid}{sender} ne SENDER_AXIS && $report_list{$objid}{sender} ne SENDER_UDP)
			|| $sender eq SENDER_AXIS
			|| $sender eq SENDER_UDP)
		{
			# queue report if state changed from|to OK status
			log_debug ("queue for ID=$objid: old->$report_list{$objid}{old}, new->$state, prev_sender=".$report_list{$objid}{sender}.", new_sender=$sender");
			$report_list{$objid}{new} = $state;
			$report_list{$objid}{sender} = $sender;
		}
		else {
			log_debug("queue skip state for ID=$objid: last state has privelegie-".SENDER_AXIS);
		}
    }

    # accumulate changed OK states
    return if ( timegm(gmtime) - $last_reported < OM_REPORT_FREQ ); # return if no time to report

    log_debug ("send queued items");
    foreach my $id  (sort keys %report_list)
    {
	my $st = $report_list{$id}{new};
    	log_debug ("state2OM::send_update objid=$id, state=$st, will=".(defined $st ? 'process' : 'skip'));
	next if (! defined $id || ! defined $st);
	log_debug ("$id: old->$report_list{$id}{old}, new->$report_list{$id}{new}");
	#next if ( $report_list{$id}{old} eq  $report_list{$id}{new});

	my $ret_str = $om_client->send_update($id, {'health' => $st});
	if ($ret_str) {
		$log->error ("state2OM failed: $ret_str");
		die $ret_str;
    	}
    	else {
	    #log_debug("state2OM OK");
	    #$report_list{$id} = undef; # remove report need
	    $report_list{$id}{old} = $report_list{$id}{new};
	}

    }

    $last_reported = timegm(gmtime);
}

sub checkMaster
{
    if (! -f "$ENV{APL}/var/conf/master/s_master")
    {
        return 0;
    }
    return 1;
}



# -------------------------------------------------------------- nonblock -----
#   nonblock($socket) puts socket into nonblocking mode
sub nonblock {
    my $socket = shift;
    my $flags;

    $flags = fcntl($socket, F_GETFL, 0)
            or die "Can't get flags for socket: $!\n";
    fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
            or die "Can't make socket nonblocking: $!\n";
}

# -------------------------------------------------------------- prepare_socket -----
#   create server socket
sub prepare_socket
{
	die (CONF_FILE.' required file NOT FOUND') if ! -e CONF_FILE;

    log_debug("loading config file ".CONF_FILE);

	open CONF, '<'.CONF_FILE;
	$socket_path = undef;
	while (<CONF>) {
		chomp;
		log_debug("read [$_]\n");
		if (/SOCKET_NAME=(.+)/) { $socket_path = $1;  }
	}
	die ('SOCKET_NAME=<path> parameter required inside '.CONF_FILE) unless length $socket_path;

    unlink $socket_path;
    my $server = new IO::Socket::UNIX->new( Local => $socket_path, Type  => SOCK_DGRAM );
	if (! defined ($server)) { log_fatal("Couldn't open port $socket_path: $!"); die; }
    nonblock($server);
    $log->info("Socket for path [$socket_path] -> ".$server);

    return $server;
}


END { unlink $socket_path if $socket_path }

# todo: thread to periodically check last time camera's state was updated
