#!/usr/bin/perl
#  $Id: camerahealth_d.pl 12484 2008-10-28 15:52:07Z starostin $
# -----------------------------------------------------------------------------
# Axis daemon detects 'video loss' and 'video restore' status of the Axis cameras and 
# forward the notification to camera health daemon
# -----------------------------------------------------------------------------
#  Author: Yaroslav Sinyatkin (yas@videonext.net)
#  Edited by: 
#  QA by:  
#  Copyright: (c) videoNEXT LLC, 2005
# -----------------------------------------------------------------------------
# ------------------------------------------------------------------------------
package AxisDaemon;

use strict;
use IO::Socket;
use IO::Select;
use LWP;
use Data::Dumper;
use Time::Local;
use NextCAM::Conf;

#use constant LISTEN_PORT => 4445; # listen notifacations from Axis cameras
use constant CONF_FILE	 => $ENV{APL}.'/mgears/etc/cam_stat.conf'; # config file 
use constant LOCAL_SOCK	 => '/tmp/local_axis.soc'; # socket for server's answer

use constant STATE_UNKNOWN      => 0; # 
use constant STATE_CONNECTED    => 1; # connected but not streaming yet
use constant STATE_STREAMING    => 2; # streaming OK
use constant STATE_OFFLINE      => 3; # disconnected  

use constant MAIN_LOOP_SLEEP    => 15;
use constant HTTP_TIMEOUT       => 3; # seconds
my $DEBUG = shift || 0; # provide any argument to turn on logging to STDOUT
my $APL	  = $ENV{APL} || '/opt/sarch';
my $Daemon;

use Log::Log4perl "get_logger";
Log::Log4perl::init_and_watch("$ENV{APL}/common/etc/logger.camstat", 60);
my $log = get_logger('HM::CAMSTAT');
# ==============================================================================
sub log_debug { $log->debug(@_); print "@_\n" if $DEBUG; }
sub log_die   { $log->fatal(@_); die @_; }
# ==============================================================================

$SIG{TERM} = $SIG{INT} = sub { exit 0 };

$SIG{HUP} = \&reinit;

# testing
#ssend ("axis_video_lost objid=1");
#ssend ("axis_video_restored objid=1");


sub new
{
  	my $class = shift;
	my %cam_state = (); # (axis camera OBJID) -> STATE_XXX
	my %cam_ip = (); # (axis camera OBJID) -> IP
	my $http_agent = LWP::UserAgent->new;

	$http_agent->timeout(HTTP_TIMEOUT);
  	my $self = {
              out_peer	=> undef,
			  outsock	=> undef,
			  cam_state => {},
			  cam_ip	=> {},
			  cam_creds	=> {},
			  http_agent=> $http_agent
               };

	prepare_out_socket($self);

	log_debug ("socket -> peer=$self->{out_peer}(".(defined $self->{out_peer} ? 1 : 0)."), sock=$self->{outsock}");

	log_die ("failed to init output socket") unless $self->{outsock} || $self->{out_peer};

  	return defined $self->{outsock} && defined $self->{out_peer} ? bless($self, $class) : undef;
}

sub process_requests
{
	my $self = shift;
	my $camstate = 0;
	my $res = undef;

    while(1) 
	{
		$self->readconf();
		while (my ($camid, $wasstate) = each %{$self->{cam_state}}) 
		{
	    	#log_debug("process_req($camid),state=$wasstate, def?".(defined $wasstate ? 1 : 0));
			next unless defined $wasstate;
			$camstate = $self->get_state ($camid);
			log_debug ("ID=$camid, states: $wasstate -> $camstate");
			next unless defined $camstate;

			if ($camstate ne STATE_STREAMING ) { #&& $wasstate ne $camstate
        		$res = $self->ssend ("axis_video_lost objid=$camid");
            }
            elsif ($wasstate ne STATE_STREAMING && $camstate eq STATE_STREAMING ) {
                $res = $self->ssend ("axis_video_restored objid=$camid");
       		}
			$self->{cam_state}{$camid} = $camstate if defined $res ;
		}
    	sleep (MAIN_LOOP_SLEEP);

    } # while(1)
} #     process_requests


# ---------------------------------------------------------------- handle -----
#   handle($socket) deals with all pending requests for $client
sub get_state 
{
    my $self 	= shift;
    my $objid 	= shift;
	my $retval = undef;
	
	my ($ip, $port, $input) = ($self->{cam_ip}{$objid}, $self->{http_port}{$objid}, $self->{video_input}{$objid});
	my $url = "http://$ip:$port/axis-cgi/view/videostatus.cgi?status=$input";
	#$http_agent->credentials( $cam_ip{$objid},    'realm-name',    'root', 'pass'  );
	my $request = HTTP::Request->new(GET => $url);
    $request->authorization_basic($self->{cam_creds}{$objid}[0], $self->{cam_creds}{$objid}[1]);

    #log_debug("get_state($objid), URL=[$url]");

    my $resp = $self->{http_agent}->request($request);


    #log_debug("response: success=".$resp->is_success.", cont=[".$resp->content."]");

	if (! $resp->is_success || $resp->code ne 200) {
		#$self->{cam_state}{$objid} 	= undef;
		#$self->{cam_ip}{$objid} 	= undef;
	    $log->error("get_state($objid), failed: OKflag=".($resp->is_success ? 1 : 0).", ErrCode=(".$resp->code."), Message: ".$resp->message);
	    #log_debug("Will ignore $objid. To reinit send SIGHUP, state=".$self->{cam_state}{$objid});
		return $retval;
	}

	my $content = $resp->content;
	if ($content =~ /=\svideo/) 		{ $retval = STATE_STREAMING; }
	elsif ($content =~ /=\sno\svideo/) 	{ $retval = STATE_OFFLINE;	}
	else 								{ $retval = STATE_UNKNOWN;  }

    log_debug("get_state($objid)=$retval");

	return $retval;
}

sub ssend
{
	my $self = shift;
    my $text = shift;
	
	log_debug ("sending to socket -> [$text]");
    my $res = send ($self->{outsock}, $text."\n", 0, $self->{out_peer});
	$log->error ("send failed : $!") unless $res;
	return $res;
}


sub prepare_out_socket
{
	my ($self) = @_;

	my $socket_path = undef;

	# loading config with socket path
	die (CONF_FILE.' required file NOT FOUND') if ! -e CONF_FILE;
    print("loading config file ".CONF_FILE."\n");
	open CONF, '<'.CONF_FILE;
	while (<CONF>) {
		chomp;
		if (/SOCKET_NAME=(.+)/) { $socket_path = $1;  }
	}
	die ('SOCKET_NAME=<path> parameter required inside '.CONF_FILE) unless length $socket_path;

	# creating socket 
    $self->{outsock} = new IO::Socket::UNIX->new( Local => LOCAL_SOCK, 
                        					Type  => SOCK_DGRAM
                    ) or die("Couldn't open socket for path [$socket_path]: $!");

	$self->{out_peer} = sockaddr_un($socket_path);

    $log->warn("Ready to write in port $socket_path, peer=[$self->{out_peer}]");
    
    return $self->{outsock};
    #print("Writing [$command] to socket ...\n");
    #send ($server, $command."\n", 0, $peer) or die ("Send failed: $!");
	#print("OK\n");
}


# readconf
sub readconf
{
    log_debug ("start readconf()");   
    my $self = shift;
    
    my %conf = GetCfgs('DEVICETYPE'=>'CAMERA');
    
    # Update camera cache
    foreach my $dev (keys %conf) {
	my $cfg = $conf{$dev};
	# Skip everything except Axis cameras
	next unless $cfg->{CAMERAMODEL} =~ /^Axis$/i;
	# Skip deleted devices
	next if $cfg->{LOCATION} eq '@garbage';
	# !!! OBJID==DEVID for SKM>=2.6.1
	$self->{cam_state}{$dev} = STATE_UNKNOWN unless exists $self->{cam_state}{$dev};
	$self->{cam_ip}{$dev} = $cfg->{DEVIP};
	$self->{cam_creds}{$dev} = [$cfg->{USRNAME}, $cfg->{PASSWD}];
        $self->{video_input}{$dev} = $cfg->{CAMERA};
        $self->{http_port}{$dev} = $cfg->{HTTP_PORT};
    }
    # Wipe deleted devices from cache
    foreach my $dev (keys %{$self->{cam_state}}) {
	delete $self->{cam_state}{$dev} if 
	    not exists $conf{$dev} or 
	    $conf{$dev}{CAMERAMODEL}!~/^axis$/i or
	    $conf{$dev}{LOCATION} eq '@garbage';
    }
    log_debug ("readconf() done");   
}

sub reinit {
    $log->info("Reinit daemon");
    $Daemon->readconf();
}

# =============================================================================
# Main
sub main {
    $Daemon = AxisDaemon->new() or log_die("failed to create daemon instance");
    $Daemon->readconf();
    $Daemon->process_requests();
}

main;

END { unlink LOCAL_SOCK }
