#!/usr/bin/perl

use lib "/opt/sarch/common/lib";
use lib "/opt/sarch/fw/bin/perl/FwIface";
use lib "/opt/sarch/fw/bin/perl";
use lib "/opt/sarch/elog/bin/perl";
use lib "/opt/sarch/elog/bin/perl/ELogIface";

use strict;
use warnings;
use SKM::DB;

use Data::Dumper;

use NextCAM::Servicing::Client;
use FwIface::Types;
use ELog;

use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";

my $log = get_logger('NEXTCAM');
my $service_client = undef;
my $elog_client = undef;
my $store_path;
my $export_witnesses;
my $export_format;

sub run_shell_cmd
{
    my $shell_cmd = shift;
    
    # Usage: ionice [OPTIONS] [COMMAND [ARG]...]
    # Sets or gets process io scheduling class and priority.
    #   -n      Class data (typically 0-7, lower being higher prio)
    #   -c      Scheduling class
    #               1: realtime, 2: best-effort, 3: idle
    #   -p      Process pid
    #   -h      This help page
    
    # http://linux.die.net/man/1/ionice:
    # The priority within the best effort class will be dynamically derived from the cpu nice level of the process: io_priority = (cpu_nice + 20) / 5

    my $result = `nice -n 15 $shell_cmd 2>&1`;
    if ($? == -1) 
    {
        $log->error("Failed to execute '$shell_cmd': $! ($result)");
        return 0;
    }
    elsif ($? & 127) {
        my $err_str = sprintf "Child died with signal %d, %s coredump",
                            ($? & 127),  ($? & 128) ? 'with' : 'without';
        $log->error("$err_str ($result)");
        return 0;
    }
    elsif ($? >> 8 != 0)
    {
        my $err_str = sprintf "Child exited with value %d", $? >> 8;

        $log->error("$err_str ($result)");
        return 0;
    }
    
    $log->info("'$shell_cmd' executed successfuly");
    return 1;
}

sub do_export
{
    my $event = shift;
    my $dbh   = shift;
    my $audio_objid = 0;
    
    $log->info("Processing event: $event->{eventid}_$event->{utc_when}_$event->{objid}");

    if ($event->{associated})
    {
        # Try to find first audio associated device 
        my $query = "SELECT obj, name FROM _objs WHERE obj in ($event->{associated}) and name LIKE 'a%'";
        my @audio = $dbh->selectrow_array($query);
        if ($audio[0])
        {
            $log->info("Found audio obj: $audio[0]=>$audio[1]");
            $audio_objid = $audio[0];
        }
    }

    my $when = $event->{utc_when}; $when =~ s/:/-/g;
    my $out_filename = "$store_path/$event->{eventid}_${when}_$event->{objid}.$export_format";
    my $out_signed_filename = "$store_path/$event->{eventid}_$event->{utc_when}_$event->{objid}_signed.$export_format";
    my $export_cmd = "$ENV{APL}/mgears/bin/storage_export -m $event->{eventid} -v $event->{objid} -o '$out_filename'";

    if ($audio_objid) {
        $export_cmd .= " -a $audio_objid";
    }
    
    
    $log->info("Running $export_cmd");
    
    my $property_migrate = "2"; # success
    if (!run_shell_cmd($export_cmd)) {
        $property_migrate = "3"; # failed
    }
	
    if (not defined $service_client) {
        $service_client = NextCAM::Servicing::Client->new("s_master", 10000);
    }
    if (not defined $elog_client) {
        $elog_client = ELogClient->new($service_client->getProtocol("skm.eventlog"));
    }

    my $resp;
    my $action = Action->new({
                                  name       => "update",
                              parameters => {"eventid"           => $event->{eventid},
                                             "property.MIGRATE"  => $property_migrate
                                            }
                             });
    
    eval
    {
        $resp = $elog_client->submitActionA($action);
    };
    if ($@)
    {
        $log->error("FwIface::FwException: ". $@->{errorId}.": ".$@->{what}. "\n");
        $service_client = undef;
        $elog_client = undef;
    } 
}


while (1)
{
    my $dbh;
    my @arr_result;
    my $query;

    eval {
        $dbh = DBMaster({PrintError => 0, RaiseError => 1});

        # getting and checking path for store 
        $query = "SELECT val FROM _obj_attr WHERE attr='ELOG_EXPORT_LOC' and obj=53";
        @arr_result = $dbh->selectrow_array($query);
        $store_path = $arr_result[0];
        if ($store_path eq "/") {
           die("Root partition is not allowed for store exported data");   
        }

        # getting export format
        $query = "SELECT val FROM _obj_attr WHERE attr='MEDIA_EXPORT_FORMAT' and obj=53";
        @arr_result = $dbh->selectrow_array($query);
        $export_format = $arr_result[0];

        if ( ! `df $store_path 2>/dev/null`) {
            `echo '<?xml version="1.0" encoding="us-ascii"?><data><elog_migration>0</elog_migration></data>' > $ENV{APL_VAR}/elog/elog_migration.xml`;          
        #    die("$store_path is not mounted");
        }
        else {
            `echo '<?xml version="1.0" encoding="us-ascii"?><data><elog_migration>1</elog_migration></data>' > $ENV{APL_VAR}/elog/elog_migration.xml`;
        
            # getting export parameters
            $query = "SELECT val FROM _obj_attr WHERE attr='ELOG_EXPORT_MODE' and obj=53";
            @arr_result = $dbh->selectrow_array($query);
            my $val = $arr_result[0];
            if ($val eq "Originator and Witnesses") {
                $export_witnesses = 1;
            } else {
                $export_witnesses = 0;
            }
            $log->info("Export options: $val");

            $log->info("Using path for store exported data: $store_path");

            # getting events marked for migration
            $query = "SELECT DISTINCT event.eventid, event.utc_when, eventwitness_video.objid, _obj_attr_associate.val as associated
                      FROM event, eventproperty, eventwitness eventwitness_video,    _objs, _obj_attr, _obj_attr  _obj_attr_associate
                      WHERE eventproperty.name='MIGRATE' AND eventproperty.value='1' AND event.eventid=eventproperty.eventid
                      AND (_objs.obj=eventwitness_video.objid AND _objs.otype='D' AND _objs.subtype='C' 
                           AND _obj_attr.obj=_objs.obj AND _obj_attr.attr='MEDIA_FORMAT' AND _obj_attr.val != 'audio')
                      AND (_obj_attr_associate.obj=eventwitness_video.objid AND _obj_attr_associate.attr='ASSOCIATE')
                      AND eventwitness_video.eventid=event.eventid";

            if (!$export_witnesses) {
                $query .= " AND event.objid=eventwitness_video.objid";
            }

            my $events = $dbh->selectall_arrayref($query, { Slice => {} });
      
            foreach my $event (@$events) {
                do_export($event, $dbh);
            }

        }
    };
    $log->error("$@") if $@; 

    
    $dbh->disconnect() if $dbh; 
    
    sleep 60;    
}
