SET search_path TO apl;

CREATE TYPE QS_Event as
(
    eventid              integer,
    objid                integer, 
    eventsource          integer, 
    utc_when             bigint,
    utc_from             bigint,
    utc_to               bigint,
    priority             integer, 
    state                integer, 
    message              character varying,
    note                 character varying, 
    lifespan             integer,
    eventtype            integer,
    updated_at           float,
    eventproperty_name   character varying,
    eventproperty_value  character varying,
    bookmark_utc_when    bigint,
    bookmark_text        character varying
);

CREATE OR REPLACE FUNCTION qs_insert_event(_objid integer,
                                           _eventsource integer,
                                           _workflow integer,
                                           _state integer,
                                           _utc_when bigint,
                                           _utc_from bigint,
                                           _utc_to bigint,
                                           _priority integer,
                                           _lifespan integer,
                                           _eventtype integer,
                                           _note character varying,
                                           _message character varying) RETURNS integer AS
$$
BEGIN

INSERT INTO event (objid, eventsource, workflow, state, utc_when, utc_from, utc_to, priority, lifespan, eventtype, note, message)
VALUES (_objid, _eventsource, _workflow, _state,
        TIMESTAMP 'epoch' + _utc_when * INTERVAL '1 second',
        TIMESTAMP 'epoch' + _utc_from * INTERVAL '1 second',
        TIMESTAMP 'epoch' + _utc_to * INTERVAL '1 second',
        _priority, _lifespan, _eventtype, _note, _message);

RETURN (SELECT currval_seqeventid());

END;
$$ LANGUAGE 'plpgsql';

CREATE OR REPLACE FUNCTION qs_update_event(_eventid integer,
                                           _utc_from bigint,
                                           _utc_to bigint,
                                           _state integer,
                                           _priority integer,
                                           _lifespan integer,
                                           _note character varying,
                                           _message character varying) RETURNS void AS
$$
DECLARE 
   query  text;
   res    text;
BEGIN

query = '';

IF _utc_from IS NOT NULL THEN
   query = query || 'UPDATE event SET utc_from = TIMESTAMP ''epoch'' + ' || _utc_from || '* INTERVAL ''1 second'' WHERE eventid = ' || _eventid || ';';
END IF;

IF _utc_to IS NOT NULL THEN
   query = query || 'UPDATE event SET utc_to = TIMESTAMP ''epoch'' + ' || _utc_to || '* INTERVAL ''1 second'' WHERE eventid = ' || _eventid || ';';
END IF;

IF _state IS NOT NULL THEN
   query = query || 'UPDATE event SET state =' ||  _state || ' WHERE eventid = ' || _eventid || ';';
END IF;

IF _priority IS NOT NULL THEN
   query = query || 'UPDATE event SET priority =' ||  _priority || ' WHERE eventid = ' || _eventid || ';';
END IF;

IF _lifespan IS NOT NULL THEN
   query = query || 'UPDATE event SET lifespan =' ||  _lifespan || ' WHERE eventid = ' || _eventid || ';';
END IF;

IF _note IS NOT NULL THEN
   query = query || 'UPDATE event SET note =''' ||  _note || ''' WHERE eventid = ' || _eventid || ';';
END IF;

IF _message IS NOT NULL THEN
   query = query || 'UPDATE event SET message =''' ||  _message || ''' WHERE eventid = ' || _eventid || ';';
END IF;

IF query = '' THEN
   RETURN;
END IF;

--RAISE EXCEPTION '%', query;

EXECUTE query;

END;
$$ LANGUAGE 'plpgsql';
-- SELECT qs_update_event(-1, 1346233117, null, 0, null, null, null, null);

CREATE OR REPLACE FUNCTION qs_insert_or_update_property(_eventid integer,
                                                        _name character varying,
                                                        _value character varying) RETURNS void AS
$$
DECLARE 
   query  text;
   res    text;
BEGIN

query = 'UPDATE eventproperty SET value=''' || _value || ''' WHERE eventid=' || _eventid || ' AND name=''' || _name || ''';
      INSERT INTO eventproperty(eventid, name, value)   
                   SELECT ' || _eventid || ', ''' || _name || ''', ''' || _value || '''   
                   WHERE NOT EXISTS (SELECT 1 FROM eventproperty WHERE eventid=' || _eventid || ' AND name=''' || _name || ''');';

EXECUTE query;

END;
$$ LANGUAGE 'plpgsql';

CREATE OR REPLACE FUNCTION qs_insert_or_update_bookmark(_eventid integer,
                                                        _utc_when bigint,
                                                        _bookmark character varying) RETURNS void AS
$$
DECLARE 
   query  text;
   res    text;
BEGIN

query = 'UPDATE eventbookmark SET bookmark=''' || _bookmark || ''' WHERE eventid=' || _eventid || ' AND utc_when=TIMESTAMP ''epoch'' + ' || _utc_when || ' * INTERVAL ''1 second''; 
      INSERT INTO eventbookmark(eventid, utc_when, bookmark) 
                   SELECT ' || _eventid || ', TIMESTAMP ''epoch'' + ' || _utc_when || ' * INTERVAL ''1 second'', ''' || _bookmark || '''
                   WHERE NOT EXISTS (SELECT 1 FROM eventbookmark WHERE eventid=' || _eventid || ' AND utc_when=TIMESTAMP ''epoch'' + ' || _utc_when || ' * INTERVAL ''1 second'');';

EXECUTE query;

END;
$$ LANGUAGE 'plpgsql';

-- Returns set of rows with event info, properties and bookmarks
CREATE OR REPLACE FUNCTION qs_get_events(_eventids integer[]) RETURNS setof QS_Event AS
$$
DECLARE
   query  text;
   res    text;
   eventids_text text;
   result QS_Event;
BEGIN

IF _eventids IS NULL OR array_length(_eventids, 1) IS NULL THEN
   RETURN;
END IF;

   SELECT array_to_string(_eventids, ',') INTO eventids_text;
   eventids_text = 'ARRAY[' || eventids_text || ']';
   query = '
      SELECT  
               event.eventid,
               event.objid, 
               eventsource, 
               extract(''epoch'' from utc_when) + extract(timezone from current_time) as utc_when,
               extract(''epoch'' from utc_from) + extract(timezone from current_time) as utc_from,
               extract(''epoch'' from utc_to)   + extract(timezone from current_time) as utc_to, 
               priority, 
               state, 
               message, 
               note, 
               lifespan, 
               eventtype,
               extract(''epoch'' from updated_at) + extract(timezone from current_time) as updated_at,
               eventproperty.name as eventproperty_name, 
               eventproperty.value as eventproperty_value, 
               -- fields bellow used in next SELECT
               NULL as bookmark_utc_when, 
               NULL as bookmark_text 
    FROM event 
    LEFT JOIN eventproperty ON eventproperty.eventid=event.eventid
    WHERE event.eventid=ANY(' || eventids_text || ')' ||
    'UNION
    SELECT eventid, NULL, NULL, NULL as utc_when, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, -1 as updated_at, NULL, NULL,
           extract(''epoch'' from eventbookmark.utc_when) + extract(timezone from current_time) as bookmark_utc_when,
           bookmark as bookmark_text
    FROM eventbookmark
    WHERE eventid=ANY(' || eventids_text || ')
    ORDER BY updated_at DESC, utc_when DESC;';

    FOR result in
        EXECUTE query
    LOOP
    RETURN NEXT result;
    END LOOP;

END
$$ LANGUAGE 'plpgsql';

CREATE OR REPLACE FUNCTION qs_search_by_property(_roleid integer, _name character varying, _value character varying, _global_search integer) RETURNS setof QS_Event AS
$$
DECLARE
   query  text;
   res    text;
   eventids integer[];
   objects integer[];
   result QS_Event;
BEGIN

   -- getting objects belong to role
   SELECT array(SELECT DISTINCT obj from getobjectsfromrole(_roleid, 'D', 'C')) INTO objects;
   IF objects IS NULL OR array_length(objects, 1) IS NULL THEN
      objects = array_append(objects, 0);
   END IF;   

   SELECT array(SELECT DISTINCT event.eventid 
                  FROM eventproperty, event 
                  WHERE eventproperty.name=_name AND eventproperty.value=_value
                  AND event.eventsource=7 AND event.objid=ANY(objects) 
                  AND event.eventid=eventproperty.eventid)
    INTO eventids;

FOR result in
    SELECT * FROM qs_get_events(eventids)
LOOP
RETURN NEXT result;
END LOOP;

END
$$ LANGUAGE 'plpgsql';

CREATE OR REPLACE FUNCTION qs_search_full_text(_roleid integer, _text character varying, _global_search integer) RETURNS setof QS_Event AS
$$
DECLARE
   eveventids integer[];
   objects integer[];
   result QS_Event;
BEGIN

   -- getting objects belong to role
   SELECT array(SELECT DISTINCT obj from getobjectsfromrole(_roleid, 'D', 'C')) INTO objects;
   IF objects IS NULL OR array_length(objects, 1) IS NULL THEN
      objects = array_append(objects, 0);         
   END IF;
   
   SELECT array(
      SELECT DISTINCT event.eventid 
      FROM event
      LEFT JOIN eventproperty ON eventproperty.eventid=event.eventid
      LEFT JOIN eventbookmark ON eventbookmark.eventid=event.eventid
      WHERE (event.note  @@ to_tsquery(_text)
            OR event.message  @@ to_tsquery(_text)
            OR eventproperty.value @@ to_tsquery(_text)
            OR eventbookmark.bookmark @@ to_tsquery(_text)) AND event.eventsource=7 AND event.objid=ANY(objects))
   INTO eveventids;         

FOR result in
    SELECT * FROM qs_get_events(eveventids)
LOOP
RETURN NEXT result;
END LOOP;

END
$$ LANGUAGE 'plpgsql';

CREATE OR REPLACE FUNCTION qs_get_remote_rtsp_endpoint(_objid integer) RETURNS TEXT AS
$$
DECLARE
   result text;
BEGIN

SELECT _obj_attr.val || ':' || _obj_attr2.val 
FROM   _obj_attr, _obj_attr _obj_attr2, _objs node, _objs device 
WHERE  device.obj=_objid
 AND device.node_id=node.obj AND _obj_attr.attr='IP'
 AND _obj_attr2.attr='RTSP_PORT' AND _obj_attr.obj=device.node_id AND _obj_attr2.obj=device.node_id INTO result;

RETURN result;

END
$$ LANGUAGE 'plpgsql';

