-- ========== transdb ==========

SET search_path TO transdb;

-- this table will contain a log of updates for remote replication to AVATARs
CREATE TABLE eventrepllog (
    eventid     integer not null,
    oper        char,
	obj         integer,
    updated_at  timestamp without time zone not null,
    CONSTRAINT c_eventrepllog_operation CHECK (oper IN ('I','U','D'))
);
CREATE INDEX idx_eventrepllog_obj_upd_ts ON eventrepllog(updated_at,obj);
GRANT insert,delete,select ON eventrepllog TO apl;

CREATE OR REPLACE FUNCTION event_repl_log() RETURNS TRIGGER AS $event$
DECLARE
    op      char;
    evid    integer;
    upd_at  timestamp without time zone;
	objid   integer;
	cnt     integer;
BEGIN
    IF (TG_OP = 'DELETE') THEN
        evid = OLD.eventid;
        upd_at = OLD.updated_at;
		objid = OLD.objid;
		op = 'D';
    ELSE
        evid = NEW.eventid;
        upd_at = NEW.updated_at;
		objid = NEW.objid;
		IF(TG_OP = 'INSERT') THEN
			op = 'I';
		ELSE
			op = 'U';
		END IF;
    END IF;
    INSERT INTO eventrepllog VALUES(evid, op, objid, upd_at);

	SELECT nextval('eventrepllog_seq') INTO cnt;
    IF(cnt > 500) THEN -- purge once in 500 executions
		PERFORM setval('eventrepllog_seq',0);
		SELECT count(*) FROM eventrepllog INTO cnt;
		cnt = cnt - 10000; -- maintain 10_000 entries in replication log, purge the rest
		IF( cnt  > 0 ) THEN
			DELETE FROM eventrepllog WHERE ctid IN (SELECT ctid FROM eventrepllog ORDER BY updated_at LIMIT cnt);
		END IF;
    END IF;
    
	RETURN NULL; -- "everything is OK"
END;
$event$ LANGUAGE plpgsql;

CREATE TRIGGER trg_event_repl
    AFTER INSERT OR UPDATE OR DELETE ON event
    FOR EACH ROW EXECUTE PROCEDURE event_repl_log();

-- triggers for cascading "updated_at" update 

CREATE FUNCTION upd_timestamp_cascading_event() RETURNS TRIGGER AS $$
BEGIN
    IF (TG_OP = 'DELETE') THEN
	    UPDATE event SET updated_at=now() at time zone 'UTC' WHERE eventid=OLD.eventid;
		RETURN OLD;
	ELSE
	    UPDATE event SET updated_at=now() at time zone 'UTC' WHERE eventid=NEW.eventid;
		RETURN NEW;
	END IF;
END;
$$ LANGUAGE plpgsql;

CREATE TRIGGER trg_event_upd_ts_eventwitness BEFORE UPDATE ON eventwitness FOR EACH ROW EXECUTE PROCEDURE upd_timestamp_cascading_event();
CREATE TRIGGER trg_event_upd_ts_eventproperty BEFORE UPDATE ON eventproperty FOR EACH ROW EXECUTE PROCEDURE upd_timestamp_cascading_event();
CREATE TRIGGER trg_event_upd_ts_eventbookmark BEFORE UPDATE ON eventbookmark FOR EACH ROW EXECUTE PROCEDURE upd_timestamp_cascading_event();



-- ========== apl ==========

SET search_path TO apl;

-- re-create UPDATE rule to add "externalid", "workflow"
CREATE OR REPLACE RULE v_event_upd AS ON UPDATE TO event
    DO INSTEAD
    UPDATE transdb.event
 	SET
        utc_when = NEW.utc_when, 
        utc_from = NEW.utc_from,
        utc_to = NEW.utc_to,    
        priority = NEW.priority, 
        state = NEW.state, 
        message = NEW.message,
        note = NEW.note, 
        lifespan = NEW.lifespan,
        externalid = NEW.externalid,
        workflow = NEW.workflow
    WHERE eventid=OLD.eventid;

-- re-create INSERT rule to add "returning" clause
CREATE OR REPLACE RULE v_event_ins AS ON INSERT TO event
    DO INSTEAD
	INSERT INTO transdb.event VALUES (NEW.*)
	RETURNING *;

-- this is for transdb.trg_event_repl top operate.. very non-intuitive!
CREATE SEQUENCE eventrepllog_seq START WITH 0 MINVALUE 0 INCREMENT BY 1;
GRANT select,update ON eventrepllog_seq TO apl;
	
-- give access to replication log
CREATE OR REPLACE VIEW eventrepllog AS SELECT *,ctid FROM transdb.eventrepllog;
GRANT select,insert,delete ON eventrepllog TO apl;

CREATE OR REPLACE RULE v_eventrepllog_ins AS ON INSERT TO eventrepllog
 	DO INSTEAD
	INSERT INTO transdb.eventrepllog VALUES (NEW.eventid, NEW.oper, NEW.obj, NEW.updated_at);

CREATE OR REPLACE RULE v_eventrepllog_del AS ON DELETE TO eventrepllog
 	DO INSTEAD
	DELETE FROM transdb.eventrepllog WHERE updated_at=OLD.updated_at;

