SET search_path TO transdb;

CREATE TABLE camera_status (
    status      integer not null,
    description varchar,
    PRIMARY KEY (status)
);

--
-- Basic values for camera status
--

INSERT INTO camera_status (status, description) VALUES
(0, 'Online and streaming'),
(1, 'Turned off by user'),
(2, 'Turned off by scheduler'),
(3, 'Broken');


CREATE TABLE camera_downtime (
    objid	integer not null,
    utc_from	timestamp without time zone not null,
    utc_to		timestamp without time zone,
    status	integer,
    PRIMARY KEY (objid, utc_from),
    FOREIGN KEY (status) REFERENCES camera_status,
    CONSTRAINT c_interval CHECK (utc_from < utc_to)
);

CREATE INDEX idx_camera_downtime_objid ON camera_downtime(objid);

--
-- Views and rules
--
SET search_path TO apl;
CREATE OR REPLACE VIEW camera_status as select * from transdb.camera_status;
-- INSERT, UPDATE and DELETE forbidden
CREATE OR REPLACE RULE v_camstatus_ins AS ON INSERT TO camera_status
        DO INSTEAD NOTHING;
CREATE OR REPLACE RULE v_camstatus_upd AS ON UPDATE TO camera_status
 	DO INSTEAD NOTHING;
CREATE OR REPLACE RULE v_camstatus_del AS ON DELETE TO camera_status
        DO INSTEAD NOTHING;

CREATE OR REPLACE VIEW camera_downtime AS SELECT * FROM transdb.camera_downtime;
CREATE OR REPLACE RULE v_cdt_ins AS ON INSERT TO camera_downtime
        DO INSTEAD INSERT INTO transdb.camera_downtime VALUES (NEW.*);
CREATE OR REPLACE RULE v_cdt_upd AS ON UPDATE TO camera_downtime
    DO INSTEAD 
	UPDATE transdb.camera_downtime
	SET
	    utc_to=NEW.utc_to,
	    status=NEW.status
	WHERE objid=OLD.objid AND utc_from=OLD.utc_from;
CREATE OR REPLACE RULE v_cdt_del AS ON DELETE TO camera_downtime
    DO INSTEAD
        DELETE FROM transdb.camera_downtime
        WHERE objid=OLD.objid AND utc_from=OLD.utc_from;
--
-- Public interface
--
CREATE TYPE downtime_period AS (objid integer, ts_from bigint, duration integer, status integer);

CREATE OR REPLACE FUNCTION getCameraDowntimeSummary(objids integer[], ts_from bigint, ts_to bigint, granularity varchar) RETURNS SETOF downtime_period AS $$
DECLARE
    result downtime_period;
    row camera_downtime%ROWTYPE;
    u_from bigint;
    u_to bigint;
    ol_from bigint;
    ol_to bigint;
    inter interval;
    ts_ol_from timestamp without time zone;
    ts_ol_to timestamp without time zone;
    ts_loop_from timestamp without time zone;
    ts_loop_to timestamp without time zone;
    ts_loop timestamp without time zone;
    ts_loop_ol_from timestamp without time zone;
    ts_loop_ol_to timestamp without time zone;
BEGIN
    result := (0, 0, 0, 0);
    inter := ('1 ' || granularity)::interval;
    FOR row IN 
        SELECT * FROM camera_downtime
	WHERE objids @> array[objid] AND extract('epoch' from coalesce(utc_to,now() at time zone 'UTC')) > ts_from 
	    AND extract('epoch' from utc_from) < ts_to
	ORDER BY utc_from ASC
    LOOP
	u_from := extract('epoch' from row.utc_from)::int;
	u_to := extract('epoch' from coalesce(row.utc_to,now() at time zone 'UTC'))::int;
	-- Find overlap of given interval and current one
	IF u_from < ts_from THEN ol_from := ts_from; ELSE ol_from := u_from; END IF;
	IF u_to > ts_to THEN ol_to := ts_to; ELSE ol_to := u_to; END IF;
	-- convert to timestamp
	ts_ol_from = timestamp 'epoch' + ol_from * interval '1 second';
	ts_ol_to = timestamp 'epoch' + ol_to * interval '1 second';
	ts_loop_from := date_trunc(granularity, timestamp 'epoch' + ol_from * interval '1 second');
	ts_loop_to := date_trunc(granularity, timestamp 'epoch' + ol_to * interval '1 second') + inter;
	
	-- inner loop
	ts_loop := ts_loop_from;
	WHILE ts_loop < ts_loop_to
	LOOP
	    IF (ts_loop, ts_loop + inter) overlaps (ts_ol_from, ts_ol_to)
	    THEN
		-- Find interval overlap and compute number of seconds in that overlap
		IF ts_loop < ts_ol_from 
		THEN 
		    ts_loop_ol_from := ts_ol_from; 
		ELSE 
		    ts_loop_ol_from := ts_loop; 
		END IF;
		IF ts_loop + inter > ts_ol_to 
		THEN 
		    ts_loop_ol_to := ts_ol_to; 
		ELSE 
		    ts_loop_ol_to := ts_loop + inter; 
		END IF;
		-- Found resulting interval, add it to function result
		result.objid := row.objid;
		result.ts_from := extract(epoch from ts_loop)::int;
		result.duration := extract(epoch from ts_loop_ol_to - ts_loop_ol_from)::int;
		result.status := row.status;
		RETURN NEXT result;
	    END IF;
	    ts_loop := ts_loop + inter;
	END LOOP;
	--RETURN NEXT result;
    END LOOP;
RETURN;
END
$$ LANGUAGE 'plpgsql';

-- Grant permissions
--
GRANT select ON camera_status to apl;
GRANT select,insert,update,delete ON camera_downtime to apl;
