SET search_path TO transdb;

-- List of chunks migrated to Cloud storage
-- e.g. (12345,101,102-01-131028120000.mjpg,online)
--
CREATE TABLE cloud_chunks (
	eventid		integer 	not null,
	objid		integer 	not null,
	chunkid 	varchar 	not null,
	location	varchar		not null default 'online'
);

CREATE INDEX idx_cloud_chunks_eventid ON cloud_chunks(eventid);

-- Tasks for cloud storage cleanup for sm_cloud_wipe
--
CREATE TABLE cloud_wipe_tasks (
	eventid		integer		not null,
	created_at	timestamp without time zone	default (now() at time zone 'UTC')
);
CREATE UNIQUE INDEX idx_cloud_wipe_tasks_eventid ON cloud_wipe_tasks(eventid);

-- Tasks for local storage cleanup for sm_wipe
-- e.g. ('{101,102,123}', 2013-10-05 22:00)
--
CREATE TABLE wipe_tasks (
	objids 		integer[] 	not null, -- array of object IDs
	dt_hour 	timestamp without time zone	not null -- hour-truncated date
);


SET search_path TO apl;

-- Create helper function and aggregate for wipe_tasks
-- Aggregate is used for creating arrays of objids without duplicates
--
CREATE OR REPLACE function array_append_nodup (a integer[], num integer) returns integer[] as $$
DECLARE
    res integer[];
BEGIN
    IF array[num] <@ a THEN
	res = a;
    ELSE
	res := a || num;
    END IF;
    return res;
END;
$$ LANGUAGE 'plpgsql';

CREATE AGGREGATE array_agg_nodup (integer) (
    SFUNC = array_append_nodup,
    STYPE = integer[],
    INITCOND = '{}'
);

-- Create views and rules in 'apl' schema
--
-- cloud_chunks
--
CREATE OR REPLACE VIEW cloud_chunks as select * from transdb.cloud_chunks;

CREATE OR REPLACE RULE v_cloud_chunks_ins AS ON INSERT TO cloud_chunks
    DO INSTEAD
	INSERT INTO transdb.cloud_chunks VALUES (NEW.*);

CREATE OR REPLACE RULE v_cloud_chunks_upd AS ON UPDATE TO cloud_chunks
    DO INSTEAD 
	UPDATE transdb.cloud_chunks
	SET
	    location=NEW.location
	WHERE eventid=OLD.eventid AND objid=OLD.objid AND chunkid=OLD.chunkid;

CREATE OR REPLACE RULE v_cloud_chunks_del AS ON DELETE TO cloud_chunks
 	DO INSTEAD
	DELETE FROM transdb.cloud_chunks WHERE eventid=OLD.eventid AND objid=OLD.objid AND chunkid=OLD.chunkid;

ALTER VIEW cloud_chunks ALTER COLUMN location SET DEFAULT 'online';

-- cloud_wipe_tasks
--
CREATE OR REPLACE VIEW cloud_wipe_tasks as select * from transdb.cloud_wipe_tasks;

CREATE OR REPLACE RULE v_cloud_wipe_tasks_ins AS ON INSERT TO cloud_wipe_tasks
    DO INSTEAD
	INSERT INTO transdb.cloud_wipe_tasks VALUES (NEW.*);

CREATE OR REPLACE RULE v_cloud_wipe_tasks_del AS ON DELETE TO cloud_wipe_tasks
 	DO INSTEAD
	DELETE FROM transdb.cloud_wipe_tasks WHERE eventid = OLD.eventid;


-- wipe_tasks
--
CREATE OR REPLACE VIEW wipe_tasks as select * from transdb.wipe_tasks;

CREATE OR REPLACE RULE v_wipe_tasks_ins AS ON INSERT TO wipe_tasks
    DO INSTEAD
	INSERT INTO transdb.wipe_tasks VALUES (NEW.*);

CREATE OR REPLACE RULE v_wipe_tasks_del AS ON DELETE TO wipe_tasks
 	DO INSTEAD
	DELETE FROM transdb.wipe_tasks WHERE dt_hour = old.dt_hour;
	

-- Grant permissions
--
GRANT select,insert,update,delete ON cloud_chunks to apl;
GRANT select,insert,update,delete ON wipe_tasks to apl;
GRANT select,insert,update,delete ON cloud_wipe_tasks to apl;
