------------------------------------------------------------------
--  GISmo module DB Table creation 	--
--  object's in 'transdb' schema					--
------------------------------------------------------------------

--
-- Set current schema
-- Following objects will be assigned to it
--
-- Here goes 'private' section.
-- Tables, sequences, associated triggers should be placed
-- into 'confdb' or 'transdb' schemas
--
SET search_path TO transdb;

--
-- First create sequence for primary key field of our table
--
CREATE SEQUENCE seq_gismo_data start 100;

-- Lookup table (contains foreign keys for out main table)
--
CREATE TABLE gismo_type (
    typeid char(1) NOT NULL CONSTRAINT pk_gismo_type PRIMARY KEY,
    name   varchar NOT NULL
);

INSERT INTO gismo_type (typeid, name) VALUES('g', 'gismo_type');

-- Main table
--
CREATE TABLE gismo_data (
    id      integer DEFAULT nextval('seq_gismo_data') CONSTRAINT pk_gismo_data PRIMARY KEY,
--    udid    varchar(22) DEFAULT NULL UNIQUE,
--    name    varchar(255) NOT NULL default '',
    type    char(1) NOT NULL REFERENCES gismo_type(typeid) ON DELETE CASCADE,
    stime   timestamp without time zone default (now() at time zone 'UTC'),
    deleted integer default 0,
 	--    
	time bigint,
	userId text,
	userName text,
	objId text,
	sessionId text,
	cameraFovX text,
	cameraFovY text,
	cameraTiltAngel text,
	cameraAzimuthAngle text,
	cameraZoom text,
	cameraFocus text,
	cameraRangeFinderValue text,
	cameraLatitude text,
	cameraLongitude text,
	cameraHeight text,
	cameraResolutionWidth text,
	cameraResolutionHeight text,
	imageX text,
	imageY text,
	targetId text,
	targetSymbolId text,
	targetSymbolDescr text,
	targetLatitude text,
	targetLongitude text,
	targetAltitude text,
	targetDistance text,
	targetType text,
	targetSpeed text,
	targetCourse text,
	targetSource text,
	targetElEventId integer NOT NULL,
	constraint FK_GISMO_EVENT foreign key (targetElEventId) references event(eventid)
	
	--    
 );

CREATE INDEX idx_gismo_data_eventid ON gismo_data(targetElEventId);
 	
-- Trigger
-- Resides in private namespace ('transdb' schema)
--
CREATE OR REPLACE FUNCTION gismo_upd_timestamp() RETURNS TRIGGER AS $$
BEGIN
    if OLD.stime = NEW.stime then
        NEW.stime := now() at time zone 'UTC';
    end if;
    return NEW;
END;
$$ LANGUAGE plpgsql;

CREATE TRIGGER trg_gismo_data_upd_time BEFORE UPDATE
    ON gismo_data FOR EACH ROW
    EXECUTE PROCEDURE gismo_upd_timestamp();


--
-- Here goes 'public' section
-- Following objects will be accessible to apl user and should be placed into 'apl' schema
-- 1. Views
--    1a. associated rules
-- 2. API functions 
--    2a. associated composite types
--    2b. associated aggregates
--
SET search_path TO apl;

--
-- Create views and rules for lookup table
-- Assume that it contains static data that cannot be changed after installation
-- Create rules for INSERT, UPDATE and DELETE that does nothing. Only SELECT is allowed to apl
--
CREATE OR REPLACE VIEW gismo_type as select * from transdb.gismo_type;
CREATE OR REPLACE RULE v_gismo_type_ins AS ON INSERT TO gismo_type
    DO INSTEAD NOTHING;
CREATE OR REPLACE RULE v_gismo_type_upd AS ON UPDATE TO gismo_type
    DO INSTEAD NOTHING;
CREATE OR REPLACE RULE v_gismo_type_del AS ON DELETE TO gismo_type
 	DO INSTEAD NOTHING;

--
-- Create views and rules for main table
-- INSERT/UPDATE/DLETE is allowed to apl
--
CREATE OR REPLACE VIEW gismo_data as select * from transdb.gismo_data;
CREATE OR REPLACE RULE v_gismo_data_ins AS ON INSERT TO gismo_data
    DO INSTEAD
	INSERT INTO transdb.gismo_data VALUES (NEW.*);
CREATE OR REPLACE RULE v_gismo_data_upd AS ON UPDATE TO gismo_data
 	DO INSTEAD
 	UPDATE transdb.gismo_data
 	SET
-- 	    udid=NEW.udid,
-- 	    name=NEW.name,
 	    deleted=NEW.deleted,
 	    stime=NEW.stime,  
--
	time=NEW.time,
	userId=NEW.userId,
	userName=NEW.userName,
	objId=NEW.objId,
	sessionId=NEW.sessionId,
	cameraFovX=NEW.cameraFovX,
	cameraFovY=NEW.cameraFovY,
	cameraTiltAngel=NEW.cameraTiltAngel,
	cameraAzimuthAngle=NEW.cameraAzimuthAngle,
	cameraZoom=NEW.cameraZoom,
	cameraFocus=NEW.cameraFocus,
	cameraRangeFinderValue=NEW.cameraRangeFinderValue,
	cameraLatitude=NEW.cameraLatitude,
	cameraLongitude=NEW.cameraLongitude,
	cameraHeight=NEW.cameraHeight,
	cameraResolutionWidth=NEW.cameraResolutionWidth,
	cameraResolutionHeight=NEW.cameraResolutionHeight,
	imageX=NEW.imageX,
	imageY=NEW.imageY,
	targetId=NEW.targetId,
	targetSymbolId=NEW.targetSymbolId,
	targetSymbolDescr=NEW.targetSymbolDescr,
	targetLatitude=NEW.targetLatitude,
	targetLongitude=NEW.targetLongitude,
	targetAltitude=NEW.targetAltitude,
	targetDistance=NEW.targetDistance,
	targetType=NEW.targetType,
	targetSpeed=NEW.targetSpeed,
	targetCourse=NEW.targetCourse,
	targetSource=NEW.targetSource,
	targetElEventId=NEW.targetElEventId
-- 	    
 	WHERE id=OLD.id AND deleted=OLD.deleted;
CREATE OR REPLACE RULE v_gismo_data_del AS ON DELETE TO gismo_data
 	DO INSTEAD
	DELETE FROM transdb.gismo_data 
	WHERE id=OLD.id AND deleted=OLD.deleted;
--
-- Assign default values to view cols as the original table has
--
--ALTER VIEW gismo_data ALTER COLUMN udid SET DEFAULT NULL;
--ALTER VIEW gismo_data ALTER COLUMN name SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN stime SET DEFAULT (now() at time zone 'UTC');
ALTER VIEW gismo_data ALTER COLUMN deleted SET DEFAULT 0;

ALTER VIEW gismo_data ALTER COLUMN time SET DEFAULT 0;
ALTER VIEW gismo_data ALTER COLUMN userId SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN userName SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN objId SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN sessionId SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN cameraFovX SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN cameraFovY SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN cameraTiltAngel SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN cameraAzimuthAngle SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN cameraZoom SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN cameraFocus SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN cameraRangeFinderValue SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN cameraLatitude SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN cameraLongitude SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN cameraHeight SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN cameraResolutionWidth SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN cameraResolutionHeight SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN imageX SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN imageY SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN targetId SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN targetSymbolId SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN targetSymbolDescr SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN targetLatitude SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN targetLongitude SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN targetAltitude SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN targetDistance SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN targetType SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN targetSpeed SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN targetCourse SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN targetSource SET DEFAULT '';
ALTER VIEW gismo_data ALTER COLUMN targetElEventId SET DEFAULT 0;

--
-- Create replacement views for sequences
-- For now algorythm is as follows:
--
-- 1. Create view with a single field of integer type
-- 2. Create ON SELECT rule for that view which returns currval('your_private_sequence');
-- 3. Create ON UPDATE rule which calls nextval('your_private_sequence');
-- 4. Create public proxy functions that replaces calls to currval and nextval of original sequence
--    E.g. nextval_seq_extable and currval_seq_extable
CREATE OR REPLACE VIEW v_seq_gismo_data (currval) AS SELECT 0::bigint;

CREATE OR REPLACE RULE "_RETURN" AS ON SELECT TO v_seq_gismo_data DO INSTEAD
    SELECT currval('transdb.seq_gismo_data');
CREATE OR REPLACE RULE v_seq_gismo_data_nextval AS ON UPDATE TO v_seq_gismo_data
DO INSTEAD
    SELECT nextval('transdb.seq_gismo_data');

CREATE OR REPLACE FUNCTION nextval_seq_gismo_data() RETURNS int AS $$
DECLARE
    val bigint;
BEGIN
    UPDATE v_seq_gismo_data SET currval=DEFAULT;
    SELECT currval FROM v_seq_gismo_data INTO val;
    RETURN val;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION currval_seq_gismo_data() RETURNS int AS $$
DECLARE
    val bigint;
BEGIN
    SELECT currval FROM v_seq_gismo_data INTO val;
    RETURN val;
END;
$$ LANGUAGE plpgsql;

-- Now we can set default value of the 'id' column of our view to proxy function return value
--
ALTER VIEW gismo_data ALTER COLUMN id SET DEFAULT nextval_seq_gismo_data();

--
-- Create public routines and associated composite types
--
CREATE TYPE gismo_ctype AS (id integer, udid varchar, name varchar, typename varchar);

CREATE OR REPLACE FUNCTION getByType(typeid char(1)) RETURNS SETOF gismo_ctype AS $$
DECLARE
   result gismo_ctype;
BEGIN
    FOR result in
        SELECT e.id, e.udid, e.name, et.name
            FROM gismo_data e INNER JOIN gismo_type et ON e.type=et.typeid
            WHERE
        	et.typeid=typeid
            ORDER BY e.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- 
-- Finally, grant permissions on newly created views
-- DDL is forbidden for apl
-- Grant select only on lookup tables and select/insert/update/delete on others
--
GRANT select ON gismo_type to apl;
GRANT select,insert,update,delete ON gismo_data to apl;
--
-- Grant permissions on sequences
-- We must grant apl select and update on transdb original sequences, too
--
GRANT select,update ON v_seq_gismo_data TO apl;
GRANT select,update ON transdb.seq_gismo_data TO apl;