--DROP FUNCTION el_get_event(integer);
--DROP FUNCTION el_merge_witnesses(integer, integer, integer[]);
--DROP FUNCTION el_insert_event(integer, integer, integer, integer, bigint, bigint, bigint, integer, integer, integer, character varying);
--DROP FUNCTION el_update_event(bigint, bigint, integer, integer, integer, character varying, character varying, integer);
--DROP FUNCTION el_get_event_for_merge(integer, integer);
--DROP FUNCTION el_insert_property(integer, character varying, character varying);

--DROP TYPE EL_Event;

--DROP TABLE event;
--DROP TABLE eventproperty;
--DROP TABLE eventwitness;

--DROP SEQUENCE seqEventId;



CREATE SEQUENCE seqEventId;

CREATE TABLE event (
	eventid			integer		not null default nextval('seqEventId'::text),
	objid			integer		not null,
	eventsource		integer		not null,
	utc_when		timestamp without time zone 		not null,
	utc_from		timestamp without time zone 		not null,
	utc_to			timestamp without time zone 		not null,
	eventtype		integer     not null,
	priority		integer		not null,
	priority_systhld			integer,
	workflow		integer		not null,
	state			integer		not null,
	message			varchar,
	note			varchar,
	history			varchar,
	lifespan		integer		not null default 1,
	lifespan_systhld	integer,
	updated_at		timestamp without time zone default	('now'::text)::timestamp(6) without time zone,
	--
	CONSTRAINT c_event_dates1 CHECK (utc_from < utc_to),
	CONSTRAINT c_event_dates2 CHECK (utc_from <= utc_when),
	CONSTRAINT c_event_dates3 CHECK (utc_when <= utc_to)
);
CREATE UNIQUE INDEX idx_pk_event ON event(eventid);
CREATE INDEX idx_event_utc_from ON event(utc_from);
CREATE INDEX idx_event_utc_to ON event(utc_to);
CREATE INDEX idx_event_objid_ls ON event(objid, lifespan);


CREATE TRIGGER trg_event_upd_time BEFORE UPDATE ON event FOR EACH ROW EXECUTE PROCEDURE upd_timestamp();


CREATE TABLE eventwitness (
	eventid			integer		not null,
	objid			integer		not null
);
CREATE UNIQUE INDEX idx_pk_eventwitness ON eventwitness(eventid, objid);



CREATE TABLE eventproperty (
	eventid			integer		not null,
	name			varchar		not null,
	value			varchar
);
CREATE UNIQUE INDEX idx_pk_eventproperty ON eventproperty(eventid, name);




CREATE TYPE EL_Event as
(
    objid                integer, 
    eventsource          integer, 
    utc_when             bigint,
    utc_from             bigint,
    utc_to               bigint,
    priority             integer, 
    state                integer, 
    message              character varying,
    note                 character varying, 
    lifespan             integer,
    eventtype            integer,
    eventproperty_name   character varying,
    eventproperty_value  character varying,
    witness_objid        integer,
    witness_name         character varying(255),
    witness_otype        character(1), 
    witness_subtype      character(1),
    witness_timedst      character varying,
    witness_tz           character varying
);


-- Returns set of rows with event info, properties and witnesses 
CREATE OR REPLACE FUNCTION el_get_event(integer) RETURNS setof EL_Event AS
$$
DECLARE 
   result EL_Event;
BEGIN

FOR result in
    SELECT  
               event.objid, 
               eventsource, 
               extract('epoch' from utc_when) + extract(timezone from current_time),
               extract('epoch' from utc_from) + extract(timezone from current_time),
               extract('epoch' from utc_to)   + extract(timezone from current_time), 
               priority, 
               state, 
               message, 
               note, 
               lifespan, 
               eventtype,
               eventproperty.name, 
               eventproperty.value, 
               -- fields bellow used in next SELECT
               NULL as witness_objid, 
               NULL as witness_name, 
               NULL as witness_otype, 
               NULL as witness_subtype, 
               NULL as witness_timedst, 
               NULL as witness_tz 
    FROM event 
    LEFT JOIN eventproperty ON eventproperty.eventid=event.eventid
    WHERE event.eventid=$1
    UNION
    SELECT NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
              eventwitness.objid, 
              _objs.name, 
              otype, 
              subtype, 
              _obj_attr.val,
              0       
    FROM event, _objs, eventwitness, _obj_attr
    WHERE event.eventid=$1 AND eventwitness.eventid=event.eventid AND _objs.obj = eventwitness.objid
      AND _obj_attr.obj=eventwitness.objid AND _obj_attr.attr='TIME_ZONE'
    ORDER BY objid ASC
LOOP
RETURN NEXT result;
END LOOP;
 
RETURN;

END
$$ LANGUAGE 'plpgsql';


-- Merge passed witnesses with ASSOCIATED objects and returns set of rows with witness info (if any)
CREATE OR REPLACE FUNCTION el_merge_witnesses(integer, integer, integer[]) RETURNS setof EL_Object_Info AS
$$
DECLARE 
   event_id         ALIAS FOR $1;
   obj_id           ALIAS FOR $2;
   witnesses_       ALIAS FOR $3;
   witnesses        integer[];
   merged_witnesses text[]; 
   result           EL_Object_Info;
BEGIN

-- postgres 8.1.x requires this
IF witnesses_ IS NULL THEN
   witnesses = ARRAY[obj_id];
ELSE
   witnesses = witnesses_;
END IF;

merged_witnesses := (SELECT COALESCE(
                              string_to_array(replace(val, ' ', ''), ','), -- just for case removing spaces between comas
                              CAST(ARRAY[obj_id] as text[]) -- if previous expression is NULL keep array of original obj_id. 
                            ) || CAST(witnesses AS text[]) || CAST(ARRAY[obj_id] as text[]) -- adding objid of event to witnesses too
                    FROM _obj_attr 
                    WHERE obj=obj_id AND attr='ASSOCIATE');

INSERT INTO eventwitness 
(
   SELECT DISTINCT 
      event_id,
      CAST (merged_witnesses[i] AS integer)
   FROM generate_series(1, array_upper(merged_witnesses, 1)) as i
);

IF FOUND THEN

  FOR result in
      SELECT o.obj, o.name, o.otype, o.subtype, COALESCE(a1.val, 'UTC') as tz, 0, COALESCE(a2.val, '0') as vl_rate
        FROM eventwitness w
       INNER JOIN _objs o ON o.obj = w.objid
        LEFT OUTER JOIN _obj_attr a1 ON a1.obj=o.obj AND a1.attr='TIME_ZONE'
        LEFT OUTER JOIN _obj_attr a2 ON a2.obj=o.obj AND a2.attr='STAT_VL_RATE_LAST'
       WHERE w.eventid = event_id
  LOOP
     RETURN NEXT result;
  END LOOP;
END IF;

RETURN;

END
$$ LANGUAGE 'plpgsql';



CREATE OR REPLACE FUNCTION el_insert_event(integer, integer, integer, integer, bigint, bigint, bigint, integer, integer, integer, character varying) RETURNS integer AS
$$
BEGIN

INSERT INTO event (objid, eventsource, workflow, state, utc_when, utc_from, utc_to, priority, lifespan, eventtype, message)
VALUES ($1, $2, $3, $4, 
        TIMESTAMP 'epoch' + $5 * INTERVAL '1 second',
        TIMESTAMP 'epoch' + $6 * INTERVAL '1 second',
        TIMESTAMP 'epoch' + $7 * INTERVAL '1 second',
        $8, $9, $10, $11);

RETURN (SELECT currval('seqEventId'));

END;
$$ LANGUAGE 'plpgsql';

CREATE OR REPLACE FUNCTION el_update_event(bigint, bigint, integer, integer, integer, character varying, character varying, integer) RETURNS void AS
$$
BEGIN

UPDATE event SET 
          utc_from = TIMESTAMP 'epoch' + $1 * INTERVAL '1 second',
          utc_to   = TIMESTAMP 'epoch' + $2 * INTERVAL '1 second',
          state    = $3, 
          priority = $4, 
          lifespan = $5, 
          message  = $6, 
          note     = $7 
          WHERE eventid = $8;

END;
$$ LANGUAGE 'plpgsql';

-- Returns event id of candidate to merge
CREATE OR REPLACE FUNCTION el_get_event_for_merge(integer, integer) RETURNS integer AS
$$
DECLARE 
   obj_id  ALIAS FOR $1; 
   event_id integer;
BEGIN

   event_id := (SELECT MAX(eventid) FROM event WHERE objid=$1 AND eventsource=$2);
   RETURN event_id;

END;
$$ LANGUAGE 'plpgsql';


CREATE OR REPLACE FUNCTION el_insert_property(integer, character varying, character varying) RETURNS void AS
$$
BEGIN

INSERT INTO eventproperty(eventid, name, value) VALUES($1, $2, $3);

END;
$$ LANGUAGE 'plpgsql';
