SET search_path TO confdb;
-- Set UDID=null on delete, Set UDID=null if UDID=''
CREATE OR REPLACE FUNCTION SET_UDID_ON_DELETE_UPDATE() RETURNS TRIGGER AS $$
BEGIN
    NEW.udid = upper(NEW.udid);
    if NEW.deleted = 1 then
        NEW.udid := null;
    end if;
    if NEW.udid = '' then
        NEW.udid := null;
    end if;
    return NEW;
END;
$$ LANGUAGE plpgsql;

DROP TRIGGER IF EXISTS objs_TBU ON _objs;
CREATE TRIGGER objs_TBU BEFORE update
    ON _objs FOR EACH ROW
    EXECUTE PROCEDURE SET_UDID_ON_DELETE_UPDATE();

-- write attr from _obj_attr to _objs on update
CREATE OR REPLACE FUNCTION updateAttributes() RETURNS TRIGGER AS $$
BEGIN
    if NEW.attr = 'NAME' then
        UPDATE _objs
            SET name = NEW.val
            WHERE obj = NEW.obj;
    end if;
    if NEW.attr = 'LOCATION' then
        UPDATE _objs
            SET location = NEW.val
            WHERE obj = NEW.obj;
    end if;
    if NEW.attr = 'DESCRIPTION' then
        UPDATE _objs
            SET description = NEW.val
            WHERE obj = NEW.obj;
    end if;
    if NEW.attr = 'UDID' then
        NEW.val = upper(NEW.val);
        if NEW.val = '' then
            NEW.val = null;
        end if;
        UPDATE _objs
            SET udid = NEW.val
            WHERE obj = NEW.obj;
    end if;
    NEW.val = COALESCE(NEW.val, '');
    return NEW;
END;
$$ LANGUAGE plpgsql;

-- write udid to _objs
CREATE OR REPLACE FUNCTION updateUDID() RETURNS TRIGGER AS $$
BEGIN
    -- do not write udid for nodes
    if NOT (NEW.otype = 'D' AND NEW.subtype = 'N') then
        if COALESCE(NEW.udid, '') = '' then
            NEW.udid = upper(NEW.otype || NEW.subtype || NEW.obj);
        end if;
    end if;
    return NEW;
END;
$$ LANGUAGE plpgsql;

UPDATE _objs SET udid = null WHERE deleted = 1;

-- add avatar to set link type
INSERT INTO _link_types VALUES ('V2S', 'V', 'S', False, True,  'Avatar to Set');

-- add set for avatars
INSERT INTO _objs (obj, otype, subtype, protected) values (18, 'S', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (18, 'NAME', 'All Avatars');
INSERT INTO _obj_attr (obj, attr, val) values (18, 'DESCRIPTION', 'All Avatars');

-- set default avatar set
UPDATE _obj_otype_subtype SET defaultsetobj = 18 WHERE otype = 'V' AND subtype = '*';

-- "Avatar" -> "All Avatars"
INSERT INTO _links
    SELECT
        _objs.obj as obj_res, 18 as obj_cons, 1 as protected
        FROM _objs
        WHERE
            _objs.otype = 'V' AND _objs.subtype = '*';

-- add "All Avatars" to "ADMIN ROLE"
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (18, 31, 1, 'M');

-- add special credential for changing avatar profile
INSERT INTO _cred_types VALUES ('x', 'V', '*', 'Avatar Profile', 'Change Avatar Profile');
-- add special credential for changing avatar profile to Manage permission
UPDATE permission_type SET credentials = 'LVvApaeDMrCSPsmx' WHERE permission = 'M';


SET search_path TO apl;
-- correct getting objects from set (use any type to set, not only D2S, X2S, G2S)

-- select object credentials for user
CREATE OR REPLACE FUNCTION getObject(userobj integer, objectobj integer) RETURNS SETOF object_list_with_credentials AS $$
DECLARE
   result object_list_with_credentials;
BEGIN
    FOR result in
        SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected, concatFields(set2role.permission) as permission, concatFields(permission_type.credentials || set2role.special_credentials) as credentials
            FROM _objs, permission_type, _links as set2role, _links as user2role, _links as object2set
            WHERE
                set2role.link_type = 'S2R'
                AND user2role.link_type = 'U2R'
                AND object2set.link_type LIKE '_2S'
                AND _objs.obj = object2set.obj_res
                AND set2role.obj_cons = user2role.obj_cons
                AND object2set.obj_cons = set2role.obj_res
                AND set2role.permission = permission_type.permission
                AND user2role.obj_res = userobj
                AND object2set.obj_res = objectobj
            GROUP BY _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- select objects with credentials for user and role with specified otype and subtype (roleobj, objtype and objsubtype can be specified as NULL)
CREATE OR REPLACE FUNCTION getObjects(userobj integer, roleobj integer, objtype char(1), objsubtype char(1)) RETURNS SETOF object_list_with_credentials AS $$
DECLARE
   result object_list_with_credentials;
BEGIN
    FOR result in
        SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected, concatFields(set2role.permission) AS permission, concatFields(permission_type.credentials || set2role.special_credentials) as credentials
            FROM _objs, permission_type, _links as set2role, _links as user2role, _links as object2set
            WHERE
                (objtype IS NULL OR _objs.otype = objtype)
                AND (objsubtype IS NULL OR _objs.subtype = objsubtype)
                AND _objs.deleted = 0
                AND set2role.link_type = 'S2R'
                AND user2role.link_type = 'U2R'
                AND object2set.link_type LIKE '_2S'
                AND _objs.obj = object2set.obj_res
                AND set2role.obj_cons = user2role.obj_cons
                AND object2set.obj_cons = set2role.obj_res
                AND set2role.permission = permission_type.permission
                AND user2role.obj_res = userobj
                AND (roleobj IS NULL OR user2role.obj_cons = roleobj)
            GROUP BY _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- select objects from set with credentials for user and role with specified otype and subtype (roleobj, objtype and objsubtype can be specified as NULL)
CREATE OR REPLACE FUNCTION getObjects(userobj integer, roleobj integer, setobj integer, objtype char(1), objsubtype char(1)) RETURNS SETOF object_list_with_credentials AS $$
DECLARE
   result object_list_with_credentials;
BEGIN
    FOR result in
        SELECT object.obj, object.udid, object.name, object.description, object.location, object.otype, object.subtype, object.protected, object.permission, object.credentials
            FROM (
                SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected, concatFields(set2role.permission) AS permission, concatFields(permission_type.credentials || set2role.special_credentials) as credentials
                    FROM _objs, permission_type, _links as set2role, _links as user2role, _links as object2set
                    WHERE
                        (objtype IS NULL OR _objs.otype = objtype)
                        AND (objsubtype IS NULL OR _objs.subtype = objsubtype)
                        AND _objs.deleted = 0
                        AND set2role.link_type = 'S2R'
                        AND user2role.link_type = 'U2R'
                        AND object2set.link_type LIKE '_2S'
                        AND _objs.obj = object2set.obj_res
                        AND set2role.obj_cons = user2role.obj_cons
                        AND object2set.obj_cons = set2role.obj_res
                        AND set2role.permission = permission_type.permission
                        AND user2role.obj_res = userobj
                        AND (roleobj IS NULL OR user2role.obj_cons = roleobj)
                    GROUP BY _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected
            ) AS object, _links as object2set
            WHERE
                object2set.link_type LIKE '_2S'
                AND object.obj = object2set.obj_res
                AND object2set.obj_cons = setobj
            ORDER BY object.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- select objects from set, without credentials with specified otype and subtype (otype and subtype can be specified as NULL)
CREATE OR REPLACE FUNCTION getObjectsFromSet(setobj integer, otype char(1), subtype char(1)) RETURNS SETOF object_list_with_credentials AS $$
DECLARE
   result object_list_with_credentials;
BEGIN
    FOR result in
        SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected
            FROM _objs, _links as object2set
            WHERE
                ($2 IS NULL OR _objs.otype = $2)
                AND ($3 IS NULL OR _objs.subtype = $3)
                AND _objs.deleted = 0
                AND object2set.link_type LIKE '_2S'
                AND object2set.obj_res = _objs.obj
                AND object2set.obj_cons = $1
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- select objects from role with specified otype and subtype (otype and subtype can be specified as NULL)
CREATE OR REPLACE FUNCTION getObjectsFromRole(roleobj integer, otype char(1), subtype char(1)) RETURNS SETOF object_list_without_credentials AS $$
DECLARE
   result object_list_without_credentials;
BEGIN
    FOR result in
        SELECT set2role.obj_res as setobj, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected
            FROM _objs, _links as set2role, _links as object2set
            WHERE
                ($2 IS NULL OR _objs.otype = $2)
                AND ($3 IS NULL OR _objs.subtype = $3)
                AND _objs.deleted = 0
                AND set2role.link_type = 'S2R'
                AND object2set.link_type LIKE '_2S'
                AND object2set.obj_res = _objs.obj
                AND set2role.obj_res = object2set.obj_cons
                AND set2role.obj_cons = $1
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- select all sets with object
CREATE OR REPLACE FUNCTION getSets(objectobj integer) RETURNS SETOF object_list AS $$
DECLARE
   result object_list;
BEGIN
    FOR result in
        SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.protected
            FROM _objs, _links as object2set
            WHERE
                _objs.deleted = 0
                AND _objs.obj NOT IN (SELECT defaultsetobj FROM _obj_otype_subtype WHERE _obj_otype_subtype.disabled = 1)
                AND _objs.otype = 'S'
                AND object2set.obj_cons = _objs.obj
                AND object2set.obj_res = objectobj
                AND object2set.link_type LIKE '_2S'
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- select sets for user, role and object (role and object can be null)
CREATE OR REPLACE FUNCTION getSets(userobj integer, roleobj integer, objectobj integer) RETURNS SETOF object_list AS $$
DECLARE
   result object_list;
BEGIN
    FOR result in
        SELECT DISTINCT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.protected
            FROM _objs, _links as user2role, _links as set2role, _links as object2set
            WHERE
                _objs.deleted = 0
                AND _objs.obj NOT IN (SELECT defaultsetobj FROM _obj_otype_subtype WHERE _obj_otype_subtype.disabled = 1)
                AND user2role.link_type = 'U2R'
                AND set2role.link_type = 'S2R'
                AND _objs.obj = set2role.obj_res
                AND _objs.otype = 'S'
                AND user2role.obj_res = userobj
                AND (roleobj IS NULL OR user2role.obj_cons = roleobj)
                AND set2role.obj_cons = user2role.obj_cons
                AND (
                    objectobj IS NULL OR (
                        object2set.obj_res = objectobj
                        AND object2set.obj_cons = set2role.obj_res
                        AND object2set.link_type LIKE '_2S'
                    )
                )
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';
